/**CFile***********************************************************************

  FileName    [satResult.c]

  PackageName [sat]

  Synopsis    [Sat-related result structure]

  Description [The generic sat result type is defined in this module.]

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``sat'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/


#include "sat.h"
#include "satInt.h" /* for 'options' */


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Private definition for the Sat result type.]

  Description [To use this structure you must handle a Sat_Result_ptr type. 
  A Sat_Result instance can be converted into a trace by using the 
  Bmc_Trace structure.]

  SeeAlso     [Sat_SatResult_ptr,  Bmc_Trace]

******************************************************************************/
typedef struct Sat_SatResult_TAG {
  lsList propModel;  /* the propositional model */ 
  sat_result result; /* the sat solver succedeed or not */
} Sat_SatResult;


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Declarations of internal functions                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Sat_Result constructor]

  Description []

  SideEffects []

  SeeAlso     [Sat_SatResult_Delete]

******************************************************************************/
Sat_SatResult_ptr 
Sat_SatResult_Create()
{
  Sat_SatResult_ptr self = ALLOC(Sat_SatResult, 1);
  nusmv_assert(self != NULL);

  self->propModel = lsCreate();
  self->result = SAT_UNAVAILABLE;

  return self;
}

/**Function********************************************************************

  Synopsis    [Sat_Result destructor]

  Description [self pointer will be no longer usable after you called this 
  method.]

  SideEffects []

  SeeAlso     [Sat_SatResult_Create]

******************************************************************************/
void Sat_SatResult_Delete(Sat_SatResult_ptr* self_ref)
{
  lsDestroy((*self_ref)->propModel, NULL);
  FREE(*self_ref);
  *self_ref = NULL;
}


/**Function********************************************************************

  Synopsis    [Returns an empty list which can be used in order to 
  generate the propositional model]

  Description [This method is called by the Sat solver layer]

  SideEffects []

  SeeAlso     []

******************************************************************************/
lsList 
Sat_SatResult_GetListToCreatePropositionalModel(const Sat_SatResult_ptr self)
{
  nusmv_assert(self->propModel != NULL); /* Did you called Create before? */
  return self->propModel;
}


/**Function********************************************************************

  Synopsis    [Returns the propositional model generated by the sat solver]

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
lsList Sat_SatResult_GetPropModel(const Sat_SatResult_ptr self)
{
  nusmv_assert(self->propModel != NULL); /* Did you called Create before? */
  return self->propModel;
}


/**Function********************************************************************

  Synopsis    [Returns true if the result represents a satisfiable problem, 
  false otherwise]

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean Sat_SatResult_IsSatisfiable(const Sat_SatResult_ptr self)
{
  return (self->result == SAT_SATISFIABLE_PROBLEM)? true : false;
}


/**Function********************************************************************

  Synopsis    [Returns true if the result is unsatisfiable, false otherwise]

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean Sat_SatResult_IsUnsatisfiable(const Sat_SatResult_ptr self)
{
  return (self->result == SAT_UNSATISFIABLE_PROBLEM)? true : false;
}


/**Function********************************************************************

  Synopsis    [Returns true if the sat solver call had produced an internal 
  error, or was not available]

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean Sat_SatResult_IsError(const Sat_SatResult_ptr self)
{
  return ((self->result == SAT_INTERNAL_ERROR)? true : false)
    || Sat_SatResult_IsNotAvailable(self);
}


/**Function********************************************************************

  Synopsis    [Returns true if the required sat solver was not available 
  to produce a valid result]

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean  Sat_SatResult_IsNotAvailable(const Sat_SatResult_ptr self)
{
  return (self->result == SAT_UNAVAILABLE)? true : false;
}


/**Function********************************************************************

  Synopsis    [Marks the given result as satisfiable]

  Description []

  SideEffects []

  SeeAlso     [self will change]

******************************************************************************/
void Sat_SatResult_SetSatisfiable(Sat_SatResult_ptr self)
{
  self->result = SAT_SATISFIABLE_PROBLEM;
}


/**Function********************************************************************

  Synopsis    [Marks the given result as unsatisfiable]

  Description []

  SideEffects []

  SeeAlso     [self will change]

******************************************************************************/
void Sat_SatResult_SetUnsatisfiable(Sat_SatResult_ptr self)
{
  self->result = SAT_UNSATISFIABLE_PROBLEM;
}


/**Function********************************************************************

  Synopsis    [Marks the given result as internal error]

  Description []

  SideEffects []

  SeeAlso     [self will change]

******************************************************************************/
void Sat_SatResult_SetInternalError(Sat_SatResult_ptr self)
{
  self->result = SAT_INTERNAL_ERROR;
}


/**Function********************************************************************

  Synopsis    [Marks the given result as not available]

  Description []

  SideEffects []

  SeeAlso     [self will change]

******************************************************************************/
void Sat_SatResult_SetNotAvailable(Sat_SatResult_ptr self)
{
  self->result = SAT_UNAVAILABLE;
}


/**AutomaticEnd***************************************************************/
