/**CFile***********************************************************************

  FileName    [satPkg.c]

  PackageName [sat]

  Synopsis    [Sat.Pkg module]

  Description [This module contains all the sat package handling functions]

  SeeAlso     []

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``sat'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst and University of Trento. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "sat.h"
#include "satInt.h" /* for options */
#include "satUtils.h"

#include "opt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable*******************************************************************
  Synopsis           [This private variable keeps all the sat solver instances
  that are initially created by NuSMV]
  Description        []
  SideEffects        []
  SeeAlso            []
******************************************************************************/
#include "satSolver.h"
Sat_SatSolver_ptr solvers[] = {
  NULL
# ifdef SAT_ENABLE_ZCHAFF_SOLVER	       
  , NULL
# endif
};

/**Variable*******************************************************************
  Synopsis           [The Sim solver instance]
  Description        []
  SideEffects        []
  SeeAlso            []
******************************************************************************/
#include "satSimSolver.h"
static Sat_SimSolver_ptr sim = NULL;

/**Variable*******************************************************************
  Synopsis           [The ZChaff solver instance]
  Description        []
  SideEffects        []
  SeeAlso            []
******************************************************************************/
#ifdef SAT_ENABLE_ZCHAFF_SOLVER
# include "satZChaffSolver.h"
  static Sat_ZChaffSolver_ptr zchaff = NULL;
#endif


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the Sat package inner structures]

  Description        [Creates the Sim and ZChaff instances. The latter 
  is created only if explicitly required.]

  SideEffects        []

  SeeAlso            []
******************************************************************************/
void Sat_Init()
{
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Initializing the Sat package... \n");
  }

  sim = Sat_SimSolver_Create();
  solvers[Sat_Sim] = Sat_SimSolver_GetSatSolver(sim);
  nusmv_assert(solvers[Sat_Sim] != NULL);

# ifdef SAT_ENABLE_ZCHAFF_SOLVER
  zchaff = Sat_ZChaffSolver_Create();
  solvers[Sat_ZChaff] = Sat_ZChaffSolver_GetSatSolver(zchaff);
  nusmv_assert(solvers[Sat_ZChaff] != NULL);
# endif
}

/**Function********************************************************************

  Synopsis           [Frees all resources allocated for the Sat package]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Sat_Quit()
{
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Resetting the Sat module... \n");
  }

  Sat_SimSolver_Delete(&sim);
  solvers[Sat_Sim] = NULL;
# ifdef SAT_ENABLE_ZCHAFF_SOLVER
  Sat_ZChaffSolver_Delete(&zchaff);
  solvers[Sat_ZChaff] = NULL;
# endif
}


/**Function********************************************************************

  Synopsis           [Returns the currently selected sat solver instance]

  Description        [type is the 'symbolic' name of the solver you required
  an instance]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
Sat_SatSolver_ptr Sat_GetSelectedSolver(Sat_SatSolver_Type type)
{
  int curr_idx = (int) type;
  nusmv_assert((curr_idx >= 0) && (curr_idx < GET_ARRAY_LENGTH(solvers)));
  nusmv_assert(solvers[curr_idx] != NULL);
  return solvers[curr_idx];
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**AutomaticEnd***************************************************************/

