/**CFile***********************************************************************

  FileName    [rbcFormula.c]

  PackageName [rbc]

  Synopsis    [Formula constructors.]

  Description [External functions included in this module:
		<ul>
		<li> <b>Rbc_GetOne()</b>        logical truth
		<li> <b>Rbc_GetZero()</b>       logical falsity
		<li> <b>Rbc_GetIthVar</b>       variables
		<li> <b>Rbc_MakeNot()</b>       negation
		<li> <b>Rbc_MakeAnd()</b>       conjunction
		<li> <b>Rbc_MakeOr()</b>        disjunction
		<li> <b>Rbc_MakeIff()</b>       coimplication
		<li> <b>Rbc_MakeXor()</b>       exclusive disjunction
		<li> <b>Rbc_MakeIte()</b>       if-then-else
		<li> <b>Rbc_GetLeftOpnd()</b>   return left operand
		<li> <b>Rbc_GetRightOpnd()</b>  return right operand
		<li> <b>Rbc_GetVarIndex()</b>   return the variable index 
		<li> <b>Rbc_Mark()</b>          make a vertex permanent
		<li> <b>Rbc_Unmark()</b>        make a vertex volatile
		</ul>]
		
  SeeAlso     []

  Author      [Armando Tacchella]

  Copyright   [
  This file is part of the ``rbc'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by University of Genova. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

  Revision    [v. 1.0]

******************************************************************************/

#include "rbcInt.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static Rbc_t * Reduce(Rbc_Manager_t * rbcManager, int op, Rbc_t * left, Rbc_t * right);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of external functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Logical constant 1 (truth).]

  Description [Returns the rbc that stands for logical truth.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_GetOne(
  Rbc_Manager_t * rbcManager)
{
  
  return (rbcManager -> one);

} /* End of Rbc_GetOne. */


/**Function********************************************************************

  Synopsis    [Logical constant 0 (falsity).]

  Description [Returns the rbc that stands for logical falsity.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_GetZero(
  Rbc_Manager_t * rbcManager)
{
  
  return (rbcManager -> zero);

} /* End of Rbc_GetZero. */


/**Function********************************************************************

  Synopsis    [Returns a variable.]

  Description [Returns a pointer to an rbc node containing the requested 
               variable. Works in three steps:
	       <ul>
	       <li> the requested variable index exceeds the current capacity:
	            allocated more room up to the requested index;
	       <li> the variable node does not exists: inserts it in the dag
	            and makes it permanent;
	       <li> returns the variable node.
	       </ul>]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_GetIthVar(
  Rbc_Manager_t * rbcManager,
  int             varIndex)
{
  int     i;

  /* Allocate more room for the varTable if needed. */
  if (rbcManager -> varCapacity <= varIndex) {
    rbcManager -> varTable = 
      REALLOC(Rbc_t*, rbcManager -> varTable, varIndex + 1);
    for (i = rbcManager -> varCapacity; i < varIndex + 1; i ++) {
      rbcManager -> varTable[i] = NIL(Rbc_t);
    }
    rbcManager -> varCapacity = varIndex + 1;
  }

  /* Create the variable if needed. */
  if (rbcManager -> varTable[varIndex] == NIL(Rbc_t)) {
    rbcManager -> varTable[varIndex] = 
      Dag_VertexInsert(rbcManager -> dagManager, 
		       RBCVAR, (char*)varIndex, (lsList)NULL);
    /* Make the node permanent. */
    Dag_VertexMark(rbcManager -> varTable[varIndex]);
    ++(rbcManager -> stats[RBCVAR_NO]);
  }

  /* Return the variable as rbc node. */
  return rbcManager -> varTable[varIndex];

} /* End of Rbc_GetIthVar. */


/**Function********************************************************************

  Synopsis    [Returns the complement of an rbc.]

  Description [Returns the complement of an rbc.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_MakeNot(
  Rbc_Manager_t * rbcManager,            
  Rbc_t         * left)
{

  return RbcId(left, RBC_FALSE);

} /* End of Rbc_MakeNot. */


/**Function********************************************************************

  Synopsis    [Makes the conjunction of two rbcs.]

  Description [Makes the conjunction of two rbcs.
               Works in three steps:
	       <ul>
	       <li> performs boolean simplification: if successfull, returns
	            the result of the simplification;
	       <li> orders left and right son pointers;
	       <li> looks up the formula in the dag and returns it.
	       </ul>]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_MakeAnd(
  Rbc_Manager_t * rbcManager,
  Rbc_t         * left,
  Rbc_t         * right,
  Rbc_Bool_c      sign)
{
  Rbc_t * rTemp;
  lsList  sons;

  /* First, perform the reduction stage. */
  if ((rTemp = Reduce(rbcManager, RBCAND, left, right)) != NIL(Rbc_t)) {
    return RbcId(rTemp, sign);
  }
  
  /* Order the vertices. */
  if (right < left) {
    rTemp = right; right = left; left = rTemp;
  }

  /* Create the list of sons. */
  sons = lsCreate();
  (void) lsNewEnd(sons, (lsGeneric) left, LS_NH);
  (void) lsNewEnd(sons, (lsGeneric) right, LS_NH);

  /* Lookup the formula in the dag. */
  rTemp =  Dag_VertexLookup(rbcManager -> dagManager, 
			     RBCAND, NIL(char), sons); 

  return RbcId(rTemp, sign);

} /* End of Rbc_MakeAnd. */


/**Function********************************************************************

  Synopsis    [Makes the disjunction of two rbcs.]

  Description [Makes the disjunction of two rbcs: casts the connective to
               the negation of a conjunction using De Morgan's law.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_MakeOr(
  Rbc_Manager_t * rbcManager,
  Rbc_t         * left,
  Rbc_t         * right,
  Rbc_Bool_c      sign)
{
  /* Use De Morgan's law. */
  return Rbc_MakeAnd(rbcManager,
		     RbcId(left, RBC_FALSE),
		     RbcId(right, RBC_FALSE),
		     (Rbc_Bool_c)((int)sign ^ (int)RBC_FALSE));

} /* End of Rbc_MakeOr. */


/**Function********************************************************************

  Synopsis    [Makes the coimplication of two rbcs.]

  Description [Makes the coimplication of two rbcs.
               Works in four steps:
	       <ul>
	       <li> performs boolean simplification: if successfull, returns
	            the result of the simplification;
	       <li> orders left and right son pointers;
	       <li> re-encodes the negation 
	       <li> looks up the formula in the dag and returns it.
	       </ul>]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_MakeIff(
  Rbc_Manager_t * rbcManager,
  Rbc_t         * left,
  Rbc_t         * right,
  Rbc_Bool_c      sign)
{
  Rbc_t * rTemp;
  lsList  sons;

  /* First, perform the reduction stage. */
  if ((rTemp = Reduce(rbcManager, RBCIFF, left, right)) != NIL(Rbc_t)) {
    return RbcId(rTemp, sign);
  }
  
  /* Order the  vertices. */
  if (right < left) {
    rTemp = right; right = left; left = rTemp;
  }

  /* Negation always on top. */
  sign = sign ^ Dag_VertexIsSet(left) ^ Dag_VertexIsSet(right);
  Dag_VertexClear(left);
  Dag_VertexClear(right);

  /* Create the list of sons. */
  sons = lsCreate();
  (void) lsNewEnd(sons, (lsGeneric) left, LS_NH);
  (void) lsNewEnd(sons, (lsGeneric) right, LS_NH);

  /* Lookup the formula in the dag. */
  rTemp =  Dag_VertexLookup(rbcManager -> dagManager, 
			     RBCIFF, NIL(char), sons); 

  return RbcId(rTemp, sign);

} /* End of Rbc_MakeIff. */


/**Function********************************************************************

  Synopsis    [Makes the exclusive disjunction of two rbcs.]

  Description [Makes the exclusive disjunction of two rbcs: casts the
               connective as the negation of a coimplication.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_MakeXor(
  Rbc_Manager_t * rbcManager,
  Rbc_t         * left,
  Rbc_t         * right,
  Rbc_Bool_c      sign)
{
  /* Simply a negation of a coimplication. */
  return Rbc_MakeIff(rbcManager, left, right,
		     (Rbc_Bool_c)((int)sign ^ 
				  (int)RBC_FALSE));

} /* End of Rbc_MakeXor. */


/**Function********************************************************************

  Synopsis    [Makes the if-then-else of three rbcs.]

  Description [Makes the if-then-else of three rbcs: expands the connective
              into the corresponding product-of-sums.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t * 
Rbc_MakeIte(
  Rbc_Manager_t * rbcManager,
  Rbc_t         * c,
  Rbc_t         * t,
  Rbc_t         * e,
  Rbc_Bool_c      sign)
{

  /* (if c then t else e) becomes (-c or t) and (c or e). */
  return Rbc_MakeAnd(rbcManager,
                     Rbc_MakeAnd(rbcManager, 
				 c, RbcId(t, RBC_FALSE), RBC_FALSE),
                     Rbc_MakeAnd(rbcManager, 
				 RbcId(c, RBC_FALSE), 
				 RbcId(e, RBC_FALSE), 
				 RBC_FALSE),
		     sign);

} /* End of Rbc_MakeIte. */


/**Function********************************************************************

  Synopsis    [Gets the left operand.]

  Description [Gets the left operand.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t *
Rbc_GetLeftOpnd(
  Rbc_t * f)
{

  if (Dag_VertexGetRef(f) -> outList != (lsList)NULL) {
    /* Reusing f to avoid introduction of new variables. */
    (void) lsFirstItem(Dag_VertexGetRef(f) -> outList, (lsGeneric*) &f, LS_NH);
  }
  return f;

} /* End of Rbc_GetLeftOpnd. */


/**Function********************************************************************

  Synopsis    [Gets the right operand.]

  Description [Gets the right operand.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
Rbc_t *
Rbc_GetRightOpnd(
  Rbc_t * f)
{

  if (Dag_VertexGetRef(f) -> outList != (lsList)NULL) {
    /* Reusing f to avoid introduction of new variables. */
    (void) lsLastItem(Dag_VertexGetRef(f) -> outList, (lsGeneric*) &f, LS_NH);
  }
  return f;

} /* End of Rbc_GeRightOpnd. */


/**Function********************************************************************

  Synopsis    [Gets the variable index.]

  Description [Returns the variable index, 
               -1 if the rbc is not a variable.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
int
Rbc_GetVarIndex(
  Rbc_t * f)
{

  if (Dag_VertexGetRef(f) -> symbol == RBCVAR) {
    return (int)(Dag_VertexGetRef(f) -> data);
  }
  return -1;

} /* End of Rbc_GetVarIndex. */


/**Function********************************************************************

  Synopsis    [Makes a node permanent.]

  Description [Marks the vertex in the internal dag. This saves the rbc
               from being wiped out during garbage collection.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
void 
Rbc_Mark(
  Rbc_Manager_t * rbc,         
  Rbc_t        * f)
{
  
  /* To avoid calling another function, do it directly! */
  ++(Dag_VertexGetRef(f) -> mark);
  return;

} /* End of Rbc_Mark. */


/**Function********************************************************************

  Synopsis    [Makes a node volatile.]

  Description [Unmarks the vertex in the internal dag. This exposes the rbc
               to garbage collection.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
void  
Rbc_Unmark(
  Rbc_Manager_t * rbc,         
  Rbc_t * f)
{

  if (Dag_VertexGetRef(f) -> mark > 0) {
    --(Dag_VertexGetRef(f) -> mark);
  }
  return;

} /* End of Rbc_Unmark. */


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Reduction (simplification) of rbcs.]

  Description [Reduction (simplification) of rbcs.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
static Rbc_t *
Reduce(
 Rbc_Manager_t * rbcManager,         
 int             op,
 Rbc_t         * left,
 Rbc_t         * right)
{

  switch (op) {
  case RBCAND:
    if (left == right) {
      return left;
    } else if (left == RbcId(right, RBC_FALSE)) {
      return rbcManager -> zero;
    } else if ((left == rbcManager -> zero) || 
	       (right == rbcManager -> zero)) {
      return rbcManager -> zero;
    } else if (left == rbcManager -> one) {
      return right;
    } else if (right == rbcManager -> one) {
      return left;
    } else {
      return NIL(Rbc_t);
    }
  case RBCIFF:
    if (left == right) {
      return rbcManager -> one;
    } else if (left == RbcId(right, RBC_FALSE)) {
      return rbcManager -> zero;
    } else if (left == rbcManager -> zero) {
      return RbcId(right, RBC_FALSE);
    } else if (right == rbcManager -> zero) {
      return RbcId(left, RBC_FALSE);
    } else if (left == rbcManager -> one) {
      return right;
    } else if (right == rbcManager -> one) {
      return left;
    } else {
      return NIL(Rbc_t);
    }
  }

  return NIL(Rbc_t);

} /* End of Reduce. */
