/**CFile***********************************************************************

  FileName    [rbcCnf.c]

  PackageName [rbc]

  Synopsis    [Conjunctive Normal Form (CNF) conversions.]

  Description [External functions included in this module:
		<ul>
		<li> <b>Rbc_Convert2Cnf()</b>  
		</ul>]
		
  SeeAlso     []

  Author      [Armando Tacchella]

  Copyright   [
  This file is part of the ``rbc'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by University of Genova. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

  Revision    [v. 1.0]

******************************************************************************/

#include "rbcInt.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************
  Synopsis      [Data passing in cnf-DFS.]
  Description   [Data passing in cnf-DFS.]
  SeeAlso       []
******************************************************************************/
struct CnfDfsData {
  Rbc_Manager_t * rbcManager;
  int             maxVar;
  lsList          clauses;
  lsList          vars;
  int             result;
};


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

typedef struct CnfDfsData CnfDfsData_t;


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int CnfSet(Rbc_t * f, char * cnfData, int sign);
static void CnfFirst(Rbc_t * f, char * cnfData, int sign);
static void CnfBack(Rbc_t * f, char * cnfData, int sign);
static void CnfLast(Rbc_t * f, char * cnfData, int sign);
static lsGeneric SwapSign(lsGeneric data);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of external functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Translates the rbc into the corresponding (equisatisfiable)
               set of clauses.]

  Description [Given `rbcManager' and `f', `clauses' is filled with the
               disjunctions corresponding to the rbc nodes according to 
	       the rules:
	       
	       f = A & B => -f A              f = A <-> B =>  f  A  B
	                    -f B                              f -A -B
			     f -A -B                         -f -A  B
                                                             -f  A -B

	       `vars' is filled with the variables that occurred in `f' 
	       (original or model variables). It is user's responsibility 
	       to create `clauses' and `vars' *before* calling the function.
               New variables are added by the conversion: the maximum
	       index (the last added variable) is returned by the function.
	       The function returns 0 when `f' is true or false.]
							     
  SideEffects [`clauses' and `vars' are filled up. `clauses' is the empty
               list if `f' was true, and contains a single empty clause if
	       `f' was false.]

  SeeAlso     []

******************************************************************************/
int
Rbc_Convert2Cnf(
  Rbc_Manager_t * rbcManager,
  Rbc_t         * f,
  lsList          clauses,
  lsList          vars)
{
  int                i, maxVar;
  lsList             fClause;
  Dag_DfsFunctions_t cnfFunctions;
  CnfDfsData_t       cnfData;

  /* Handling special cases: f is the constant true or false. */
  if (f == Rbc_GetOne(rbcManager)) {
    return 0;
  }
  if (f == Rbc_GetZero(rbcManager)) {
    fClause = lsCreate();
    (void) lsNewBegin(clauses, (lsGeneric) fClause, LS_NH);
    return 0;
  }

  /* Determine the current maximum variable index. */
  maxVar = 0;
  for (i = 0; i < rbcManager -> varCapacity; i++) {
    if (rbcManager -> varTable[i] != NIL(Rbc_t)) {
      maxVar = i;
    }
  }

  /* Cleaning the user fields. */
  Dag_Dfs(f, &dag_DfsClean, NIL(char));

  /* Setting up the DFS functions. */
  cnfFunctions.Set        = CnfSet;
  cnfFunctions.FirstVisit = CnfFirst;
  cnfFunctions.BackVisit  = CnfBack;
  cnfFunctions.LastVisit  = CnfLast;
 
  /* Setting up the DFS data. */
  cnfData.rbcManager = rbcManager; 
  cnfData.maxVar      = maxVar + 1;
  cnfData.clauses     = clauses;
  cnfData.vars        = vars;
  cnfData.result      = 0;

  /* Calling DFS on f. */
  Dag_Dfs(f, &cnfFunctions, (char*)(&cnfData));

  /* Add the unit clause that stands for f. */
  fClause = lsCreate();
  (void) lsNewEnd(fClause, (lsGeneric) cnfData.result, LS_NH);
  (void) lsNewBegin(clauses, (lsGeneric) fClause, LS_NH);

  return (cnfData.maxVar - 1);
  
} /* End of Rbc_Convert2Cnf. */


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Dfs Set for CNF conversion.]

  Description [Dfs Set for CNF conversion.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
static int
CnfSet(
 Rbc_t  * f,
 char   * cnfData,
 int      sign)
{
  CnfDfsData_t * cd = (CnfDfsData_t*)cnfData;

  /* Set the current integer reference as result. */
  cd -> result = (sign != 0 ? -1 * (f -> iRef) : f -> iRef);

  /* All nodes should be visited once and only once. */
  return (0);

} /* End of CnfSet. */


/**Function********************************************************************

  Synopsis    [Dfs FirstVisit for CNF conversion.]

  Description [Dfs FirstVisit for CNF conversion.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
static void
CnfFirst(
 Rbc_t  * f,
 char   * cnfData,
 int      sign)
{
  CnfDfsData_t * cd = (CnfDfsData_t*)cnfData;

  if (f -> symbol != RBCVAR) { 
    /* Create a temporary list (use vertex own general reference). */
    f -> gRef = (char*) lsCreate();
  } else {
    /* Fill in vars list. */
    (void) lsNewEnd(cd -> vars, (lsGeneric)(f -> data), LS_NH);
  }

  return;

} /* End of CnfFirst. */


/**Function********************************************************************

  Synopsis    [Dfs BackVisit for CNF conversion.]

  Description [Dfs BackVisit for CNF conversion.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
static void
CnfBack(
 Rbc_t  * f,
 char   * cnfData,
 int      sign)
{
  CnfDfsData_t * cd   = (CnfDfsData_t*)cnfData;

  /* Get the current result and add it (negated) to the temporary list. */
  (void) lsNewEnd((lsList)(f -> gRef), 
		  (lsGeneric)( -1 * (cd -> result)), 
		  LS_NH);

  return;

} /* End of CnfBack. */


/**Function********************************************************************

  Synopsis    [Dfs LastVisit for CNF conversion.]

  Description [Dfs LastVisit for CNF conversion.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
static void
CnfLast(
 Rbc_t  * f,
 char   * cnfData,
 int      sign)
{
  lsList    tmpCl;
  lsGen     gen;
  int       newVar, s, pol;

  CnfDfsData_t * cd   = (CnfDfsData_t*)cnfData;
  lsList         sons = (lsList)(f -> gRef);

  if (f -> symbol == RBCVAR) {

    /* Leave the variable index in iRef. */
    f -> iRef = (int)(f -> data);

  } else {
    
    /* Generate a new variable and leave it in iRef. */
    newVar = (cd -> maxVar)++;
    f -> iRef = newVar;

    /* Generate and append clauses. */
    if (f -> symbol == RBCAND) {

      /* Add the binary clauses {-f s_i} */
      gen = lsStart(sons);
      while (lsNext(gen, (lsGeneric*) &s, LS_NH) == LS_OK) {
	tmpCl = lsCreate();
	(void) lsNewEnd(tmpCl, (lsGeneric)(-1 * newVar), LS_NH);
	(void) lsNewEnd(tmpCl, (lsGeneric)(-1 * s), LS_NH);
	(void) lsNewBegin(cd -> clauses, (lsGeneric)tmpCl, LS_NH);
      }
      lsFinish(gen);
      
      /* Add the clause {f -s_1 -s_2} */
      (void) lsNewBegin(sons, (lsGeneric)newVar, LS_NH);
      (void) lsNewBegin(cd -> clauses, (lsGeneric)sons, LS_NH);

    } else {

      /* Add the clause {-f s_1 -s_2} */
      tmpCl = lsCreate();
      (void) lsNewEnd(tmpCl, (lsGeneric)(-1 * newVar), LS_NH);
      gen = lsStart(sons);
      pol = -1;
      while (lsNext(gen, (lsGeneric*) &s, LS_NH) == LS_OK) {
	(void) lsNewEnd(tmpCl, (lsGeneric)(s * pol), LS_NH);
	pol *= -1;
      }
      (void) lsNewBegin(cd -> clauses, (lsGeneric)tmpCl, LS_NH);

      /* Add the clause {-f -s_1 s_2} */
      tmpCl = lsCreate();
      (void) lsNewEnd(tmpCl, (lsGeneric)(-1 * newVar), LS_NH);
      gen = lsStart(sons);
      pol = 1;
      while (lsNext(gen, (lsGeneric*) &s, LS_NH) == LS_OK) {
	(void) lsNewEnd(tmpCl, (lsGeneric)(s * pol), LS_NH);
	pol *= -1;
      }
      (void) lsNewBegin(cd -> clauses, (lsGeneric)tmpCl, LS_NH);

      /* Add the clause {f s_1 s_2} */
      tmpCl = lsCopy(sons, SwapSign);
      (void) lsNewBegin(tmpCl, (lsGeneric)newVar, LS_NH);
      (void) lsNewBegin(cd -> clauses, (lsGeneric)tmpCl, LS_NH);

      /* Add the clause {f -s_1 -s_2} */
      (void) lsNewBegin(sons, (lsGeneric)newVar, LS_NH);
      (void) lsNewBegin(cd -> clauses, (lsGeneric)sons, LS_NH);

    }

  }
  
  /* Adjust the sign of the result. */
  cd -> result = (sign != 0 ? -1 * (f -> iRef) : f -> iRef);

  return;

} /* End of CnfLast. */


/**Function********************************************************************

  Synopsis    [Swaps the sign of the argument.]

  Description [Swaps the sign of the argument.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
static lsGeneric
SwapSign(
  lsGeneric data)	    
{
 
  return (lsGeneric)(-1 * (int)data);

} /* End of CnfSwapSign. */
    
	       



