/**CFile***********************************************************************

  FileName    [propVerify.c]

  PackageName [prop]

  Synopsis    [Interface to model checking routines]

  Description [This file provides an interface to the model checking
  routines. In particular functions calling the model checking
  routines for each property are provided.]

  SeeAlso     [mcMc.c mcInvar.c mcAGonly.c ltl.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``prop'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "propInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: propVerify.c,v 1.1.1.1 2003/02/06 19:01:21 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Verifies a given property]

  Description        [Depending the property, different model checking
  algorithms are called. The status of the property is updated
  accordingly to the result of the verification process.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_Db_VerifyProp(Prop_Ptr prop)
{
  if (Prop_GetStatus(prop) != Prop_Wrong) {
    if (Prop_GetStatus(prop) == Prop_Unchecked)  {
      switch(Prop_GetType(prop)){
      case Prop_Ctl: 
        {
          if(!(opt_ag_only(options) && opt_forward_search(options))) {
            Mc_CheckCTLSpec(prop);
          }
          else {
            Mc_CheckAGOnlySpec(prop);
          }
          break;
        }
      case Prop_Compute:
        {
          Mc_CheckCompute(prop);
          break;
        }
      case Prop_Invar:
        {
          Mc_CheckInvar(prop);
          break;
        }
      case Prop_Ltl:
        {
          Ltl_CheckLtlSpec(prop);
          break;
        }
      default:
        {
          fprintf(nusmv_stderr, "Unknown property type.\n");
          break;
        }
      }
    }
  }
}

/**Function********************************************************************

  Synopsis           [Verifies a given property]

  Description        [The DB of properties is searched for a property
  whose unique identifier match the identifier provided and then if
  such a property exists it will be verified calling the appropriate
  model checking algorithm. If the property was checked before, then
  the property is not checked again.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_Db_VerifyPropIndex(int prop)
{
  int s = Prop_Db_GetSize();

  if (s < prop){
    fprintf(nusmv_stderr, 
	    "Property indexed by %d not present in the database.\n", prop);
    fprintf(nusmv_stderr, 
	    "Valid index are in the range [0..%d]\n", s-1);
    nusmv_exit(1);
  }
  else {
    Prop_Ptr p = Prop_Db_GetNum(prop);
    Prop_Db_VerifyProp(p);
  }
}


/**Function********************************************************************

  Synopsis           [Verifies all properties of a given type]

  Description        [The DB of properties is searched for a property
  of the given type. All the found properties are then verified
  calling the appropriate model checking algorithm. Properties already
  checked will be ignored.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_DbVerifyAllType(Prop_Type type)
{
  int i;
  int s = Prop_Db_GetSize();

  for(i=0; i < s; i++) {
    Prop_Ptr p = Prop_Db_GetNum(i);

    if (Prop_GetType(p) == type) {
      Prop_Db_VerifyProp(p);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Verifies all CTL properties]

  Description        [The DB of properties is searched for a property
  of type CTL, and all the unchecked CTL properties found will be
  submitted to model checking.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_DbVerifyAllTypeCtl() 
{
  Prop_DbVerifyAllType(Prop_Ctl);
}

/**Function********************************************************************

  Synopsis           [Verifies all COMPUTE properties]

  Description        [The DB of properties is searched for a property
  of type COMPUTE, and all the unchecked COMPUTE properties found will be
  submitted to model checking.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_DbVerifyAllTypeCompute() 
{
  Prop_DbVerifyAllType(Prop_Compute);
}

/**Function********************************************************************

  Synopsis           [Verifies all LTL properties]

  Description        [The DB of properties is searched for a property
  of type LTL, and all the unchecked LTL properties found will be
  submitted to model checking.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_DbVerifyAllTypeLtl() 
{
  Prop_DbVerifyAllType(Prop_Ltl);
}

/**Function********************************************************************

  Synopsis           [Verifies all INVAR properties]

  Description        [The DB of properties is searched for a property
  of type INVAR, and all the unchecked INVAR properties found will be
  submitted to model checking.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_DbVerifyAllTypeInvar() 
{
  Prop_DbVerifyAllType(Prop_Invar);
}

/**Function********************************************************************

  Synopsis           [Verifies all the properties in the DB]

  Description        [All the properties stored in the database not
  yet verified will be verified. The properties are verified following
  the order CTL/COMPUTE/LTL/INVAR.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_Db_VerifyAll()
{
  Prop_DbVerifyAllType(Prop_Ctl);
  Prop_DbVerifyAllType(Prop_Compute);
  Prop_DbVerifyAllType(Prop_Ltl);
  Prop_DbVerifyAllType(Prop_Invar);
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
