/**CFile***********************************************************************

  FileName    [propProp.c]

  PackageName [prop]

  Synopsis    [Main routines for the prop data structure]

  Description [Main routines for the manipulation of the prop data
  structure. A "master" property is also defined to be used to
  represent the whole system, for instance to perform reachability or
  to perform simulation. Moreover a primitives to create, query and
  manipulate a database of property is provided.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``prop'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "propInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: propProp.c,v 1.1.1.1 2003/02/06 19:01:21 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
static array_t * Prop_Db_Array = (array_t *)NULL;
static Prop_Ptr Prop_Master = (Prop_Ptr)NULL;

EXTERN DdManager* dd_manager;

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/
/*
  Used to encode that a property MIN/MAX has not yet been checked.
*/
#define PROP_UNCHECKED -2

/*
  Used to encode the infinite distanca between two set of states in MIN/MAX properties
*/
#define PROP_INFINITE -1

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void print_prop ARGS((FILE *, node_ptr));
static Prop_Ptr Prop_CreatePropPartial ARGS((int, node_ptr, Prop_Type));
static char * PropTypeGetParsingType ARGS((Prop_Type));

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Allocate a property]

  Description        [Allocate a property. If no more room is
  available then a call to <tt>numsv_exit</tt> is performed. All the
  fields of the prop structure are initialized to either NULL or the
  corresponding default type (e.g. Prop_NoType for property type).]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
Prop_Ptr Prop_Alloc() {
  Prop_Ptr p = ALLOC(Prop_Rec, 1);

  if (p == NIL(Prop_Rec)) {
    fprintf(nusmv_stderr, "Prop_Alloc: Unable to allocate memory.\n");
    nusmv_exit(1);
    return NIL(Prop_Rec);
  }
  p->index = 0;
  p->prop = (node_ptr)NULL;
  p->cone = (node_ptr)NULL;
  p->type = Prop_NoType;
  p->status = Prop_NoStatus;
  p->number = PROP_UNCHECKED;
  p->trace = 0;
  p->scalar_fsm = (Fsm_SexpPtr)NULL;
  p->bool_fsm = (Fsm_SexpPtr)NULL;
  p->bdd_fsm = (Fsm_BddPtr)NULL;
  p->be_fsm = (Bmc_Fsm_ptr)NULL;
  return p;
}

/**Function********************************************************************

  Synopsis           [Destroy the elements of a property]

  Description        [Destroy the elements of a property]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_Destroy(Prop_Ptr p) {
  if (p->scalar_fsm != (Fsm_SexpPtr)NULL)
    Compile_FsmSexpFree(p->scalar_fsm);
  if (p->bool_fsm != (Fsm_SexpPtr)NULL)
    Compile_FsmSexpFree(p->bool_fsm);
  if (p->bdd_fsm != (Fsm_BddPtr)NULL) 
    Compile_FsmBddFree(p->bdd_fsm);
  if (p->be_fsm != NULL) Bmc_Fsm_Delete(&(p->be_fsm));
}

/**Function********************************************************************

  Synopsis           [Free a property]

  Description        [Free a property. Notice that before freeing the
  property all the elements of the property that needs to be freed
  will be automatically freed.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_Free(Prop_Ptr n) {
  /* Consider the possibility to free all the elements of the
    structure appropriately */
  nusmv_assert(n != NIL(Prop_Rec));
  Prop_Destroy(n);
  FREE(n);
}

/**Function********************************************************************

  Synopsis           [Returns the index of a property]

  Description        [Returns the unique identifier of a property. It
  is used in the database of property to identify properties.]

  SideEffects        []

******************************************************************************/
int Prop_GetIndex(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->index;
}

/**Function********************************************************************

  Synopsis           [Sets the index of a property]

  Description        [Sets the unique identifier of a property]

  SideEffects        []

******************************************************************************/
void Prop_SetIndex(Prop_Ptr p, int index)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->index = index;
}

/**Function********************************************************************

  Synopsis           [Returns the property]

  Description        [Returns the property stored in the prop]

  SideEffects        []

******************************************************************************/
node_ptr Prop_GetProp(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->prop;
}

/**Function********************************************************************

  Synopsis           [Sets the property]

  Description        [Stores the property in the prop structure.]

  SideEffects        []

******************************************************************************/
void Prop_SetProp(Prop_Ptr p, node_ptr e)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->prop = e;
}

/**Function********************************************************************

  Synopsis           [Returns the cone of a property]

  Description        [If the cone of influence of a property has been
  computed, this function returns it.]

  SideEffects        []

******************************************************************************/
Set_t Prop_GetCone(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->cone;
}

/**Function********************************************************************

  Synopsis           [Sets the cone of a property]

  Description        [Stores the cone of influence of the property]

  SideEffects        []

******************************************************************************/
void Prop_SetCone(Prop_Ptr p, Set_t c)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->cone = c;
}

/**Function********************************************************************

  Synopsis           [Returns the property type]

  Description        [Returns the property kind of the stroed
  property, i.e. CTL, LTL, ...]

  SideEffects        []

******************************************************************************/
Prop_Type Prop_GetType(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->type ;
}

/**Function********************************************************************

  Synopsis           [Sets the property type.]

  Description        [Stores the property type.]

  SideEffects        []

******************************************************************************/
void Prop_SetType(Prop_Ptr p, Prop_Type t)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->type = t;
}

/**Function********************************************************************

  Synopsis           [Returns the status of the property]

  Description        [Returns the status of the property]

  SideEffects        []

******************************************************************************/
Prop_Status Prop_GetStatus(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->status;
}

/**Function********************************************************************

  Synopsis           [Sets the status of the property]

  Description        [Sets the status of the property]

  SideEffects        []

******************************************************************************/
void Prop_SetStatus(Prop_Ptr p, Prop_Status s)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->status = s;
}

/**Function********************************************************************

  Synopsis           [Returns the number of the property]

  Description        [For COMPUTE properties returns the number
  resulting from the evaluation of the property.]

  SideEffects        []

******************************************************************************/
int Prop_GetNumber(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->number;
}

/**Function********************************************************************

  Synopsis           [Sets the number of the property]

  Description        [Sets the number resulting from the
  evaluation of the property.]

  SideEffects        []

******************************************************************************/
void Prop_SetNumber(Prop_Ptr p, int n)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->number = n;
}

/**Function********************************************************************

  Synopsis           [Sets the number of the property to INFINITE]

  Description        [Sets the to INFINITE the number resulting from the
  evaluation of the property.]

  SideEffects        []

******************************************************************************/
void Prop_SetNumberInfinte(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->number = PROP_INFINITE;
}

/**Function********************************************************************

  Synopsis           [Returns the trace number associated to a property]

  Description        [For unsatisfied properties, the trace number of
  the asscociated counterexample is returned.]

  SideEffects        []

******************************************************************************/
int Prop_GetTrace(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->trace;
}

/**Function********************************************************************

  Synopsis           [Sets the trace number]

  Description        [Sets the trace number for an unsatisfied property.]

  SideEffects        []

******************************************************************************/
void Prop_SetTrace(Prop_Ptr p, int t)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->trace = t;
}

/**Function********************************************************************

  Synopsis           [Returns the scalar FSM in sexp of a property]

  Description        [Resturns the scalar FSM in sexp associated to
  the property.]

  SideEffects        []

******************************************************************************/
Fsm_SexpPtr Prop_GetScalarFsm(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->scalar_fsm;
}

/**Function********************************************************************

  Synopsis           [Sets the scalar FSM in sexp of a property]

  Description        [Sets the scalar FSM in sexp associated to the property.]

  SideEffects        []

******************************************************************************/
void Prop_SetScalarFsm(Prop_Ptr p, Fsm_SexpPtr f)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->scalar_fsm= f;
}

/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in sexp of a property]

  Description        [Resturns the boolean FSM in sexp associated to
  the property.]

  SideEffects        []

******************************************************************************/
Fsm_SexpPtr Prop_GetBoolFsm(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->bool_fsm;
}

/**Function********************************************************************

  Synopsis           [Sets the boolean FSM in sexp of a property]

  Description        [Sets the boolean FSM in sexp associated to the property.]

  SideEffects        []

******************************************************************************/
void Prop_SetBoolFsm(Prop_Ptr p, Fsm_SexpPtr f)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->bool_fsm= f;
}

/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in BDD of a property]

  Description        [Resturns the boolean FSM in BDD associated to
  the property.]

  SideEffects        []

******************************************************************************/
Fsm_BddPtr Prop_GetBddFsm(Prop_Ptr p)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  return p->bdd_fsm;
}

/**Function********************************************************************

  Synopsis           [Sets the boolean FSM in BDD of a property]

  Description        [Sets the boolean FSM in BDD associated to the property.]

  SideEffects        []

******************************************************************************/
void Prop_SetBddFsm(Prop_Ptr p, Fsm_BddPtr fsm)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->bdd_fsm = fsm;
}

/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in BE of a property]

  Description        [Returns the boolean FSM in BE associated to
  the property]

  SideEffects        []

******************************************************************************/
Bmc_Fsm_ptr Prop_GetBeFsm(Prop_Ptr prop)
{
  nusmv_assert(prop != (Prop_Ptr)NULL);
  return prop->be_fsm;
}


/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in BE of a property, 
  also taking into accout of the Cone Of Influence applied to the property]

  Description        []

  SideEffects        []

******************************************************************************/
Bmc_Fsm_ptr Prop_GetBeFsmCoi(Prop_Ptr prop) 
{
  Bmc_Fsm_ptr fsm;
  node_ptr spec;
  
  /* checks that a property was selected: */
  nusmv_assert (prop != (Prop_Ptr)NULL);

  fsm = Prop_GetBeFsm(prop);
  spec = Prop_GetProp(prop);

  if (fsm == (Bmc_Fsm_ptr)NULL) {

    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Building the Be model...\n");
    }
    if (opt_cone_of_influence(options) == true) {
      add_ptr one = add_one(dd_manager);
      node_ptr jc = cmp_struct_get_justice(cmps);
      node_ptr cc = cmp_struct_get_justice(cmps);
      Set_t spec_dep = Formulae_GetDependencies(spec, jc, cc);
      Set_t cone = ComputeCOI(spec_dep);

      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "Cone of influence\n");
      }

      Prop_SetScalarFsm(prop, Compile_MakeFsmSexpr(Set_Set2List(cone), jc, cc));
      Prop_SetBoolFsm(prop, Compile_MakeFsmBExpr(Set_Set2List(cone), jc, cc));
      Prop_SetCone(prop, cone);

      { /* constructing the fsm in be format */
	
	/* Note: here we use the variables manager held by the master be fsm
	   in order to build the new fsm. 
	   Notice that currently a single variable manager instance exists, 
	   and it is handled by the BMC package as a private global variable. 
	   Current implementation is temporary kept in this format. */	   
	Bmc_Fsm_ptr be_fsm = Bmc_Fsm_CreateFromSexprFSM(
	                       Bmc_Fsm_GetVarsManager(Prop_MasterGetBeFsm()),
			       Prop_GetBoolFsm(prop));
	Prop_SetBeFsm(prop, be_fsm);
      }

      fsm = Prop_GetBeFsm(prop);
    }
    else {
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "No cone of influence, using whole model\n");
      }
      Prop_SetFsmToMaster(prop);
      fsm = Prop_GetBeFsm(prop);
    }
  }
  else {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Using previously built Be model...\n");
    }
  }
  return(fsm);
}

/**Function********************************************************************

  Synopsis           [Sets the boolean FSM in BE of a property]

  Description        [Sets the boolean FSM in BE associated to the property.]

  SideEffects        []

******************************************************************************/
void Prop_SetBeFsm(Prop_Ptr p, Bmc_Fsm_ptr fsm)
{
  nusmv_assert(p != (Prop_Ptr)NULL);
  p->be_fsm = fsm;
}

/**Function********************************************************************

  Synopsis           [Prints a property]

  Description        [Prints a property on the specified FILE
  stream. Some of the information stored in the property structure are
  printed out (e.g. property, property kind, property status, ...).]

  SideEffects        []

******************************************************************************/
void Prop_PrintProp(FILE * file, Prop_Ptr prop) 
{
  if (prop != (Prop_Ptr)NULL) {
    fprintf(file, "%.3d : ", prop->index);
    print_prop(file, prop->prop);
    fprintf(file, "\n");
    fprintf(file, "\t[%-15s", Prop_DecodeType(prop->type));
    if (prop->type == Prop_Compute) {
      if (prop->number == PROP_UNCHECKED) fprintf(file, "%-15s", "Unchecked");
      else if (prop->number == PROP_INFINITE) fprintf(file, "%-15s", "Infinite");
	else fprintf(file, "%-15d", prop->number);
    }
    else fprintf(file, "%-15s", Prop_DecodeStatus(prop->status));
    fprintf(file, (prop->trace == 0 ? "N/A ]\n" : "T%-3d]\n"), prop->trace);
  }
}

/**Function********************************************************************

  Synopsis           [Creates the master prop]

  Description        [Crates the master prop to store the whole FSM]

  SideEffects        []

******************************************************************************/
void Prop_InitPropMaster()
{
  Prop_Master = Prop_Alloc();
}

/**Function********************************************************************

  Synopsis           [Free the master prop]

  Description        [Free the master prop]

  SideEffects        []

******************************************************************************/
void Prop_QuitPropMaster()
{
  Prop_Free(Prop_Master);
}

/**Function********************************************************************

  Synopsis           [Returns the scalar FSM]

  Description        [Returns the scalar FSM stored in the master prop]

  SideEffects        []

******************************************************************************/
Fsm_SexpPtr Prop_MasterGetScalarFsm()
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  return Prop_Master->scalar_fsm;
}

/**Function********************************************************************

  Synopsis           [Set the scalar FSM]

  Description        [Set the scalar FSM of the master prop]

  SideEffects        []

******************************************************************************/
void Prop_MasterSetScalarFsm(Fsm_SexpPtr fsm)
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  Prop_Master->scalar_fsm = fsm;
}

/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in sexp]

  Description        [Returns the boolean FSM in sexp stored in the master prop]

  SideEffects        []

******************************************************************************/
Fsm_SexpPtr Prop_MasterGetBoolFsm()
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  return Prop_Master->bool_fsm;
}

/**Function********************************************************************

  Synopsis           [Set the boolean FSM in sexp]

  Description        [Set the boolean FSM in sexp of the master prop]

  SideEffects        []

******************************************************************************/
void Prop_MasterSetBoolFsm(Fsm_SexpPtr fsm)
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  Prop_Master->bool_fsm = fsm;
}

/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in BDD]

  Description        [Returns the boolean FSM in BDD stored in the master prop]

  SideEffects        []

******************************************************************************/
Fsm_BddPtr Prop_MasterGetBddFsm()
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  return Prop_Master->bdd_fsm;
}

/**Function********************************************************************

  Synopsis           [Set the boolean FSM in BDD]

  Description        [Set the boolean FSM in BDD of the master prop]

  SideEffects        []

******************************************************************************/
void Prop_MasterSetBddFsm(Fsm_BddPtr fsm)
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  Prop_Master->bdd_fsm = fsm;
}

/**Function********************************************************************

  Synopsis           [Returns the boolean FSM in BE]

  Description        [Returns the boolean FSM in BE stored in the master prop]

  SideEffects        []

******************************************************************************/
Bmc_Fsm_ptr Prop_MasterGetBeFsm()
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  return Prop_Master->be_fsm;
}

/**Function********************************************************************

  Synopsis           [Set the boolean FSM in BE]

  Description        [Set the boolean FSM in BE of the master prop]

  SideEffects        []

******************************************************************************/
void Prop_MasterSetBeFsm(Bmc_Fsm_ptr fsm)
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  Prop_Master->be_fsm = fsm;
}

/**Function********************************************************************

  Synopsis           [Fills all the FSM fields of the master property]

  Description        [Fills all the FSM fields of the master property,
  with the exception of the BE fsm, of a property. If vars
  corresponds to all the system variables then the whole system is
  computed, otherwise only the FSM corresponding to the variables
  specified will be included in then model (e.g. cone of influence).]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_BuildMasterProp(node_ptr vars, node_ptr justice_expr, 
			  node_ptr compassion_expr, add_ptr assumption)
{
  nusmv_assert(Prop_Master != (Prop_Ptr)NULL);
  Prop_MasterSetScalarFsm(Compile_MakeFsmSexpr(vars, justice_expr, compassion_expr));
  Prop_MasterSetBoolFsm(Compile_MakeFsmBExpr(vars, justice_expr, compassion_expr));
  Prop_MasterSetBddFsm(Compile_MakeFsmBdd(vars, justice_expr, compassion_expr, assumption));
}

/**Function********************************************************************

  Synopsis           [Fills all the FSM fields of a property]

  Description        [Fills all the FSM fields, with the exception of the BE
  fsm, of a property. If vars corresponds to all the system variables
  then the whole system is computed, otherwise only the FSM
  corresponding to the variables specified will be included in then
  model (e.g. cone of influence).]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_BuildFsm(Prop_Ptr prop, node_ptr vars, node_ptr justice_expr, 
		   node_ptr compassion_expr, add_ptr assumption) 
{
  nusmv_assert(prop != (Prop_Ptr)NULL);
  /*
    if the build model has not been executed before the call below
    fills the data structures properly.
  */
  Compile_InitializeBuildModel();
  Prop_SetScalarFsm(prop, Compile_MakeFsmSexpr(vars, justice_expr, compassion_expr));
  Prop_SetBoolFsm(prop, Compile_MakeFsmBExpr(vars, justice_expr, compassion_expr));
  Prop_SetBddFsm(prop, Compile_MakeFsmBdd(vars, justice_expr, compassion_expr, assumption));
}

/**Function********************************************************************

  Synopsis           [Copies master prop FSM data into prop]

  Description        [Copies the FSM informations stored in the master
  prop into the corresponding fields of the given prop structure.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Prop_SetFsmToMaster(Prop_Ptr prop)
{
  if (Prop_MasterGetScalarFsm()) {
    Prop_SetScalarFsm(prop, Compile_FsmSexpDup(Prop_MasterGetScalarFsm()));
  }
  if (Prop_MasterGetBoolFsm()) {
    Prop_SetBoolFsm(prop, Compile_FsmSexpDup(Prop_MasterGetBoolFsm()));
  }
  if (Prop_MasterGetBddFsm()) {
    Prop_SetBddFsm(prop, Compile_FsmBddDup(Prop_MasterGetBddFsm()));
  }
  if (Prop_MasterGetBeFsm()) {
    Prop_SetBeFsm(prop, Bmc_Fsm_Dup(Prop_MasterGetBeFsm()));
  }
}

/**Function********************************************************************

  Synopsis           [Initializes the DB of properties]

  Description        [Initializes the DB of properties to an empty DB]

  SideEffects        []

******************************************************************************/
void Prop_Db_Init() {
  Prop_Db_Array = array_alloc(Prop_Rec, 1);
  if (Prop_Db_Array == (array_t *)ARRAY_OUT_OF_MEM) {
    fprintf(nusmv_stderr, "Prop_Db_Init: Unable to initialize Properties DB.\n");
    nusmv_exit(1);
  }
}

/**Function********************************************************************

  Synopsis           [Disposes the DB of properties]

  Description        [Disposes the DB of properties]

  SideEffects        []

******************************************************************************/
void Prop_Db_Quit() {
  int i;
  Prop_Ptr prop;
  for(i = 0; i < array_n(Prop_Db_Array); i++) {
    prop = array_fetch_p(Prop_Rec, Prop_Db_Array, i);
    Prop_Destroy(prop);
  }
  array_free(Prop_Db_Array);
}

/**Function********************************************************************

  Synopsis           [Fills the DB of properties]

  Description        [Given for each kind of property a list of
  respective fomulae, this function is responsible to fill the DB with
  them.]

  SideEffects        []

******************************************************************************/
void Prop_FillPropDb(node_ptr ctlspec, node_ptr computespec,
                      node_ptr ltlspec, node_ptr invarspec)
{
  node_ptr l;
  
  for(l = ctlspec; l != Nil; l = cdr(l)) {
    Prop_Db_PropCreateAndAdd(car(l), Prop_Ctl);
  }
  for(l = computespec; l != Nil; l = cdr(l)) {
    Prop_Db_PropCreateAndAdd(car(l), Prop_Compute);
  }
  for(l = ltlspec; l != Nil; l = cdr(l)) {
    Prop_Db_PropCreateAndAdd(car(l), Prop_Ltl);
  }
  for(l = invarspec; l != Nil; l = cdr(l)) {
    Prop_Db_PropCreateAndAdd(car(l), Prop_Invar);
  }
}

/**Function********************************************************************

  Synopsis           [Inserts a property in the DB of properties]

  Description        [Insert a property in the DB of properties. It
  returns 1 if out of memory.]

  SideEffects        []

******************************************************************************/
int Prop_Db_Add(Prop_Ptr p) {
  nusmv_assert(Prop_Db_Array != (array_t *)NULL);
  return array_insert_last(Prop_Rec, Prop_Db_Array, *p) == ARRAY_OUT_OF_MEM;
}

/**Function********************************************************************

  Synopsis           [Inserts a property in the DB of properties]

  Description        [Given a formula and its type, a property is
  created and stored in the DB of properties. It returns either -1 in
  case of failure, or the index of the inserted property.]

  SideEffects        []

******************************************************************************/
int Prop_Db_PropCreateAndAdd(node_ptr prop, Prop_Type type)
{
  int retval;
  int index = Prop_Db_GetSize();
  Prop_Ptr p = Prop_CreatePropPartial(index,prop,type);

  if (p == (Prop_Ptr)NULL)  return 1;
  if (opt_verbose_level_gt(options, 3)) {
    fprintf(nusmv_stdout, 
	    "Attempting to add %s property (index %d) to property list.\n", 
            Prop_DecodeType(type), p->index);
  }
  retval = Prop_Db_Add(p);
  if (opt_verbose_level_gt(options, 3)) {
    if (retval == 1) {
      fprintf(nusmv_stdout, \
              "Failing to add %s property (index %d) to property list.\n", \
              Prop_DecodeType(type), p->index);
    }
    else {
      fprintf(nusmv_stdout, \
              "%s property (index %d) successfully added to property list.\n",\
              Prop_DecodeType(type), p->index);
    }
  }
  retval = (retval == 1) ? -1 : index;
  return retval;
}

/**Function********************************************************************

  Synopsis           [Returns the last entered property in the DB]

  Description        [Returns the last entered property in the DB of properties.]

  SideEffects        []

******************************************************************************/
Prop_Ptr Prop_Db_GetLast() {
  Prop_Ptr result;
  nusmv_assert(Prop_Db_Array != (array_t *)NULL);
  result = array_fetch_last_p(Prop_Rec, Prop_Db_Array); 
  return result;
}

/**Function********************************************************************

  Synopsis           [Returns the property indexed by index]

  Description        [Returns the property whose unique identifier is
  provided in input.last entered property in the DB of properties.]

  SideEffects        []

******************************************************************************/
Prop_Ptr Prop_Db_GetNum(int num) {
  nusmv_assert(Prop_Db_Array != (array_t *)NULL);
  if (num > array_n(Prop_Db_Array)) return (Prop_Ptr)NULL;
  return (Prop_Ptr) array_fetch_p(Prop_Rec, Prop_Db_Array, num);
}

/**Function********************************************************************

  Synopsis           [Returns the size of the DB]

  Description        [Returns the size (i.e. the number of entries)
  stored in the DB of properties.]

  SideEffects        []

******************************************************************************/
int Prop_Db_GetSize() {
  nusmv_assert(Prop_Db_Array != (array_t *)NULL);
  return array_n(Prop_Db_Array);
}

/**Function********************************************************************

  Synopsis           [Prints the specified property from the DB]

  Description        [Prints on the given file stream the property
  whose unique identifier is specified]

  SideEffects        []

******************************************************************************/
int Prop_PrintPropNum(FILE * file, int num)
{
  int retval = 0;
  Prop_Ptr prop = Prop_Db_GetNum(num);

  if (prop != (Prop_Ptr)NULL) {
    Prop_PrintProp(file, prop);
  } else {
    retval = 1;
  }
  return retval;
}

/**Function********************************************************************

  Synopsis           [Prints all the properties stored in the DB]

  Description        [Prints on the given file stream all the property
  stored in the DB of properties.]

  SideEffects        []

******************************************************************************/
void Prop_DbPrintAll(FILE *file)
{
  int i;
  int size;

  nusmv_assert(Prop_Db_Array != (array_t *)NULL);
  size = Prop_Db_GetSize();
  for(i = 0; i < size; i++) {
    Prop_PrintPropNum(file, i);
  }
  if (i == 0) {
    fprintf(file, "The properties DB is empty.\n");
  }
}


/**Function********************************************************************

  Synopsis           [Prints all the properties stored in the DB]

  Description        [Prints on the given file stream all the property
  stored in the DB of properties whose type and status match the
  requested ones.]

  SideEffects        []

******************************************************************************/
void Prop_DbPrintAllStatusType(FILE * file, Prop_Type type, Prop_Status status)
{
  int i, s;

  s = Prop_Db_GetSize();
  for(i = 0; i < s; i++){
    Prop_Ptr p = Prop_Db_GetNum(i);

    if ((Prop_GetType(p) == type) && (Prop_GetStatus(p) == status)) {
      Prop_PrintProp(file, p);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Prints all the properties stored in the DB]

  Description        [Prints on the given file stream all the property
  stored in the DB of properties whose type match the requested one.]

  SideEffects        []

******************************************************************************/
void Prop_DbPrintAllType(FILE * file, Prop_Type type)
{
  int i, s;

  s = Prop_Db_GetSize();
  for(i = 0; i < s; i++){
    Prop_Ptr p = Prop_Db_GetNum(i);

    if (Prop_GetType(p) == type) {
      Prop_PrintProp(file, p);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Prints all the properties stored in the DB]

  Description        [Prints on the given file stream all the property
  stored in the DB of properties whose status match the requested one.]

  SideEffects        []

******************************************************************************/
void Prop_DbPrintAllStatus(FILE * file, Prop_Status status)
{
  int i, s;

  s = Prop_Db_GetSize();
  for(i = 0; i < s; i++){
    Prop_Ptr p = Prop_Db_GetNum(i);

    if (Prop_GetStatus(p) == status) {
      Prop_PrintProp(file, p);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Returns the a string associated to a property type]

  Description        [Returns the string corresponding to a property
  type for printing it.]

  SideEffects        []

******************************************************************************/
const char * Prop_DecodeType(Prop_Type t) 
{
  switch(t){
  case Prop_NoType:
    return PROP_NOTYPE_STRING;
  case Prop_Ctl:
    return PROP_CTL_STRING;
  case Prop_Ltl:
    return PROP_LTL_STRING;
  case Prop_Invar:
    return PROP_INVAR_STRING;
  case Prop_Compute:
    return PROP_COMPUTE_STRING;
  default:
    return "!!!!! Unknown property type !!!!!";
  }
}

/**Function********************************************************************

  Synopsis           [Returns the a string associated to a property status]

  Description        [Returns the string corresponding to a property
  status for printing it.]

  SideEffects        []

******************************************************************************/
const char * Prop_DecodeStatus(Prop_Status t) 
{
  switch(t){
  case Prop_NoStatus:
    return PROP_NOSTATUS_STRING;
  case Prop_Unchecked:
    return PROP_UNCHECKED_STRING;
  case Prop_True:
    return PROP_TRUE_STRING;
  case Prop_False:
    return PROP_FALSE_STRING;
  case Prop_Wrong:
    return PROP_WRONG_STRING;
  case Prop_Number:
    return PROP_NUMBER_STRING;
  default:
    return "!!!!! Unknown property type !!!!!";
  }
}

/**Function********************************************************************

  Synopsis           [Prints the header of the property list]

  Description        [optional]

  SideEffects        []

******************************************************************************/
void Prop_PrintPropHeader(FILE * file)
{
  fprintf(file, "**** PROPERTY LIST [ Type, Status, Counterex. Trace ] ****\n");
  fprintf(file, "--------------------------  PROPERTY LIST  -------------------------\n");
}

/**Function********************************************************************

  Synopsis           [Return the list of properties of a given type]

  Synopsis           [Given a property type returns the list of properties
  of that type currently located into the property database]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
lsList Prop_Db_GetPropsOfType(const Prop_Type type) 
{
  int i, db_size;
  lsList result = lsCreate();

  nusmv_assert(result != NULL);

  db_size = Prop_Db_GetSize();

  for(i=0; i<db_size; ++i) {
    Prop_Ptr p = Prop_Db_GetNum(i);
    if (Prop_GetType(p) == type) {
      lsNewEnd(result, (lsGeneric)p, LS_NH);
    }
  }

  return result;
} 

/**Function********************************************************************

  Synopsis           [Add a property to the database from a string and a type]

  Description        [Parses and creates a property of a given type from 
  a string. If the formula is correct, it is added to the 
  property database and its index is returned.
  Otherwise, -1 is returned.
  Valid types are Prop_Ctl, Prop_Ltl, Prop_Invar and Prop_Compute.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Prop_Db_PropParseAndAdd(const char* str, const Prop_Type type)
{
  char* argv[2];
  int argc = 2;

  nusmv_assert(str);
  
  argv[0] = NIL(char);
  argv[1] = (char *)str;

  return Prop_Db_PropParseFromArgAndAdd(argc, argv, type);
}

/**Function********************************************************************

  Synopsis           [Get a valid property index from a string]

  Description        [Gets the index of a property form a string.
  If the string does not contain a valid index, an error message is emitted
  and -1 is returned.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Prop_GetPropIdxFromString(const char* idx)
{
  int idxTemp;

  if ( Prop_Db_GetSize()<=0 ) {
    if (cmp_struct_get_flatten_hrc(cmps) == 0) {
      fprintf(nusmv_stderr,
	      "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    }
    else {
      fprintf(nusmv_stderr,"Error: there isn\'t any property available.\n");
    }
    return -1;
  }

  if (util_str2int(idx, &idxTemp) != 0) {
    fprintf(nusmv_stderr, 
	    "Error: \"%s\" is not a valid value (must be integer).\n", idx);
    return -1;
  }

  if ( (idxTemp < 0) || (idxTemp >= Prop_Db_GetSize()) ) {
    fprintf(nusmv_stderr, 
	    "Error: \"%d\" is not a valid value, must be in the range [0,%d].\n", 
	    idxTemp, Prop_Db_GetSize()-1);
    return -1;
  }

  return idxTemp;
}

/**Function********************************************************************

  Synopsis           [Check if a property in the database is of a given type]

  Description        [Checks if a property in the database is of a given type.
  If the type is correct, value 0 is returned, otherwise an error message 
  is emitted and value 1 is returned.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Prop_CheckType (Prop_Ptr property, Prop_Type type) 
{ 
  
  nusmv_assert(property != (Prop_Ptr)NULL);

  if (Prop_GetType(property) != type) {
    fprintf(nusmv_stderr,
	    "Error: specified property is not of type %s as expected\n", 
	    Prop_DecodeType(type));
    return 1;
  }

  return 0;
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Add a property to the database from an arg structure 
  and a type]

  Description        [Parses and creates a property of a given type from 
  an arg structure. If the formula is correct, it is added to the 
  property database and its index is returned.
  Otherwise, -1 is returned.
  Valid types are Prop_Ctl, Prop_Ltl, Prop_Invar and Prop_Compute.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Prop_Db_PropParseFromArgAndAdd(int argc, char **argv, const Prop_Type type)
{
  switch(type) {
  case Prop_Ctl:
  case Prop_Ltl:
  case Prop_Invar:
  case Prop_Compute:
    /* All ok */
    break;
    
  case Prop_NoType:
    fprintf(nusmv_stderr, "Required to parse a property of unknonw type.\n");
    return -1;
    break;

  default:
    fprintf(nusmv_stderr, "Required to parse a property of unsupported type.\n");
    return -1;
    break;
  } /* switch */ 
  
  {
    node_ptr parsed_command = Nil;
    char * parsing_type = PropTypeGetParsingType(type);
    int parse_result = 
      Parser_ReadCmdFromString(2, argv, parsing_type, ";\n", &parsed_command);
    
    if (parse_result != 0 || parsed_command == Nil) {
	fprintf(nusmv_stderr, "Parsing error: expected an \"%s\" expression.\n",\
		Prop_DecodeType(type));
        return -1;
    }
    else {
      node_ptr property = car(parsed_command);
      
      if (node_get_type(property) != CONTEXT) {
	property = find_node(CONTEXT, Nil, property);
      }
      return Prop_Db_PropCreateAndAdd(property, type);
    }
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Prints a property]

  Description        [Prints a property]

  SideEffects        []

******************************************************************************/
static void print_prop(FILE * file, node_ptr p) {
  node_ptr context = Nil;

  if (node_get_type(p) == CONTEXT) {
    context = car(p);
    p = cdr(p);
  }
  indent_node(file, " ", p, " ");
  if (context != Nil) {
    fprintf(file, "IN ");
    print_node(file, context);
  }
}

/**Function********************************************************************

  Synopsis           [Creates a property]

  Description        [Creates a property structure filling only the
  property and property type fields]

  SideEffects        []

******************************************************************************/
static Prop_Ptr Prop_CreatePropPartial(int index, node_ptr prop, Prop_Type type)
{
  Prop_Ptr p = Prop_Alloc();

  nusmv_assert(p != NIL(Prop_Rec));
  p->index = index;
  p->status = Prop_Unchecked;
  p->prop = prop;
  p->type = type;
  return p;
}

/**Function********************************************************************

  Synopsis           [Returns the parsing type given the property type]

  Description        [Returns the parsing type given the property type]

  SideEffects        []

******************************************************************************/
static char * PropTypeGetParsingType(Prop_Type type)
{
  char *pt[] = {"CTLWFF ", "LTLWFF ", "SIMPWFF ", "COMPWFF "};
    
  switch (type) {
  case Prop_Ctl:
    return pt[0];
    break;
  case Prop_Ltl:
    return pt[1];
    break;
  case Prop_Invar:
    return pt[2];
    break;
  case Prop_Compute:
    return pt[3];
    break;
  default:
    break;
  }
  return pt[2];  
}
