/**CFile***********************************************************************

  FileName    [parserUtil.c]

  PackageName [parser]

  Synopsis    [Parser utilities]

  Description [This file contains some parser utilities that allows
  for example to parse from a string, instead that from a file.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``parser'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "parserInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: parserUtil.c,v 1.1.1.1 2003/02/06 19:01:15 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Open a file and inform the parser to read from it]

  Description        [Open a file and inform the parser to start
  reading tokens from this file. If no input file is provided, then it
  inform the parser to start reading tokens from the standard input.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Parser_OpenInput(const char *filename)
{
  if (filename != (char *)NULL) {
    if(!(yyin = fopen(filename,"r")))
      rpterr("cannot open %s for input",filename);
    yylineno = 1;
  }
  if (yyin == NULL) yyin = stdin;
  /* Flushes the current input buffer */
  (void) yy_switch_to_buffer(yy_create_buffer(yyin, 16384));
  /*                                                ^^^^^ */
  /* Taken from the input.c flex generated file.          */
  (void) yyrestart(yyin);
}

/**Function********************************************************************

  Synopsis           [Open a file and inform the parser to read from it]

  Description        [Open a file, pre-process it using CPP, and inform 
  the parser to start reading tokens from this file. If no input file 
  is provided, then it inform the parser to start reading tokens 
  from the standard input.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Parser_OpenInputCpp(const char *filename)
{
  char * cpp_exec = NULL;
  char * cpp_cmd = NULL;
#if HAVE_GETENV
  cpp_exec = getenv("CPP");
#endif
#ifdef CPP
  if (cpp_exec == NULL) {
    cpp_exec = MACRO_STRINGIZE(CPP);
  }  
#else
# ifdef CPP_CMD
  if (cpp_exec == NULL) {
    cpp_exec = MACRO_STRINGIZE(CPP_CMD);
  }      
# endif
#endif

  if (cpp_exec == NULL) {
    internal_error("The CPP pre-proprocessor could not be found. \n");
  }

  if (filename != (char *)NULL) {
    /* Test whether it is possible to open the file */
    if(!(yyin = fopen(filename,"r")))
      rpterr("cannot open %s for input",filename);
    (void) fclose(yyin);
  }

  if (filename != (char *)NULL) {
    cpp_cmd = ALLOC(char, strlen(cpp_exec)+strlen(filename)+2);
  }
  else {
    cpp_cmd = ALLOC(char, strlen(cpp_exec)+1);
  }

  if (cpp_cmd == NIL(char)) {
    fprintf(nusmv_stderr, "Parser_OpenInputCpp: unable to allocate memory\n");
    nusmv_exit(1);
  }

  strcpy(cpp_cmd, cpp_exec);
  if (filename != (char *)NULL) {
    strcat(cpp_cmd, " ");
    strcat(cpp_cmd, filename);
    yylineno = 1;
  }

  if (opt_verbose_level_gt(options, 3)) {  
    fprintf(nusmv_stderr, "\nInvoking the command <%s>...\n", cpp_cmd);
  }
  if(!(yyin = popen(cpp_cmd,"r"))) {
    rpterr("error executind command \"%s\"", cpp_cmd);
  }

  FREE(cpp_cmd);

  /* Flushes the current input buffer */
  (void) yy_switch_to_buffer(yy_create_buffer(yyin, 16384));
  /*                                                ^^^^^ */
  /* Taken from the input.c flex generated file.          */
  (void) yyrestart(yyin);
}

/**Function********************************************************************

  Synopsis           [Close the input file]

  Description        [Closes the input file used from parser to read tokens.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Parser_CloseInput()
{
  (void) fclose(yyin);
}

/**Function********************************************************************

  Synopsis           [Close the input file]

  Description        [Closes the input file used from parser to read tokens.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Parser_CloseInputCpp()
{
  (void) pclose(yyin);
}

/**Function********************************************************************

  Synopsis           [Parse SMV code from a given file.]

  Description        [Parse SMV code from a given file. If 
  no file is provided, parse from stdin. If a parsing error occurs then
  return 1, else return 0. The result of parsing is stored in
  the global variable <tt>parse_tree</tt> to be used from the caller.]

  SideEffects        []

  SeeAlso            []

*****************************************************************************/
int Parser_ReadSMVFromFile(const char *filename)
{
  int retval = 0;
  if (opt_run_cpp(options)) {
    Parser_OpenInputCpp(filename);
  } 
  else {
    Parser_OpenInput(filename);
  }

  parse_tree = Nil;
  parse_command_flag = 0;

  if (yyparse()) {
    retval = 1;
  } 
  else {
    yylineno = 0;
    parse_tree = reverse(parse_tree);
  }

  if (opt_run_cpp(options)) {
    Parser_CloseInputCpp();
  } 
  else {
    Parser_CloseInput();
  }

  return retval;
}

/**Function********************************************************************

  Synopsis           [Parse a comand from a given string.]

  Description        [Create a string for a command, and then call
  <tt>yyparse</tt> to read from the created string. 
  If a parsing error occurs than return 1, else return 0. 
  The result of parsing is stored in <tt>pc</tt> to be used from the caller.]

  SideEffects        []

  SeeAlso            []

*****************************************************************************/
int Parser_ReadCmdFromString(int argc, char **argv, char * head, char *tail, node_ptr *pc)
{
  int i;
  char * old_input_file;
  int l = strlen(head);
  int status = 0;
  char * cmd = NIL(char);
  
  for (i = 1; i < argc; i++) l += strlen(argv[i]) + 1;
  l += strlen(tail) +1;
  cmd = ALLOC(char, l);
  if (cmd == NIL(char)) {
    fprintf(nusmv_stderr, "Parser_ReadFromString: unable to allocate \"cmd\"\n");
    nusmv_exit(1);
  }
  sprintf(cmd, "%s", head);
  for (i = 1; i < argc; i++) sprintf(cmd, "%s%s ", cmd, argv[i]);
  sprintf(cmd, "%s%s", cmd, tail);
  if (opt_verbose_level_gt(options, 3)) fprintf(nusmv_stderr, "%s\n", cmd);

  old_input_file = get_input_file(options);
  set_input_file(options, "<command-line>");

  parse_tree_int = Nil;
  parse_command_flag = 1;
  yy_scan_string(cmd);
  if (yyparse() == 0) {
    status = 0;
  } else {
    status = 1;
  }
  
  FREE(cmd); /* this deallocation must be checked yet */

  /* We need to reset the input buffer */
  yyrestart(yyin);
  set_input_file(options, old_input_file);
  *pc = parse_tree_int;
  return(status);
}
