/**CFile***********************************************************************

  FileName    [nodeWffPrint.c]

  PackageName [node]

  Synopsis    [Pretty printing of formulas represented using node struct.]

  Description [This file conatins the code to perform pretty printing
  of a formula represented with a node struct.]

  SeeAlso     [node.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``node'' package of NuSMV version 2.
  Copyright (C) 1998-2001 by CMU and ITC-irst.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "nodeInt.h"
#include "ustring.h"


static char rcsid[] UTIL_UNUSED = "$Id: nodeWffPrint.c,v 1.1.1.1 2003/02/06 19:01:16 flerda Exp $";

#define LINE_LENGTH 100000

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int fprint_node_recur ARGS((FILE *stream, node_ptr n, int p));
static int print_case_body ARGS((FILE *stream, node_ptr n));
static int print_case ARGS((FILE *stream, node_ptr n));
/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Pretty print a formula on a file]

  Description        [Pretty print a formula on a file]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int print_node(FILE *stream, node_ptr n)
{ return print_node_atcol(stream,n,0); }

/**Function********************************************************************

  Synopsis           [Pretty print a formula in a given file at given column]

  Description        [Pretty print a formula in a given file at given column]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int print_node_atcol(FILE *stream, node_ptr n, int col)
{
  int p;
  p=fprint_node_recur(stream,n,0);
  return(col + p);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [print string on stream.]

  Description        [print string on stream.]

  SideEffects        []

******************************************************************************/
static int util_streamprint(FILE *stream,char *s1)
{
  if (*s1){
    return(fprintf(stream,s1));
  }
  else
    return 1;
}

/**Function********************************************************************

  Synopsis           [Recursive step of print_node_atcol]

  Description        [Recursive step of print_node_atcol]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int fprint_node_recur(FILE *stream, node_ptr n, int p)
{
  char *op = 0;
  int priority = 0;
  int arity = 0;     /* 0: unary, 1: binary, 2: terciary, 3:quad */  int brckts = 0;
  if(n == Nil) return(1);
  if(n == (node_ptr)(-1)) return(util_streamprint(stream, "*no value*"));
  switch(node_get_type(n)){
  case TRUEEXP:
    return(util_streamprint(stream, "TRUE"));
  case FALSEEXP:
    return(util_streamprint(stream, "FALSE"));
  case SELF:
    return(util_streamprint(stream, "self"));
  case ATOM:
    if(!util_streamprint(stream, get_text((string_ptr)car(n))))return(0);
    if(cdr(n)){
      char buf[20];

      sprintf(buf, "_%d", (int)cdr(n));
      return(util_streamprint(stream, buf));
    }
    return(1);
  case NUMBER:
    {
      char buf[20];

      sprintf(buf,"%d", (int)car(n));
      return(util_streamprint(stream, buf));
    }
  case DOT:
    if(car(n) == Nil) return(print_node(stream, cdr(n)));
    return(print_node(stream, car(n))
     && util_streamprint(stream, ".")
     && print_node(stream, cdr(n)));
  case CONS:
    return(print_node(stream, car(n))
     && ((cdr(n) == Nil)
         || (util_streamprint(stream, " , ")
       && print_node(stream, cdr(n)))));
  case CASE:
    return(print_case(stream, n));
  case ARRAY:
    return(print_node(stream, car(n))
     && util_streamprint(stream, "[")
     && print_node(stream, cdr(n))
     && util_streamprint(stream, "]"));
  case TWODOTS: op = ".."; priority = 3; arity = 1; break;
  case IMPLIES: op = "->"; priority = 4; arity = 1; p = 5; break;
  case IFF: op = "<->"; priority = 4; arity = 1; p = 5; break;
  case OR: op = "|"; priority = 5; arity = 1;  p = 6; break;
  case AND: op = "&"; priority = 6; arity = 1; p = 7; break;
  case XOR: op = "xor"; priority = 4; arity = 1; p = 5; break;
  case XNOR: op = "xnor"; priority = 4; arity = 1; p = 5; break;
  case NOT: op = "!"; priority = 7; arity = 0; break;
  case EX: op = "EX "; priority = 8; arity = 0; break;
  case AX: op = "AX "; priority = 8; arity = 0; break;
  case EF: op = "EF "; priority = 8; arity = 0; break;
  case AF: op = "AF "; priority = 8; arity = 0; break;
  case EG: op = "EG "; priority = 8; arity = 0; break;
  case AG: op = "AG "; priority = 8; arity = 0; break;
  case OP_NEXT: op = " X "; priority = 8; arity = 0; break;
  case OP_PREC: op = " Y "; priority = 8; arity = 0; break;
  case OP_NOTPRECNOT: op = " Z "; priority = 8; arity = 0; break;
  case OP_GLOBAL: op = " G "; priority = 8; arity = 0; break;
  case OP_HISTORICAL: op = " H "; priority = 8; arity = 0; break;
  case OP_FUTURE: op = " F "; priority = 8; arity = 0; break;
  case OP_ONCE: op = " O "; priority = 8; arity = 0; break;
  case UNTIL: op = "U"; priority = 8; p = 9; arity = 1; break;
  case SINCE: op = "S"; priority = 8; p = 9; arity = 1; break;
  case RELEASES: op = "V"; priority = 8; p = 9; arity = 1; break;
  case TRIGGERED: op = "T"; priority = 8; p = 9; arity = 1; break;
  case EU:
    if(!util_streamprint(stream, "E")) return(0);
    op = "U"; priority = 8; p = 9; arity = 1; brckts = 1; break;
  case AU:
    if(!util_streamprint(stream, "A")) return(0);
    op = "U"; priority = 8; p = 9; arity = 1; brckts = 1; break;
  case EBU:
    if(!util_streamprint(stream, "E")) return(0);
    op = "BU"; priority = 8; p = 9; arity = 3; brckts = 1; break;
  case ABU:
    if(!util_streamprint(stream, "A")) return(0);
    op = "BU"; priority = 8; p = 9; arity = 3; brckts = 1; break;
  case EBF: op = "EBF "; priority = 8; arity = 2; break;
  case ABF: op = "ABF "; priority = 8; arity = 2; break;
  case EBG: op = "EBG "; priority = 8; arity = 2; break;
  case ABG: op = "ABG "; priority = 8; arity = 2; break;
  case MINU:
    if(!util_streamprint(stream, "MIN")) return(0);
    op = ","; priority = 8; p = 9; arity = 1; brckts = 1; break;
  case MAXU:
    if(!util_streamprint(stream, "MAX")) return(0);
    op = ","; priority = 8; p = 9; arity = 1; brckts = 1; break;
  case EQUAL: op = "="; priority = 9; arity = 1; break;
  case NOTEQUAL: op = "!="; priority = 9; arity = 1; break;
  case LT:    op = "<"; priority = 9; arity = 1; break;
  case GT:    op = ">"; priority = 9; arity = 1; break;
  case LE:    op = "<="; priority = 9; arity = 1; break;
  case GE:    op = ">="; priority = 9; arity = 1; break;
  case UNION: op = "union"; priority = 10; arity = 1; break;
  case SETIN: op = "in"; priority = 11; arity = 1; break;
  case MOD:   op = "mod"; priority = 12; arity = 1; break;
  case PLUS:  op = "+"; priority = 13; arity = 1; break;
  case MINUS: op = "-"; priority = 13; arity = 1; break;
  case TIMES: op = "*"; priority = 14; arity = 1; break;
  case DIVIDE: op = "/"; priority = 14; arity = 1; break;
  case NEXT:
    if (!util_streamprint(stream, "next")) return(0);
    op = ""; priority = 0; p = 1; arity = 0; break;
  case SMALLINIT:
    if (!util_streamprint(stream, "init")) return(0);
    op = ""; priority = 0; p = 1; arity = 0; break;
  default:
    internal_error("fprint_node_recur: type = %d", node_get_type(n));
  }
  if (brckts == 1 && priority < p && !util_streamprint(stream, " [ ")) return(0);
  if (brckts == 0 && priority < p && !util_streamprint(stream, "(")) return(0);
  switch(arity){
  case 0:
    if (!util_streamprint(stream, op))return(0);
    if (!fprint_node_recur(stream, car(n), priority)) return(0);
    break;
  case 1:
    if (!fprint_node_recur(stream, car(n), priority)) return(0);
    if (!util_streamprint(stream, " ")) return(0);
    if (!util_streamprint(stream, op)) return(0);
    if (!util_streamprint(stream, " ")) return(0);
    if (!fprint_node_recur(stream, cdr(n), priority))return(0);
    break;
  case 2:
    /* EF a..b f */
    if (!util_streamprint(stream, op)) return(0);                /* EF */
    if (!fprint_node_recur(stream,  car(cdr(n)), priority))
                                       return(0);         /* a */
    if (!util_streamprint(stream, "..")) return(0);
    if (!fprint_node_recur(stream,  cdr(cdr(n)), priority))
                                       return(0);         /* b */
    if (!util_streamprint(stream, " ")) return(0);
    if (!fprint_node_recur(stream,  car(n), priority)) return(0); /* f */
    break;
  case 3:
    /* E[f BU a..b g] */
    if (!fprint_node_recur(stream,  car(car(n)), priority))
                                       return(0);         /* f */
    if (!util_streamprint(stream, " ")) return(0);
    if (!util_streamprint(stream, op)) return(0);                /* BU */
    if (!util_streamprint(stream, " ")) return(0);
    if (!fprint_node_recur(stream,  car(cdr(n)), priority))
                                       return(0);         /* a */
    if (!util_streamprint(stream, "..")) return(0);
    if (!fprint_node_recur(stream,  cdr(cdr(n)), priority))
                                       return(0);         /* b */
    if (!util_streamprint(stream, " ")) return(0);
    if (!fprint_node_recur(stream,  car(cdr(n)), priority))
                                       return(0);         /* g */
    break;
  }
  if (brckts == 0 && priority < p && !util_streamprint(stream, ")")) return(0);
  if (brckts == 1 && priority < p && !util_streamprint(stream, " ] ")) return(0);
  return(1);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
static int print_case(FILE *stream, node_ptr n){
  return(util_streamprint(stream, "case\n") &&
         print_case_body(stream, n) &&
         util_streamprint(stream, "esac"));
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
static int print_case_body(FILE *stream, node_ptr n){
  int res = 0;

  nusmv_assert(n != Nil);
  res = print_node(stream, car(car(n))) &&
        util_streamprint(stream, " : ")      &&
        print_node(stream, cdr(car(n))) &&
        util_streamprint(stream, ";\n");
  if (res != 0) {
    /* This case has been inserted to keep track of BDD printing */
    if ((cdr(n) != Nil) && (node_get_type(cdr(n)) == FALSEEXP)) {
      res = util_streamprint(stream, " 1 : ") &&
            print_node(stream, cdr(n))   &&
            util_streamprint(stream, " ;\n");
    }
    else {
      if ((cdr(n) != Nil) && (node_get_type(cdr(n)) != TRUEEXP)) {
        res = print_case_body(stream, cdr(n));
      }
    }
  }
  return(res);
}

