/**CFile***********************************************************************

  FileName    [mcPrint.c]

  PackageName [mc]

  Synopsis    [This file contain the routines for the printing of states.]

  Description [This file contain the routines for the printing of states.]

  SeeAlso     []

  Author      [Flavio Lerda]

  Copyright   [
  This file is part of the ``mc'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "mcInt.h"
#include <dlfcn.h>

static char rcsid[] UTIL_UNUSED = "";

extern node_ptr proc_selector_internal_vname;
static void (*McPrint_StartStateSet) ARGS((void));
static void (*McPrint_EndStateSet) ARGS((void));
static void (*McPrint_StartState) ARGS((void));
static void (*McPrint_EndState) ARGS((void));
static void (*McPrint_PrintSymbol) ARGS((char *symbol, char *value));
static int loaded_hooks = 0;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void McPrint_PrintState ARGS((bdd_ptr s, node_ptr symbols, int changes_only));
static void McPrint_StartStateSetDefault ARGS((void));
static void McPrint_EndStateSetDefault ARGS((void));
static void McPrint_StartStateDefault ARGS((void));
static void McPrint_EndStateDefault ARGS((void));
static void McPrint_PrintSymbolDefault ARGS((char *symbol, char *value));
static void McPrint_LoadHooks ARGS((void));
static int McPrint_LoadDynamicStatePrinter ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/
void mcPrintStates(bdd_ptr states)
{
  extern bdd_ptr minterm_vars[];
  int j;
  int array_of_states_size =  
    bdd_count_minterm(dd_manager, reachable_states_bdd, get_minterm_vars_dim());
  bdd_ptr *array_of_states = ALLOC(bdd_ptr, array_of_states_size);

  bdd_pick_all_terms(dd_manager, reachable_states_bdd, 
      minterm_vars, get_minterm_vars_dim(),
      array_of_states, array_of_states_size);

  McPrint_LoadHooks();

  clear_print_hash();
  McPrint_StartStateSet();
  for(j=0; j < array_of_states_size; j++)
    McPrint_PrintState(array_of_states[j], all_symbols, 0);
  McPrint_EndStateSet();
  clear_print_hash();

  for(j=0; j < array_of_states_size; j++)
    bdd_free(dd_manager,array_of_states[j]);
  FREE(array_of_states);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
static void McPrint_PrintState(bdd_ptr s, node_ptr symbols, int changes_only)
{
  node_ptr proc_name;
  node_ptr process_selector_data;
  add_ptr add_state, tmp;
  add_ptr process_selector_enc;
  node_ptr los = symbols;

  add_state = bdd_to_add(dd_manager, s);
  process_selector_data = lookup_symbol_hash(proc_selector_internal_vname);
  if (process_selector_data != NULL) {
    process_selector_enc = (add_ptr)car(process_selector_data);
    tmp = add_if_then(dd_manager, add_state, process_selector_enc);
    proc_name = add_value(dd_manager, tmp);
    add_free(dd_manager, tmp);

    if ((proc_name != Nil) && (proc_name != sym_intern("main"))) {
      return;
    }
  }
  McPrint_StartState();
  while (los) {
    node_ptr cur_sym_value;
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);

    los = cdr(los);
    tmp = add_if_then(dd_manager, add_state, cur_sym_vals);
    cur_sym_value = add_value(dd_manager, tmp);
    add_free(dd_manager, tmp);
    add_free(dd_manager, cur_sym_vals);
    if (changes_only) {
      if (cur_sym_value == lookup_print_hash(cur_sym)) continue;
      insert_print_hash(cur_sym, cur_sym_value);
    }
    {
      char *symbol = node2str(cur_sym);
      char *value = node2str(cur_sym_value);
      McPrint_PrintSymbol(symbol, value);
      free(symbol);
      free(value);
    }
  }
  McPrint_EndState();

  add_free(dd_manager, add_state);
}

static void McPrint_StartStateSetDefault(void)
{
}

static void McPrint_EndStateSetDefault(void)
{
  fprintf(nusmv_stdout, "---------------------\n");
}

static void McPrint_StartStateDefault(void)
{
  fprintf(nusmv_stdout, "---------------------\n");
}

static void McPrint_EndStateDefault(void)
{
}

static void McPrint_PrintSymbolDefault(char *symbol, char *value)
{
  fprintf(nusmv_stdout, "    %s = %s\n", symbol, value);
}

static void McPrint_LoadHooks(void)
{
  if(!loaded_hooks) {
    if(!McPrint_LoadDynamicStatePrinter()) {
      McPrint_StartStateSet = McPrint_StartStateSetDefault;
      McPrint_EndStateSet = McPrint_EndStateSetDefault;
      McPrint_StartState = McPrint_StartStateDefault;
      McPrint_EndState = McPrint_EndStateDefault;
      McPrint_PrintSymbol = McPrint_PrintSymbolDefault;
    } else {
      fprintf(nusmv_stdout, "Loaded state printer from StatePrinter.sys\n");
    }
    loaded_hooks = 1;
  }
}

static int McPrint_LoadDynamicStatePrinter(void)
{
  void *handle = dlopen("StatePrinter.sys", RTLD_NOW);

  if(handle == NULL) return 0;

  McPrint_StartStateSet = dlsym(handle, "start_state_set");
  if(McPrint_StartStateSet == NULL) {
    dlclose(handle);
    return 0;
  }

  McPrint_EndStateSet = dlsym(handle, "end_state_set");
  if(McPrint_EndStateSet == NULL) {
    dlclose(handle);
    return 0;
  }

  McPrint_StartState = dlsym(handle, "start_state");
  if(McPrint_StartState == NULL) {
    dlclose(handle);
    return 0;
  }

  McPrint_EndState = dlsym(handle, "end_state");
  if(McPrint_EndState == NULL) {
    dlclose(handle);
    return 0;
  }

  McPrint_PrintSymbol = dlsym(handle, "print_symbol");
  if(McPrint_PrintSymbol == NULL) {
    dlclose(handle);
    return 0;
  }

  return 1;
}
