#define DEBUG 0
#define NO_PRINT_STATES
// #define EXPLICIT_PHASE1 "true"

#include "mcInt.h" 
#include "../node/node.h"
#include "../img/imgInt.h"

extern node_ptr proc_selector_internal_vname;

bdd_ptr invar_bdd_po = (bdd_ptr) NULL;
bdd_ptr fwd_vars_po = (bdd_ptr) NULL;
bdd_ptr bwd_vars_po = (bdd_ptr) NULL;
bdd_ptr frontier_po = (bdd_ptr) NULL;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void McPo_Phase1 ARGS((DPList list));
static void McPo_Phase2 ARGS((DPList list));
static void McPo_ImProviso ARGS((DPList list1, DPList list2));
static void McPo_ImProvisoSaturation ARGS((DPList list1, DPList list2));
static void McPo_ImProvisoSaturationRecur ARGS((DPList list1, int layer, DPList list2));
static bdd_ptr McPo_Forward ARGS((CPList Clist, bdd_ptr S, int *));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

void McPo_compute_reachable_states(Fsm_BddPtr fsm)
{
  bdd_ptr init_bdd;
  DPList list1;
  DPList list2;

  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, "\ncomputing reachable state space by po reduction...\n");

  /* set of initial states */
  init_bdd = Compile_FsmBddGetInit(fsm);

  /* set of invariant states */
  invar_bdd_po = Compile_FsmBddGetInvar(fsm);

  /* during forward quantify over the present input and state variables */
  fwd_vars_po = bdd_and(dd_manager, input_variables_bdd,
      state_variables_bdd);

  /* during backward quantify over the next input and state variables */
  bwd_vars_po = bdd_and(dd_manager, next_input_variables_bdd,
      next_state_variables_bdd);

  /* initially the reachable states are the initial states */
  reachable_states_bdd = bdd_dup(init_bdd);
    /* Flavio Lerda */
#ifndef NO_PRINT_STATES
    fprintf(nusmv_stdout, "@@@@ Initial states\n");
    {
      bdd_ptr tmp = bdd_dup(reachable_states_bdd);
      bdd_and_accumulate(dd_manager, &tmp, invar_bdd_po);
      mcPrintStates(tmp);
      bdd_free(dd_manager, tmp);
    }
    fprintf(nusmv_stdout, "@@@@\n");
#endif
    /* Flavio Lerda */

  /* initially the frontier contains all the initial states */
  frontier_po = bdd_dup(init_bdd);

  {
    DPTrans_Ptr phase1 = Compile_FsmBddGetPhase1(fsm);

    list1 = DPTransGetForward(phase1);
    list1 = DPListDup(list1);
    list1 = DPListReverse(list1);
  }

  {
    DPTrans_Ptr phase2 = Compile_FsmBddGetPhase2(fsm);

    list2 = DPTransGetForward(phase2);
    list2 = DPListDup(list2);
    list2 = DPListReverse(list2);
  }

  {
    boolean fix_point;

    do {
      if(opt_saturation(options))
	McPo_ImProvisoSaturation(list1, list2);
      else
	McPo_ImProviso(list1, list2);

      fix_point = bdd_count_states(dd_manager, frontier_po) == 0;
    } while(!fix_point);

    fprintf(nusmv_stderr,
	"  finished: BDD size = %d, states = %g\n",
	bdd_size(dd_manager, reachable_states_bdd),
	bdd_count_states(dd_manager, reachable_states_bdd));
    print_reachable_states(fsm, false);

    bdd_free(dd_manager, frontier_po);
  }

  /* This should be a DPList free... */
  /* free the local variables */
  DPListFree(dd_manager, list1);
  DPListFree(dd_manager, list2);

  /* free the static variables */
  bdd_free(dd_manager, fwd_vars_po);
  bdd_free(dd_manager, bwd_vars_po);

    /* Flavio Lerda */
#ifndef NO_PRINT_STATES
    fprintf(nusmv_stdout, "@@@@ Reachable states\n");
    {
      bdd_ptr tmp = bdd_dup(reachable_states_bdd);
      bdd_and_accumulate(dd_manager, &tmp, invar_bdd_po);
      mcPrintStates(tmp);
      bdd_free(dd_manager, tmp);
    }
    fprintf(nusmv_stdout, "@@@@\n");
#endif
    /* Flavio Lerda */
  if(opt_dump_bdds(options))
  {
    FILE *fp = fopen("reachable.dot", "w");
    Cudd_DumpDot(dd_manager, 1, &reachable_states_bdd, NULL, NULL, fp);
    fclose(fp);
    Cudd_PrintDebug(dd_manager, reachable_states_bdd, 4, 4);
  }
}

void McPo_Phase1(DPList list)
{
  /* represents the states visited so far during phase 1 */
  bdd_ptr visited = bdd_zero(dd_manager);
  /* Flavio Lerda */
#if DEBUG
  int i = 0;
#endif
  /* Flavio Lerda */

  for(; list != Nil; list = cdr(list))
  {
    /* represents the states visited in this iteration */
    bdd_ptr stack = bdd_dup(frontier_po);
    /* the frontier to be added at the next loop iteration */
    bdd_ptr loop_frontier = bdd_zero(dd_manager);
    /* Descriptor for the current process */
    DPDesc *desc = DPListGetItem(list);
    /* Gets the support variable for the equality-reduced BDD*/
    int *support_vars_backward = DPDescGetSupportVector(desc);
    /* The transition relation for one process */
    CPList Clist = DPDescGetCList(desc);
    /* did we reach a fix point? */
    boolean fix_point;
    /* the last frontier */

    /* Flavio Lerda */
#if DEBUG
    fprintf(nusmv_stdout, "@@@@ Phase1: R_%d\n", i++);
    fprintf(nusmv_stdout, "@@@@\n");
#endif

#if DEBUG > 2
    fprintf(nusmv_stdout, "@@@@ Frontier\n");
    {
      bdd_ptr tmp = bdd_dup(frontier_po);
      bdd_and_accumulate(dd_manager, &tmp, invar_bdd_po);
      mcPrintStates(tmp);
      bdd_free(dd_manager, tmp);
    }
    fprintf(nusmv_stdout, "@@@@\n");
#endif
    /* Flavio Lerda */

    do {
      bdd_and_accumulate(dd_manager, &frontier_po, invar_bdd_po);
      bdd_ptr image = McPo_Forward(Clist, frontier_po, support_vars_backward);
      bdd_free(dd_manager, frontier_po);

      /* Flavio Lerda */
#if DEBUG > 2
      fprintf(nusmv_stdout, "@@@@ Image\n");
      {
	bdd_ptr tmp = bdd_dup(image);
	bdd_and_accumulate(dd_manager, &tmp, invar_bdd_po);
	mcPrintStates(tmp);
	bdd_free(dd_manager, tmp);
      }
      fprintf(nusmv_stdout, "@@@@\n");
#endif
      /* Flavio Lerda */

      {
	bdd_ptr tmp = bdd_and(dd_manager, stack, image);

	if(bdd_isnot_zero(dd_manager, tmp))
	{
	  bdd_or_accumulate(dd_manager, &loop_frontier, tmp);

	  {
	    bdd_ptr not_tmp = bdd_not(dd_manager, tmp);

	    bdd_and_accumulate(dd_manager, &image, not_tmp);

	    bdd_free(dd_manager, not_tmp);
	  }
	}

	bdd_free(dd_manager, tmp);
      }

      bdd_or_accumulate(dd_manager, &stack, image);
      bdd_or_accumulate(dd_manager, &visited, image);

      frontier_po = image;

      {
	static int i = 0;
	if (opt_verbose_level_gt(options, 1))
	{
	  bdd_ptr tmp = bdd_dup(reachable_states_bdd);
	  bdd_or_accumulate(dd_manager, &tmp, visited);
	  fprintf(nusmv_stderr,
	      "      iteration %d: BDD size = %d, frontier size = %d, states = %g\n",
	      i, bdd_size(dd_manager, tmp),
	      bdd_size(dd_manager, frontier_po),
	      bdd_count_states(dd_manager, tmp));
	  bdd_free(dd_manager, tmp);
	}
	i++;

	if(opt_dump_bdds(options))
	{
	  FILE *fp;
	  char *fname = malloc(7 + 10);
	  sprintf(fname, "S-%d.dot", i);
	  fp = fopen(fname, "w");
	  free(fname);
	  Cudd_DumpDot(dd_manager, 1, &reachable_states_bdd, NULL, NULL, fp);
	  fclose(fp);
	  fprintf(nusmv_stderr, "States:\n");
	  Cudd_PrintDebug(dd_manager, reachable_states_bdd, 4, 4);
	}

	if(opt_dump_bdds(options))
	{
	  FILE *fp;
	  char *fname = malloc(7 + 10);
	  sprintf(fname, "F-%d.dot", i);
	  fp = fopen(fname, "w");
	  free(fname);
	  Cudd_DumpDot(dd_manager, 1, &frontier_po, NULL, NULL, fp);
	  fclose(fp);
	  fprintf(nusmv_stderr, "Frontier:\n");
	  Cudd_PrintDebug(dd_manager, frontier_po, 4, 4);
	}
      }

      /* Flavio Lerda */
#if DEBUG > 2
      fprintf(nusmv_stdout, "@@@@ Frontier\n");
      {
	bdd_ptr tmp = bdd_dup(frontier_po);
	bdd_and_accumulate(dd_manager, &tmp, invar_bdd_po);
	mcPrintStates(tmp);
	bdd_free(dd_manager, tmp);
      }
      fprintf(nusmv_stdout, "@@@@\n");
#endif
      /* Flavio Lerda */

      fix_point = bdd_count_states(dd_manager, frontier_po) == 0;
    } while(!fix_point);

    bdd_free(dd_manager, frontier_po);
    /* updates the frontier */
    frontier_po = loop_frontier;
  }

  bdd_or_accumulate(dd_manager, &reachable_states_bdd, visited);
  bdd_free(dd_manager, visited);
}

void McPo_Phase2(DPList list)
{
  boolean fix_point;
  DPList iter;
  bdd_ptr previous_reachable_states_bdd;
  /* Flavio Lerda */
  int i = 0;
  /* Flavio Lerda */

  previous_reachable_states_bdd = bdd_dup(reachable_states_bdd);

  bdd_and_accumulate(dd_manager, &frontier_po, invar_bdd_po);
  for(iter = list; iter != Nil; iter = cdr(iter))
  {
  /* Flavio Lerda */
#if DEBUG
    fprintf(nusmv_stdout, "@@@@ Phase2: R_%d\n", i);
    fprintf(nusmv_stdout, "@@@@\n");
    i++;
#endif
  /* Flavio Lerda */
    /* Descriptor for the current process */
    DPDesc *desc = DPListGetItem(list);
    /* The array to shift the support variable backward */
    int *support_vars_backward = DPDescGetSupportVector(desc);
    /* The transition relation for one process */
    CPList Clist = DPDescGetCList(desc);
    /* set of new states */
    bdd_ptr new_states = (bdd_ptr) NULL;

    /* computes the forward image from the current reachable states */
    new_states = McPo_Forward(Clist, frontier_po, support_vars_backward);

    /* add the new states to the set of reachable states */
    bdd_or_accumulate(dd_manager, &reachable_states_bdd, new_states);

    bdd_free(dd_manager, new_states);

    {
      static int i = 0;
      if (opt_verbose_level_gt(options, 1))
      {
	fprintf(nusmv_stderr,
	    "      iteration %d: BDD size = %d, frontier size = %d, states = %g\n",
	    i, bdd_size(dd_manager, reachable_states_bdd),
	    bdd_size(dd_manager, frontier_po),
	    bdd_count_states(dd_manager, reachable_states_bdd));
      }
      i++;

      if(opt_dump_bdds(options))
      {
	FILE *fp;
	char *fname = malloc(7 + 10);
	sprintf(fname, "S-%d.dot", i);
	fp = fopen(fname, "w");
	free(fname);
	Cudd_DumpDot(dd_manager, 1, &reachable_states_bdd, NULL, NULL, fp);
	fclose(fp);
	fprintf(nusmv_stderr, "States:\n");
	Cudd_PrintDebug(dd_manager, reachable_states_bdd, 4, 4);
      }

      if(opt_dump_bdds(options))
      {
	FILE *fp;
	char *fname = malloc(7 + 10);
	sprintf(fname, "F-%d.dot", i);
	fp = fopen(fname, "w");
	free(fname);
	Cudd_DumpDot(dd_manager, 1, &frontier_po, NULL, NULL, fp);
	fclose(fp);
	fprintf(nusmv_stderr, "Frontier:\n");
	Cudd_PrintDebug(dd_manager, frontier_po, 4, 4);
      }
    }
  }

  bdd_free(dd_manager, frontier_po);

  {
    bdd_ptr tmp = bdd_not(dd_manager, previous_reachable_states_bdd);
    frontier_po = bdd_and(dd_manager, reachable_states_bdd, tmp);
    bdd_free(dd_manager, tmp);
  }

#if DEBUG > 2
    fprintf(nusmv_stdout, "@@@@ Frontier\n");
    {
      bdd_ptr tmp = bdd_dup(frontier_po);
      bdd_and_accumulate(dd_manager, &tmp, invar_bdd_po);
      mcPrintStates(tmp);
      bdd_free(dd_manager, tmp);
    }
    fprintf(nusmv_stdout, "@@@@\n");
#endif

  bdd_free(dd_manager, previous_reachable_states_bdd);
}

void McPo_ImProviso(DPList list1, DPList list2)
{
#ifdef EXPLICIT_PHASE1
  {
    extern bdd_ptr minterm_vars[];
    int nstates = bdd_count_minterm(dd_manager, frontier_po,
	get_minterm_vars_dim());
    bdd_ptr *states = ALLOC(bdd_ptr, nstates);
    bdd_pick_all_terms(dd_manager, frontier_po, minterm_vars,
	get_minterm_vars_dim(), states, nstates);
    bdd_free(dd_manager, frontier_po);
    {
      int index;
      bdd_ptr frontier = bdd_zero(dd_manager);

      for(index = 0; index < nstates; index++)
      {
	frontier_po = states[index];
	McPo_Phase1(list1);
	bdd_or_accumulate(dd_manager, &frontier, frontier_po);
	bdd_free(dd_manager, frontier_po);
      }

      frontier_po = frontier;
    }
  }
#else
  McPo_Phase1(list1);
#endif
  McPo_Phase2(list2);
}

void McPo_ImProvisoSaturation(DPList list1, DPList list2)
{
  boolean fix_point;
  DPList iter;
  bdd_ptr previous_reachable_states_bdd;

  previous_reachable_states_bdd = bdd_dup(reachable_states_bdd);

  bdd_and_accumulate(dd_manager, &frontier_po, invar_bdd_po);

  {
    /* lengths of the transition relation list */
    int layers = DPListLength(list2);
    DPList reversed_lists[layers];
    DPList iter;
    int layer;

    iter = list2;
    for(layer = 0; layer < layers; layer++)
    {
      reversed_lists[layers - layer - 1] = iter;
      iter = cdr(iter);
    }

    for(layer = 0; layer < layers; layer++)
    {
      {
	char *name = DPDescGetName(DPListGetItem(reversed_lists[layer]));
	// fprintf(nusmv_stderr, "  starting layer %d (%s)\n", layer, name);
      }

      McPo_ImProvisoSaturationRecur(reversed_lists[layer], layer, list1);

      // fprintf(nusmv_stderr, "  finished layer %d: BDD size = %d, states = %g\n", layer, bdd_size(dd_manager, reachable_states_bdd), bdd_count_states(dd_manager, reachable_states_bdd));
    }
  }

  bdd_free(dd_manager, frontier_po);

  {
    bdd_ptr tmp = bdd_not(dd_manager, previous_reachable_states_bdd);
    frontier_po = bdd_and(dd_manager, reachable_states_bdd, tmp);
    bdd_free(dd_manager, tmp);
  }

  bdd_free(dd_manager, previous_reachable_states_bdd);
}

static void McPo_ImProvisoSaturationRecur(DPList list2, int layer, DPList list1)
{
  if(!DPListIsNotEmpty(list2))
  {
    int level = -1;
    // fprintf(nusmv_stderr, "    starting saturation of level %d for layer %d (%s)\n", level, layer, "Phase1");
    McPo_Phase1(list1);
    // fprintf(nusmv_stderr, "    finished saturation of level %d for layer %d\n", level, layer);
    return;
  }

  {
    /* frontier generated by last iteration */
    bdd_ptr next_frontier = bdd_zero(dd_manager);
    /* true if last outer iteration reached a fix point */
    int fix_point;
    /* level at which we are currently working */
    int level = DPListLength(list2)-1;
    /* number of outer iterations before reaching a fix point */
    int outer;
    /* Descriptor of current process */
    DPDesc *desc = DPListGetItem(list2);

    {
      char *name = DPDescGetName(desc);
      // fprintf(nusmv_stderr, "    starting saturation of level %d for layer %d (%s)\n", level, layer, name);
    }

    do {
      /* saturates lower levels */
      McPo_ImProvisoSaturationRecur(DPListNext(list2), layer, list1);

      /* Adds the returned frontier to the frontier of the next level */
      bdd_or_accumulate(dd_manager, &next_frontier, frontier_po);

      /* set of states reachable before image computation */
      bdd_ptr previous_reachable_states_bdd;

      /* The bdd of the support variables */
      bdd_ptr support_bdd_current = DPDescGetSupportBDD(desc);
      /* The array to shift the support variable backward */
      int *support_vars_backward = DPDescGetSupportVector(desc);
      /* The transition relation for this layer */
      CPList trans = DPDescGetCList(desc);
      /* Name of the process */
      char *name = DPDescGetName(desc);

      /* the reachable states before last step */
      previous_reachable_states_bdd = bdd_dup(reachable_states_bdd);

      /* intersect the frontier with normal assignments  */
      bdd_and_accumulate(dd_manager, &frontier_po, invar_bdd_po);

      {
	/* set of new states */
	bdd_ptr new_states;

	// fprintf(nusmv_stderr, "      image computation at level %d for layer %d\n", level, layer);
	if(opt_equality_reduced(options))
	{
	  bdd_ptr tmp = (bdd_ptr) CPImageBoth(dd_manager, trans,
	      frontier_po, support_bdd_current);
	  new_states = bdd_shift_set(dd_manager, tmp, support_vars_backward);
	  bdd_free(dd_manager, tmp);
	}
	else
	{
	  bdd_ptr tmp = (bdd_ptr) CPImageBoth(dd_manager, trans,
	      frontier_po, fwd_vars_po);
	  new_states = bdd_shift_backward(dd_manager, tmp);
	  bdd_free(dd_manager, tmp);
	}

	/* add the new states to the set of reachable states */
	bdd_or_accumulate(dd_manager, &reachable_states_bdd, new_states);

	bdd_free(dd_manager, new_states);
      }

      /* free the last frontier */
      bdd_free(dd_manager, frontier_po);

      /* computes the new frontier */
      {
	bdd_ptr tmp = bdd_not(dd_manager, previous_reachable_states_bdd);
	frontier_po = bdd_and(dd_manager, reachable_states_bdd, tmp);
	bdd_free(dd_manager, tmp);
      }

      bdd_free(dd_manager, previous_reachable_states_bdd);

      fix_point = bdd_count_states(dd_manager, frontier_po) == 0;

    } while(!fix_point);

    bdd_free(dd_manager, frontier_po);

    frontier_po = next_frontier;

    // fprintf(nusmv_stderr, "    finished saturation of level %d for layer %d\n", level, layer);
  }
}

static bdd_ptr McPo_Forward(CPList Clist, bdd_ptr S, int *support_vars_backward)
{
  // Added support for equality reduced BDDs.
  bdd_ptr tmp = CPImageBoth(dd_manager, Clist, S, fwd_vars_po);
  bdd_ptr result;
  if(opt_equality_reduced(options))
    result = bdd_shift_set(dd_manager, tmp, support_vars_backward);
  else
    result = bdd_shift_backward(dd_manager, tmp);

  bdd_free(dd_manager, tmp);

  return result;
}
