#define NO_PRINT_STATES

/**CFile***********************************************************************

  FileName    [mcMc.c]

  PackageName [mc]

  Synopsis    [Fair CTL model checking routines.]

  Description [Fair CTL model checking routines.]

  SeeAlso     [mcExplain.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``mc'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "mcInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: mcMc.c,v 1.8 2003/10/28 17:29:22 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The set of reachable states.]

  Description [The set of reachable states.]

  SeeAlso     []

******************************************************************************/
bdd_ptr reachable_states_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [Slices of reachable states.]

  Description [This is the list of the slices of reachable states
  accumulated during computation of the reachable states.]

  SeeAlso     []

******************************************************************************/
node_ptr reachable_states_layers = Nil;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static bdd_ptr fair_iter ARGS((Fsm_BddPtr, bdd_ptr g, node_ptr fc));
static void warn_reachable_states_not_computed ARGS((FILE *file));
static void Mc_compute_reachable_states ARGS((Fsm_BddPtr fsm));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Verifies that M,s0 |= alpha ]

  Description [Verifies that M,s0 |= alpha using the fair CTL model checking.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void Mc_CheckCTLSpec(Prop_Ptr prop){
  node_ptr exp;
  bdd_ptr s0, tmp_1, tmp_2;
  node_ptr spec  = Prop_GetProp(prop);
  Fsm_BddPtr fsm = Prop_GetBddFsm(prop);

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_spec(nusmv_stderr, spec);
    fprintf(nusmv_stderr, "\n");
  }

  if (fsm == (Fsm_BddPtr)NULL) {
    if (opt_cone_of_influence(options) == true) {
      add_ptr one = add_one(dd_manager);
      node_ptr jc = cmp_struct_get_justice(cmps);
      node_ptr cc = cmp_struct_get_compassion(cmps);
      Set_t spec_dep = Formulae_GetDependencies(spec, jc, cc);
      Set_t cone = ComputeCOI(spec_dep);

      Prop_BuildFsm(prop, Set_Set2List(cone), jc, cc, one);
      Prop_SetCone(prop, cone);
      fsm = Prop_GetBddFsm(prop);
      add_free(dd_manager, one);
    }
    else {
      Prop_SetFsmToMaster(prop);
      fsm = Prop_GetBddFsm(prop);
    }
  }
  
  /* Evaluates the spec */
  s0 = eval_spec(fsm, spec, Nil);
  /*
    Intersect with initial states (invar are normal assignments)
  */
  { /*
      We quantify out the input variables from the result of the
      formula evaluation.
    */
    bdd_ptr tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);

    bdd_free(dd_manager, s0);
    s0 = tmp;
  }
  tmp_1 = bdd_not(dd_manager, s0);
  tmp_2 = bdd_and(dd_manager, Compile_FsmBddGetInvar(fsm), tmp_1);
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, s0);
  s0 = bdd_and(dd_manager, Compile_FsmBddGetInit(fsm), tmp_2);
  bdd_free(dd_manager, tmp_2);
  /* Prints out the result, if not true explain. */
  fprintf(nusmv_stdout, "-- ");
  print_spec(nusmv_stdout, spec);

  if (bdd_is_zero(dd_manager, s0)) { 
    fprintf(nusmv_stdout, "is true\n");
    Prop_SetStatus(prop, Prop_True);
  }
  else {
    fprintf(nusmv_stdout, "is false\n");
    Prop_SetStatus(prop, Prop_False);
    /* Flavio Lerda
     * 
     * Do not print the counter-example
     *
    tmp_1 = bdd_pick_one_state(dd_manager, s0);
    bdd_free(dd_manager, s0);
    s0 = tmp_1;
    bdd_ref(s0);
    exp = reverse(explain(fsm, cons((node_ptr)s0, Nil), spec, Nil));
    if (exp == Nil) {
      // The counterexample consists of one initial state
      exp = cons((node_ptr)s0, Nil);
    }
    Prop_SetTrace(prop, get_trace_number());
    print_explanation(exp, all_symbols);
    walk_dd(dd_manager, bdd_free, exp);
    */
  }
  bdd_free(dd_manager, s0);
}

/**Function********************************************************************

  Synopsis    [Compute quantitative characteristics on the model.]

  Description [Compute the given quantitative characteristics on the model.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void Mc_CheckCompute(Prop_Ptr prop)
{
  int s0;
  node_ptr  spec = Prop_GetProp(prop);
  Fsm_BddPtr fsm = Prop_GetBddFsm(prop);

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_spec(nusmv_stderr, spec);
    fprintf(nusmv_stderr, "\n");
  }

  if (fsm == (Fsm_BddPtr)NULL) {
    if (opt_cone_of_influence(options) == true) {
      add_ptr one = add_one(dd_manager);
      node_ptr jc = cmp_struct_get_justice(cmps);
      node_ptr cc = cmp_struct_get_compassion(cmps);
      Set_t spec_dep = Formulae_GetDependencies(spec, jc, cc);
      Set_t cone = ComputeCOI(spec_dep);

      Prop_BuildFsm(prop, Set_Set2List(cone), jc, cc, one);
      Prop_SetCone(prop, cone);
      fsm = Prop_GetBddFsm(prop);
      add_free(dd_manager, one);
    }
    else {
      Prop_SetFsmToMaster(prop);
      fsm = Prop_GetBddFsm(prop);
    }
  }
  
  s0 = eval_compute(fsm, spec, Nil);

  if (reachable_states_bdd == (bdd_ptr)NULL) {
    warn_reachable_states_not_computed(nusmv_stdout);
  }

  fprintf(nusmv_stdout, "-- ");
  print_compute(nusmv_stdout, spec);
  if (s0 == -1) {
    fprintf(nusmv_stdout, "is infinity\n");
    Prop_SetNumberInfinte(prop);
    Prop_SetStatus(prop, Prop_Number);
  }
  else {
    fprintf(nusmv_stdout, "is %d\n", s0);
    Prop_SetNumber(prop, s0);
    Prop_SetStatus(prop, Prop_Number);
  }
}

/**Function********************************************************************

  Synopsis    [Warn that the reachable states have not been computed]

  Description [Emit a warning that the reachable states have not been
  computed and that the following result may be wrong. Emit information
  on how to generate the reachable states.]

  SideEffects []

  SeeAlso     [compute_reachable_states]

******************************************************************************/
static void warn_reachable_states_not_computed(FILE *file)
{
  fprintf(file, "WARNING *** Reachable states have not been computed.   ***\n");
  fprintf(file, "WARNING *** The following result may be wrong.         ***\n");
  if (opt_batch(options)) {
    fprintf(file, "WARNING *** Use options -f or -r to compute the        ***\n");

    fprintf(file, "WARNING *** reachable states.                          ***\n");
  }
  else {
    fprintf(file, "WARNING *** Call functions compute_reachable or        ***\n");
    fprintf(file, "WARNING *** print_reachable_state to compute the       ***\n");
    fprintf(file, "WARNING *** reachable states.                          ***\n");
  }
}

/**Function********************************************************************

  Synopsis    [Check that the transition relation is total]

  Description [Check that the transition relation is total. If not the
  case than a deadlock state is printed out.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void check_transition_relation(Fsm_BddPtr fsm)
{
  bdd_ptr deadlock_states;
   
  if (opt_verbose_level_gt(options, 0))
    fprintf(nusmv_stderr, "Checking for transition relation totality.\n");
  deadlock_states = trans_is_total(fsm);
  if (bdd_isnot_zero(dd_manager, deadlock_states)) {
    bdd_ptr ds = bdd_pick_one_state(dd_manager, deadlock_states);

    fprintf(nusmv_stdout, "##########################################################\n");
    if (reachable_states_bdd == (bdd_ptr)NULL) {
      warn_reachable_states_not_computed(nusmv_stdout);
    }
    fprintf(nusmv_stdout, "The Transition relation is not total. A deadlock state is:\n");
    print_state(ds, all_variables, 0);
    fprintf(nusmv_stdout, "##########################################################\n");
    bdd_free(dd_manager, ds);
  } else {
    fprintf(nusmv_stdout, "##########################################################\n");
    fprintf(nusmv_stdout, "The Transition relation is total.\n");
    fprintf(nusmv_stdout, "##########################################################\n");
  }
  bdd_free(dd_manager, deadlock_states);
}

/**Function********************************************************************

  Synopsis           [Compute reachable states]

  Description        [  This function computes reachable states.<br>
  According to the specified options, the system can compute the
  reachable states in different ways:<br>
  <ul>
  <li> <code>reachable_states</code> is the result</li>
  <li> <code>from_lower_bound</code> is the frontier</li>
  <li> <code>from_upper_bound</code> is the previous value of
       <code>reachable_states</code></li>
  <li> <code>not_from_upper_bound</code> is the complement of
       <code>from_upper_bound</code></li> 
  </ul>
]

  SideEffects        [The global variable <code>reachable_states</code> is updated.] 

  SeeAlso            []

******************************************************************************/
void compute_reachable_states(Fsm_BddPtr fsm)
{ 
  if(opt_po(options))
    McPo_compute_reachable_states(fsm);
  else if(opt_saturation(options))
    McSaturation_compute_reachable_states(fsm);
  else
    Mc_compute_reachable_states(fsm);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EX(g)</i>.]

  Description        [Computes the set of states satisfying <i>EX(g)</i>.]

  SideEffects        []

  SeeAlso            [eu ef eg]

******************************************************************************/
bdd_ptr ex(Fsm_BddPtr fsm, bdd_ptr g)
{
  bdd_ptr tmp;
  bdd_ptr fair_states_bdd;
  bdd_ptr invar_bdd;
  bdd_ptr result = bdd_dup(g);
  
  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  invar_bdd = Compile_FsmBddGetInvar(fsm);

  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &result, fair_states_bdd);
  }
  tmp = bdd_and(dd_manager, result, invar_bdd);
  bdd_free(dd_manager, result);
  result = Img_ImageBwd(fsm, tmp);
  bdd_free(dd_manager, tmp);

  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &result, reachable_states_bdd);
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>E\[ f U g \]</i>.]

  Description        [Computes the set of states satisfying <i>E\[ f U g \]</i>.]

  SideEffects        []

  SeeAlso            [ebu]

******************************************************************************/
bdd_ptr eu(Fsm_BddPtr fsm, bdd_ptr f, bdd_ptr g)
{
  bdd_ptr fair_states_bdd;
  bdd_ptr new, oldY;
  bdd_ptr Y = bdd_dup(g);
  int n = 1;

  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  }
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  }
  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "eu: computing fixed point approximations for ",
                get_the_node()," ...\n");
  oldY = bdd_dup(Y);
  new = bdd_dup(Y);
  while(bdd_isnot_zero(dd_manager, new)){
    bdd_ptr tmp_1, tmp_2;
    
    if (opt_verbose_level_gt(options, 1)) {
      double states = bdd_count_states(dd_manager, Y);
      int size = bdd_size(dd_manager, Y);

      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n", n++, states,size);
    }
    bdd_free(dd_manager, oldY);
    oldY = bdd_dup(Y);
    tmp_1 = ex(fsm, new);
    tmp_2 = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, tmp_1);
    bdd_or_accumulate(dd_manager, &Y, tmp_2);
    bdd_free(dd_manager, tmp_2);
    tmp_1 = bdd_not(dd_manager, oldY);
    bdd_free(dd_manager, new);
    new = bdd_and(dd_manager, Y, tmp_1);
    bdd_free(dd_manager, tmp_1);
  }
  bdd_free(dd_manager, new);
  bdd_free(dd_manager, oldY);
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EF(g)</i>.]

  Description        [Computes the set of states satisfying <i>EF(g)</i>.]

  SideEffects        []

  SeeAlso            [eu ex]

******************************************************************************/
bdd_ptr ef(Fsm_BddPtr fsm, bdd_ptr g)
{
  bdd_ptr result, one;

  one = bdd_one(dd_manager);
  result = eu(fsm, one, g);
  bdd_free(dd_manager, one);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EG(g)</i>.]

  Description        [Computes the set of states satisfying <i>EG(g)</i>.]

  SideEffects        []

  SeeAlso            [eu ex ef]

******************************************************************************/
bdd_ptr eg(Fsm_BddPtr fsm, bdd_ptr g)
{
  int n = 1;
  bdd_ptr tmp_1;
  bdd_ptr Y = bdd_dup(g);
  bdd_ptr oldY = bdd_zero(dd_manager);
  node_ptr fairness_constraints = Compile_FsmBddGetJustice(fsm);

  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "eg: computing fixed point approximations for ",
                get_the_node(), " ...\n");
  while (Y != oldY) {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    bdd_free(dd_manager, oldY);
    oldY = bdd_dup(Y);
    {
      bdd_ptr Z = fair_iter(fsm, Y, fairness_constraints);

      bdd_and_accumulate(dd_manager, &Y, Z);
      bdd_free(dd_manager, Z);
    }
    tmp_1 = ex(fsm, Y);
    bdd_and_accumulate(dd_manager, &Y, tmp_1);
    bdd_free(dd_manager, tmp_1);
  }
  bdd_free(dd_manager, oldY);
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>A\[f U g\]</i>.]

  Description        [Computes the set of states satisfying <i>A\[f U g\]</i>.]

  SideEffects        []

  SeeAlso            [ax af ex ef]

******************************************************************************/
bdd_ptr au(Fsm_BddPtr fsm, bdd_ptr f, bdd_ptr g)
{
  bdd_ptr result, tmp_1, tmp_2, tmp_3, tmp_4;
  
  tmp_1 = bdd_not(dd_manager, f);
  tmp_2 = bdd_not(dd_manager, g);
  tmp_3 = eg(fsm, tmp_2);
  tmp_4 = bdd_and(dd_manager, tmp_1, tmp_2);
  bdd_free(dd_manager, tmp_1);
  tmp_1 = eu(fsm, tmp_2, tmp_4);
  bdd_free(dd_manager, tmp_2);
  tmp_2 = bdd_or(dd_manager, tmp_1, tmp_3);
  result = bdd_not(dd_manager, tmp_2);
  
  bdd_free(dd_manager, tmp_1);
  bdd_free(dd_manager, tmp_2);
  bdd_free(dd_manager, tmp_3);
  bdd_free(dd_manager, tmp_4);

  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the set of justice constraints BDD.]

  Description        [Computes the set of justice constraints BDD, in
  order to use them during the model checking phase.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Mc_ComputeJusticeConstraints(Fsm_BddPtr fsm, node_ptr justice_exp) {
  node_ptr result;
  
  result = eval_formula_list(fsm, justice_exp, Nil);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the set of compassion constraints BDD pairs.]

  Description        [Computes the set of compassion constraints BDD pairs, in
  order to use them during the model checking phase.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Mc_ComputeCompassionConstraints(Fsm_BddPtr fsm, 
					 node_ptr compassion_exp) {
  node_ptr result;
  
  result = eval_formula_list(fsm, compassion_exp, Nil);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of <code>fair_states</code>.]

  Description        [Computes the set of <code>fair_states</code>.]

  SideEffects        []

  SeeAlso            [eg]

******************************************************************************/
bdd_ptr Mc_ComputeFairStates(Fsm_BddPtr fsm) 
{
  bdd_ptr result, one;
  extern node_ptr one_number;

  set_the_node(one_number);
  one = bdd_one(dd_manager);
  result = eg(fsm, one);
  bdd_free(dd_manager, one);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>E\[f U^{inf..sup} g\]</i>.]

  Description        [Computes the set of states satisfying
                      <i>E\[f U^{inf..sup} g\]</i></i>.]

  SideEffects        []

  SeeAlso            [eu]

******************************************************************************/
bdd_ptr ebu(Fsm_BddPtr fsm, bdd_ptr f, bdd_ptr g, int inf, int sup)
{
  int i;
  bdd_ptr fair_states_bdd;
  bdd_ptr Y, oldY, tmp_1, tmp_2;
  int n = 1;

  if (inf > sup || inf < 0) return(bdd_zero(dd_manager));
  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  Y = bdd_dup(g);
  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  }
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  }
  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "ebu: computing fixed point approximations for ",
                get_the_node()," ...\n");

  /* compute Y = g | (f & ex(Y)) for states within the bound */
  for (i = sup; i > inf; i--) {
    /* There are more states within the bounds */
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = ex(fsm, Y);
    tmp_2 = bdd_and(dd_manager, f, tmp_1);
    bdd_or_accumulate(dd_manager, &Y, tmp_2);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, tmp_2);

    if (Y == oldY) {
      /* fixpoint found. collect garbage, and goto next phase */
      i = inf + 1;
    }
  }

  /* compute Y = f & ex(Y) for states before the bound */
  for (i = inf; i > 0; i--) {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = bdd_dup(Y);
    tmp_1 = ex(fsm, Y);
    bdd_free(dd_manager, Y);
    Y = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, oldY);
    if (Y == oldY) {
      /* fixpoint found. collect garbage, and finish */
      i = 1;
    }
  }
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EG^{inf..sup}(g)</i>.]

  Description        [Computes the set of states satisfying
                      <i>EG^{inf..sup}(g)</i>.]

  SideEffects        []

  SeeAlso            [eg]

******************************************************************************/
bdd_ptr ebg(Fsm_BddPtr fsm, bdd_ptr g, int inf, int  sup)
{
  int i;
  bdd_ptr fair_states_bdd;
  bdd_ptr Y, oldY, tmp_1;
  int n = 1;


  if (inf > sup || inf < 0) return(bdd_zero(dd_manager));
  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  Y = bdd_dup(g);
  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  }
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  }
  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "ebg: computing fixed point approximations for ",
                get_the_node()," ...\n");
  /* compute Y = g & ex(Y) for states within the bound */
  for (i = sup; i > inf; i--) {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = ex(fsm, Y);
    bdd_and_accumulate(dd_manager, &Y, tmp_1);
    bdd_free(dd_manager, tmp_1);
    if (Y == oldY) {
      /* fixpoint found. goto next phase */
      i = inf + 1;
    }
  }
  /* compute Y = ex(Y) for states before the bound */
  for (i = inf; i > 0; i--) {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = ex(fsm, Y);
    bdd_free(dd_manager, Y);
    Y = tmp_1;
    if (Y == oldY) {
      /* fixpoint found. */
      i = 1;
    }
  }
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>A\[f U^{inf..sup} g\]</i>.]

  Description        [Computes the set of states satisfying
                     <i>A\[f U^{inf..sup} g\]</i>.]

  SideEffects        []

  SeeAlso            [au]

******************************************************************************/
bdd_ptr abu(Fsm_BddPtr fsm, bdd_ptr f, bdd_ptr g, int inf, int sup)
{
  int i;
  bdd_ptr fair_states_bdd;
  bdd_ptr Y, oldY, tmp_1, tmp_2;
  int n = 1;

  if (inf > sup || inf < 0) return(bdd_zero(dd_manager));
  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  Y = bdd_dup(g);
  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, fair_states_bdd);
  }
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Y, reachable_states_bdd);
  }
  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "abu: computing fixed point approximations for ",
                get_the_node(), " ...\n");
  /* compute Y = g | (f & ax(Y)) for states within the bound */
  for (i = sup; i > inf; i--) {
    if (opt_verbose_level_gt(options, 1)){
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = Y;
    tmp_1 = bdd_not(dd_manager, Y);
    tmp_2 = ex(fsm, tmp_1);
    bdd_free(dd_manager, tmp_1);
    tmp_1 = bdd_not(dd_manager, tmp_2);
    bdd_free(dd_manager, tmp_2);
    tmp_2 = bdd_and(dd_manager, f, tmp_1);
    bdd_or_accumulate(dd_manager, &Y, tmp_2);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, tmp_2);
    if (Y == oldY) {
      /* fixpoint found. goto next phase */
      i = inf + 1;
    }
  }
  /* compute Y = f & ax(Y) for states before the bound */
  for (i = inf; i > 0; i--) {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Y%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Y), bdd_size(dd_manager, Y));
    }
    oldY = bdd_dup(Y);
    tmp_1 = bdd_not(dd_manager, Y);
    tmp_2 = ex(fsm, tmp_1);
    bdd_free(dd_manager, tmp_1);
    tmp_1 = bdd_not(dd_manager, tmp_2);
    bdd_free(dd_manager, tmp_2);
    bdd_free(dd_manager, Y);
    Y = bdd_and(dd_manager, f, tmp_1);
    bdd_free(dd_manager, oldY);
    bdd_free(dd_manager,tmp_1);
    
    if (Y == oldY) {
      /* fixpoint found. finish */
      i = 1;
    }
  }
  return(Y);
}

/**Function********************************************************************

  Synopsis           [Set of states satisfying <i>EF^{inf..sup}(g)</i>.]

  Description        [Computes the set of states satisfying
                     <i>EF^{inf..sup}(g)</i>.]

  SideEffects        []

  SeeAlso            [ef]

******************************************************************************/
bdd_ptr ebf(Fsm_BddPtr fsm, bdd_ptr g, int inf, int sup)
{
  bdd_ptr one, result;

  one = bdd_one(dd_manager);
  result = ebu(fsm, one, g, inf, sup);
  bdd_free(dd_manager, one);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the minimum length of the shortest path
  from <i>f</i> to <i>g</i>.]

  Description        [This function computes the minimum length of the
  shortest path from <i>f</i> to <i>g</i>.<br>
  Starts from <i>f</i> and proceeds forward until finds a state in <i>g</i>.
  Notice that this function works correctly only if <code>-f</code> 
  option is used.]

  SideEffects        []

  SeeAlso            [maxu]

******************************************************************************/
int minu(Fsm_BddPtr fsm, bdd_ptr arg_f, bdd_ptr arg_g)
{
  int i;
  int n = 1;
  bdd_ptr fair_states_bdd;
  bdd_ptr R, Rp, tmp_1, tmp_2;
  bdd_ptr f = bdd_dup(arg_f);
  bdd_ptr g = bdd_dup(arg_g);
  bdd_ptr invar_bdd = Compile_FsmBddGetInvar(fsm);
  
  R = (bdd_ptr)NULL;
  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &g, fair_states_bdd);
  }
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &f, reachable_states_bdd);
  }
  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "minu: computing fixed point approximations for ",
                get_the_node(), " ...\n");
  i = 0;
  Rp = bdd_and(dd_manager, f, invar_bdd); /* starts searching from f */
  do {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Rp%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Rp), bdd_size(dd_manager, Rp));
    }
    tmp_1 = bdd_and(dd_manager, Rp, g);
    if (bdd_isnot_zero(dd_manager, tmp_1)) {
      /* If current frontier intersects g return minimum */
      bdd_free(dd_manager, tmp_1);
      bdd_free(dd_manager, f);
      bdd_free(dd_manager, g);
      bdd_free(dd_manager, Rp);
      if (R != (bdd_ptr)NULL) bdd_free(dd_manager, R);
      return(i);
    }
    bdd_free(dd_manager, tmp_1);
    if (R != (bdd_ptr)NULL) bdd_free(dd_manager, R);
    R = Rp;
    /* go forward */
    tmp_1 = Img_ImageFwd(fsm, R);
    tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
    bdd_free(dd_manager, tmp_1);
    Rp = bdd_or(dd_manager, R, tmp_2);
    bdd_free(dd_manager, tmp_2);
    i++;
  } while ( Rp != R );
  /* could not find g anywhere. A fixpoint has been found. g will not be
     ever found, so return infinity. */
  bdd_free(dd_manager, f);
  bdd_free(dd_manager, g);
  bdd_free(dd_manager, Rp);
  bdd_free(dd_manager, R);
  return(-1);
}

/**Function********************************************************************

  Synopsis           [This function computes the maximum length of the
  shortest path from <i>f</i> to <i>g</i>.]

  Description        [This function computes the maximum length of the
  shortest path from <i>f</i> to <i>g</i>. It starts from !g and
  proceeds backward until no states in <i>f</i> can be found. In other
  words, it looks for the maximum length of <i>f->AG!g</i>.
  Notice that this function works correctly only if <code>-f</code> 
  option is used.]

  SideEffects        []

  SeeAlso            [minu]

******************************************************************************/
int maxu(Fsm_BddPtr fsm, bdd_ptr f, bdd_ptr g)
{
  int i;
  int n = 1;
  bdd_ptr fair_states_bdd;
  bdd_ptr R, Rp;
  bdd_ptr notg, tmp_1;
  bdd_ptr invar_bdd;

  invar_bdd = Compile_FsmBddGetInvar(fsm);
  fair_states_bdd = Compile_FsmBddGetFairStates(fsm);
  if (opt_verbose_level_gt(options, 1))
    indent_node(nusmv_stderr, "maxu: computing fixed point approximations for ",
                get_the_node()," ...\n");
  tmp_1 = bdd_not(dd_manager, g);
  notg = bdd_and(dd_manager, tmp_1, invar_bdd);
  bdd_free(dd_manager, tmp_1);
  i = 0;
  R = bdd_one(dd_manager);
  Rp = bdd_dup(notg); /* starts from !g */
  if (fair_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Rp, fair_states_bdd);
  }
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &Rp, reachable_states_bdd);
  }
  do {
    if (opt_verbose_level_gt(options, 1)) {
      indent(nusmv_stderr);
      fprintf(nusmv_stderr, "size of Rp%d = %g states, %d BDD nodes\n",
              n++, bdd_count_states(dd_manager, Rp), bdd_size(dd_manager, Rp));
    }
    tmp_1 = bdd_and(dd_manager, Rp, f);
    if (bdd_is_zero(dd_manager, tmp_1)) {
      /* !g does not intersect f anymore. The maximum length of a path
         completely in !g is i. This is the maximum. */
      bdd_free(dd_manager, tmp_1);
      bdd_free(dd_manager, R);
      bdd_free(dd_manager, Rp);
      bdd_free(dd_manager, notg);
      return(i);
    };
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, R);
    R = Rp;
    tmp_1 = ex(fsm, R);
    Rp = bdd_and(dd_manager, tmp_1, notg);
    bdd_free(dd_manager, tmp_1);
    i++;
  } while (R != Rp);
  /* a fixpoint has been found in which !g & f holds, so return infinity */
  bdd_free(dd_manager, R);
  bdd_free(dd_manager, Rp);
  bdd_free(dd_manager, notg);
  return(-1);
}

/**Function********************************************************************

  Synopsis           [Checks the totality of the transition relation.]

  Description        [This function checks that the transition
  relation is total. If not, the set of deadlock states is
  returned. The checking is performed by computing <em> (not(EX(T)) and
  INVAR)</em> if the result of such a computation is the set of
  deadlock states. The reachable states must be computed before. This
  is because considering EX(true) we can found states that are not
  reachable, and thus does not affect the model checking algorithm.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr trans_is_total(Fsm_BddPtr fsm)
{
  bdd_ptr states_with_succ, deadlock_states;
  bdd_ptr invar_bdd;
  bdd_ptr one;

  invar_bdd = Compile_FsmBddGetInvar(fsm);
  
  one = bdd_one(dd_manager);
  states_with_succ = ex(fsm, one);
  bdd_free(dd_manager,one);
  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, "The # of states with a successor are: %g\n",
            bdd_count_states(dd_manager, states_with_succ));

  deadlock_states = bdd_not(dd_manager, states_with_succ);
  bdd_and_accumulate(dd_manager, &deadlock_states, invar_bdd);
  if (reachable_states_bdd != (bdd_ptr)NULL) {
    bdd_and_accumulate(dd_manager, &deadlock_states, reachable_states_bdd);
  }
  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, "The # of states without a successor are: %g\n",
            bdd_count_states(dd_manager, deadlock_states));
  bdd_free(dd_manager, states_with_succ);
  return(deadlock_states);
}

/**Function********************************************************************

  Synopsis           [Prints statistical information about reachable states.]

  Description        [Prints statistical information about reachable
  states, i.e. the real number of reachable states. It is computed
  taking care of the encoding and of the indifferent variables in the
  encoding.]

  SideEffects        []

  SeeAlso            [compute_reachable_states]

******************************************************************************/
void print_reachable_states(Fsm_BddPtr fsm, int verbose)
{
  int bits_encoding_var;

  if (opt_print_reachable(options)) {
    bdd_ptr invar_bdd, reachable;
    bdd_ptr mask, tmp;
    double reached_cardinality;
    double search_space_cardinality;
    int bitvars = 0;
    node_ptr l = real_state_variables;

    mask = bdd_one(dd_manager);  
    invar_bdd = Compile_FsmBddGetInvar(fsm);

    if (reachable_states_bdd == (bdd_ptr)NULL) {
      compute_reachable_states(fsm);
    }
    while (l != Nil) {
      /*
        We use "2" as last argument of the above call, because we
        encode current and next variables as successive variables in
        the order. If we choose to change this, we need to modify
        the "make_var_mask" implementation.
      */
      add_ptr tmp_1 = make_var_mask(dd_manager, (add_ptr)car(l),
                                    &bits_encoding_var, 2);
      bdd_ptr tmp_2 = add_to_bdd(dd_manager, tmp_1);

      add_free(dd_manager, tmp_1);
      bitvars += bits_encoding_var;
      bdd_and_accumulate(dd_manager, &mask, tmp_2);
      bdd_free(dd_manager, tmp_2);
      l = cdr(l);
    }
      
    tmp = bdd_forall(dd_manager, invar_bdd, input_variables_bdd);
    reachable = bdd_and(dd_manager, reachable_states_bdd, tmp);
    bdd_free(dd_manager, tmp);

    bdd_and_accumulate(dd_manager, &reachable, mask);

    reached_cardinality = bdd_count_minterm(dd_manager, reachable, bitvars);
    search_space_cardinality = bdd_count_minterm(dd_manager, mask, bitvars);

    fprintf(nusmv_stdout, "reachable states: %g (2^%g) out of %g (2^%g)\n",
            reached_cardinality, log(reached_cardinality)/log(2.0),
            search_space_cardinality, log(search_space_cardinality)/log(2.0));

    if (verbose) print_states(reachable);    

    bdd_free(dd_manager, reachable);
  }
}

/**Function********************************************************************

  Synopsis           [Prints out a CTL specification]

  Description        [Prints out a CTL specification]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_spec(FILE *file, node_ptr n)
{
  node_ptr context = Nil;
  
  if (node_get_type(n) == CONTEXT) {
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "specification ", n, " ");
  if(context) {
    (void) fprintf(file, "(in module ");
    (void) print_node(file, context);
    (void) fprintf(file, ") ");
  }
}


/**Function********************************************************************

  Synopsis           [Prints out a COMPUTE specification]

  Description        [Prints out a COMPUTE specification]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_compute(FILE *file, node_ptr n)
{
  node_ptr context = Nil;

  if (node_get_type(n) == CONTEXT){
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "the result of ", n, " ");
  if (context) {
    fprintf(file, "(in module ");
    print_node(file, context);
    fprintf(file, ") ");
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Compute reachable states]

  Description        [  This function computes reachable states.<br>
  According to the specified options, the system can compute the
  reachable states in different ways:<br>
  <ul>
  <li> <code>reachable_states</code> is the result</li>
  <li> <code>from_lower_bound</code> is the frontier</li>
  <li> <code>from_upper_bound</code> is the previous value of
       <code>reachable_states</code></li>
  <li> <code>not_from_upper_bound</code> is the complement of
       <code>from_upper_bound</code></li> 
  </ul>
]

  SideEffects        [The global variable <code>reachable_states</code> is updated.] 

  SeeAlso            []

******************************************************************************/
static void Mc_compute_reachable_states(Fsm_BddPtr fsm)
{
  int i = 0;
  /* from_lower_bound is the frontier */
  bdd_ptr from_lower_bound, init_bdd, invar_bdd;

  init_bdd = Compile_FsmBddGetInit(fsm);
  invar_bdd = Compile_FsmBddGetInvar(fsm);
  from_lower_bound = bdd_dup(init_bdd);
  reachable_states_layers = Nil;
  reachable_states_bdd = bdd_dup(init_bdd);
  if (opt_verbose_level_gt(options, 1))
    fprintf(nusmv_stderr, "\ncomputing reachable state space...\n");
  while (bdd_isnot_zero(dd_manager, from_lower_bound)) {
    bdd_ptr from_upper_bound = bdd_dup(reachable_states_bdd);
    reachable_states_layers = cons((node_ptr)bdd_dup(reachable_states_bdd), reachable_states_layers);

    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "  iteration %d: BDD size = %d, frontier size = %d, states = %g\n",
              i, bdd_size(dd_manager, reachable_states_bdd),
              bdd_size(dd_manager, from_lower_bound),
              bdd_count_states(dd_manager, reachable_states_bdd));
    
    /* intersect the frontier with normal assignments  */
    bdd_and_accumulate(dd_manager, &from_lower_bound, invar_bdd);
    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "  invariant combined, size = %d\n",
              bdd_size(dd_manager, from_lower_bound));
    {
      bdd_ptr image = Img_ImageFwd(fsm, from_lower_bound);
      bdd_or_accumulate(dd_manager, &reachable_states_bdd, image);
      bdd_free(dd_manager, image);
    }
    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "  forward step done, size = %d\n",
              bdd_size(dd_manager, reachable_states_bdd));
    {
      bdd_ptr not_from_upper_bound = bdd_not(dd_manager, from_upper_bound);
      
      bdd_free(dd_manager, from_lower_bound);
      from_lower_bound = bdd_and(dd_manager, reachable_states_bdd, not_from_upper_bound);
      bdd_free(dd_manager, not_from_upper_bound);
			
    }
    if (opt_verbose_level_gt(options, 1))
      fprintf(nusmv_stderr, "  new frontier computed, size = %d\n",
              bdd_size(dd_manager, from_lower_bound));
    bdd_free(dd_manager, from_upper_bound);
    i++;
  }
  bdd_free(dd_manager, from_lower_bound);

  /* Flavio Lerda */
#ifndef NO_PRINT_STATES
    fprintf(nusmv_stdout, "@@@@ Reachable states\n");
    {
      bdd_ptr tmp = bdd_dup(reachable_states_bdd);
      bdd_and_accumulate(dd_manager, &tmp, invar_bdd);
      mcPrintStates(tmp);
      bdd_free(dd_manager, tmp);
    }
    fprintf(nusmv_stdout, "@@@@\n");
#endif
  /* Flavio Lerda */
}

/**Function********************************************************************

  Synopsis           [Auxiliary function to  handle <tt>CTL</tt> model
  checking under <em>FAIRNESS</em> constraints.]

  Description        [This function is used in the computation of
  <tt>EG alpha</tt> under <em>FAIRNESS</em> constraints. For each
  fairness constraint <tt>f</tt> in the list <code>fc</code>, it
  computes the set of states satisfying <tt>E \[g U f\]<tt>, then 
  intersects the corresponding partial result and the resulting set is
  returned back.]

  SideEffects        []

  SeeAlso            [eg eu]

******************************************************************************/
static bdd_ptr fair_iter(Fsm_BddPtr fsm, bdd_ptr g, node_ptr fc)
{
  if(fc == Nil) return(bdd_one(dd_manager));
  if(node_get_type(fc) == CONS){
    bdd_ptr left, right, res;

    left = fair_iter(fsm, g, car(fc));
    right = fair_iter(fsm, g, cdr(fc));
    res = bdd_and(dd_manager, left, right);
    bdd_free(dd_manager, left);
    bdd_free(dd_manager, right);
    return(res);
  }
  if(node_get_type(fc) == BDD){
    bdd_ptr tmp_1, tmp_2, res;
 
    tmp_1 = bdd_dup((bdd_ptr)car(fc));
    tmp_2 = bdd_and(dd_manager, g, tmp_1);
    res = eu(fsm, g, tmp_2);
    bdd_free(dd_manager, tmp_1);
    bdd_free(dd_manager, tmp_2);
    return(res);
  }
  internal_error("fair_iter: fc->type = %d", node_get_type(fc));
}

