/**CFile***********************************************************************

  FileName    [mcCmd.c]

  PackageName [mc]

  Synopsis    [Model checking commands.]

  Description [This file contains all the shell command to deal with
  model checking and for counterexample navigation.]

  SeeAlso     [cmdCmd.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``mc'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "mcInt.h" 
#include "error.h" /* for CATCH */

static char rcsid[] UTIL_UNUSED = "$Id: mcCmd.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/* prototypes of the command functions */
int CommandCheckTrans ARGS((int argc, char **argv));
int CommandCheckSpec ARGS((int argc, char **argv));
int CommandCheckInvar ARGS((int argc, char **argv));
int CommandCheckCompute ARGS((int argc, char **argv));
int CommandComputeReachable ARGS((int argc, char **argv));
int CommandPrintReachableStates ARGS((int argc, char **argv));
int CommandGotoState ARGS((int argc, char **argv));
int CommandPrintCurrentState ARGS((int argc, char **argv));
int CommandAssign ARGS((int argc, char **argv));
int CommandStep ARGS((int argc, char **argv));
int CommandEval ARGS((int argc, char **argv));

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [It's used to store the current state when the interactive
  model inspection feature is enabled.]

  Description [It's used to store the current state when the interactive
  model inspection feature is enabled.]

******************************************************************************/
static bdd_ptr current_state_bdd = (bdd_ptr)NULL;

bdd_ptr current_state_bdd_get(void) {
  return((current_state_exist() == true) ? bdd_dup(current_state_bdd) : (bdd_ptr)NULL);
}
void current_state_bdd_set(bdd_ptr state) {
  if (current_state_exist() == true) current_state_bdd_free();
  current_state_bdd = bdd_dup(state);
}
boolean current_state_bdd_exist(void) {
  return((current_state_bdd != (bdd_ptr)NULL) ? true : false);
}
void current_state_bdd_free(void) {
  if (current_state_bdd_exist() == true) bdd_free(dd_manager, current_state_bdd);
}
void current_state_bdd_reset(void) {
  if (current_state_bdd_exist() == true) current_state_bdd = (bdd_ptr)NULL;
}

/**Variable********************************************************************

  Synopsis    [It's used to store the current state label when the interactive
  model inspection feature is enabled.]

  Description [It's used to store the current state label when the interactive
  model inspection feature is enabled.]

******************************************************************************/
static node_ptr current_state_label = Nil;

node_ptr current_state_label_get(void) {
  return((current_state_exist() == true) ? current_state_label : (node_ptr)NULL);
}
void current_state_label_set(node_ptr label) {current_state_label = label;}
void current_state_set(bdd_ptr state, node_ptr label) {
  current_state_bdd_set(state);
  current_state_label_set(label);
}
boolean current_state_label_exist(void) {
  return((current_state_label != Nil) ? true : false);
}
boolean current_state_exist(void) {
  return(((current_state_label_exist() == true) && (current_state_bdd_exist() == true)) ? true : false);
}
void current_state_label_reset(void) {
  if (current_state_label_exist() == true) current_state_label = Nil;
}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int UsageCheckTrans ARGS((void));
static int UsageCheckSpec ARGS((void));
static int UsageCheckInvar ARGS((void));
static int UsageCheckCompute ARGS((void));
static int UsageComputeReachable ARGS((void));
static int UsagePrintReachableStates ARGS((void));
static int UsageGotoState  ARGS((void));
static int UsagePrintCurrentState ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the mc package.]

  Description        [Initializes the mc package.]

  SideEffects        []

******************************************************************************/
void Mc_Init(void){
  Cmd_CommandAdd("check_trans", CommandCheckTrans, 0);
  Cmd_CommandAdd("check_spec", CommandCheckSpec, 0);
  Cmd_CommandAdd("compute", CommandCheckCompute, 0);
  Cmd_CommandAdd("check_invar", CommandCheckInvar, 0);
  Cmd_CommandAdd("compute_reachable", CommandComputeReachable, 0);
  Cmd_CommandAdd("print_reachable_states", CommandPrintReachableStates, 0);
  Cmd_CommandAdd("goto_state", CommandGotoState, 0);
  Cmd_CommandAdd("print_current_state", CommandPrintCurrentState, 0);
}

/**Function********************************************************************

  Synopsis           [Quit the mc package]

  Description        [Quit the mc package]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Mc_End(void)
{
  if (reachable_states_bdd != (bdd_ptr)NULL) bdd_free(dd_manager, reachable_states_bdd);
  if (reachable_states_layers != Nil)  walk_dd(dd_manager, bdd_free, reachable_states_layers);
  current_state_bdd_free();
}

/**Function********************************************************************

  Synopsis           [Checks the transition relation for totality.]

  CommandName        [check_trans] 	   

  CommandSynopsis    [Checks the transition relation for totality.]  

  CommandArguments   [\[-h\] \[-m | -o output-file\]]

  CommandDescription [
  Checks if the transition relation is total. If the transition
  relation is not total then a potential deadlock state is shown out.
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command to the program
            specified by the <tt>PAGER</tt> shell variable if
            defined, else through the UNIX command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command to the file
       <tt>output-file</tt>.
  </dl>
  At the beginning reachable states are computed in order to guarantee 
  that deadlock states are actually reachable.]

  SideEffects        []

******************************************************************************/
int CommandCheckTrans(int argc, char **argv)
{
  int c;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckTrans());
    case 'o':
      if (useMore == 1) return(UsageCheckTrans());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckTrans());
      useMore = 1;
      break;
    default:  return(UsageCheckTrans());
    }
  }

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if (Img_CheckNoCone() == 0) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }
  
  if (argc != util_optind) return(UsageCheckTrans());

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  check_transition_relation(Prop_MasterGetBddFsm());
  set_forward_search(options);

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  return(0);
}

static int UsageCheckTrans()
{
  fprintf(nusmv_stderr, "usage: check_trans [-h] [-m | -o file]\n");
  fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\tPipes output through the program specified by\n");
  fprintf(nusmv_stderr, "      \t\tthe \"PAGER\" environment variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\tWrites the generated output to \"file\".\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Performs fair CTL model checking.]

  CommandName        [check_spec] 	   

  CommandSynopsis    [Performs fair CTL model checking.]  

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[-n number | -p "ctl-expr \[IN context\]"\]]  

  CommandDescription [Performs fair CTL model checking.<p>

  A <tt>ctl-expr</tt> to be checked can be specified at command line 
  using option <tt>-p</tt>. Alternatively, option <tt>-n</tt> can be used
  for checking a particular formula in the property database. If 
  neither <tt>-n</tt> nor <tt>-p</tt> are used, all the SPEC formulas in
  the database are checked.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>SPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the <tt>UNIX</tt> command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>SPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>-p "ctl-expr \[IN context\]"</tt>
       <dd> A CTL formula to be checked. <tt>context</tt> is the module 
       instance name which the variables in <tt>ctl-expr</tt> must 
       be evaluated in.       
    <dt> <tt>-n number</tt>
       <dd> Checks the CTL property with index <tt>number</tt> in the property
            database. 
  </dl><p>

  If the <tt>ag_only_search</tt> environment variable has been set, and
  the set of reachable states has been already computed, then a
  specialized algorithm to check AG formulas is used instead of the
  standard model checking algorithms.]  

  SideEffects        []

******************************************************************************/
int CommandCheckSpec(int argc, char **argv)
{
  int c;
  int prop_no = -1;
  char * formula = NIL(char);
  int status = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:n:p:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckSpec());
    case 'n':
      {
        if (formula != NIL(char)) return(UsageCheckSpec());
        if (prop_no != -1) return(UsageCheckSpec());

	prop_no = Prop_GetPropIdxFromString(util_optarg);
	if (prop_no == -1) return(1);

        break;
      }
    case 'p':
      {
        if (prop_no != -1) return(UsageCheckSpec());
        if (formula != NIL(char)) return(UsageCheckSpec());

	formula = util_strsav(util_optarg);
        break;
      }
    case 'o':
      if (useMore == 1) return(UsageCheckSpec());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckSpec());
      useMore = 1;
      break;
    default:  return(UsageCheckSpec());
    }
  }
  if (argc != util_optind) return(UsageCheckSpec());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, 
	    "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if (Img_Check() == 0) {
    (void)fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n",\
                  method2str(get_partition_method(options)));
    (void)fprintf(nusmv_stderr, \
                  "Use \t \"build_model -f -m %s\"\nto build the partitioned  transitioned relation.\n",\
                  method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
      nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }
  
  if (formula != NIL(char)) {
    prop_no = Prop_Db_PropParseAndAdd(formula, Prop_Ctl);
    if (prop_no == -1) return(1); /* error */
    CATCH {
      Prop_Db_VerifyPropIndex(prop_no);
    }
    FAIL {
      status = 1;
    }
  }
  else if (prop_no != -1) {
    if (Prop_CheckType(Prop_Db_GetNum(prop_no), Prop_Ctl) != 0) {
      status = 1;
    }
    else {
      CATCH {
	Prop_Db_VerifyPropIndex(prop_no);
      }
      FAIL {
	status = 1;
      }
    }
  }
  else {
    CATCH {
      Prop_DbVerifyAllTypeCtl();
    }
    FAIL {
      status = 1;
    }
  }

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }

  return(status);
}

static int UsageCheckSpec()
{
  fprintf(nusmv_stderr, "usage: check_spec [-h] [-m | -o file] [-n number | -p \"ctl-expr\"]\n");
  fprintf(nusmv_stderr, "   -h \t\t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\t\tPipes output through the program specified\n");
  fprintf(nusmv_stderr, "      \t\t\tby the \"PAGER\" environment variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\t\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "   -n number\t\tChecks only the SPEC with the given index number.\n");
  fprintf(nusmv_stderr, "   -p \"ctl-expr\"\tChecks only the given CTL formula.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Performs model checking of invariants]

  CommandName        [check_invar] 	   

  CommandSynopsis    [Performs model checking of invariants]  

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[-n number | -p "invar-expr \[IN context\]"\]]  

  CommandDescription [Performs invariant checking on the given
  model. An invariant is a set of states. Checking the invariant is
  the process of determining that all states reachable from the
  initial states lie in the invariant.

  Invariants to be verified can be provided as simple formulas
  (without any temporal operators) in the input file via the
  <tt>INVARSPEC</tt> keyword or directly at command line, using the option 
  <tt>-p</tt>.<p>

  Option <tt>-n</tt> can be used for checking a particular invariant
  of the model. If neither <tt>-n</tt> nor <tt>-p</tt> are
  used, all the invariants are checked.<p>

  During checking of invariant all the fairness conditions associated 
  with the model are ignored.<p>

  If an invariant does not hold, a proof of failure is demonstrated.
  This consists of a path starting from an initial state to a state
  lying outside the invariant. This path has the property that it is the
  shortest path leading to a state outside the invariant.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the program in processing
           <tt>INVARSPEC</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the UNIX command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>INVARSPEC</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>-p "invar-expr \[IN context\]"</tt>
       <dd> The command line specified invariant formula to be verified.
       <tt>context</tt> is the module instance name which the variables 
       in <tt>invar-expr</tt> must be evaluated in.
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckInvar(int argc, char **argv)
{
  int c;
  int prop_no = -1;
  char * formula = NIL(char);
  int status = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:n:p:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckInvar());
    case 'n':
      {
        if (formula != NIL(char)) return(UsageCheckInvar());
        if (prop_no != -1) return(UsageCheckInvar());

	prop_no = Prop_GetPropIdxFromString(util_optarg);
	if (prop_no == -1) return(1);

        break;
      }
    case 'p':
      {
        if (prop_no != -1) return(UsageCheckInvar());
        if (formula != NIL(char)) return(UsageCheckInvar());

	formula = util_strsav(util_optarg);
        break;
      }
    case 'o':
      if (useMore == 1) return(UsageCheckInvar());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckInvar());
      useMore = 1;
      break;
    default:  return(UsageCheckInvar());
    }
  }
  if (argc != util_optind) return(UsageCheckInvar());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, 
	    "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if (Img_Check() == 0) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model [-f] -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  if (formula != NIL(char)) {
    prop_no = Prop_Db_PropParseAndAdd(formula, Prop_Invar);
    if (prop_no == -1) return(1); /* error */
    CATCH {
      Prop_Db_VerifyPropIndex(prop_no);
    }
    FAIL {
      status = 1;
    }
  }
  else if (prop_no != -1) {
    if (Prop_CheckType(Prop_Db_GetNum(prop_no), Prop_Invar) != 0) {
      status = 1;
    }
    else {
      CATCH {
	Prop_Db_VerifyPropIndex(prop_no);
      }
      FAIL {
	status = 1;
      }
    }
  }
  else {
    CATCH {
      Prop_DbVerifyAllTypeInvar();
    }
    FAIL {
      status = 1;
    }
  }

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }

  return(status);
}

static int UsageCheckInvar()
{
  fprintf(nusmv_stderr, "usage: check_invar [-h] [-m| -o file] [-n number | -p \"invar-expr\"]\n");
  fprintf(nusmv_stderr, "   -h \t\t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\t\tPipes output through the program specified by\n");
  fprintf(nusmv_stderr, "      \t\t\tthe \"PAGER\" shell variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\t\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "   -n number\t\tchecks only the INVARSPEC with the given index number.\n");
  fprintf(nusmv_stderr, "   -p \"invar-expr\"\tchecks only the given invariant formula.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Performs computation of quantitative characteristics]

  CommandName        [compute] 	   

  CommandSynopsis    [Performs computation of quantitative characteristics]

  CommandArguments   [\[-h\] \[-m | -o output-file\] \[-n number | -p "compute-expr \[IN context\]"\]]  

  CommandDescription [This command deals with the computation of
  quantitative characteristics of real time systems. It is able to
  compute the length of the shortest (longest) path from two given set
  of states.<p>
  
  <center><code>MAX \[ alpha , beta \]</code></center><br>
  <center><code>MIN \[ alpha , beta \]</code></center><p>
  
  Properties of the above form can be specified in the input file via
  the keyword <code>COMPUTE</code> or directly at command line,
  using option <tt>-p</tt>.<p>

  Option <tt>-n</tt> can be used for computing a particular expression
  in the model. If neither <tt>-n</tt> nor <tt>-p</tt> are
  used, all the COMPUTE specifications are computed.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-m</tt>
       <dd> Pipes the output generated by the command in processing
           <tt>COMPUTE</tt>s to the program specified by the
           <tt>PAGER</tt> shell variable if defined, else
           through the UNIX command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command in processing
           <tt>COMPUTE</tt>s to the file <tt>output-file</tt>.
    <dt> <tt>-p "compute-expr \[IN context\]"</tt>
       <dd> A COMPUTE formula to be checked. <tt>context</tt> is the module 
       instance name which the variables in <tt>compute-expr</tt> must 
       be evaluated in. 
    <dt> <tt>-n number</tt>
       <dd> Computes only the property with index <tt>number</tt>
  </dl>]  

  SideEffects        []

******************************************************************************/
int CommandCheckCompute(int argc, char **argv)
{
  int c;
  int prop_no = -1;
  char * formula = NIL(char);
  int status = 0;
  int useMore = 0;
  char * dbgFileName = NIL(char);
#if HAVE_GETENV
  char * pager;
#endif
  FILE * old_nusmv_stdout = NIL(FILE);
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hmo:n:p:")) != EOF) {
    switch (c) {
    case 'h': return(UsageCheckCompute());
    case 'n':
      {
        if (formula != NIL(char)) return(UsageCheckCompute());
        if (prop_no != -1) return(UsageCheckCompute());

	prop_no = Prop_GetPropIdxFromString(util_optarg);
	if (prop_no == -1) return(1);

        break;
      }
    case 'p':
      {
        if (prop_no != -1) return(UsageCheckCompute());
        if (formula != NIL(char)) return(UsageCheckCompute());

	formula = util_strsav(util_optarg);
        break;
      }
    case 'o':
      if (useMore == 1) return(UsageCheckCompute());
      dbgFileName = util_strsav(util_optarg);
      fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageCheckCompute());
      useMore = 1;
      break;
    default:  return(UsageCheckCompute());
    }
  }
  if (argc != util_optind) return(UsageCheckCompute());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, 
	    "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }
    
  if (Img_Check() == 0) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model [-f] -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
     nusmv_stdout = popen(pager, "w"); 
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  if (formula != NIL(char)) {
    prop_no = Prop_Db_PropParseAndAdd(formula, Prop_Compute);
    if (prop_no == -1) return(1); /* error */
    CATCH {
      Prop_Db_VerifyPropIndex(prop_no);
    }
    FAIL {
      status = 1;
    }
  }
  else if (prop_no != -1) {
    if (Prop_CheckType(Prop_Db_GetNum(prop_no), Prop_Compute) != 0) {
      status = 1;
    }
    else {
      CATCH {
	Prop_Db_VerifyPropIndex(prop_no);
      }
      FAIL {
	status = 1;
      }
    }
  }
  else {
    CATCH {
      Prop_DbVerifyAllTypeCompute();
    }
    FAIL {
      status = 1;
    }
  }

  if (useMore) {
    pclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }
  if (dbgFileName != NIL(char)) {
    fflush(nusmv_stdout);
    fclose(nusmv_stdout);
    nusmv_stdout = old_nusmv_stdout;
  }

  return(status);
}

static int UsageCheckCompute()
{
  fprintf(nusmv_stderr, "usage: compute [-h] [-m | -o file] [-n number | -p \"compute-expr\"]\n");
  fprintf(nusmv_stderr, "   -h \t\t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -m \t\t\tPipes output through the program specified by\n");
  fprintf(nusmv_stderr, "      \t\t\tthe \"PAGER\" shell variable if defined,\n");
  fprintf(nusmv_stderr, "      \t\t\telse through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "   -o file\t\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "   -n number\t\tConsiders only the compute expression with the given index number.\n");
  fprintf(nusmv_stderr, "   -p \"compute-expr\"\tComputes the given expression.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Computes the set of reachable states]

  CommandName        [compute_reachable] 	   

  CommandSynopsis    [Computes the set of reachable states]  

  CommandArguments   [\[-h\]]  

  CommandDescription [Computes the set of reachable states. The result
  is then used to simplify image and preimage computations. This can
  result in improved performances for models with sparse state
  spaces. Sometimes this option may slow down the performances because
  the computation of reachable states may be very expensive. The
  environment variable <tt>forward_search</tt> is set during the execution
  of this command.]

  SideEffects        []

******************************************************************************/
int CommandComputeReachable(int argc, char **argv)
{
  int c;
  
  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return(1);
  }
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"h")) != EOF) {
    switch (c) {
    case 'h': return(UsageComputeReachable());
    default:  return(UsageComputeReachable());
    }
  }
  if (argc != util_optind) return(UsageComputeReachable());

  if (Img_CheckNoCone() == 0) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }
  
  if (reachable_states_bdd == (bdd_ptr)NULL) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Computation of reachable states...");
    }
    compute_reachable_states(Prop_MasterGetBddFsm());
    set_forward_search(options);
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "...done.\n");
    }
  }
  else {
    fprintf(nusmv_stderr, "Reachable States already computed.\n");
  }
  return(0);
}

static int UsageComputeReachable()
{
  fprintf(nusmv_stderr, "usage: compute_reachable [-h]\n");
  fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Prints the reachable states.]

  CommandName        [print_reachable_states] 	   

  CommandSynopsis    [Prints out the number of reachable states. In verbose mode,
  prints also the list of reachable states.]

  CommandArguments   [\[-h\] \[-v\]]  

  CommandDescription [Prints the number of reachable states of the
  given model. In verbose mode, prints also the list of all reachable states.
  The reachable states are computed if needed.]

  SideEffects        []

******************************************************************************/
int CommandPrintReachableStates(int argc, char **argv)
{
  int c;
  int verbose = 0;
  /*
   * Parse the command line.
  */
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "hv")) != EOF) {
    switch (c) {
    case 'h': return(UsagePrintReachableStates());
    case 'v': 
      verbose = 1;
      break;
    default:
      return(UsagePrintReachableStates());
    }
  }
  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be built before. Use the \"build_model\" command.\n");
    return(1);
  }

  if (Img_CheckNoCone() == 0) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  set_forward_search(options);
  set_print_reachable(options);
  fprintf(nusmv_stdout, "######################################################################\n");
  (void) print_reachable_states(Prop_MasterGetBddFsm(),verbose);
  fprintf(nusmv_stdout, "######################################################################\n");
  return(0);
}

static int UsagePrintReachableStates()
{
  fprintf(nusmv_stderr, "usage: print_reachable_states [-h] [-v]\n");
  fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -v \t\tPrints the list of reachable states.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Goes to a given state of a trace]

  CommandName        [goto_state] 	   

  CommandSynopsis    [Goes to a given state of a trace]  

  CommandArguments   [\[-h\] state]  

  CommandDescription [Makes <tt>state</tt> the <em>current
  state</em>. This command is used to navigate alongs traces
  produced by NuSMV. During the navigation, there is a <em>current
  state</em>, and the <em>current trace</em> is the trace the
  <em>current state</em> belongs to.]  

  SideEffects        [<em>state</em> became the current state.]

******************************************************************************/
int CommandGotoState(int argc, char **argv)
{
  int c;
  int status = 0;
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"h")) != EOF) {
    switch (c) {
    case 'h': return(UsageGotoState());
    default:  return(UsageGotoState());
    }
  }
  if (argc == 1) return(UsageGotoState());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }

  if (Img_CheckNoCone() == 0) {
    fprintf(nusmv_stderr, "The current partition method %s has not yet be computed.\n", method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  {
    node_ptr parsed_command = Nil;
    
    argv += util_optind-1;
    argc -= util_optind-1;

    if (Parser_ReadCmdFromString(argc, argv, "GOTO ", ";\n", &parsed_command) == 0) {
      if ((parsed_command != Nil) && (node_get_type(parsed_command) == GOTO)) {
        node_ptr label = car(parsed_command);
        bdd_ptr state   = (bdd_ptr)lookup_traces_hash(label);
          
        if (state) {
          if (current_state_bdd != (bdd_ptr)NULL)
            bdd_free(dd_manager, current_state_bdd);
	  { /* Copying the source trace */
	    int from_trace = trace_label_get_trace(label);
	    int from_state = trace_label_get_state(label);
	    int i;
	    for (i = 1; i <= from_state; i++)
	    {
	      node_ptr from_label = trace_mk_statelabel(from_trace, i);
	      current_state_label = trace_get_new_statelabel(i);
	      current_state_bdd = (bdd_ptr)(lookup_traces_hash(from_label));
	      nusmv_assert(state);
	      insert_traces_hash(current_state_label, 
				 (node_ptr)bdd_dup(current_state_bdd));
	    }
	  }
	  /* Dup of current_state_bdd (this variable is global) */
	  bdd_dup(current_state_bdd); 
          fprintf(nusmv_stdout, "The starting state for new trace is:\n");
          fprintf(nusmv_stdout, "-> State ");
          print_node(nusmv_stdout, current_state_label);
          fprintf(nusmv_stdout, " <-\n");
          clear_print_hash();
	  set_indent_size(2);
	  print_state(state, all_symbols, 1);
	  reset_indent_size();
          inc_trace_number();
	}
        else {
          indent_node(nusmv_stderr, "The label ", label, " is undefined.\n");
        }
      }
      else {
        fprintf(nusmv_stderr, "Parsing error: expected \"goto_state state\".\n");
        status = 1;
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: expected \"goto_state state\".\n");
      status = 1;
    }
  }
  return(status);
}

static int UsageGotoState()
{
  fprintf(nusmv_stderr, "usage: goto_state [-h] state\n");
  fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   state \tSets current state to \"state\".\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Prints the current state]

  CommandName        [print_current_state] 	   

  CommandSynopsis    [Prints out the current state]  

  CommandArguments   [\[-h\] \[-v\]]  

  CommandDescription [Prints the name of the <em>current state</em> if
  defined.<p>

  Command options:<p>
  <dl>
    <dt> <tt>-v</tt>
       <dd> Prints the value of all the state variables of the <em>current 
       state</em>.
  </dl>
  ]  

  SideEffects        []

******************************************************************************/
int CommandPrintCurrentState(int argc, char ** argv)
{
  int c;
  int Verbosely = 1;
  
  util_getopt_reset();
  while ((c = util_getopt(argc,argv,"hv")) != EOF) {
    switch (c) {
    case 'h': return(UsagePrintCurrentState());
    case 'v': {
      Verbosely = 0;
      break;
    }
    default:  return(UsagePrintCurrentState());
    }
  }
  if (argc != util_optind) return(UsagePrintCurrentState());
  if ((current_state_label != Nil) && (current_state_bdd != (bdd_ptr)NULL)) {
    fprintf(nusmv_stdout, "Current state is ");
    print_node(nusmv_stdout, current_state_label);
    fprintf(nusmv_stdout, "\n");
    if (Verbosely == 0) print_state(current_state_bdd, all_symbols, Verbosely);
  }
  else {
    if (get_trace_number() > 0) {
      fprintf(nusmv_stdout, "The current state has not yet been defined.\n");
      fprintf(nusmv_stdout, "Use \"goto_state\" to define the current state.\n");
    }
    else {
      fprintf(nusmv_stdout, "There is no trace actually stored in the system.\n");
      fprintf(nusmv_stdout, "Use \"pick_state\" to define the current state.\n");
    }
    return(1);
  }
  return(0);
}

static int UsagePrintCurrentState()
{
  fprintf(nusmv_stderr, "usage: print_current_state [-h] [-v]\n");
  fprintf(nusmv_stderr, "   -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "   -v \t\tPrints the value of each state variable in the current state.\n");
  return(1);
}
