/**CFile***********************************************************************

  FileName    [mcAGonly.c]

  PackageName [mc]

  Synopsis    [This file contains the code to deal with AG formulas in a special way.]

  Description [This file contains the code to deal with AG formulas
  only, using special purpose algorithms. This functionality is invoked
  with the -AG option and works only in conjunction with the -f
  (forward search) option.]

  Comments [None]

  SeeAlso     [mcMc.c mcEval.c mcExplain.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``mc'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "mcInt.h" 
#include "mc.h"

static char rcsid[] UTIL_UNUSED = "$Id: mcAGonly.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static node_ptr cont_AG_counterexample ARGS((Fsm_BddPtr, node_ptr, node_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [This function checks for SPEC of the form AG
  alpha in "context".]

  Description [The implicit assumption is that "spec" must be an AG
  formula (i.e. it must contain only conjunctions and AG's).  No attempt
  is done to normalize the formula (e.g. push negations). The AG mode
  relies on the previous computation and storage of the reachable
  state space (<tt>reachable_states_layers</tt>), they are used in
  counterexample computation.] 

  SideEffects        []

  SeeAlso            [check_spec]

******************************************************************************/
void Mc_CheckAGOnlySpec(Prop_Ptr prop) {
  int status;
  node_ptr  spec = Prop_GetProp(prop);
  Fsm_BddPtr fsm = Prop_GetBddFsm(prop);

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_spec(nusmv_stderr, spec);
    fprintf(nusmv_stderr, "\n");
  }

  if (fsm == (Fsm_BddPtr)NULL) {
    if (opt_cone_of_influence(options) == true) {
      add_ptr one = add_one(dd_manager);
      node_ptr jc = cmp_struct_get_justice(cmps);
      node_ptr cc = cmp_struct_get_compassion(cmps);      
      Set_t spec_dep = Formulae_GetDependencies(spec, jc, cc);
      Set_t cone = ComputeCOI(spec_dep);

      Prop_BuildFsm(prop, Set_Set2List(cone), jc, cc, one);
      Prop_SetCone(prop, cone);
      fsm = Prop_GetBddFsm(prop);
      add_free(dd_manager, one);
    }
    else {
      Prop_SetFsmToMaster(prop);
      fsm = Prop_GetBddFsm(prop);
    }
  }

  status = check_AG_only(fsm, spec, Nil);

  if (status == 1) {
    /* The property holds */
    Prop_SetStatus(prop, Prop_True);
  }
  else {
    /* The property holds */
    Prop_SetStatus(prop, Prop_False);
    /* The trace number is the previous one, since get_trace_number
       returns the trace number for further traces. */
    Prop_SetTrace(prop, get_trace_number()-1);
  }
}

/**Function********************************************************************

  Synopsis           [This function checks for SPEC of the form AG
  alpha in "context".]

  Description [The implicit assumption is that "spec" must be an AG
  formula (i.e. it must contain only conjunctions and AG's).  No attempt
  is done to normalize the formula (e.g. push negations). The AG mode
  relies on the previous computation and storage of the reachable
  state space (<tt>reachable_states_layers</tt>), they are used in
  counterexample computation.] 

  SideEffects        []

  SeeAlso            [check_spec]

******************************************************************************/
int check_AG_only(Fsm_BddPtr fsm, node_ptr spec, node_ptr context)
{
  if (spec == Nil) return(1);
  switch(node_get_type(spec)) {
  case CONTEXT:
    return(check_AG_only(fsm, cdr(spec),car(spec)));
  case AND:
    return(check_AG_only(fsm, car(spec), context) &
           check_AG_only(fsm, cdr(spec), context));
  case AG:
    {
      bdd_ptr tmp_1, tmp_2, acc;
      bdd_ptr init_bdd, invar_bdd;
      bdd_ptr s0 = eval_spec(fsm, car(spec),context);

      { /*
          We quantify out the input variables from the result of the
          formula evaluation.
        */
        bdd_ptr tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);
        
        bdd_free(dd_manager, s0);
        s0 = tmp;
      }
      init_bdd = Compile_FsmBddGetInit(fsm);
      invar_bdd = Compile_FsmBddGetInvar(fsm);
      tmp_1 = bdd_not(dd_manager, s0);
      tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
      bdd_free(dd_manager, tmp_1);
      acc = bdd_and(dd_manager, reachable_states_bdd, tmp_2);
      bdd_free(dd_manager, tmp_2);
      fprintf(nusmv_stdout, "-- ");
      print_spec(nusmv_stdout, spec);
      if (bdd_is_zero(dd_manager, acc)){
        fprintf(nusmv_stdout, "is true\n");
        bdd_free(dd_manager, s0);
        bdd_free(dd_manager, acc);
        return(1);
      }
      {
	node_ptr l;

        fprintf(nusmv_stdout, "is false\n");        
        l = make_AG_counterexample(fsm, reachable_states_layers, acc);
        bdd_free(dd_manager, acc);
        bdd_free(dd_manager, s0);
	print_explanation(l, all_variables);
	return(0);
      }
    }
  default:
    if (opt_verbose_level_gt(options, 0)) {    
      fprintf(nusmv_stdout, "*** WARNING - ");
      print_spec(nusmv_stdout, spec);
      fprintf(nusmv_stdout, "skipped: it is not an AG-only formula\n");
    }
    return(0);
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [This function constructs a counterexample
  starting from state s0.]

  Description        [Compute a counterexample starting from a given state.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr make_AG_counterexample(Fsm_BddPtr fsm, node_ptr l, bdd_ptr s0)
{
  node_ptr p;
  bdd_ptr s, tmp_1;

  if (l == Nil) return(Nil);
  p = make_AG_counterexample(fsm, cdr(l), s0);
  if (p != Nil) return(p);
  tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(l), s0);
  s = bdd_pick_one_state(dd_manager, tmp_1);
  bdd_free(dd_manager, tmp_1);
  if (bdd_is_zero(dd_manager, s)) {
    bdd_free(dd_manager, s);
    return(Nil);
  }
  return(cont_AG_counterexample(fsm, cdr(l), cons((node_ptr)s, Nil)));
}

/**Function********************************************************************

  Synopsis           [Auxiliary function to compute a counterexample.]

  Description        [Auxiliary function to compute a counterexample.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr cont_AG_counterexample(Fsm_BddPtr fsm, node_ptr l, node_ptr p)
{
  bdd_ptr s, tmp_1, tmp_2;
  bdd_ptr invar_bdd;

  invar_bdd = Compile_FsmBddGetInvar(fsm);
  if (l == Nil) return(p);
  tmp_1 = Img_ImageBwd(fsm, (bdd_ptr)car(p));
  tmp_2 = bdd_and(dd_manager, invar_bdd, tmp_1);
  bdd_free(dd_manager, tmp_1);
  tmp_1 = bdd_and(dd_manager, (bdd_ptr)car(l), tmp_2);
  bdd_free(dd_manager, tmp_2);
  s = bdd_pick_one_state(dd_manager, tmp_1);
  bdd_free(dd_manager, tmp_1);
  if (bdd_is_zero(dd_manager, s))
    internal_error("make_AG_counterexample: s == ZERO");
  return(cont_AG_counterexample(fsm, cdr(l),cons((node_ptr)s,p)));
}

