/**CFile***********************************************************************

  FileName    [ltlCompassion.c]

  PackageName [ltl]

  Synopsis    [Routines to perform strongly fair LTL model checking]

  Description [The technique adopted has been taken from [1].
  <ol>
    <li>O. Y. Kesten, A. Pnueli, and L. Raviv. Algorithmic
        Verification of Linear Temporal Logic Specifications. In
        K.G. Larsen, S. Skyum, and G. Winskel, editors,

        Proceedings of the 25th International Colloquium on Automata,
        Languages, and Programming (ICALP 1998), volume 1443 of
        Lecture Notes in Computer Science, pages
        1-16. Springer-Verlag, 1998.
    </li>
  </ol>]

  SeeAlso     [mc]

  Author      [Rik Eshuis]

  Copyright   [
  This file is part of the ``ltl'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "ltlInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: ltlCompassion.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static bdd_ptr successor ARGS((bdd_ptr from, bdd_ptr relation));
static bdd_ptr successors ARGS((bdd_ptr from, bdd_ptr relation));
static bdd_ptr predecessor ARGS((bdd_ptr from, bdd_ptr relation));
static bdd_ptr predecessors ARGS((bdd_ptr from, bdd_ptr relation));
static node_ptr path ARGS((bdd_ptr source, bdd_ptr dest, bdd_ptr R));
static bdd_ptr GetTrans ARGS((Fsm_BddPtr fsm));

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Check for feasability]

  Description        [Checks whether the model has a fair path and returns
  the initial state of the path.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr feasible(Fsm_BddPtr fsm)
{
  bdd_ptr new, old, R, tmp_1, tmp_2, init_bdd, trans_bdd, invar_bdd, next_invar_bdd;
  node_ptr justice_bdd_expr, compassion_bdd_expr;
  node_ptr exp;

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "checking feasability...");
    fprintf(nusmv_stderr, "\n");
  }
 
  init_bdd=Compile_FsmBddGetInit(fsm);
  trans_bdd=GetTrans(fsm);
  invar_bdd=Compile_FsmBddGetInvar(fsm);
  next_invar_bdd = bdd_shift_forward(dd_manager, invar_bdd);

  bdd_and_accumulate(dd_manager, &init_bdd, invar_bdd);
  bdd_and_accumulate(dd_manager, &trans_bdd, invar_bdd);
  bdd_and_accumulate(dd_manager, &trans_bdd, next_invar_bdd);

  justice_bdd_expr= Compile_FsmBddGetJustice(fsm);
  compassion_bdd_expr = Compile_FsmBddGetCompassion(fsm);
  
  old = bdd_zero(dd_manager);
  R = bdd_dup (trans_bdd);
  new = successors(init_bdd,R);
  while (new != old) {
    bdd_free (dd_manager, old);
    old =bdd_dup(new);    
    {
      node_ptr l = justice_bdd_expr;
      while(l) { /* Loop over specifications */
	bdd_ptr spec;
	spec = bdd_dup((bdd_ptr) car(car(l)));
	l = cdr(l);
	if (opt_verbose_level_gt(options, 0)) {
	  fprintf(nusmv_stderr, "evaluating justice constraint");
	  print_sexp(nusmv_stderr,car(l));
	  fprintf(nusmv_stderr, "\n");
	}
	{
	  bdd_ptr result;	  
	  bdd_and_accumulate (dd_manager, &spec, new);
	  result = successors(spec,R);
	  bdd_free (dd_manager,spec);
	  bdd_free (dd_manager,new);
	  new = bdd_dup (result);
	  bdd_free (dd_manager,result);
	}
	{
	  bdd_ptr nextnew;
	  nextnew = bdd_shift_forward(dd_manager,new);
	  bdd_and_accumulate(dd_manager,&R,nextnew);
	  bdd_free(dd_manager,nextnew);
	}
      }    
    }
    {
      node_ptr l = compassion_bdd_expr;
      while(l) { /* Loop over specifications */
	bdd_ptr p_spec;
	bdd_ptr q_spec;
	p_spec  = bdd_dup((bdd_ptr)car(car(car(l))));	
	q_spec  = bdd_dup((bdd_ptr)car(cdr(car(l)))); 
	l = cdr(l);  
	if (opt_verbose_level_gt(options, 0)) {
	  fprintf(nusmv_stderr, "evaluating compassion constraint");
	  fprintf(nusmv_stderr, "\n");
	}
	{
	  bdd_ptr result1,result2, temp;
	  
	  temp=bdd_not(dd_manager,p_spec);
	  result1= bdd_and(dd_manager, new, temp);
	  bdd_free(dd_manager,temp);
	  bdd_and_accumulate (dd_manager, &q_spec, new);
	  result2 = successors(q_spec,R);
	  bdd_free (dd_manager,p_spec);
	  bdd_free (dd_manager,q_spec);
	  bdd_free (dd_manager,new); 	    
	  new = bdd_or (dd_manager,result1, result2);  	    
	  bdd_free (dd_manager,result1);
	  bdd_free (dd_manager,result2);
	  {
	    bdd_ptr nextnew;
	    nextnew = bdd_shift_forward(dd_manager,new);
	    bdd_and_accumulate(dd_manager,&R,nextnew);
	    bdd_free(dd_manager,nextnew);
	    
	  }
	  
	}
      }    
    }
    {
      bdd_ptr temp, succ;
      succ = successor(new,R);
      temp = bdd_and (dd_manager,new, succ);
      while (new != temp){
	bdd_free (dd_manager, new);
	new = bdd_dup (temp);
	bdd_free (dd_manager,temp);
	bdd_free (dd_manager,succ);
	succ = successor(new,R);
	temp = bdd_and (dd_manager,new, succ);
      }
      bdd_free (dd_manager,temp);
      bdd_free (dd_manager,succ);
    }
 }
 return new;  
}


/**Function********************************************************************

  Synopsis           [Compute a withness of feasability]

  Description        [Computes fair path from one of the states 
  passed as parameter.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr witness(Fsm_BddPtr fsm, bdd_ptr feasib){
  bdd_ptr final, R, state, zero, init_bdd, trans_bdd, invar_bdd, next_invar_bdd;
  node_ptr prefix, period;
  node_ptr justice_bdd_expr, compassion_bdd_expr;

  init_bdd=Compile_FsmBddGetInit(fsm);
  trans_bdd=GetTrans(fsm);
  invar_bdd=Compile_FsmBddGetInvar(fsm);
  next_invar_bdd = bdd_shift_forward(dd_manager, invar_bdd);

  bdd_and_accumulate(dd_manager, &init_bdd, invar_bdd);
  bdd_and_accumulate(dd_manager, &trans_bdd, invar_bdd);
  bdd_and_accumulate(dd_manager, &trans_bdd, next_invar_bdd);

  justice_bdd_expr= Compile_FsmBddGetJustice(fsm);
  compassion_bdd_expr = Compile_FsmBddGetCompassion(Prop_MasterGetBddFsm());

  final = bdd_dup(feasib);
  R= bdd_and(dd_manager,trans_bdd,final);
  state = bdd_pick_one_state_rand(dd_manager,final);
  zero = bdd_zero(dd_manager);
  {
    bdd_ptr succ, prec, not, diff;
    succ = successors (state,R);
    prec = predecessors (R,state);
    not = bdd_not(dd_manager,succ); 
    diff = bdd_and(dd_manager,prec,not);
    while (diff != zero){
      bdd_free (dd_manager, state);
      state = bdd_pick_one_state_rand(dd_manager,diff);
      bdd_free (dd_manager, succ);
      bdd_free (dd_manager, prec);
      bdd_free (dd_manager, diff);
      bdd_free(dd_manager,not);
      succ  = successors(state,R); 
      prec  = predecessors(R,state);
      not = bdd_not(dd_manager,succ);
      diff = bdd_and(dd_manager,prec,not);	
    } 
  
    bdd_free(dd_manager,diff);
    bdd_free(dd_manager,final);
    final = bdd_and(dd_manager,succ,prec);
    bdd_free(dd_manager,not);
    bdd_free(dd_manager,succ);
    bdd_free(dd_manager,prec);
    bdd_and_accumulate(dd_manager, &R, final);
    {
      bdd_ptr next_final;
      next_final = bdd_shift_forward (dd_manager,final);
      bdd_and_accumulate(dd_manager, &R, next_final);
    }
  }
  
  prefix = reverse(path(init_bdd,final,trans_bdd));
  period = cons(last(prefix),Nil);
  
  {
    node_ptr l = justice_bdd_expr;
    
    while(l) { /* Loop over specifications */
      bdd_ptr spec;
      spec = bdd_dup((bdd_ptr)car(car(l)));
      l = cdr(l);
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "evaluating ");
	fprintf(nusmv_stderr, "\n");
      }
      {
	bdd_ptr result;
	node_ptr curr_period;
	int found = 0;
	curr_period=period;
	while (! found) {
	  found = (bdd_and(dd_manager,spec,(bdd_ptr)car(curr_period)) != zero);
	  curr_period=cdr(curr_period);
	  if (curr_period==Nil) break;
	}
	if (!found){
	  period = append(period,
			  reverse(path((bdd_ptr)last(period),
				       bdd_and(dd_manager,final,spec),R)));
	}

      }
      bdd_free(dd_manager,spec);
    }    
  }
  {
    node_ptr l = compassion_bdd_expr;

    while(l) { /* Loop over specifications */
      bdd_ptr p_spec;
      bdd_ptr q_spec;
      p_spec  = bdd_dup((bdd_ptr)car(car(car(l))));   
      q_spec  = bdd_dup((bdd_ptr)car(cdr(car(l)))); 
      l = cdr(l);
      {
	bdd_ptr result;
	node_ptr curr_period;
	int found = 0;
	curr_period=period;

	while (! found) {
	  found = (bdd_and(dd_manager,q_spec,
			   (bdd_ptr)car(curr_period)) != zero);
	  curr_period=cdr(curr_period);
	  if (curr_period==Nil) break;
	}
	if (!found){
	  if (bdd_and(dd_manager,final,p_spec)!=zero){
	    period = append(period,
			    reverse(path((bdd_ptr)last(period),
					 bdd_and(dd_manager,final,q_spec),R)));
	  }
	}
      }
      bdd_free(dd_manager,p_spec);  
      bdd_free(dd_manager,q_spec);
    }    
  }

  {
    node_ptr temp;
    temp = reverse(path((bdd_ptr)last(period),(bdd_ptr)last(prefix),R));
    temp = cdr(temp);
    period =append(period,temp);
  }

  prefix = append(prefix,cdr(period));
  return prefix;
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function*******************************************************************

  Synopsis     [Compute the direct successor of a state]

  Description [Given a state from and transition relation, compute the
  direct successor state.]

  SideEffects        []

******************************************************************************/
static bdd_ptr successor(bdd_ptr from, bdd_ptr relation)
{
  bdd_ptr result, next_result, vars, temp;
  vars = bdd_dup(state_variables_bdd);
  next_result = bdd_and_abstract(dd_manager, relation, from, vars);
  result = bdd_shift_backward(dd_manager, next_result);
  bdd_free(dd_manager, next_result);
  bdd_free(dd_manager,vars);
  return(result); 
}

/**Function*******************************************************************

  Synopsis     [Compute the direct and indirect successors of a state]

  Description [Given a state from and transition relation, compute the
  direct and indirect successor states (transitive closure of
  successor).]

  SideEffects        []

******************************************************************************/
static bdd_ptr successors(bdd_ptr from, bdd_ptr relation)
{
  bdd_ptr old, new;
  old = bdd_zero(dd_manager);
  new = bdd_dup (from);
  while (old != new){
    bdd_ptr image; 
    bdd_free (dd_manager,old);
    old = bdd_dup (new);
    image  = successor(old,relation);
    bdd_or_accumulate(dd_manager, &new, image);
    bdd_free (dd_manager,image);
  }
  bdd_free(dd_manager, old);
  return(new); 
}

/**Function*******************************************************************

  Synopsis     [Compute the direct predecessor of a state]

  Description [Given a state to and a transition relation, compute the
  direct predecessor state.]

  SideEffects        []

******************************************************************************/
static bdd_ptr predecessor(bdd_ptr relation, bdd_ptr to)
{
  bdd_ptr result, next_to, vars;
  next_to = bdd_shift_forward(dd_manager, to);
  vars = bdd_dup(next_state_variables_bdd);
  result = bdd_and_abstract(dd_manager, relation, next_to, vars);
  bdd_free(dd_manager, next_to);
  bdd_free(dd_manager, vars);
  return(result); 
}

/**Function*******************************************************************

  Synopsis     [Compute the direct and indirect predecessors of a state]

  Description [Given a state to and a transition relation, compute the
  direct and indirect predecessor states (transitive closure of
  predecessor).]

  SideEffects        []

******************************************************************************/
static bdd_ptr predecessors(bdd_ptr relation, bdd_ptr to)
{
  bdd_ptr old, new;
  old = bdd_zero(dd_manager);
  new = bdd_dup (to);
  while (old != new){
    bdd_ptr bwdimage;
    bdd_free (dd_manager,old);
    old = bdd_dup (new);
    bwdimage = predecessor(relation,old);
    bdd_or_accumulate(dd_manager, &new, bwdimage);
    bdd_free (dd_manager,bwdimage);
  }
  bdd_free(dd_manager, old);
  return(new); 
}

/**Function********************************************************************

  Synopsis           [Compute a path from source to destination]

  Description        [Computes a path given the bdds representind the source
  states, the target states, and the transition relation.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr path(bdd_ptr source, bdd_ptr dest, bdd_ptr R){
  bdd_ptr start,zero,test;
  node_ptr L;
  bdd_ptr f,s;

  L= Nil;
  zero = bdd_zero(dd_manager);
  start=bdd_dup(source);
  f=predecessor(R,dest);
  while (bdd_and(dd_manager,start,f)==zero){
    bdd_ptr fold;
    fold=f;
    f=predecessor(R,fold);
    bdd_free(dd_manager,fold);
  }
  { 
    bdd_ptr temp;
    temp = bdd_and(dd_manager,start,f);
    s = bdd_pick_one_state_rand(dd_manager,temp);     
    bdd_free(dd_manager,temp);     
  }
  bdd_ref(s);
  L= cons((node_ptr)s,L);
  
  bdd_free(dd_manager,start);
  start= successor(s,R);
  test = bdd_and(dd_manager,start,dest);
  bdd_free(dd_manager,f);
  test = bdd_and(dd_manager,start,dest);
  while (test == zero){
    f=predecessor(R,dest);
    while (bdd_and(dd_manager,start,f)==zero){
      bdd_ptr fold;
      fold=f;
      f=predecessor(R,fold);
      bdd_free(dd_manager,fold);
    }     
    { 
      bdd_ptr temp;
      temp = bdd_and(dd_manager,start,f);
      s = bdd_pick_one_state_rand(dd_manager,temp);
      bdd_free(dd_manager,temp);      
    }     
    bdd_ref(s);
    L= cons((node_ptr)s,L);
    bdd_free(dd_manager,start);
    start= successor(s,R);
    bdd_free(dd_manager,test);
    test = bdd_and(dd_manager,start,dest);
    bdd_free(dd_manager,f);
    
  }
  
  {
    bdd_ptr last;
    last = bdd_pick_one_state_rand(dd_manager,bdd_and(dd_manager,start,dest));
    bdd_ref(last);
    L = cons((node_ptr)last,L);
    bdd_free(dd_manager,last);
    bdd_free(dd_manager,zero);
    return L;
  }
}

/**Function********************************************************************

  Synopsis           [Get the monolothic version of a transition relation]

  Description        [Gets the bdd corresponding to the monolothic
  version of the transition relation of a given fsm.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static bdd_ptr GetTrans(Fsm_BddPtr fsm){
  Partition_Method part;
  CPTrans_Ptr cp_trans = NULL; /* just to be sure we capture ecceptions */
  CPList cp_fwd;

  /* Update the transition relation */
  part = get_partition_method(options);  
  switch(part) {
  case Monolithic: 
    cp_trans = Compile_FsmBddGetMonoTrans(fsm);
    break;
  case Threshold:
    cp_trans = Compile_FsmBddGetThreshold(fsm);
    break;
  case Iwls95CP:
    cp_trans = Compile_FsmBddGetIwls95CP(fsm);
    break;
  default: 
    rpterr("ltlPropAddTableau: unknown partition method\n");
  }

  nusmv_assert(cp_trans != (CPTrans_Ptr)NULL);

  cp_fwd = CPTransGetForward(cp_trans);
  return CPListBuildMonolithicBDDFromCPlist(cp_fwd);
}

