/**CFile***********************************************************************

  FileName    [ltl.c]

  PackageName [ltl]

  Synopsis    [Routines to perform reduction of LTL model checking to
  CTL model checking.]

  Description [Here we perform the reduction of LTL model checking to
  CTL model checking. The technique adopted has been taken from [1].
  <ol>
    <li>O. Grumberg E. Clarke and K. Hamaguchi. "Another Look at LTL Model Checking".
       <em>Formal Methods in System Design</em>, 10(1):57--71, February 1997.</li>
  </ol>]

  SeeAlso     [mc]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``ltl'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "ltlInt.h" 
#include "error.h" /* for CATCH */

static char rcsid[] UTIL_UNUSED = "$Id: ltl.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/
/**Macro***********************************************************************

  Synopsis     [The LTL to NuSMV translator.]

  Description  [The name of the external program that performs
  translation of a LTL formula to the NuSMV code that represents the
  corresponding tableau.]

  SideEffects []

******************************************************************************/
#define LTL_TRANSLATOR "ltl2smv"

/**Macro***********************************************************************

  Synopsis     [The base module name declared by the <tt>LTL_TRANSLATOR</tt>.]

  Description  [The base module name declared by the <tt>LTL_TRANSLATOR</tt>.]

******************************************************************************/
#define LTL_MODULE_BASE_NAME "ltl_spec_"

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The counter of the LTL SPEC.]

  Description [The counter of LTL SPEC. It is used as seed to generate
  variables and model name by the <tt>LTL_TRANSLATOR</tt>.]

******************************************************************************/
static int ltl_spec_counter = 0;
static void inc_ltl_spec_counter() 
{
  ltl_spec_counter++;
}
static int ltl_spec_counter_get() 
{
  return(ltl_spec_counter);
}

/**Variable********************************************************************

  Synopsis    [Variables to dump the state]

  Description [These variables are used by <tt>ltlPushStatus</tt>
  and <tt>ltlPopStatus</tt> to save the system state.]

  SeeAlso     [ltlPushStatus ltlPopStatus]

******************************************************************************/
static node_ptr old_state_variables = Nil;
static node_ptr old_all_variables = Nil;
static node_ptr old_all_symbols = Nil;
static node_ptr old_input_variables = Nil;
static node_ptr old_real_state_variables = Nil;
static node_ptr old_boolean_variable_list = Nil;

static bdd_ptr old_input_variables_bdd = (bdd_ptr)NULL;
static bdd_ptr old_next_input_variables_bdd = (bdd_ptr)NULL;
static bdd_ptr old_state_variables_bdd = (bdd_ptr)NULL;
static bdd_ptr old_next_state_variables_bdd = (bdd_ptr)NULL;

static add_ptr old_input_variables_add = (add_ptr)NULL;
static add_ptr old_next_input_variables_add = (add_ptr)NULL;
static add_ptr old_state_variables_add = (add_ptr)NULL;
static add_ptr old_next_state_variables_add = (add_ptr)NULL;

static int old_minterm_vars_dim;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static char * ltlMakeTempFileName ARGS((void));
static void ltlInsertModuleHashReadModule ARGS((node_ptr));
static void ltlPushStatus ARGS((void));
static void ltlPopStatus ARGS((void));
static void ltlPropAddTableau ARGS((Prop_Ptr, node_ptr, node_ptr, node_ptr, node_ptr, node_ptr));
static int ltlBuildTableauAndPropFsm ARGS((Prop_Ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis    [The main routine to perform LTL model checking.]

  Description [The main routine to perform LTL model checking. It
  first takes the LTL formula, prints it in a file. It calls the LTL2SMV
  translator on it an reads in the generated tableau. The tableau is
  instantiated, compiled and then conjoined with the original model
  (both the set of fairness conditions and the transition relation are
  affected by this operation, for this reason we save the current
  model, and after the verification of the property we restore the
  original one).]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void Ltl_CheckLtlSpec(Prop_Ptr prop) {
  Fsm_BddPtr fsm;
  node_ptr spec;
  bdd_ptr s0;
  int tableau_result;
  int full_fairness = 0;
  node_ptr spec_formula = Nil;
  
  spec = Prop_GetProp(prop);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "evaluating ");
    print_ltlspec(nusmv_stderr, spec);
    fprintf(nusmv_stderr, "\n");
  }

  fsm = Prop_GetBddFsm(prop);
  /* We need to extract the context from the formula to propertly
     contextualize the tableau */

  inc_ltl_spec_counter();
  
  if (fsm == (Fsm_BddPtr)NULL) {
    if (opt_cone_of_influence(options) == true) {
      add_ptr one = add_one(dd_manager);
      node_ptr jc = cmp_struct_get_justice(cmps);
      node_ptr cc = cmp_struct_get_compassion(cmps);
      Set_t spec_dep = Formulae_GetDependencies(spec, jc, cc);
      Set_t cone = ComputeCOI(spec_dep);

      Prop_BuildFsm(prop, Set_Set2List(cone), jc, cc, one);
      Prop_SetCone(prop, cone);
      add_free(dd_manager, one);
    }
    else {
      Prop_SetFsmToMaster(prop);
    }
  }

  ltlPushStatus();

  CATCH {
    tableau_result = ltlBuildTableauAndPropFsm(prop);
  } FAIL {
    ltlPopStatus();
    nusmv_exit(1);
  }

  if (tableau_result) {
    ltlPopStatus();
    fprintf(nusmv_stderr, 
	    "Ltl_CheckLtlSpec: Problems in Tableau generation.\n");
    nusmv_exit(1);
  }

  fsm = Prop_GetBddFsm(prop);

  full_fairness = (Compile_FsmBddGetCompassion(fsm) != Nil);

  if (full_fairness) {
    bdd_ptr tmp;
    s0= feasible(fsm);   
    tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);
    bdd_free(dd_manager, s0);
    s0 = tmp;
  }
  else { /* Old Clarke/Grumberg/Hamaguchi algorithm */
    bdd_ptr tmp;
    spec_formula = 
      new_node(NOT,
	       new_node(EG,
			new_node(TRUEEXP,Nil,Nil), Nil), Nil);

    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "Checking CTL ");
      print_spec(nusmv_stderr, spec_formula);
      fprintf(nusmv_stderr, " generated from the tableau.\n");
    }

    CATCH {
      s0 = eval_spec(fsm, spec_formula, Nil);
    } FAIL {
      ltlPopStatus();
      fprintf(nusmv_stderr, 
	      "Ltl_CheckLtlSpec: Problems in Tableau verification.\n");
      return;
    }

    /* Quantify out the input variables from the result of the
       formula evaluation.  */
    tmp = bdd_forsome(dd_manager, s0, input_variables_bdd);
    bdd_free(dd_manager, s0);
    /* Negate the result */
    s0 = bdd_not(dd_manager,tmp);
    bdd_free(dd_manager, tmp);
    /* Intersect with init and invar states */
    bdd_and_accumulate(dd_manager,&s0,Compile_FsmBddGetInit(fsm));
    bdd_and_accumulate(dd_manager,&s0,Compile_FsmBddGetInvar(fsm));
  }
  /* Prints out the result, if not true explain. */
  fprintf(nusmv_stdout, "-- ");
  print_spec(nusmv_stdout, spec);
  
  if (bdd_is_zero(dd_manager, s0)) { 
    fprintf(nusmv_stdout, "is true\n");
    Prop_SetStatus(prop, Prop_True);
  }
  else {
    node_ptr exp;
    fprintf(nusmv_stdout, "is false\n");
    Prop_SetStatus(prop, Prop_False);
    if (full_fairness) {
      exp = witness (fsm,s0); 
    }
    else {
      bdd_ptr tmp = bdd_pick_one_state(dd_manager, s0);
      bdd_free(dd_manager, s0);
      s0 = tmp;
      bdd_ref(s0);
      exp = reverse(explain(fsm, cons((node_ptr)s0, Nil), spec_formula, Nil));
    }
    if (exp == Nil) {
      /* The counterexample consists of one initial state */
      exp = cons((node_ptr)s0, Nil);
    }
    Prop_SetTrace(prop, get_trace_number());
    print_explanation(exp, old_all_symbols);
    walk_dd(dd_manager, bdd_free, exp);
  }
  bdd_free(dd_manager, s0);

  ltlPopStatus();
}

/**Function********************************************************************

  Synopsis           [Print the LTL specification.]

  Description        [Print the LTL specification.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void print_ltlspec(FILE *file, node_ptr n)
{
  node_ptr context = Nil;
  
  if (node_get_type(n) == CONTEXT) {
    context = car(n);
    n = cdr(n);
  }
  indent_node(file, "LTL specification ", n, " ");
  if(context) {
    fprintf(file, "(in module ");
    print_node(file, context);
    fprintf(file, ") ");
  }
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Add the tableau module to the list of know modules]

  Description        [Add the tableau module to the list of know modules]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void ltlInsertModuleHashReadModule(node_ptr parse_tree)
{ /* We insert the definition of the current module in the module_hash in
     order to make it available for the Compile_FlattenHierarchy routines. */
  node_ptr cur_module = car(reverse(parse_tree));
  node_ptr name = find_atom(car(car(cur_module)));
  node_ptr params = cdr(car(cur_module));
  node_ptr def = cdr(cur_module);

  if (lookup_module_hash(name)) error_redefining(name);
  insert_module_hash(name, new_node(LAMBDA, params, reverse(def)));
}

/**Function********************************************************************

  Synopsis           [Save the status of global variables]

  Description        [Save the status of global variables]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void ltlPushStatus()
{
  node_ptr tmp;

  old_state_variables      = state_variables;
  state_variables          = Nil;
  old_all_variables        = all_variables;
  all_variables            = Nil;
  old_all_symbols          = all_symbols;
  all_symbols              = Nil;
  old_input_variables      = input_variables;
  input_variables          = Nil;
  old_real_state_variables = real_state_variables;
  real_state_variables     = Nil;
  old_boolean_variable_list= Nil;

  tmp = get_boolean_variable_list();
  while(tmp) {
    old_boolean_variable_list = cons(car(tmp),old_boolean_variable_list);
    tmp = cdr(tmp);
  }
  old_boolean_variable_list = reverse(old_boolean_variable_list);
  
  old_state_variables_bdd       = bdd_dup(state_variables_bdd);
  old_state_variables_add       = add_dup(state_variables_add);

  old_next_state_variables_bdd  = bdd_dup(next_state_variables_bdd);
  old_next_state_variables_add  = add_dup(next_state_variables_add);

  old_input_variables_bdd       = bdd_dup(input_variables_bdd);
  old_input_variables_add       = add_dup(input_variables_add);

  old_next_input_variables_bdd  = bdd_dup(next_input_variables_bdd);
  old_next_input_variables_add  = add_dup(next_input_variables_add);

  old_minterm_vars_dim = get_minterm_vars_dim();
}

/**Function********************************************************************

  Synopsis           [Restore the status of global variables previously saved]

  Description        [Restore the status of global variables previously saved]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void ltlPopStatus()
{
  state_variables      = old_state_variables;
  all_variables        = old_all_variables;
  all_symbols          = old_all_symbols;
  input_variables      = old_input_variables;
  real_state_variables = old_real_state_variables;

  free_boolean_variable_list();
  set_boolean_variable_list(old_boolean_variable_list);

  bdd_free(dd_manager, state_variables_bdd);
  state_variables_bdd = old_state_variables_bdd;

  add_free(dd_manager, state_variables_add);
  state_variables_add = old_state_variables_add;

  bdd_free(dd_manager, next_state_variables_bdd);
  next_state_variables_bdd  = old_next_state_variables_bdd;

  add_free(dd_manager, next_state_variables_add);
  next_state_variables_add = old_next_state_variables_add;

  bdd_free(dd_manager, input_variables_bdd);
  input_variables_bdd = old_input_variables_bdd;

  add_free(dd_manager, input_variables_add);
  input_variables_add = old_input_variables_add;

  bdd_free(dd_manager, next_input_variables_bdd);
  next_input_variables_bdd  = old_next_input_variables_bdd;

  add_free(dd_manager, next_input_variables_add);
  next_input_variables_add  = old_next_input_variables_add;

  set_minterm_vars_dim(old_minterm_vars_dim);
}

/**Function********************************************************************

  Synopsis           [Main routine to add the tableau to the FSM]

  Description        [Main routine to add the tableau to the FSM]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void ltlPropAddTableau(Prop_Ptr prop, node_ptr trans_expr, node_ptr init_expr, 
			      node_ptr invar_expr, node_ptr justice_expr, 
			      node_ptr compassion_expr)
{
  Partition_Method part;
  Fsm_BddPtr fsm;
  add_ptr tmp_add;
  bdd_ptr tmp_bdd;

  CPTrans_Ptr cp_trans;

  node_ptr justice_constraints, compassion_constraints;
  bdd_ptr init, invar, fair_states;
  
  /* We update the cubes for quantification */
  bdd_and_accumulate(dd_manager, &state_variables_bdd, old_state_variables_bdd);
  add_and_accumulate(dd_manager, &state_variables_add, old_state_variables_add);
  
  bdd_and_accumulate(dd_manager, &next_state_variables_bdd, old_next_state_variables_bdd);
  add_and_accumulate(dd_manager, &next_state_variables_add, old_next_state_variables_add);

  bdd_and_accumulate(dd_manager, &input_variables_bdd, old_input_variables_bdd);
  add_and_accumulate(dd_manager, &input_variables_add, old_input_variables_add);

  bdd_and_accumulate(dd_manager, &next_input_variables_bdd, old_next_input_variables_bdd);
  add_and_accumulate(dd_manager, &next_input_variables_add, old_next_input_variables_add);

  fsm = Prop_GetBddFsm(prop);
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);

  cp_trans = NULL; /* just to be sure we capture ecceptions */

  /* Update the transition relation */
  part = get_partition_method(options);  
  switch(part) {
  case Monolithic: 
    {
      cp_trans = Compile_FsmBddGetMonoTrans(fsm);
      break;
    }
  case Threshold:
    {
      cp_trans = Compile_FsmBddGetThreshold(fsm);
      break;
    }
  case Iwls95CP:
    {
      cp_trans = Compile_FsmBddGetIwls95CP(fsm);
      break;
    }
  default: 
    {
      rpterr("ltlPropAddTableau: unknown partition method\n");
    }
  }

  nusmv_assert(cp_trans != (CPTrans_Ptr)NULL);

  tmp_add = eval(trans_expr, Nil);
  tmp_bdd = add_to_bdd(dd_manager,tmp_add);
  add_free(dd_manager, tmp_add);
  cp_trans = Compile_CPAddClusterAndFillSchedule(cp_trans, tmp_bdd);
  bdd_free(dd_manager, tmp_bdd);

  switch(part) {
  case Monolithic: 
    {
      Compile_FsmBddSetMonoTrans(fsm, cp_trans);
      break;
    }
  case Threshold:
    {
      Compile_FsmBddSetThreshold(fsm, cp_trans);
      break;
    }
  case Iwls95CP:
    {
      Compile_FsmBddSetIwls95CP(fsm, cp_trans);
      break;
    }
  default: 
    {
      rpterr("ltlPropAddTableau: unknown partition method\n");
    }
  }

  /* Update the init states */
  init = Compile_FsmBddGetInit(fsm);
  nusmv_assert(init != (bdd_ptr)NULL);

  tmp_add = eval(init_expr, Nil);
  tmp_bdd = add_to_bdd(dd_manager,tmp_add);
  add_free(dd_manager, tmp_add);
  init = bdd_and(dd_manager, init, tmp_bdd);
  bdd_free(dd_manager, tmp_bdd);

  Compile_FsmBddSetInit(fsm,init);

  /* Update the invar states */
  invar = Compile_FsmBddGetInvar(fsm);
  nusmv_assert(invar != (bdd_ptr)NULL);

  tmp_add = eval(invar_expr, Nil);
  tmp_bdd = add_to_bdd(dd_manager,tmp_add);
  add_free(dd_manager, tmp_add);
  invar = bdd_and(dd_manager, invar, tmp_bdd);
  bdd_free(dd_manager, tmp_bdd);

  Compile_FsmBddSetInvar(fsm,invar);

  /* Update the fairness constraints */
  justice_constraints 
    = append_ns(Compile_FsmBddGetJustice(fsm),
		Mc_ComputeJusticeConstraints(fsm, justice_expr));
  Compile_FsmBddSetJustice(fsm, justice_constraints);
  compassion_constraints 
    = append_ns(Compile_FsmBddGetCompassion(fsm),
		Mc_ComputeCompassionConstraints(fsm, compassion_expr));
  Compile_FsmBddSetCompassion(fsm, compassion_constraints);

  /* Update the fair states */
  fair_states = bdd_dup(Compile_FsmBddGetFairStates(fsm));
  if (fair_states != (bdd_ptr)NULL) bdd_free(dd_manager, fair_states);
  Compile_FsmBddSetFairStates(fsm, Mc_ComputeFairStates(fsm));
}

/**Function********************************************************************

  Synopsis           [Creates the tableau]

  Description [Creates the tableau for a LTL property.  The FSM of the
  property contains the tableau. Returns 1 if an error is encountered
  during the tableau generation.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int ltlBuildTableauAndPropFsm(Prop_Ptr prop)
{
  node_ptr spec;
  node_ptr ltl_formula, context;
  char * Tmp_Input_Fname, * Tmp_Output_Fname;
  char * old_input_file, * Ltl_Module_Name, * External_Command;
  FILE * file;
  
  spec = Prop_GetProp(prop);
  if (node_get_type(spec) == CONTEXT) {
    context     = car(spec);
    ltl_formula = cdr(spec);
  }
  else {
    context     = Nil;
    ltl_formula = spec;
  }

  /* The formula has to be negated */
  ltl_formula = new_node(NOT, ltl_formula, Nil);

  /* create and open the file where to print the LTL formula to
     provide in input to the tableau */
  Tmp_Input_Fname = ltlMakeTempFileName();
  if (Tmp_Input_Fname == NIL(char)) {
    fprintf(nusmv_stderr, "Could not create input temporary file. ");
    nusmv_exit(1);
  }
  file = fopen(Tmp_Input_Fname, "w");
  if (file == NIL(FILE)) {
    fprintf(nusmv_stderr, "Could not create input temporary file. ");
    fprintf(nusmv_stderr, "Clean up /tmp an try again.\n");
    nusmv_exit(1);
  }
  print_node(file, ltl_formula);
  fclose(file);
  
  /* create the file where the SMV code produced by the tableau is written */
  Tmp_Output_Fname = ltlMakeTempFileName();
  if (Tmp_Output_Fname == NIL(char)) {
    fprintf(nusmv_stderr, "Could not create output temporary file. ");
    unlink(Tmp_Input_Fname);
    FREE(Tmp_Input_Fname);
    nusmv_exit(1);
  }

  {
    char ltl_spec_counter_str[6]; 
    int j = 4; /* Number of spaces */

    sprintf(ltl_spec_counter_str, "%d", ltl_spec_counter);
    j += strlen(LTL_TRANSLATOR);
    j += strlen(ltl_spec_counter_str);
    j += strlen(Tmp_Input_Fname);
    j += strlen(Tmp_Output_Fname) + 4; /* For spaces */
    External_Command = ALLOC(char, j);
  }
  if (External_Command == NIL(char)) {
    fprintf(nusmv_stderr, "Unable to allocate External_Command\n");
    unlink(Tmp_Input_Fname);
    FREE(Tmp_Input_Fname);
    FREE(Tmp_Output_Fname);
    nusmv_exit(1);
  }

  /* Write the LTL formula into the temporary file. */
  if (opt_verbose_level_gt(options, 0)) 
    fprintf(nusmv_stderr, "Computing the corresponding tableau....");

  sprintf(External_Command, "%s %d %s %s", LTL_TRANSLATOR, ltl_spec_counter_get(),
                                           Tmp_Input_Fname, Tmp_Output_Fname);
  if (system(External_Command) != 0) {
    fprintf(nusmv_stderr, "Unable to execute the command:\n%s\n", External_Command);
    unlink(Tmp_Input_Fname);
    FREE(Tmp_Input_Fname);
    FREE(Tmp_Output_Fname);
    nusmv_exit(1);
  }

  /* Deleting input file for tableau construction */
  unlink(Tmp_Input_Fname);
  FREE(Tmp_Input_Fname);

  if (opt_verbose_level_gt(options, 0)) 
    fprintf(nusmv_stderr, ".... done\n");

  /* We save the current input file */
  old_input_file = get_input_file(options);
  set_input_file(options, Tmp_Output_Fname);

  /* Parse the new input file and store the result in parse_tree */
  if (opt_verbose_level_gt(options, 0)) 
    fprintf(nusmv_stderr, "Parsing the generated tableau....");
  
  if (Parser_ReadSMVFromFile(Tmp_Output_Fname)) {
    set_input_file(options, old_input_file);
    unlink(Tmp_Output_Fname);
    FREE(Tmp_Output_Fname);
    nusmv_exit(1);
  }

  /* Deleting output file of tableau construction */
  unlink(Tmp_Output_Fname);
  FREE(Tmp_Output_Fname);

  if (opt_verbose_level_gt(options, 0))
    fprintf(nusmv_stderr, "....done\n");
  
  {
    char lsc_str[6];
    int j = 0;
      
    sprintf(lsc_str, "%d", ltl_spec_counter);
    j += strlen(LTL_MODULE_BASE_NAME);
    j += strlen(lsc_str);
    Ltl_Module_Name = ALLOC(char, j+1);
    if (Ltl_Module_Name == NIL(char)) {
      fprintf(nusmv_stderr, "Unable to allocate \"Ltl_Module_Name\".\n");
      set_input_file(options, old_input_file);
      nusmv_exit(1);
    }
  }

  sprintf(Ltl_Module_Name, "%s%d", LTL_MODULE_BASE_NAME, ltl_spec_counter);

  /* We insert the definition of the current module in the module_hash
     in order to make it available for the Compile_FlattenHierarchy routines. */
  ltlInsertModuleHashReadModule(parse_tree);

  {
    node_ptr ltl_init_expr       = Nil;
    node_ptr ltl_invar_expr      = Nil;
    node_ptr ltl_trans_expr      = Nil;
    node_ptr ltl_procs_expr      = Nil;
    node_ptr ltl_justice_expr    = Nil;
    node_ptr ltl_compassion_expr = Nil;
    node_ptr ltl_spec_expr       = Nil;
    node_ptr ltl_compute_expr    = Nil;
    node_ptr ltl_ltlspec_expr    = Nil;
    node_ptr ltl_invar_spec_expr = Nil;

    /*
      We call Compile_FlattenTableau with the name of the generated
      tableau, and as root name the actual property context. In this
      way local variables of the tableau and local variables of the
      formula will be contextualized to the right module.
    */
    Compile_FlattenTableau(sym_intern(Ltl_Module_Name), context,
			   &ltl_trans_expr, &ltl_init_expr, &ltl_invar_expr,
			   &ltl_spec_expr, &ltl_compute_expr, 
			   &ltl_ltlspec_expr, &ltl_invar_spec_expr, 
			   &ltl_justice_expr, &ltl_compassion_expr,
			   &ltl_procs_expr, Nil);
    /* The above function does side effect on some global variables */
    all_symbols          = reverse(all_symbols);
    state_variables      = reverse(state_variables);
    input_variables      = reverse(input_variables);
    all_variables        = append_ns(input_variables, state_variables);
    real_state_variables = append_ns(old_real_state_variables, 
				     real_state_variables);
    ltl_justice_expr     = reverse(ltl_justice_expr);
    ltl_compassion_expr  = reverse(ltl_compassion_expr);

    /* Check if we are using an old version of ltl2smv */
    if (ltl_spec_expr != Nil) {
      fprintf(nusmv_stderr, 
	      "Error: CTL specification in tableau construction.\n");
      fprintf(nusmv_stderr,
	      "       You should probably update your version of ltl2smv.\n");
      set_input_file(options, old_input_file);
      return(1);
    }

    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Creating LTL tableau variables...\n");
    }
    { /* setting input_order_file to NULL in order to avoid 
	 duplicate variables warning */
      char * iof = get_input_order_file(options);
      set_input_order_file(options, NULL);
      CATCH {
	Compile_BuildVarsBdd();
      } FAIL { 
	set_input_order_file(options, iof); 
	set_input_file(options, old_input_file);
	return(1);
      }
      set_input_order_file(options, iof);       
    }
    build_real_state_variables();

    all_symbols     = append_ns(old_all_symbols, all_symbols);
    all_variables   = append_ns(old_all_variables, all_variables);
    state_variables = append_ns(old_state_variables, state_variables);
    input_variables = append_ns(old_input_variables, input_variables);

    /*
      The error trapping mechanism is enough in this block. All the
      other errors even external to this block are trapped and the
      else of the CATCH is executed.
    */
    CATCH { 
      ltlPropAddTableau(prop, ltl_trans_expr, ltl_init_expr, 
			      ltl_invar_expr, ltl_justice_expr,
			      ltl_compassion_expr);
    } FAIL {
      set_input_file(options, old_input_file);
      return(1);
    }

    set_input_file(options, old_input_file);

    return(0);
  }
}

/**Function********************************************************************

  Synopsis           [Return a string to be used as temporary file]

  Description        [Return a string to be used as temporary file]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static char * ltlMakeTempFileName() {
  char template[] = "/tmp/ltl2smvXXXXXX";
  int fn = mkstemp(template);
  if (fn == -1) return NULL;
  close(fn);
  return(util_strsav(template));
}



