#define NO_PRINT_STATES
#define NO_TIMES

/**CFile***********************************************************************

  FileName    [imgUtil.c]

  PackageName [img]

  Synopsis    [Routines to perform image computations.]

  Description [Routines to perform image computations.]

  Author      [Marco Roveri and Emanuele Olivetti]

  Copyright   [
  This file is part of the ``img'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "imgInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: imgUtil.c,v 1.5 2003/10/28 17:29:21 flerda Exp $";

static bdd_ptr ImgImageBwdCP ARGS((CPTrans_Ptr, bdd_ptr));
static bdd_ptr ImgImageFwdCP ARGS((CPTrans_Ptr, bdd_ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Computes the backward image of a set of states.]

  Description        [Computes the backward image of a set of states.
  The resulting set is in term of current state variables. Depending
  the partitioning heuristich choosen, the corresponding image
  function is used.]

  SideEffects        [None]

  SeeAlso            [Img_ImageFwd]

******************************************************************************/
bdd_ptr Img_ImageBwd(Fsm_BddPtr fsm, bdd_ptr g)
{
  Partition_Method part;

  part = get_partition_method(options);
  switch(part) {
  case Monolithic:
    {
      return(ImgImageBwdCP(Compile_FsmBddGetMonoTrans(fsm), g));
    }
  case Threshold:
    {
      return(ImgImageBwdCP(Compile_FsmBddGetThreshold(fsm), g));
    }
  case Iwls95CP:
    {
      return(ImgImageBwdCP(Compile_FsmBddGetIwls95CP(fsm), g));
    }
  default:
    {
      rpterr("Img_ImageBwd: partitioning method not available.");
      return((bdd_ptr)Nil);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Computes the forward image of a set of states.]

  Description        [Computes the forward image of a set of states.
  The resulting set is in term of current state variables. Depending
  the partitioning heuristich choosen, the corresponding image
  function is used.]

  SideEffects        [None]

  SeeAlso            [Img_ImageBwd]

******************************************************************************/
bdd_ptr Img_ImageFwd(Fsm_BddPtr fsm, bdd_ptr g)
{
  Partition_Method part;

  part = get_partition_method(options);
  switch(part) {
  case Monolithic:
    {
      return(ImgImageFwdCP(Compile_FsmBddGetMonoTrans(fsm), g));
    }
  case Threshold:
    {
      return(ImgImageFwdCP(Compile_FsmBddGetThreshold(fsm), g));
    }
  case Iwls95CP:
    {
      return(ImgImageFwdCP(Compile_FsmBddGetIwls95CP(fsm), g));
    }
  default:
    {
      rpterr("Img_ImageFwd: partitioning method not available.");
      return((bdd_ptr)Nil);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Check if built model exists.]

  Description        [Check if built model exists. Return zero if
  something wrong.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Img_Check(){
  if (opt_cone_of_influence(options) == true) {
    return(1);
  }
  else {
    Fsm_BddPtr fsm = Prop_MasterGetBddFsm();

    if (fsm == (Fsm_BddPtr)NULL) {
      /* It has not yet been executed build_model */
      return(0);
    }
    switch(get_partition_method(options)) {
    case Monolithic:
      {
        return((Compile_FsmBddGetMonoTrans(fsm) == (CPTrans_Ptr)NULL) ? 0 : 1);
      }
    case Threshold:
      {
        return((Compile_FsmBddGetThreshold(fsm) == (CPTrans_Ptr)NULL) ? 0 : 1);
      }
    case Iwls95CP:
      {
        return((Compile_FsmBddGetIwls95CP(fsm) == (CPTrans_Ptr)NULL) ? 0 : 1);
      }      
    default:
      {
        fprintf(nusmv_stderr, "Unknown partition method.\n");
        return(0);
      }
    }
  }
  return(0);
}

/**Function********************************************************************

  Synopsis           [Check if a model has been built and exists.]

  Description        [Check if a model has been built and exists.. Return zero if something wrong.]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
int Img_CheckNoCone(){
  Fsm_BddPtr fsm = Prop_MasterGetBddFsm();

  if (fsm == (Fsm_BddPtr)NULL) {
    /* It has not yet been executed build_model */
    return(0);
  }
  switch(get_partition_method(options)) {
  case Monolithic:
    {
      return((Compile_FsmBddGetMonoTrans(fsm) == (CPTrans_Ptr)NULL) ? 0 : 1);
    }
  case Threshold:
    {
      return((Compile_FsmBddGetThreshold(fsm) == (CPTrans_Ptr)NULL) ? 0 : 1);
    }
  case Iwls95CP:
    {
      return((Compile_FsmBddGetIwls95CP(fsm) == (CPTrans_Ptr)NULL) ? 0 : 1);
    }      
  default:
    {
      fprintf(nusmv_stderr, "Unknown partition method.\n");
      return(0);
    }
  }
  return(0);
}

/**Function********************************************************************

  Synopsis           [Computes the backward image of a set of
  states.]

  Description        [Computes the backward image of a set of states.
  It uses an implicitly conjoined transition relation. The resulting
  set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static bdd_ptr ImgImageBwdCP(CPTrans_Ptr trans, bdd_ptr g)
{
  CPList CP_Bwd_trans = CPTransGetBackward(trans);
  bdd_ptr vars = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);
  bdd_ptr tmp = bdd_shift_forward(dd_manager, g);
  bdd_ptr result = CPImageBoth(dd_manager, CP_Bwd_trans, tmp, vars);

  bdd_free(dd_manager, tmp);
  bdd_free(dd_manager, vars);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the forward image of a set of  states.]

  Description        [Computes the forward image of a set of states.
  It uses an implicitly conjoined transition relation. The resulting
  set is in term of current state variables.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static bdd_ptr ImgImageFwdCP(CPTrans_Ptr trans, bdd_ptr g)
{
  CPList CP_Fwd_trans = CPTransGetForward(trans);
  bdd_ptr vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
  bdd_ptr tmp = CPImageBoth(dd_manager, CP_Fwd_trans, g, vars);
  bdd_ptr result = bdd_shift_backward(dd_manager, tmp);

  bdd_free(dd_manager, tmp);
  bdd_free(dd_manager, vars);
  return(result);
}

/* NS : ImgBwdSaturate 
static bdd_ptr ImgImageBwdCPSaturate(CPTrans_Ptr trans, bdd_ptr g) {

	CPList CP_Bwd_trans = CPTransGetBackward(trans);
	bdd_ptr vars = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
	bdd_ptr tmp = CPImageSaturateBoth(dd_manager, CP_Bwd_trans, g, vars);
	bdd_ptr result = bdd_shift_backward(dd_manager,tmp);

  bdd_free(dd_manager, tmp);
  bdd_free(dd_manager, vars);
  return(result);

}
*/

/*NS :CPImageSaturateBoth 
static bdd_ptr CPImageSaturateBoth(DdManager *dd, CPList Clist, bdd_ptr S, bdd_ptr vars) {
	bdd_ptr result;
	int it=0; long maxSize=0, intermediateSize=0;
	bddptr cur_prod = dup(S);

	for( CPCluster* Ci = Clist.begin(); Ci != Clist.end(); Ci= Ci->next ) {

		// saturate the current cluster
		while( cur_
		bdd_ptr new_p = bdd_and_abstract( dd, cur_prod, CPClusterGetTi(Ci), CPClusterGetEi(Ci));
		it++;
		
    bdd_free(dd, cur_prod);
    cur_prod = new_p;
	}

	result = bdd_forsome(dd, cur_prod, vars);
	bdd_free(dd, cur_prod);


  return(result);
}

*/

/**Function********************************************************************

  Synopsis [Performs both forward or backward image computation.]

  Description [Depending the Cluster list passed as argument, this
  function computes respectively the forward or backward image of a
  given state. CP version.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr CPImageBoth(DdManager *dd, CPList Clist, bdd_ptr S, bdd_ptr vars)
{
  // The current implementation works also for equality-reduced BDDs
  // but we might later try to optmize it for that case, for instance
  // by doing exist abstract only for variables in <I>vars</I>.
  bdd_ptr result;
  int it = 0;
  long maxSize = 0;
  long intermediateSize = 0;
  bdd_ptr cur_prod = bdd_dup(S);
  
#ifndef NO_TIMES
  {
  fprintf(nusmv_stdout, "@@@ Begin Image Computation\n");
  time_t t_begin = time(NULL);
#endif

#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ Starting Set\n");
  dd_printminterm(dd_manager, S);
  fprintf(nusmv_stdout, "@@@@\n");
#endif

  while (CPListIsNotEmpty(Clist)) {
    CPCluster * Ci = CPListGetItem(Clist);
    // bdd_ptr e;
#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ Current Product\n");
  dd_printminterm(dd_manager, cur_prod);
  fprintf(nusmv_stdout, "@@@@\n");
#endif
#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ Ti\n");
  dd_printminterm(dd_manager, CPClusterGetTi(Ci));
  fprintf(nusmv_stdout, "@@@@\n");
#endif
#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ Ei\n");
  dd_printminterm(dd_manager, CPClusterGetEi(Ci));
  fprintf(nusmv_stdout, "@@@@\n");
#endif
#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ vars\n");
  dd_printminterm(dd_manager, vars);
  fprintf(nusmv_stdout, "@@@@\n");
#endif
  // e = bdd_dup(CPClusterGetEi(Ci));
  // bdd_or_accumulate(dd_manager, &e, vars);
#ifndef NO_PRINT_STATES
  //fprintf(nusmv_stdout, "@@@@ e\n");
  //dd_printminterm(dd_manager, e);
  //fprintf(nusmv_stdout, "@@@@\n");
#endif
bdd_ptr new_p;

#ifndef NO_TIMES
  {
  fprintf(nusmv_stdout, "@@@ Begin And Abstract\n");
  time_t t_begin = time(NULL);
#endif

    /* Flavio Lerda */
    new_p = bdd_and_abstract(dd, cur_prod, CPClusterGetTi(Ci),
                                                   CPClusterGetEi(Ci));
#ifndef NO_TIMES
    /* Flavio Lerda */
    {
      time_t t_end = time(NULL);
      fprintf(nusmv_stdout, "@@@ End And Abstract (%ld seconds)\n", (t_end - t_begin));
    }
  }
#endif

#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ new_p\n");
  dd_printminterm(dd_manager, new_p);
  fprintf(nusmv_stdout, "@@@@\n");
#endif
    it++;

    if (get_verbose_level(options) >= 1) { 
      intermediateSize = bdd_size(dd, new_p);
      (void) fprintf(nusmv_stdout, "          Size of intermediate product = %10ld (BDD nodes).\n", intermediateSize);
      if (maxSize < intermediateSize) {
        maxSize = intermediateSize;
      }
    }

    bdd_free(dd, cur_prod);
    cur_prod = new_p;
    Clist = CPListNext(Clist);
  }

#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ vars\n");
  dd_printminterm(dd_manager, vars);
  fprintf(nusmv_stdout, "@@@@\n");
#endif
  result = bdd_forsome(dd, cur_prod, vars);
  bdd_free(dd, cur_prod);
#ifndef NO_PRINT_STATES
  fprintf(nusmv_stdout, "@@@@ result\n");
  dd_printminterm(dd_manager, result);
  fprintf(nusmv_stdout, "@@@@\n");
#endif

#ifndef NO_TIMES
    /* Flavio Lerda */
    {
      time_t t_end = time(NULL);
      fprintf(nusmv_stdout, "@@@ End Image Computation (%ld seconds)\n", (t_end - t_begin));
    }
  }
#endif

  if (get_verbose_level(options) >= 1) {
    (void) fprintf(nusmv_stdout, "                 Total size of product = %10ld (BDD nodes).\n", (long)bdd_size(dd, result));
    (void) fprintf(nusmv_stdout, "Max. BDD size for intermediate product = %10ld (BDD nodes)\n", maxSize);
    fprintf(nusmv_stdout, "Variables number in intermediate product = %10d (BDD nodes).\n", bdd_size(dd_manager, bdd_support(dd_manager, result)));
  }
  return(result);
}
