/**CFile***********************************************************************

  FileName    [compileWriteFlat.c]

  PackageName [compile]

  Synopsis [Creation of an SMV file containing the flattened model.]

  Description [Creation of an SMV file containing the flattened model,
  processes will be removed by explicitly introducing a process
  variable and modifying assignments to take care of inertia.]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileWriteFlat.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void print_assign ARGS((FILE *, node_ptr, node_ptr));
static int write_flatten_vars ARGS((FILE *, node_ptr lov, char *));
static int write_flatten_define ARGS((FILE *));
static int write_flatten_assign ARGS((FILE *));
static int write_flatten_expr ARGS((FILE *, node_ptr, char *));
static int write_flatten_expr_pair ARGS((FILE *, node_ptr, char *));
static int write_process_selector_define ARGS((FILE *));

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Prints the flatten version of an SMV model.]

  Description        [[Prints on the specified file the flatten
  version of an SMV model.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Compile_WriteFlatten(FILE * out, cmp_struct_ptr s)
{
  fprintf(out, "-- Flattened model generated from %s\n\n", get_input_file(options)); 
  fprintf(out, "MODULE main\n");
  write_flatten_vars(out, input_variables, "IVAR");
  write_flatten_vars(out, state_variables, "VAR");
  write_process_selector_define(out);
  write_flatten_define(out);
  if (write_flatten_assign(out)) fprintf(out, "\n");
  if (write_flatten_expr(out, cmp_struct_get_init(s), "INIT\n"))
    fprintf(out, "\n");
  if (write_flatten_expr(out, cmp_struct_get_invar(s), "INVAR\n"))
    fprintf(out, "\n");
  if (write_flatten_expr(out, cmp_struct_get_trans(s), "TRANS\n"))
    fprintf(out, "\n");

  if (cmp_struct_get_compassion(s) != Nil) {
    if (write_flatten_expr(out, cmp_struct_get_justice(s), "JUSTICE\n"))
      fprintf(out, "\n\n");
    if (write_flatten_expr_pair(out, cmp_struct_get_compassion(s), 
				"COMPASSION\n"))
      fprintf(out, "\n\n");
  } 
  else { /* For backward compatibility */
    if (write_flatten_expr(out, cmp_struct_get_justice(s), "FAIRNESS\n"))
      fprintf(out, "\n\n");
  }

  if (write_flatten_expr(out, cmp_struct_get_spec(s), "SPEC\n"))
      fprintf(out, "\n\n");
  if (write_flatten_expr(out, cmp_struct_get_compute(s), "COMPUTE\n"))
      fprintf(out, "\n\n");
  if (write_flatten_expr(out, cmp_struct_get_ltlspec(s), "LTLSPEC\n"))
      fprintf(out, "\n\n");
  if (write_flatten_expr(out, cmp_struct_get_invar_spec(s), "INVARSPEC\n"))
      fprintf(out, "\n\n");
  return(1);
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Writes VAR declarations in SMV format on a file.]

  Description        [Writes VAR declarations in SMV format on a file.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int write_flatten_vars(FILE * out, node_ptr lov, char * str)
{
  if (lov != Nil) fprintf(out, "%s\n", str);
  else return(0);
  while (lov != Nil) {
    node_ptr name = car(lov);
    node_ptr data = lookup_symbol_hash(name);
    
    if (data != Nil && node_get_type(data) == VAR) {
      print_node(out, name);
      if (cdr(data) == boolean_type) {
        fprintf(out, " : boolean;\n");
      } 
      else {
        node_ptr l = cdr(data);

        fprintf(out, " : {");
        while (l != Nil) {
          print_node(out, car(l));
          l = cdr(l);
          if (l != Nil) fprintf(out, ", ");
        }
        fprintf(out, "};\n");
      }
    }
    lov = cdr(lov);
  }
  fprintf(out, "\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Writes DEFINE declarations in SMV format on a
  file.]

  Description        [Writes DEFINE declarations in SMV format on a
  file.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int write_process_selector_define(FILE * out) 
{
  node_ptr n;

  fprintf(out, "DEFINE\n");

  for(n = process_running_symbols; n != Nil; n = cdr(n)){
    node_ptr name = car(n);
    node_ptr def = lookup_symbol_hash(name);

    if (def != Nil && node_get_type(def) == CONTEXT) {
      node_ptr fdef = Compile_FlattenSexp(def, Nil);

      print_node(out, name);
      fprintf(out, " := ");
      print_node(out, fdef);
      fprintf(out, ";\n");
    }
    else {
      fprintf(nusmv_stderr, "write_process_selector_define:\n");
      fprintf(nusmv_stderr, "Undefined symbol: \"");
      print_node(nusmv_stderr, name);
      fprintf(nusmv_stderr, "\"\n");
    }
  }
  fprintf(out, "\n");
  return(1);
}

      

/**Function********************************************************************

  Synopsis           [Writes DEFINE declarations in SMV format on a
  file.]

  Description        [Writes DEFINE declarations in SMV format on a
  file.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int write_flatten_define(FILE * out)
{
  node_ptr names = all_symbols;
  int define_emitted = 0; /* Set to 1 once keyword DEFINE is emitted */

  while(names) {
    node_ptr name = car(names);
    node_ptr def = lookup_symbol_hash(name);

    if (def != Nil && node_get_type(def) == CONTEXT) {
      node_ptr fdef = Compile_FlattenSexp(def, Nil);
     
      /* Emit keyword DEFINE if necessary */
      if (! define_emitted) {
	fprintf(out, "DEFINE\n");
	define_emitted = 1;
      }

      if (fdef != Nil) {
        print_node(out, name);
	fprintf(out, " := ");
        print_node(out, fdef);
	fprintf(out, ";\n");
      }
      else
        fprintf(nusmv_stderr, "write_flatten_define: Flattening failed\n");
    }
    names = cdr(names);
  }
  fprintf(out, "\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Writes flattened expression in SMV format on a file.]

  Description        [Writes a generic expression prefixed by a given
  string in SMV format on a file.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int write_flatten_expr(FILE * out, node_ptr n, char * s)
{
  int res = 0;

  if (n == Nil) return(0);
  switch(node_get_type(n)) {
  case CONS:
  case AND:
    res = write_flatten_expr(out, car(n), s);
    res = write_flatten_expr(out, cdr(n), s);
    break;
  default:
    res = 1;
    fprintf(out, "%s ", s);
    if (node_get_type(n) == CONTEXT) {
      /* We are dealing with a property that has not yet been 
	 flattened before */
      node_ptr fn = Compile_FlattenSexp(n, Nil);

      print_node(out, fn);
    } else {
      print_node(out, n);
    }
    fprintf(out, "\n");
    break;
  }
  return res;
}

/**Function********************************************************************

  Synopsis           [Writes flattened expression pairs in SMV format on a file.]

  Description        [Writes a list of flattened expression pairs 
  prefixed by a given string in SMV format on a file.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int write_flatten_expr_pair(FILE * out, node_ptr l, char * s)
{
  if (l == Nil) return 0;
  while (l) {
    node_ptr n = car(l);
    l = cdr(l);
    nusmv_assert(node_get_type(n) == CONS);
    fprintf(out, "%s (", s);
    if (node_get_type(n) == CONTEXT) {
      node_ptr fn = Compile_FlattenSexp(car(n), Nil);
      print_node(out, fn);
    } else {
      print_node(out, car(n));
    }
    fprintf(out, ", ");
    if (node_get_type(n) == CONTEXT) {
      node_ptr fn = Compile_FlattenSexp(cdr(n), Nil);
      print_node(out, fn);
    } else {
      print_node(out, cdr(n));
    }
    fprintf(out, ")\n");
  }
  return 1;
}

/**Function********************************************************************

  Synopsis           [Writes flattened ASSIGN declarations in SMV format on a
  file.]

  Description        [Writes flattened ASSIGN declarations in SMV format on a
  file.]

  SideEffects        []

  SeeAlso            [write_flatten_assign_recur]

******************************************************************************/
static int write_flatten_assign(FILE * out)
{
  node_ptr vars = all_variables;

  while (vars != Nil) {
    node_ptr name = car(vars);
    node_ptr init_name = find_node(SMALLINIT, name, Nil);
    node_ptr next_name = find_node(NEXT, name, Nil);
    node_ptr invar_expr = lookup_assign_db_hash(name);
    node_ptr init_expr = lookup_assign_db_hash(init_name);
    node_ptr next_expr = lookup_assign_db_hash(next_name);

    if (invar_expr != (node_ptr)NULL) {
      invar_expr = car(invar_expr);
    }
    if (init_expr != (node_ptr)NULL) {
      init_expr = car(init_expr);
    }
    if (next_expr != (node_ptr)NULL) {
      next_expr = car(next_expr);
    }

    if ((init_expr != (node_ptr)NULL) ||
        (next_expr != (node_ptr)NULL) ||
        (invar_expr != (node_ptr)NULL)) {
      fprintf(out, "ASSIGN\n");
    }
    if (init_expr != (node_ptr)NULL) {
      print_assign(out, init_name, init_expr);
    }
    if (invar_expr != (node_ptr)NULL) {
      print_assign(out, name, invar_expr);
    }
    if (next_expr != (node_ptr)NULL) {
      print_assign(out, next_name, next_expr);
    }
    if ((init_expr != (node_ptr)NULL) ||
        (next_expr != (node_ptr)NULL) ||
        (invar_expr != (node_ptr)NULL)) {
      fprintf(out, "\n");
    }
    vars = cdr(vars);
  }
  return(1);
}

/**Function********************************************************************

  Synopsis           [Prints an assignement statement]

  Description        [[Prints an assignement statement]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void print_assign(FILE * out, node_ptr lhs, node_ptr rhs){
  print_node(out, lhs);
  fprintf(out, " := ");
  print_node(out, rhs);
  fprintf(out, ";\n");
}



