/**CFile***********************************************************************

  FileName    [compileUtil.c]

  PackageName [compile]

  Synopsis    [Routines for model computation.]

  Description [This file contains the code for the compilation of the
  flattened hierarchy into BDD:
  <ul>
  <li> Creation of the boolean variables.</li>
  <li> Creation of the BDD representing the inertia of the system when
       there are processes. In fact when a process is running the
       other processes are stopped, and their state variables don't
       change.</li>
  <li> Creation of the BDD representing what does not change in the
       system, i.e. the set of invariance. These are introduced in the
       model by the keyword "<tt>INVAR</tt>" or by the <em>normal
       assignments</em> (i.e. "<tt>ASSIGN x : = y & z;</tt>"). These
       states are not stored in the transition relation, they are
       stored in an a doc variable.
  <li> Creation of the BDD representing the set of initial states.
  <li> Creation of the BDD representing the transition relation. 
       Various ways of representing the transition relation are offered
       the users.
       <ul>
       <li> <em>Monolithic</em>: the monolithic transition relation is
            computed.</li>
       <li> <em>Conjunctive Partioned (Threshold)</em>: the transition 
            relation is stored as an implicitly conjoined list of 
            transition relation. This kind of partitioning can be 
            used only if the model considered is a synchronous one.</li>
       <li> <em>Conjunctive Partioned IWLS95</em>: as the above, but the
            heuristic proposed in \[1\] is used to order partition clusters. </li>
       </ul>
  <li> Computes the fairness constraints. I.e. each fairness constraint
       (which can be a CTL formula) is evaluated and the resulting BDD
       is stored in the list <tt>fairness_constraints_bdd</tt> to be
       then used in the model checking phase.
  </ul>
  \[1\] R. K. Ranjan and A. Aziz and B. Plessier and C. Pixley and R. K. Brayton,
      "Efficient BDD Algorithms for FSM Synthesis and Verification,
      IEEE/ACM Proceedings International Workshop on Logic Synthesis,
      Lake Tahoe (NV), May 1995.</li>
  ]

  SeeAlso     []

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/
#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileUtil.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

typedef enum {
  State_Instantiation_Mode,
  Input_Instantiation_Mode
} Instantiation_Vars_Mode_Type;

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The initial states.]

  Description [The {A|B}DD representing the initial states.]

  SeeAlso     []

******************************************************************************/
add_ptr init_add = (add_ptr)NULL;
bdd_ptr init_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The invariant states.]

  Description [The {A|B}DD representing the invariant states.]

  SeeAlso     []

******************************************************************************/
add_ptr invar_add = (add_ptr)NULL;
bdd_ptr invar_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [Associative list variable corresponding FSM sexps]

  Description [This is an associative hash. To each variable the
  corresponding FSM sexp are associated.]

  SeeAlso     []

******************************************************************************/
static hash_ptr variable_sexp_model_hash;
void init_variable_sexp_model_hash() 
{
  variable_sexp_model_hash = new_assoc();
}
void insert_variable_sexp_model_hash(node_ptr x, node_ptr y) 
{
  insert_assoc(variable_sexp_model_hash, x, y);
}
node_ptr lookup_variable_sexp_model_hash(node_ptr x) {
  return(find_assoc(variable_sexp_model_hash, x));
}

void clear_variable_sexp_model_hash() {
  clear_assoc(variable_sexp_model_hash);
}

static assoc_retval variable_sexp_model_hash_free(char *key, char *data, char * arg) {
  node_ptr element = (node_ptr)data;

  free_node(cdr(element));
  free_node(element);
  return(ASSOC_DELETE);
}

void free_variable_sexp_model_hash_free() {
  clear_assoc_and_free_entries(variable_sexp_model_hash, variable_sexp_model_hash_free);}

/**Variable********************************************************************

  Synopsis    [The list of real state variables.]

  Description [It a list of ADD, one for each state variable,
  representing its encoding. It is used in <tt>print_reachable_states</tt>.]

  SeeAlso     [variables print_reachable_states]

******************************************************************************/
node_ptr real_state_variables = Nil;

/**Variable********************************************************************

  Synopsis    [The ADD of process selector.]

  Description [The ADD of process selector. Its an ADD whose leaves
  are the names of the processes. It is used to choose which process
  run and when. It is essentially the boolean encoding of the
  <tt>PROCESS_SELECTOR_VAR_NAME</tt>.]

  SeeAlso     [running_add running_atom PROCESS_SELECTOR_VAR_NAME]

******************************************************************************/
add_ptr process_selector_add; 

/**Variable********************************************************************

  Synopsis    [The NuSMV atom <tt>running</tt>.]

  Description [The internal representation of the NuSMV atom <tt>running</tt>.]

  SeeAlso     [process_selector_add running_add]

******************************************************************************/
node_ptr running_atom; 

/**Variable********************************************************************

  Synopsis    [The NuSMV ADD for atom <em>running</em>.]

  Description [The ADD corresponding to the boolean NuSMV variable
  <em>running</em>.]

  SeeAlso     [running_atom process_selector_add]

******************************************************************************/
add_ptr running_add;       

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void compileBddFsmPrintInfo ARGS((FILE *, Fsm_BddPtr, CPTrans_Ptr));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Builds an internal representation for a given string.]

  Description        [Builds an internal representation for a given
  string. If the conversion has been performed in the past, then the
  hashed value is returned back, else a new one is created, hashed and
  returned. We hash this in order to allow the following:
  <pre>
  VAR
     x : {a1, a2, a3};
     y : {a3, a4, a5};

  ASSIGN
     next(x) := case
                 x = y    : a2;
                 !(x = y) : a1;
                 1        : a3;
                esac;
  </pre>
  i.e. to allow the equality test between x and y. This can be
  performed because we internally have a unique representation of the
  atom <tt>a3</tt>.]

  SideEffects        []

  SeeAlso            [find_atom]

******************************************************************************/
node_ptr sym_intern(char *s)
{
  return(find_node(ATOM, (node_ptr)find_string(s), Nil));
}



/**Function********************************************************************

  Synopsis           [The ADD variables are built.]

  Description        [This function creates the ADD variables needed
  to build the automaton. If an input order file is given as option,
  then the ordering of variables is read from it, else the order in
  which the variables appear in a depth first traversal of the
  hierarchy is used. The symbolic name of the variable is stored in an
  array in a position corresponding to the first index of the boolean
  variable created to encode it. The boolean variables needed to
  encode a scalar symbolic variable are grouped together, so that the
  ordering methods do not modify the internal order of them. If the
  input order file contains a subset of the variables of the model,
  than the variables not specified in the ordering file are created at
  the end of the given ordering following the order in which they
  appear in a depth first traversal of the hierarchy. If the input
  ordering file is a superset of the variables declared in the model,
  than the variables not appearing in the model are discarded.]

  SideEffects        [<tt>state_variables_add</tt> <tt>next_state_variables_add</tt>
  <tt>num_of_state_variables</tt> are modified.]

  SeeAlso            [Compile_EncodeVar Compile_ReadOrder dd_new_var_block]

******************************************************************************/
void Compile_BuildVarsBdd(void)
{
  boolean is_input;
  node_ptr read_variables, l_o_vars;

  if (get_input_order_file(options) != NULL) {
    /*
      If the input order file is defined, then the possibly partial
      variable order is read in, and stored in the variable
      "read_variables".
    */
    read_variables = Compile_ReadOrder(get_input_order_file(options));
    l_o_vars = read_variables;
  }
  else {
    l_o_vars = all_variables;
  }

  /* Initialization of the ADD variables */
  state_variables_add = add_one(dd_manager);
  next_state_variables_add = add_one(dd_manager);
  input_variables_add = add_one(dd_manager);
  next_input_variables_add = add_one(dd_manager);

  while (l_o_vars) { /* Loops over all the read or defined variables, depending the
                     input order file is provided or not, to create add/bdd variables. */
    node_ptr q;
    node_ptr vname = car(l_o_vars); /* The current variable */

    q = lookup_symbol_hash(vname);
    if (!q || node_get_type(q) != VAR) {
      /* The current variable does not appear in the read model !! */
      warning_variable_not_declared(vname);
      l_o_vars = cdr(l_o_vars);
      continue;
    }
    if (car(q)) {
      /* The current variable appear twice in the ordering file! */
      warning_var_appear_twice_in_order_file(vname);
      l_o_vars = cdr(l_o_vars);
      continue;
    }
    if (opt_verbose_level_gt(options, 2)) {
      print_node(nusmv_stderr, vname);
      fprintf(nusmv_stderr, ":\n");
    }

    /* Side effects on the hash */
    is_input = memberp(vname, input_variables) ? true : false;
    node_bdd_setcar(q, Compile_EncodeVar(vname, cdr(q), is_input, true));
    
    l_o_vars = cdr(l_o_vars); 
  }
  /* If the ordering is read from the ordering input file, then
     additional check have to be performed. */
  if (get_input_order_file(options) != NULL) {
    /*
      Check that all the variables defined in the source file (listed
      in global variable "all_variables") are listed in the order file. If
      not the case, then an a warning is printed out, and the variables
      are created at the current position in the ordering.
    */
    l_o_vars = all_variables;
    while (l_o_vars) {
      node_ptr vname = car(l_o_vars);
      node_ptr q = lookup_symbol_hash(vname);

      if (!q || node_get_type(q) != VAR) error_unknown_var_in_order_file(vname);
      if (!car(q)) { /* The current variable has not been specified in
                        the source file. We create it in the following
                        free position in the ordering, i.e. at the
                        end. */
        warning_missing_variable(vname);
        if (opt_verbose_level_gt(options, 1)) {
          print_node(nusmv_stderr, vname);
          fprintf(nusmv_stderr, ":\n");
        }
        /* Side effects on the hash */
        is_input = memberp(vname, input_variables) ? true : false;
        node_bdd_setcar(q, Compile_EncodeVar(vname, cdr(q), is_input, true));
      }
      l_o_vars = cdr(l_o_vars);
    }
  }
  if (!opt_reorder(options)) Compile_WriteOrder(get_output_order_file(options), 0);

  /* converting ADD cube to BDD cube */
  input_variables_bdd      = add_to_bdd(dd_manager, input_variables_add);
  next_input_variables_bdd = add_to_bdd(dd_manager, next_input_variables_add);
  state_variables_bdd      = add_to_bdd(dd_manager, state_variables_add);
  next_state_variables_bdd = add_to_bdd(dd_manager, next_state_variables_add);
}

/**Function********************************************************************

  Synopsis           [Computes the <tt>real_state_variables</tt>, a
  list of ADDs, one for each symbolic scalar variable.]

  Description        [Computes the <tt>real_state_variables</tt>, a
  list of ADDs, one for each symbolic scalar variable representing its
  boolean encoding.]

  SideEffects        [<tt>real_state_variables</tt> is computed.]

  SeeAlso            [eval]

******************************************************************************/
void build_real_state_variables()
{
  node_ptr var_list = state_variables;
  
  real_state_variables = Nil;
  while (var_list) {
    node_ptr var = car(var_list);
    add_ptr dd_var = eval(var, Nil);
 
    real_state_variables = cons((node_ptr)dd_var, real_state_variables);
    var_list = cdr(var_list);
  }
}

/**Function********************************************************************

  Synopsis           [Computes the BDD representing set of initial states.]

  Description        [This function evaluates all the init statements
  corresponding to the variables passed as argument. Moreover, the
  number of states and the number of BDD nodes of the global initial
  set are printed out.]

  SideEffects        []

******************************************************************************/
bdd_ptr Compile_BuildInitBdd(node_ptr list_variables)
{
  node_ptr l;
  bdd_ptr result_bdd;
  add_ptr result_add = add_one(dd_manager);

  if (opt_verbose_level_gt(options, 1)) {
    print_in_process("Evaluating init() assignments and INIT statements", Nil);
  }
  for(l = list_variables; l != Nil; l = cdr(l)) {
    add_ptr init_dd_i; 
    node_ptr var_i = car(l);
    node_ptr init_e_i = lookup_variable_sexp_model_hash(var_i);

    if (opt_verbose_level_gt(options, 2)) {
      indent_node(nusmv_stderr, " computing initial values for variable: ", var_i, "\n");
    }
    if (init_e_i != (node_ptr)NULL) {
      init_dd_i = eval(var_model_sexp_get_init(init_e_i), Nil);
    }
    else {
      init_dd_i = add_one(dd_manager);
    }
    if (opt_verbose_level_gt(options, 2)) {
      indent_node(nusmv_stderr, " size of initial ADD for variable ", var_i, " ");
      fprintf(nusmv_stderr, ": %d ADD nodes\n", add_size(dd_manager, init_dd_i));
    }
    add_and_accumulate(dd_manager, &result_add, init_dd_i);
    add_free(dd_manager, init_dd_i);
  }
  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, "size of global initial set = %g states, %d ADD nodes\n",
            add_count_states(dd_manager, result_add), add_size(dd_manager, result_add));
  }
  result_bdd = add_to_bdd(dd_manager, result_add);
  add_free(dd_manager, result_add);

  return(result_bdd);
}

/**Function********************************************************************

  Synopsis           [Computes the BDD representing model invariants.]

  Description        [This function evaluates all the model invariant
  corresponding to the variables passed as argument.  The result is
  restricted to the set <tt>assumption</tt> using the Coudert and
  Madre algorithm. Moreover the number of BDD nodes of the global set
  of invariant are printed out.]

  SideEffects        [] 

******************************************************************************/
bdd_ptr Compile_BuildInvarBdd(node_ptr list_variables, add_ptr assumption)
{
  node_ptr l;
  bdd_ptr result_bdd;
  add_ptr tmp;
  add_ptr result_add = add_one(dd_manager);

  if (opt_verbose_level_gt(options, 1)) {
    print_in_process("Evaluating invariant assignments and INVAR statements", Nil);
  }
  for(l = list_variables; l != Nil; l = cdr(l)) {
    add_ptr invar_dd_i; 
    node_ptr var_i = car(l);
    node_ptr invar_e_i = lookup_variable_sexp_model_hash(var_i);

    if (opt_verbose_level_gt(options, 2)) {
      indent_node(nusmv_stderr, " computing invariant values for variable: ", var_i, "\n");
    }
    if (invar_e_i != (node_ptr)NULL) {
      invar_dd_i = eval(var_model_sexp_get_invar(invar_e_i), Nil);
    }
    else {
      invar_dd_i = add_one(dd_manager);
    }
    if (opt_verbose_level_gt(options, 2)) {
      indent_node(nusmv_stderr, " size of invariant ADD for variable ", var_i, " ");
      fprintf(nusmv_stderr, ": %d ADD nodes\n", add_size(dd_manager, invar_dd_i));
    }
    add_and_accumulate(dd_manager, &result_add, invar_dd_i);
    add_free(dd_manager, invar_dd_i);
  }
  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, "size of global invariant set = %g states, %d ADD nodes\n",
            add_count_states(dd_manager, result_add), add_size(dd_manager, result_add));
  }
  tmp = result_add;
  result_add = add_simplify_assuming(dd_manager, tmp, assumption);
  add_free(dd_manager, tmp);
  result_bdd = add_to_bdd(dd_manager, result_add);
  add_free(dd_manager, result_add);

  return(result_bdd);
}

/**Function********************************************************************

  Synopsis           [Prints out model statistics.]

  Description        [Depending the partition method used this
  function prints out the statistics; i.e. the size of the transition
  relation or the size of the clusters used in conjunctive partitioning.]

  SideEffects        []

******************************************************************************/
void compilePrintBddModelStatistic(FILE * file, Fsm_BddPtr fsm) {
  CPTrans_Ptr c=(CPTrans_Ptr)Nil;
  switch(get_partition_method(options)) {
  case Monolithic:
    c = Compile_FsmBddGetMonoTrans(fsm);
    break;
  case Threshold:
    c = Compile_FsmBddGetThreshold(fsm);
    break;
  case Iwls95CP:
    c = Compile_FsmBddGetIwls95CP(fsm);
    break;
  default:
   rpterr("Unknown partitioning method\n");
   return;
  }
  compileBddFsmPrintInfo(file, fsm, c);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
static void compileBddFsmPrintInfo(FILE *file, Fsm_BddPtr fsm, CPTrans_Ptr c)
{
  bdd_ptr init_bdd, invar_bdd;

  init_bdd = Compile_FsmBddGetInit(fsm);
  invar_bdd = Compile_FsmBddGetInvar(fsm);
  
  if ((init_bdd != NULL) && (invar_bdd != NULL)) {
    fprintf(file, "BDD nodes representing init set of states: %d\n",
	    bdd_size(dd_manager, init_bdd));
    fprintf(file, "BDD nodes representing invar set: %d\n",
	    bdd_size(dd_manager, invar_bdd));
  }
  CPTransPrintInfo(file, c);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

node_ptr var_model_sexp_build(node_ptr init, node_ptr invar, node_ptr next)
{
  return(cons(init, cons(invar, next)));
}

node_ptr var_model_sexp_get_init(node_ptr vm)
{
  nusmv_assert(vm != Nil);
  
  return(car(vm));
}

node_ptr var_model_sexp_get_invar(node_ptr vm)
{
  nusmv_assert(vm != Nil);
  nusmv_assert(cdr(vm) != Nil);
  
  return(car(cdr(vm)));
}

node_ptr var_model_sexp_get_next(node_ptr vm)
{
  nusmv_assert(vm != Nil);
  nusmv_assert(cdr(vm) != Nil);
  
  return(cdr(cdr(vm)));
}

static node_ptr mk_true() {return(find_node(TRUEEXP, Nil, Nil));}
static node_ptr mk_and(node_ptr a, node_ptr b) {return(new_node(AND, a, b));}


/**Function********************************************************************

  Synopsis           [Initializes the build model.]

  Description [Checks correctness of the NuSMV progrma if not yet
  checked. Initializes the build model if not yet initialized.]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
void Compile_InitializeBuildModel(void) {
  if (cmp_struct_get_compile_check(cmps) == 0) {
    Compile_CheckProgram(cmp_struct_get_procs(cmps),
                         cmp_struct_get_spec(cmps),
                         cmp_struct_get_ltlspec(cmps),
                         cmp_struct_get_invar_spec(cmps),
                         cmp_struct_get_justice(cmps),
			 cmp_struct_get_compassion(cmps));
    /* We keep track that checks have been performed */
    cmp_struct_set_compile_check(cmps);
  }

  if (cmp_struct_get_build_model_setup(cmps) == 0) {
    Compile_InitializeSexpBuildModel(all_variables);
    /* We keep track that construction has been performed */
    cmp_struct_set_build_model_setup(cmps);
  }
  return;
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
void Compile_InitializeSexpBuildModel(node_ptr all_variables)
{
  node_ptr l;
  
  for(l = all_variables; l != Nil; l = cdr(l)){
    node_ptr var_name = car(l);
    node_ptr var_name_i = find_node(SMALLINIT, var_name, Nil);
    node_ptr var_name_n = find_node(NEXT, var_name, Nil);
    node_ptr init_e     = lookup_assign_db_hash(var_name_i);
    node_ptr invar_e    = lookup_assign_db_hash(var_name);
    node_ptr next_e     = lookup_assign_db_hash(var_name_n);
    node_ptr init_sexp  = mk_true();
    node_ptr invar_sexp = mk_true();
    node_ptr trans_sexp = mk_true();

    if (init_e != (node_ptr)NULL) {
      if (car(init_e) != Nil) {
        node_ptr init_assign_expr = find_node(EQDEF, var_name_i, car(init_e));

        init_sexp = mk_and(init_sexp, init_assign_expr);
      }
      if (cdr(init_e) != Nil) {
        init_sexp = mk_and(init_sexp, cdr(init_e));
      }
    }
    if (invar_e != (node_ptr)NULL) {
      if (car(invar_e) != Nil) {
        node_ptr invar_assign_expr = new_node(EQDEF, var_name, car(invar_e));

        invar_sexp = mk_and(invar_sexp, invar_assign_expr);
      }
      if (cdr(invar_e) != Nil) {
        invar_sexp = mk_and(invar_sexp, cdr(invar_e));
      }
    }
    if (next_e != (node_ptr)NULL) {
      if (car(next_e) != Nil) {
        node_ptr next_assign_expr = new_node(EQDEF, var_name_n, car(next_e));

        trans_sexp = mk_and(trans_sexp, next_assign_expr);
      }
      if (cdr(next_e) != Nil) {
        trans_sexp = mk_and(trans_sexp, cdr(next_e));
      }
    }
    insert_variable_sexp_model_hash(var_name, var_model_sexp_build(init_sexp, invar_sexp, trans_sexp));
  }
}

