/**CFile***********************************************************************

  FileName    [compileOrd.c]

  PackageName [compile]

  Synopsis    [Routines for reading and writing variable orders.]

  Description [Routines for reading and writing variable orders.]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileOrd.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [reads from a file the variable order.]

  Description        [This function reads from a file the variable
  order to be used in the system. The grammar of the order file is as
  follows:
  <pre>
    orderfile    ::=  variable NL
    variable     ::=  complex_atom
    complex_atom ::=  simple_atom | simple_atom DOT complex_atom
    simple_atom  ::=  atom
  </pre>
  Where <tt>NL</tt> is a new line, and <tt>DOT</tt> is the period character.
  ]

  SideEffects        []

  SeeAlso            [write_order]

******************************************************************************/
node_ptr Compile_ReadOrder(const char *input_order_file)
{
  if (input_order_file != NULL) {
    extern int yylex();
    int token;
    node_ptr read_variables = Nil;

    /* Save the main input file */
    char * tmp_input_file = get_input_file(options);
    set_input_file(options, (char *)input_order_file);

    Parser_OpenInput(input_order_file);
    token = yylex();
    while (token) {
      node_ptr var = Nil;

      while (1) {
	if (token != ATOM) rpterr("syntax error");
	var = find_node(DOT, var, (node_ptr)find_atom(yylval.node));
	token = yylex();
	while( token == LB ) {
	  token = yylex();
	  if ( token != NUMBER ) rpterr("syntax error");
	  var = find_node(ARRAY, var,
                                 find_node(NUMBER,
                                           (node_ptr)eval_num(yylval.node, Nil), Nil));
	  token = yylex();
	  if (token != RB ) rpterr("syntax error");
	  token = yylex();
	}
	if (token != DOT) break;
	token = yylex();
      }
      read_variables = cons(var, read_variables);
    }
    read_variables = reverse(read_variables);
    Parser_CloseInput();

    /* Restore the main input file */
    set_input_file(options, tmp_input_file);

    return(read_variables);
  }
  return(Nil);
}

/**Function********************************************************************

  Synopsis           [Writes on a file the variable order.]

  Description        [This function writes the variable order
  currently in use in the system in the specified output file. If the
  option "-reorder" has been used then, then this function exits (if
  the batch interaction mode is used). The file generated as output
  can be used as input order file for next computations. If the specified 
  output file is an empty string ("" or NULL, see util_is_string_null) output
  is redirected to stdout.]

  SideEffects        []

  SeeAlso            [Compile_ReadOrder]

******************************************************************************/
void Compile_WriteOrder(const char *output_order_file_name, int force)
{
  if (!opt_reorder(options) 
      && (is_default_order_file(options) || 
	  util_is_string_null(output_order_file_name))
      && (force == 0)) return;
  {
    FILE *oof;
    node_ptr ordering;
    
    if (!util_is_string_null(output_order_file_name)) {
      if ((oof = fopen(output_order_file_name, "w")) == NULL) {
        rpterr("output_order: unable to open file %s", output_order_file_name);
      }
    }
    else {
      /* use stdout: */
      oof = nusmv_stdout;
    }
    
    ordering = Compile_GetOrdering();

    while(ordering != Nil) {
      node_ptr name = car(ordering);
      
      print_node(oof, name);
      fprintf(oof, "\n"); 
      ordering = cdr(ordering);
    }
    
    /* close the file: */
    if (oof != nusmv_stdout) { 
      if (fclose(oof) == EOF) {
	rpterr("cannot close %s", output_order_file_name);
      }
    }

    if (opt_verbose_level_gt(options, 0)) {
      if (output_order_file_name != NULL) {
	fprintf(nusmv_stderr, "NuSMV: variable order output to file %s\n", 
		output_order_file_name);
      }
    }
    if (opt_batch(options)) 
      if (!opt_reorder(options)) nusmv_exit(0);
  }
}


