/**CFile***********************************************************************

  FileName    [compileIwls95.c]

  PackageName [compile]

  Synopsis    [Build a conjunctively decomposed transition relation
  using the clustering heuristic by Ranjan et al, Iwls95.]

  Description [Conjuncitively partitioned models are very sensitive to
  the way clusters are built and ordered. The algorithm implemented
  here takes into account the number of next state variables
  introduced, when choosing the next cluster in the order. The
  ordering of the clusters takes into account the \"index\" of the
  variables, because it was found that quantifying out a variable from
  a function becomes computationally less expensive as the depth (the
  index) of the variable in the BDD ordering increases.<br>

  <ol>
  <li> R. K. Ranjan and A. Aziz and B. Plessier and C. Pixley and
       R. K. Brayton, "Efficient BDD Algorithms for FSM Synthesis and
       Verification", IEEE/ACM Proceedings International Workshop on Logic
       Synthesis, Lake Tahoe (NV), May 1995.</li>
  </ol>]

  SeeAlso     [compileMono.c, compileThreshold.c]

  Author      [Marco Roveri and Emanuele Olivetti]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileIwls95.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/* Set this to 1 for activating debugging of IWLS95 routines */
/* It slow down the system performances. */
#define DEBUG_IWLS95 0

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [This structure contains the information about a particular
               Iwls95CP cluster, used during reordering computation.]

  Description [There is one Iwls95ClusterInfo_struct per cluster. The
  various fields of this structure contain the information for
  ordering the clusters for image computation purposes.]

  SeeAlso     []

******************************************************************************/
struct CPClusterInfoIwls95_struct {
  int i;
  bdd_ptr Supp_Ti;      /* Set of Support of Cluster T_i = S(T_i)     */
  bdd_ptr Supp_Q_Ci;    /* {v : v \in S(T_j), j != i, j \in Q}        */
  bdd_ptr PSPI_Ti;      /* {v : v \in (PS U PI) & v \in S(T_i)}       */
  bdd_ptr NS_Ti;        /* {v : v \in NS & v \in S(T_i)}              */
  double v_c;           /* Number of variables which can be existentially
                           quantified when T_i is multiplied in the product */
  double w_c;           /* Number of current and input variables in
                           the support of T_i */
  double x_c;           /* Number of current and input variables which
                           have not yet been quantified */
  double y_c;           /* Number of next variables that would be
                           introduced in the product by multiplying T_i */
  double z_c;           /* Number of next variables not yet introduced
                           in the product */
  double m_c;           /* The maximum "index" of a variable to be
                           quantified in S(T_i) */
  double M_c;           /* The maximum BDD index of a variable to be
                           quantified out in the remaining clusters */
};

typedef struct CPClusterInfoIwls95_struct CPClusterInfoIwls95;

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static float Iwls95ComputeBenefit ARGS((CPClusterInfoIwls95 *,
					Iwls95OptionStruct_t *));
static void Iwls95ClusterInfoFree ARGS((DdManager *, CPClusterInfoIwls95 *));
static void Iwls95ComputeClusterInfoAux ARGS((DdManager *, CPList, double *,
					      double *, double *, 
					      bdd_ptr, bdd_ptr));

static bdd_ptr Iwls95Compute_Supp_Q_Ci ARGS((DdManager *, CPList, CPCluster *));
static CPClusterInfoIwls95 * Iwls95ClusterInfoAlloc ARGS((void));

static void Iwls95MakePartition ARGS((DdManager * dd, CPList Ti_list, 
				CPList * new_Ti_list,
				Iwls95OptionStruct_t * Options,
				bdd_ptr PS, bdd_ptr PI, bdd_ptr NS));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [The implicitly conjoined transition relation is
  computed accordingly to the heuristic described in IWLS95.]

  Description        [Computes an implicitly conjoined transition
  relation, in which clusters are ordered accordingly the heuristic
  defined in IWLS95.<br>

  First computes the BDD representing the transition relation of the
  next function of each scalar variable (<code>next(x) := expr
  </code>). This initializes the set of clusters.<br> 

  Then, the list of clusters is ordered, and if necesessary recombine
  clusters to obtain a small number of clusters whose size in BDD
  nodes is smaller than a given threshold. Then clusters are ordered
  again.]

  SideEffects        []

  SeeAlso            [Compile_BuildModelMonolithic Compile_BuildModelThreshold]

******************************************************************************/
CPTrans_Ptr Compile_BuildModelIwls95CP(node_ptr list_variables,
                                   Iwls95OptionStruct_t * Options,
                                   add_ptr assumption)
{
  CPTrans_Ptr result;
  CPList Fwd_trans = CPListEmpty;
  CPList Bwd_trans = CPListEmpty;
  CPList cluster_bdd = CPListEmpty;

  cluster_bdd = CPListBuildElementaryBDDList(list_variables, assumption);

  {
    bdd_ptr PI = bdd_one(dd_manager);
    bdd_ptr PS = bdd_and(dd_manager, input_variables_bdd, state_variables_bdd);
    bdd_ptr NS = bdd_and(dd_manager, next_input_variables_bdd, next_state_variables_bdd);

    /* The main algorithm for generating and ordering clusters */
    /* Forward */

    Iwls95MakePartition(dd_manager, cluster_bdd, &Fwd_trans, Options, PS, PI, NS);

    /* Backward (WARNING: NS and PS MUST BE inverted compared to the forward case!!!) */
    Iwls95MakePartition(dd_manager, cluster_bdd, &Bwd_trans, Options, NS, PI, PS);

    bdd_free(dd_manager, PS);
    bdd_free(dd_manager, PI);
    bdd_free(dd_manager, NS);
    CPListFree(dd_manager, cluster_bdd);
  }
  
  result = CPTransAlloc();
  CPTransSetForward(result, Fwd_trans);
  CPTransSetBackward(result, Bwd_trans);

  if (opt_verbose_level_gt(options, 1)) {
    CPTransPrintInfo(nusmv_stderr, result);
  }


#if DEBUG_IWLS95
  compileCPTransCheckPartitionedTransition(result, list_variables, assumption);
#endif

  return(result);
}

/**Function********************************************************************

  Synopsis    [Gets the necessary options for computing the image and returns
               in the option structure.]

  Description [Gets the necessary options for computing the image and returns
               in the option structure.]

  SideEffects []

******************************************************************************/
Iwls95OptionStruct_t * Iwls95GetOptions(void)
{
  char *flagValue;
  Iwls95OptionStruct_t *option;
  
  option = ALLOC(Iwls95OptionStruct_t, 1);

  option->clusterSize = get_image_cluster_size(options);

  flagValue = Cmd_FlagReadByName("image_verbosity");
  if (flagValue == NIL(char)){
    option->verbosity = 0; /* the default value */
  }
  else {
    option->verbosity = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W1");
  if (flagValue == NIL(char)){
    option->W1 = 6; /* the default value */
  }
  else {
    option->W1 = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W2");
  if (flagValue == NIL(char)){
    option->W2 = 1; /* the default value */
  }
  else {
    option->W2 = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W3");
  if (flagValue == NIL(char)){
    option->W3 = 1; /* the default value */
  }
  else {
    option->W3 = atoi(flagValue);
  }

  flagValue = Cmd_FlagReadByName("image_W4");
  if (flagValue == NIL(char)){
    option->W4 = 2; /* the default value */
  }
  else {
    option->W4 = atoi(flagValue);
  }
  return option;
}

/**Function********************************************************************

  Synopsis           [Given a list of cluster, this function
  initializes the data structures to perform image computation.]

  Description        [This function builds the
  data structures to perform image computation. <br>
  This process consists of the following steps:<br>
  <ol>
  <li> Ordering of the clusters given as input accordingly with the
       heuristic described in IWLS95.</li>
  <li> Clustering of the result of previous step accordingly the
       threshold value stored in the option \"image_cluster_size\".</li>
  <li> Ordering of the result of previous step accordingly with the
       heuristic described in IWLS95.</li>
  </ol>]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void Iwls95MakePartition(DdManager * dd, CPList Ti_list, 
				CPList * new_Ti_list,
				Iwls95OptionStruct_t * Options,
				bdd_ptr PS, bdd_ptr PI, bdd_ptr NS)
{
  CPList Clustered = CPListEmpty;
  CPList NewClustered = CPListEmpty;
  
  *new_Ti_list = CPListEmpty;

  if (opt_iwls95_preorder(options) == true) {

    /* (pre)Ordering , clustering, reordering */
    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "...Ordering clusters...");
    }
      
    Iwls95OrderClusters(dd, Ti_list, &Clustered, PS, PI, NS, Options);
      
    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "...done\nClustering clusters...");
    }
      
    if (opt_affinity(options) != true) {     
      /* Threshold clustering */
      CPListBuildThreshold(dd, Clustered, &NewClustered, Options->clusterSize);
    }
    else {
      /* Threshold clustering with affinity */
      compileCPMakeAffinityClusters(dd, Clustered, &NewClustered, Options->clusterSize);
    }
      
    /* We free intermediate results */
    CPListFree(dd, Clustered);
      
    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "\nOrdering clusters...");
    }
      
    Iwls95OrderClusters(dd, NewClustered, new_Ti_list, PS, PI, NS, Options);

    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "...done\n");
    }
      
    CPListFree(dd, NewClustered);
  }
  else {
    /* Clustering , ordering */
    /* Here it may be interesting to use heuristic when the Cluster
       Threshold is 1 */
    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "Clustering clusters...");
    }

    if (opt_affinity(options) != true) {
      /* Threshold clustering */
      CPListBuildThreshold(dd, Ti_list, &NewClustered, Options->clusterSize);
    }
    else {
      /* Threshold clustering with affinity */
      compileCPMakeAffinityClusters(dd, Ti_list, &NewClustered, Options->clusterSize);
    }

    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "...done\nOrdering clusters...");
    }

    Iwls95OrderClusters(dd, NewClustered, new_Ti_list, PS, PI, NS, Options);

    if (opt_verbose_level_gt(options, 2)) {
      fprintf(nusmv_stderr, "...done\n");
    }
      
    CPListFree(dd, NewClustered);
  }
}

/**Function********************************************************************

  Synopsis           [This function orders the clusters given as input
  accordingly with the order heuristic described in IWLS95.]

  Description        [This function orders the clusters given as input
  accordingly with the order heuristic described in IWLS95. The
  ordering is performed according to a cost function attached to each
  cluster. It also removes further occurence of same clusters.]

  SideEffects        [The result is stored in \"Ordered\"]
  SeeAlso            [Iwls95ClusterInfo_struct,  Iwls95OptionStruct]

******************************************************************************/
void Iwls95OrderClusters(DdManager * dd, CPList Unordered, CPList *Ordered,
                         bdd_ptr PS, bdd_ptr PI, bdd_ptr NS,
                         Iwls95OptionStruct_t *Options)
{
  CPList clist = Unordered;
  CPList Olist = CPListEmpty;

  while (CPListIsNotEmpty(clist)) {
    double best_benefit = -9999;
    CPCluster * newc;
    CPCluster * Best_Cluster = (CPCluster *)NULL;
    CPList C_info_list;
    CPList C_info_list_copy;

    C_info_list = Iwls95ComputeClusterInfo(dd, clist, PS, PI, NS);
    C_info_list_copy = C_info_list;

    while (CPListIsNotEmpty(C_info_list_copy)) {
      CPCluster * current2 = CPListGetItem(C_info_list_copy);
      CPClusterInfoIwls95 * C_info = (CPClusterInfoIwls95 *)CPClusterGetInfo(current2);
      double benefit = (double)Iwls95ComputeBenefit(C_info, Options);

      if (benefit > best_benefit) {
        best_benefit = benefit;
	Best_Cluster = current2;
      }

      C_info_list_copy = CPListNext(C_info_list_copy);
    }

    newc = CPClusterDup(Best_Cluster);
    CPClusterSetInfo(newc,(void*)NULL); 
    /* infos are not duplicated and will be freed in next line */
    CPListFreeAll(dd, C_info_list, (int (*)(void *, void *))Iwls95ClusterInfoFree);
    clist = CPListDeleteCluster(clist, newc);
    Olist = CPListCons(newc, Olist);
  }
  *Ordered = CPListReverse(CPListDup(Olist));
}

/**Function********************************************************************

  Synopsis           [Computes the parameters necessary to use iwls95]

  Description        [Fill all the fields of the list of clusters taken as input.]

  SideEffects        [Side effects on <code>Q</code>]

  SeeAlso            []

******************************************************************************/
CPList Iwls95ComputeClusterInfo(
 DdManager *dd /* The DD manager */,
 CPList Q /* The list of clusters whose fields have to be filled */,
 bdd_ptr PS /* Cube of present state variables */,
 bdd_ptr PI /* Cube of input variables */,
 bdd_ptr NS /* Cube of next state variables */)
{
  CPList result=(CPList)NULL;
  CPList C_list = Q;
  bdd_ptr  PSPI = bdd_and(dd, PS, PI);
  double x_c, z_c, M_c;

  Iwls95ComputeClusterInfoAux(dd, Q, &x_c, &z_c, &M_c, PSPI, NS);
  while(CPListIsNotEmpty(C_list)) {
    CPCluster * Cur_Cluster = CPClusterDup(CPListGetItem(C_list));
    bdd_ptr Ti = CPClusterGetTi(Cur_Cluster);
    CPClusterInfoIwls95 * CI_Ti = Iwls95ClusterInfoAlloc();

    CPClusterSetInfo(Cur_Cluster, CI_Ti);
    CI_Ti->Supp_Ti     = bdd_support(dd, Ti);
    CI_Ti->PSPI_Ti     = bdd_cube_intersection(dd, CI_Ti->Supp_Ti, PSPI);
    CI_Ti->NS_Ti       = bdd_cube_intersection(dd, CI_Ti->Supp_Ti, NS);
    CI_Ti->Supp_Q_Ci   = Iwls95Compute_Supp_Q_Ci(dd, Q, Cur_Cluster);
    CPClusterSetEi(Cur_Cluster, bdd_cube_diff(dd, CI_Ti->PSPI_Ti, CI_Ti->Supp_Q_Ci));
    {
      double v_c = bdd_size(dd, CPClusterGetEi(Cur_Cluster)) - 1;
      double w_c = bdd_size(dd, CI_Ti->PSPI_Ti) - 1;
      double y_c = bdd_size(dd, CI_Ti->NS_Ti) - 1;
      double m_c = bdd_get_lowest_index(dd, CI_Ti->PSPI_Ti);

      CI_Ti->v_c = (v_c < 0) ? 0 : v_c;
      CI_Ti->w_c = (w_c < 0) ? 0 : w_c;
      CI_Ti->x_c = (x_c < 0) ? 0 : x_c;
      CI_Ti->y_c = (y_c < 0) ? 0 : y_c;
      CI_Ti->z_c = (z_c < 0) ? 0 : z_c;
      CI_Ti->m_c = m_c;
      CI_Ti->M_c = M_c;
    }

    result = CPListCons(Cur_Cluster, result);
    C_list = CPListNext(C_list);
  }
  bdd_free(dd, PSPI);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Computes the set Supp_Q_Ci.]

  Description        [Computes the set of present an primary input variables
  that belong to the set of support of cluster Ci, and do not belong to the
  set of support of each cluster Cj, for j != i and Cj belonging to the set
  of the not yet ordered clusters. The set Supp_Q_Ci is formally defined as:
  Supp_Q_Ci = {v \in (PS U PI) /\ v \not\in S(T_Cj), Cj != Ci, Cj \in Q}]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr Iwls95Compute_Supp_Q_Ci(DdManager *dd, CPList Q,
                                CPCluster * Ci)
{
  bdd_ptr Acc = bdd_one(dd);

  while (CPListIsNotEmpty(Q)) {
    CPCluster * Cj = CPListGetItem(Q);

    if (CPClusterEqual(Cj, Ci) == 0) {
      bdd_ptr Supp = bdd_support(dd, CPClusterGetTi(Cj));

      bdd_and_accumulate(dd, &Acc, Supp);
      bdd_free(dd, Supp);
    }
    Q = CPListNext(Q);
  }
  return(Acc);
}

/**Function********************************************************************

  Synopsis           [Computes some global parameters necessary in the
  ordering of clusters.]

  Description        [Computes some global parameters necessary in the
  ordering of clusters.]

  SideEffects        [The results are returned in x_c, z_c, M_c]

  SeeAlso            [Iwls95ComputeClusterInfo, Iwls95OptionStruct, Iwls95ClusterInfo_struct]

******************************************************************************/
static void Iwls95ComputeClusterInfoAux(DdManager * dd, CPList Q, double * x_c,
                                        double * z_c, double * M_c,
                                        bdd_ptr PSPI, bdd_ptr NS)
{
  bdd_ptr Acc = CPListComputeClustersCube(dd, Q);
  bdd_ptr Acc_PSPI = bdd_cube_intersection(dd, Acc, PSPI);
  bdd_ptr Acc_NS = bdd_cube_intersection(dd, Acc, NS);

  *x_c = bdd_size(dd, Acc_PSPI);
  *z_c = bdd_size(dd, Acc_NS);
  *M_c = bdd_get_lowest_index(dd, Acc_PSPI);
  bdd_free(dd, Acc);
  bdd_free(dd, Acc_PSPI);
  bdd_free(dd, Acc_NS);
}

/**Function********************************************************************

  Synopsis           [Computes the cube of the set of support of all
  the clusters.]

  Description        [Given a list of clusters, it computes their set
  of support.] 

  SideEffects        []

******************************************************************************/
bdd_ptr Iwls95ComputeClustersCube(DdManager * dd, CPList set_of_clusters)
{
  CPList soc = set_of_clusters;
  
  bdd_ptr result = bdd_one(dd);
  while (CPListIsNotEmpty(soc)) {
    CPCluster * Cur_Cluster = CPListGetItem(soc);
    bdd_ptr Supp_Ti = bdd_support(dd, CPClusterGetTi(Cur_Cluster));

    bdd_and_accumulate(dd, &result, Supp_Ti);
    bdd_free(dd, Supp_Ti);
    soc = CPListNext(soc);
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [Allocates and initializes memory for Iwls95Cluster_Info struct.]

  Description        [Allocates and initializes memory for Iwls95Cluster_Info struct.]

  SideEffects        []

******************************************************************************/
static CPClusterInfoIwls95 * Iwls95ClusterInfoAlloc(void)
{
  CPClusterInfoIwls95 * CI;

  CI = ALLOC(CPClusterInfoIwls95, 1);
  CI->Supp_Ti     = (bdd_ptr)NULL; 
  CI->Supp_Q_Ci   = (bdd_ptr)NULL;
  CI->PSPI_Ti     = (bdd_ptr)NULL;
  CI->NS_Ti       = (bdd_ptr)NULL;
  CI->v_c         = -1;
  CI->w_c         = -1;
  CI->x_c         = -1;
  CI->y_c         = -1;
  CI->z_c         = -1;
  CI->m_c         = -1;
  CI->M_c         = -1;
  return(CI);
}

/**Function********************************************************************

  Synopsis           [Frees the  memory allocated for Iwls95Cluster_Info struct.]

  Description        [Frees the  memory allocated for Iwls95Cluster_Info struct.]

  SideEffects        []

******************************************************************************/
void Iwls95ClusterInfoFree(DdManager *dd, CPClusterInfoIwls95 * CI)
{
  if (CI != (CPClusterInfoIwls95 *)NULL) {
    if (CI->Supp_Ti   != (bdd_ptr)NULL) bdd_free(dd, CI->Supp_Ti);
    if (CI->Supp_Q_Ci != (bdd_ptr)NULL) bdd_free(dd, CI->Supp_Q_Ci);
    if (CI->PSPI_Ti   != (bdd_ptr)NULL) bdd_free(dd, CI->PSPI_Ti);
    if (CI->NS_Ti     != (bdd_ptr)NULL) bdd_free(dd, CI->NS_Ti);
    FREE(CI);
  }
}

/**Function********************************************************************

  Synopsis           [Frees a list of Iwls95Cluster_Info struct.]

  Description        [Frees a list of Iwls95Cluster_Info struct.]

  SideEffects        []

******************************************************************************/
void Iwls95FreeClustersList(DdManager *dd, node_ptr CL)
{
  CPListFreeAll(dd, CL,(int (*)(void *, void *))Iwls95ClusterInfoFree);
}

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis     [Computes the benefit of the given cluster.]

  Description  [Computes the benefit function associated to each cluster.
       The objective function attached with each Ti is
       Ci =  W1*C1 + W2*C2 - W3*C3 + W4*C4
       where:
       W1 = weight attached with variables getting smoothed
       W2 = weight attached with the support count of the Ti
       W3 = weight attached with variables getting introduced
       W4 = weight attached with the max bdd id of the Ti
       C1 = v_c/w_c
       C2 = w_c/x_c
       C3 = y_c/z_c
       C4 = m_c/M_c
       v_c = number of ps and pi variables which can be existentially
             quantified when Ti is multiplied in the product;
       w_c = number of ps and pi variables in the support of Ti;
       x_c = total number of ps and pi variables remaining which have not yet
             been smoothed out;
       y_c = number of ns variables that would be introduced in the product
             by multiplying Ti;
       z_c = total number of ns variables which have not been introduced
             so far.
       m_c = value of Max bdd id of Ti
       M_c = value of Max bdd id across all the Ti's remaining to be multiplied

       Get the weights from the global option.]

  SideEffects  []

******************************************************************************/
static float Iwls95ComputeBenefit(CPClusterInfoIwls95 * Ci,
                                  Iwls95OptionStruct_t * Option)
{
  int W1, W2, W3, W4;
  float benefit;
  
  W1 = Option->W1;
  W2 = Option->W2;
  W3 = Option->W3;
  W4 = Option->W4;
 
  benefit = 0;
  benefit += (Ci->w_c ? (W1 * (float)(Ci->v_c / Ci->w_c)) : 0);
  benefit += (Ci->x_c ? (W2 * (float)(Ci->w_c / Ci->x_c)) : 0);
  benefit -= (Ci->z_c ? (W2 * (float)(Ci->y_c / Ci->z_c)) : 0);
  benefit += (Ci->M_c ? (W2 * (float)(Ci->m_c / Ci->M_c)) : 0);

  return(benefit);
}

/**Function********************************************************************

  Synopsis    [Prints the option values used in IWLS95 technique for
  image computation.]

  Description [Prints the option values used in IWLS95 technique for
  image computation.]

  SideEffects []

******************************************************************************/
int Iwls95PrintOption(FILE * fp)
{
  Iwls95OptionStruct_t * option = Iwls95GetOptions();

  if (option == NIL(Iwls95OptionStruct_t)) {
    fprintf(fp, "Iwls95 Option appear to be NULL\n");
    return(1);
  }
  fprintf(fp, "Printing Information about Image method: IWLS95\n"); 
  fprintf(fp, "   Threshold Value of Bdd Size For Creating Clusters = %d\n", option->clusterSize);
  fprintf(fp, "   Verbosity = %d\n", option->verbosity);
  fprintf(fp, "   W1 =%3d\n", option->W1);
  fprintf(fp, "   W2 =%2d\n", option->W2);
  fprintf(fp, "   W3 =%2d\n", option->W3);
  fprintf(fp, "   W4 =%2d\n", option->W4);
  fprintf(fp, "Use \"set image_cluster_size value\" to set this to desired value.\n"); 
  fprintf(fp, "Use \"set image_verbosity value\" to set this to desired value.\n");

  fprintf(fp, "Use \"set image_W? value\" to set these to the desired values.\n");
  return(0);
}
