/**CFile***********************************************************************

  FileName    [compileFsm.c]

  PackageName [compile]

  Synopsis    [FSM data structures and manipulation routines in 
  different formats.]

  Description [This file provides FSM data structure in different
  formats, i.e. scalar in sexp, boolean in sexp, boolean in RBC and
  boolean in BDD. In each format a FSM is composed of the initial
  states, the invariant conditions, the transition relation and a set
  of fairness conditions. For each data structure the corresponding
  constructors/destructors are provided.]

  SeeAlso     []

  Author      [Marco Roveri and Emanuele Olivetti]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileFsm.c,v 1.5 2003/10/28 17:29:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/
struct _Fsm_SexpRec {
  node_ptr Init;
  node_ptr Invar;
  node_ptr Trans;
  node_ptr Justice;
  node_ptr Compassion;
};

struct _Fsm_BddRec {
  bdd_ptr Init;
  bdd_ptr Invar;
  CPTrans_Ptr MonoTrans;
  CPTrans_Ptr Iwls95CP;
  CPTrans_Ptr Threshold;
  DPTrans_Ptr Saturation;
  DPTrans_Ptr Phase1;
  DPTrans_Ptr Phase2;
  node_ptr Justice;
  node_ptr Compassion;
  bdd_ptr Fairstates;
};


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/



/**Variable********************************************************************

  Synopsis    [Hash to keep track of already considered clusters.]

  Description [Hash to keep track of already considered clusters.]

  SeeAlso     []

******************************************************************************/
static st_table * sexp_model_hash = (st_table*)NULL;
static void init_sexp_model_hash() 
{
  sexp_model_hash = st_init_table(st_ptrcmp, st_ptrhash);
  if (sexp_model_hash == (st_table *)NULL) {
    fprintf(nusmv_stderr, "init_sexp_model_hash: unable to allocate local hash.\n");
    nusmv_exit(1);
  }
}
static void quit_sexp_model_hash()
{
  nusmv_assert(sexp_model_hash != (st_table *)NULL);
  st_free_table(sexp_model_hash);
}
static void insert_sexp_model_hash(node_ptr el, int c)
{
  if (st_add_direct(sexp_model_hash, (char *)el, (char *)c) == ST_OUT_OF_MEM) {
    fprintf(nusmv_stderr, "insert_sexp_model_hash: Unable to insert result in local hash.\n");
    nusmv_exit(1);
  }
}
static int query_sexp_model_hash(node_ptr el, int c)
{
  int v;
  int result = st_lookup(sexp_model_hash, (char *)el, (char **)&v);

  if (result) {
    return(v == c);
  }
  return(result);
}

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/* SEXP SCALAR AND BOOL FSM */
Fsm_SexpPtr Compile_FsmSexpAlloc()
{
  Fsm_SexpPtr fsm = ALLOC(Fsm_SexpRec, 1);

  if (fsm == (Fsm_SexpPtr)NULL) {
    fprintf(nusmv_stderr, "Compile_FsmSexpAlloc: Unable to allocate memory.\n");
    nusmv_exit(1);
    return(NIL(Fsm_SexpRec));
  }
  fsm->Init       = (node_ptr)NULL;
  fsm->Invar      = (node_ptr)NULL;
  fsm->Trans      = (node_ptr)NULL;
  fsm->Justice    = (node_ptr)NULL;
  fsm->Compassion = (node_ptr)NULL;
  return(fsm);
}

Fsm_SexpPtr Compile_FsmSexpDup(Fsm_SexpPtr fsm)
{
  Fsm_SexpPtr nfsm;

  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  nfsm = Compile_FsmSexpAlloc();
  nfsm->Init       = fsm->Init;
  nfsm->Invar      = fsm->Invar;
  nfsm->Trans      = fsm->Trans;
  nfsm->Justice    = fsm->Justice;
  nfsm->Compassion = fsm->Compassion;
  return(nfsm);
}
void Compile_FsmSexpFree(Fsm_SexpPtr fsm)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  FREE(fsm);
}

void Compile_FsmSexpSetInit(Fsm_SexpPtr fsm, node_ptr init)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  fsm->Init = init;
}
node_ptr Compile_FsmSexpGetInit(Fsm_SexpPtr fsm)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  return(fsm->Init);
}

void Compile_FsmSexpSetInvar(Fsm_SexpPtr fsm, node_ptr invar)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  fsm->Invar = invar;
}
node_ptr Compile_FsmSexpGetInvar(Fsm_SexpPtr fsm)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  return(fsm->Invar);
  
}


void Compile_FsmSexpSetTrans(Fsm_SexpPtr fsm, node_ptr trans)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  fsm->Trans = trans;
}
node_ptr Compile_FsmSexpGetTrans(Fsm_SexpPtr fsm)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  return(fsm->Trans);
}

void Compile_FsmSexpSetJustice(Fsm_SexpPtr fsm, node_ptr justice)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  fsm->Justice = justice;
}
node_ptr Compile_FsmSexpGetJustice(Fsm_SexpPtr fsm)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  return(fsm->Justice);
}
void Compile_FsmSexpSetCompassion(Fsm_SexpPtr fsm, node_ptr compassion)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  fsm->Compassion = compassion;
}
node_ptr Compile_FsmSexpGetCompassion(Fsm_SexpPtr fsm)
{
  nusmv_assert(fsm != (Fsm_SexpPtr)NULL);
  return(fsm->Compassion);
}


/* FSM BDD */
Fsm_BddPtr Compile_FsmBddAlloc()
{
  Fsm_BddPtr fsm = ALLOC(Fsm_BddRec, 1);

  if (fsm == (Fsm_BddPtr)NULL) {
    fprintf(nusmv_stderr, "Compile_FsmBddAlloc: Unable to allocate memory.\n");
    nusmv_exit(1);
    return(NIL(Fsm_BddRec));
  }
  fsm->Init        = (bdd_ptr)NULL;
  fsm->Invar       = (bdd_ptr)NULL;
  fsm->MonoTrans   = (CPTrans_Ptr)NULL;
  fsm->Iwls95CP    = (CPTrans_Ptr)NULL;
  fsm->Threshold   = (CPTrans_Ptr)NULL;
  fsm->Saturation  = (DPTrans_Ptr)NULL;
  fsm->Phase1      = (DPTrans_Ptr)NULL;
  fsm->Phase2      = (DPTrans_Ptr)NULL;
  fsm->Justice     = (node_ptr)NULL;
  fsm->Compassion  = (node_ptr)NULL;
  fsm->Fairstates  = (bdd_ptr)NULL;
  return(fsm);
}

static node_ptr dd_node_dup(node_ptr n) 
{
  nusmv_assert(node_get_type(n) == BDD);
  return(new_node(BDD,(node_ptr)bdd_dup((bdd_ptr)(car(n))),Nil));
}

static node_ptr dd_pair_node_dup(node_ptr n) 
{
  nusmv_assert(node_get_type(n) == CONS && 
	       node_get_type(car(n)) == BDD && 
	       node_get_type(cdr(n)) == BDD);
  return(cons(new_node(BDD,(node_ptr)bdd_dup((bdd_ptr)(car(car(n)))),Nil),
	      new_node(BDD,(node_ptr)bdd_dup((bdd_ptr)(car(cdr(n)))),Nil)));
}

Fsm_BddPtr Compile_FsmBddDup(Fsm_BddPtr fsm)
{
  Fsm_BddPtr nfsm;

  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  nfsm = Compile_FsmBddAlloc();

  if (fsm->Init != (bdd_ptr)NULL)     
    nfsm->Init = bdd_dup(fsm->Init);

  if (fsm->Invar != (bdd_ptr)NULL)    
    nfsm->Invar = bdd_dup(fsm->Invar);

  if (fsm->MonoTrans != (CPTrans_Ptr)NULL) 
    nfsm->MonoTrans = CPTransDup(fsm->MonoTrans);

  if (fsm->Iwls95CP != (CPTrans_Ptr)NULL)
    nfsm->Iwls95CP = CPTransDup(fsm->Iwls95CP);

  if (fsm->Threshold != (CPTrans_Ptr)NULL)
    nfsm->Threshold = CPTransDup(fsm->Threshold);

  if (fsm->Saturation != (DPTrans_Ptr)NULL)
    nfsm->Saturation = DPTransDup(fsm->Saturation);

  if (fsm->Phase1 != (DPTrans_Ptr)NULL)
    nfsm->Phase1 = DPTransDup(fsm->Phase1);

  if (fsm->Phase2 != (DPTrans_Ptr)NULL)
    nfsm->Phase2 = DPTransDup(fsm->Phase2);

  if (fsm->Justice != (node_ptr)NULL)
    nfsm->Justice = map(dd_node_dup, fsm->Justice);

  if (fsm->Compassion != (node_ptr)NULL)
    nfsm->Compassion = map(dd_pair_node_dup, fsm->Compassion);

  if (fsm->Fairstates != (bdd_ptr)NULL)
    nfsm->Fairstates = bdd_dup(fsm->Fairstates);

  return(nfsm);
}

static void dd_node_free(node_ptr n) 
{
  nusmv_assert(node_get_type(n) == BDD);
  bdd_free(dd_manager, (bdd_ptr)(car(n)));
}

static void dd_pair_node_free(node_ptr n) 
{
  nusmv_assert(node_get_type(n) == CONS && 
	       node_get_type(car(n)) == BDD && 
	       node_get_type(cdr(n)) == BDD);
  bdd_free(dd_manager, (bdd_ptr)(car(car((n)))));
  bdd_free(dd_manager, (bdd_ptr)(car(cdr((n)))));
}

void Compile_FsmBddFree(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);

  if (fsm->Init != (bdd_ptr)NULL)      bdd_free(dd_manager, fsm->Init);
  if (fsm->Invar != (bdd_ptr)NULL)     bdd_free(dd_manager, fsm->Invar);
  if (fsm->MonoTrans != (CPTrans_Ptr)NULL)  CPTransFree(fsm->MonoTrans);
  if (fsm->Iwls95CP != (CPTrans_Ptr)NULL)   CPTransFree(fsm->Iwls95CP);
  if (fsm->Threshold != (CPTrans_Ptr)NULL)  CPTransFree(fsm->Threshold);
  if (fsm->Saturation != (DPTrans_Ptr)NULL)  DPTransFree(fsm->Saturation);
  if (fsm->Phase1 != (DPTrans_Ptr)NULL)  DPTransFree(fsm->Phase1);
  if (fsm->Phase2 != (DPTrans_Ptr)NULL)  DPTransFree(fsm->Phase2);
  if (fsm->Justice != (node_ptr)NULL) {
    walk(dd_node_free, fsm->Justice);
    free_list(fsm->Justice);
  }
  if (fsm->Compassion != (node_ptr)NULL) {
    walk(dd_pair_node_free, fsm->Compassion);
    free_list(fsm->Compassion);
  }
  if (fsm->Fairstates != (bdd_ptr)NULL) bdd_free(dd_manager, fsm->Fairstates);

  FREE(fsm);
}

bdd_ptr Compile_FsmBddGetInit(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Init);
}
void Compile_FsmBddSetInit(Fsm_BddPtr fsm, bdd_ptr init)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Init = init;
}

bdd_ptr Compile_FsmBddGetInvar(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Invar);
}
void Compile_FsmBddSetInvar(Fsm_BddPtr fsm, bdd_ptr invar)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Invar = invar;
}

CPTrans_Ptr Compile_FsmBddGetMonoTrans(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->MonoTrans);
}

void Compile_FsmBddSetMonoTrans(Fsm_BddPtr fsm, CPTrans_Ptr trans)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->MonoTrans = trans;
}

CPTrans_Ptr Compile_FsmBddGetIwls95CP(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Iwls95CP);
}

void Compile_FsmBddSetIwls95CP(Fsm_BddPtr fsm, CPTrans_Ptr trans)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Iwls95CP = trans;
}

CPTrans_Ptr Compile_FsmBddGetThreshold(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Threshold);
}

void Compile_FsmBddSetThreshold(Fsm_BddPtr fsm, CPTrans_Ptr trans)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Threshold = trans;
}

DPTrans_Ptr Compile_FsmBddGetSaturation(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Saturation);
}

void Compile_FsmBddSetSaturation(Fsm_BddPtr fsm, DPTrans_Ptr trans)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Saturation = trans;
}

DPTrans_Ptr Compile_FsmBddGetPhase1(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Phase1);
}

void Compile_FsmBddSetPhase1(Fsm_BddPtr fsm, DPTrans_Ptr trans)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Phase1 = trans;
}

DPTrans_Ptr Compile_FsmBddGetPhase2(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Phase2);
}

void Compile_FsmBddSetPhase2(Fsm_BddPtr fsm, DPTrans_Ptr trans)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Phase2 = trans;
}

node_ptr Compile_FsmBddGetJustice(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Justice);
}
void Compile_FsmBddSetJustice(Fsm_BddPtr fsm, node_ptr justice)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Justice = justice;
}
node_ptr Compile_FsmBddGetCompassion(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Compassion);
}
void Compile_FsmBddSetCompassion(Fsm_BddPtr fsm, node_ptr compassion)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Compassion = compassion;
}

bdd_ptr Compile_FsmBddGetFairStates(Fsm_BddPtr fsm)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  return(fsm->Fairstates);
}
void Compile_FsmBddSetFairStates(Fsm_BddPtr fsm, bdd_ptr fair)
{
  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  fsm->Fairstates = fair;
}


/* ROUTINES TO BUILD THE SCALAR SEXP FSM */
static node_ptr mk_true() {return(find_node(TRUEEXP, Nil, Nil));}
static node_ptr mk_and(node_ptr a, node_ptr b) {return(find_node(AND, a, b));}

node_ptr compileMakeFsmExpr(node_ptr expr, int mode)
{
  node_ptr result;
  
  if ((expr == Nil) || (query_sexp_model_hash(expr, mode))) {
    result = mk_true();
  }
  else {
    switch(node_get_type(expr)) {
    case AND: 
      {
        node_ptr left  = compileMakeFsmExpr(car(expr), mode);
        node_ptr right = compileMakeFsmExpr(cdr(expr), mode);

        result = mk_and(left, right);
        break;
      }
    default: 
      {
        result = expr;
        break;
      }
    }
  }
  insert_sexp_model_hash(expr, mode);
  return(result);
}

Fsm_SexpPtr Compile_MakeFsmSexpr(node_ptr vars, 
				 node_ptr justice_expr,
				 node_ptr compassion_expr)
{
  node_ptr l;
  Fsm_SexpPtr fsm;
  node_ptr init_expr  = mk_true();
  node_ptr invar_expr = mk_true();
  node_ptr next_expr  = mk_true();
  
  init_sexp_model_hash();
  for(l = vars; l != Nil; l = cdr(l)){
    node_ptr var_i = car(l);
    node_ptr expr_var_i = lookup_variable_sexp_model_hash(var_i);

    if (expr_var_i != (node_ptr)NULL) {
      if (var_model_sexp_get_init(expr_var_i) != Nil) {
        init_expr = mk_and(init_expr, compileMakeFsmExpr(var_model_sexp_get_init(expr_var_i), INIT));
      }
      if (var_model_sexp_get_invar(expr_var_i) != Nil) {
        invar_expr = mk_and(invar_expr, compileMakeFsmExpr(var_model_sexp_get_invar(expr_var_i), INVAR));
      }
      if (var_model_sexp_get_next(expr_var_i) != Nil) {
        next_expr = mk_and(next_expr, compileMakeFsmExpr(var_model_sexp_get_next(expr_var_i), TRANS));
      }
    }
  }
  quit_sexp_model_hash();
  /* Building the Scalar Sexp FSM */
  fsm = Compile_FsmSexpAlloc();
  Compile_FsmSexpSetInit(fsm, init_expr);
  Compile_FsmSexpSetInvar(fsm, invar_expr);
  Compile_FsmSexpSetTrans(fsm, next_expr);
  Compile_FsmSexpSetJustice(fsm, justice_expr);
  Compile_FsmSexpSetCompassion(fsm, compassion_expr);
  return(fsm);
}


/* ROUTINES TO BUILD THE BOOLEAN SEXP FSM */
node_ptr compileMakeFsmBExpr(node_ptr expr, int mode)
{
  node_ptr result;
  
  if ((expr == Nil) || (query_sexp_model_hash(expr, mode))) {
    result = mk_true();
  }
  else {
    switch(node_get_type(expr)) {
    case AND: 
      {
        node_ptr left  = compileMakeFsmBExpr(car(expr), mode);
        node_ptr right = compileMakeFsmBExpr(cdr(expr), mode);

        result = mk_and(left, right);
        break;
      }
    default: 
      {
        result = expr2bexpr(expr);
        break;
      }
    }
  }
  insert_sexp_model_hash(expr, mode);
  return(result);
}
  
Fsm_SexpPtr Compile_MakeFsmBExpr(node_ptr vars, 
				 node_ptr justice_expr,
				 node_ptr compassion_expr)
{
  node_ptr l;
  Fsm_SexpPtr fsm;
  int ovl;
  node_ptr init_expr  = mk_true();
  node_ptr invar_expr = mk_true();
  node_ptr next_expr  = mk_true();

  /* We set the verbose level to 0 and then we restore the original
     value. This because booleanization uses eval */
  ovl = get_verbose_level(options);
  set_verbose_level(options, 0);
  init_sexp_model_hash();
  for(l = vars; l != Nil; l = cdr(l)){
    node_ptr var_i = car(l);
    node_ptr expr_var_i = lookup_variable_sexp_model_hash(var_i);

    if (expr_var_i != (node_ptr)NULL) {
      /* INIT */
      if (var_model_sexp_get_init(expr_var_i) != Nil) {
        init_expr = mk_and(init_expr, compileMakeFsmBExpr(var_model_sexp_get_init(expr_var_i), INIT));
      }
      /* INVAR */
      if (var_model_sexp_get_invar(expr_var_i) != Nil) {
        invar_expr = mk_and(invar_expr, compileMakeFsmBExpr(var_model_sexp_get_invar(expr_var_i), INVAR));
      }
      /* TRANS */
      if (var_model_sexp_get_next(expr_var_i) != Nil) {
        next_expr = mk_and(next_expr, compileMakeFsmBExpr(var_model_sexp_get_next(expr_var_i), TRANS));
      }
    }
  }
  quit_sexp_model_hash();
  /* we restore the verbose level to the original value */
  set_verbose_level(options, ovl);
  /* Building the Scalar Sexp FSM */
  fsm = Compile_FsmSexpAlloc();
  Compile_FsmSexpSetInit(fsm, init_expr);
  Compile_FsmSexpSetInvar(fsm, invar_expr);
  Compile_FsmSexpSetTrans(fsm, next_expr);
  Compile_FsmSexpSetJustice(fsm, expr2bexpr(justice_expr));
  Compile_FsmSexpSetCompassion(fsm, expr2bexpr(compassion_expr));
  return(fsm);
}


/* BUILD THE FSM BDD */
void Compile_BuildFsmBddPartitionMethod(Fsm_BddPtr fsm, node_ptr vars,
                                        Partition_Method method, add_ptr assumption)
{

  nusmv_assert(fsm != (Fsm_BddPtr)NULL);

  if(!opt_po(options))
  {
    switch(method){
      case Monolithic:
	{
	  if (fsm->MonoTrans == (CPTrans_Ptr)NULL) {
	    fsm->MonoTrans = Compile_BuildModelMonolithic(vars, assumption);
	  }
	  break;
	}
      case Threshold:
	{
	  if (fsm->Threshold == (CPTrans_Ptr)NULL) {
	    fsm->Threshold = Compile_BuildModelThreshold(vars, get_conj_part_threshold(options), assumption);
	  }
	  break;
	}
      case Iwls95CP:
	{
	  if (fsm->Iwls95CP == (CPTrans_Ptr)NULL) {
	    fsm->Iwls95CP = Compile_BuildModelIwls95CP(vars, Iwls95GetOptions(), assumption);
	  }
	  break;
	}
      default: 
	{
	  rpterr("Compile_BuildFsmBddPartitionMethod: Unknown partitioning method.\n");
	  break;
	}
    }
  }

  if(opt_po(options))
  {
    if (fsm->Phase1 == (DPTrans_Ptr)NULL) {
      fsm->Phase1 = Compile_BuildModelPhase1(vars,
	  cmp_struct_get_procs(cmps),
	  assumption);
    }
    if (fsm->Phase2 == (DPTrans_Ptr)NULL) {
      fsm->Phase2 = Compile_BuildModelPhase2(vars,
	  cmp_struct_get_procs(cmps),
	  assumption);
    }
  }
  else if(opt_saturation(options))
  {
    if (fsm->Saturation == (DPTrans_Ptr)NULL) {
      fsm->Saturation = Compile_BuildModelSaturation(vars,
	  cmp_struct_get_procs(cmps),
	  assumption);
    }
  }
}

Fsm_BddPtr Compile_MakeFsmBdd(node_ptr vars, 
			      node_ptr justice_expr, 
			      node_ptr compassion_expr, 
			      add_ptr assumption)
{
  Partition_Method pm = get_partition_method(options);
  Fsm_BddPtr fsm = Compile_FsmBddAlloc();

  nusmv_assert(fsm != (Fsm_BddPtr)NULL);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Computing set of initial states...");
  }
  fsm->Init  = Compile_BuildInitBdd(vars);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "...done\nComputing set of invariant states...");
  }
  fsm->Invar = Compile_BuildInvarBdd(vars, assumption);
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "...done\nComputing transition relation...");
  }
  Compile_BuildFsmBddPartitionMethod(fsm, vars, pm, assumption);

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "...done\nComputing fair states...");
  }
  fsm->Justice = Mc_ComputeJusticeConstraints(fsm, justice_expr);
  fsm->Compassion = Mc_ComputeCompassionConstraints(fsm, compassion_expr);
  /* Flavio Lerda */
#if 0
  fsm->Fairstates = Mc_ComputeFairStates(fsm);
#else
  fsm->Fairstates = bdd_one(dd_manager);
#endif
  /* Flavio Lerda */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "...done");
  }
  return(fsm);
}



/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


