/**CFile***********************************************************************

  FileName    [compileEncode.c]

  PackageName [compile]

  Synopsis    [Routines that performs the encoding of the symbolic variables
  into BDD.]

  Description [Routines that performs the encoding of the symbolic variables
  into BDD. At now the only available encoding is the binary one.]

  SeeAlso     [compileEval.c, compileOrd.c]

  Author      [Marco Roveri]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileEncode.c,v 1.3 2003/09/25 21:04:42 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [The cube of input variables.]

  Description [The cube of input variables to be used in image forward.]

  SeeAlso     []

******************************************************************************/
add_ptr input_variables_add = (add_ptr)NULL;
bdd_ptr input_variables_bdd = (bdd_ptr)NULL;


/**Variable********************************************************************

  Synopsis    [The cube of input variables.]

  Description [The cube of input variables to be used in image backward.]

  SeeAlso     []

******************************************************************************/
add_ptr next_input_variables_add = (add_ptr)NULL;
bdd_ptr next_input_variables_bdd = (bdd_ptr)NULL;


/**Variable********************************************************************

  Synopsis    [The cube of state variables.]

  Description [The cube of state variables to be used in image forward.]

  SeeAlso     []

******************************************************************************/
add_ptr state_variables_add = (add_ptr)NULL;
bdd_ptr state_variables_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The cube of state variables.]

  Description [The cube of state variables to be used in image backward.]

  SeeAlso     []

******************************************************************************/
add_ptr next_state_variables_add = (add_ptr)NULL;
bdd_ptr next_state_variables_bdd = (bdd_ptr)NULL;

/**Variable********************************************************************

  Synopsis    [The array of symbolic variable names.]

  Description [The array of symbolic variable names. Each element <tt>i</tt>
  contains the symbolic name associated to the boolean variable with
  index <tt>i</tt>.]

******************************************************************************/
node_ptr variable_names[MAX_VAR_INDEX];

/**Variable********************************************************************

  Synopsis    [The array of symbolic boolean variable names.]

  Description [The array of symbolic boolean variable names. Each
  element <tt>i</tt> contains the symbolic name associated to the
  boolean variable with index <tt>i</tt>.]

******************************************************************************/
static node_ptr bool_variable_names[MAX_VAR_INDEX];


/**Variable********************************************************************

  Synopsis    [The list of the boolean variables]

  Description [The list of the boolean variables created by the encoding.]

  SeeAlso     [optional]

******************************************************************************/
static node_ptr boolean_variables_names_list = Nil;
node_ptr get_boolean_variable_list()
{
  return(boolean_variables_names_list);
}
void reset_boolean_variable_list()
{
  boolean_variables_names_list = Nil;
}
void set_boolean_variable_list(node_ptr val)
{
  boolean_variables_names_list = val;
}
void insert_boolean_variable(node_ptr b) 
{
  boolean_variables_names_list = append(boolean_variables_names_list, cons(b, Nil));
}
void free_boolean_variable_list()
{
  node_ptr l = boolean_variables_names_list;

  while(l != Nil) {
    node_ptr m = l;
    l = cdr(l);
    free_node(m);
  }
}

/**Variable********************************************************************

  Synopsis    [required]

  Description [optional]

  SeeAlso     [optional]

******************************************************************************/
node_ptr input_boolean_variables = Nil;

/**Variable********************************************************************

  Synopsis    [required]

  Description [optional]

  SeeAlso     [optional]

******************************************************************************/
node_ptr state_boolean_variables = Nil;

/**Variable********************************************************************

  Synopsis    [Correspondence between current and next variables.]

  Description [These arrays are used to maintain correspondence
  between current and next variables. Position <em>i</em> contains the
  index of the corresponding next state variable. They are used to
  perform forward and backward shifting respectively.]

  SeeAlso     []

******************************************************************************/
static int current2next[MAX_VAR_INDEX];
static int next2current[MAX_VAR_INDEX];

/**Variable********************************************************************

  Synopsis    [Array used to pick up a minterm from a given BDD.]

  Description [This array should contain at least all variables in the
  support of the BDD of which we want extract a minterm.]

  SeeAlso     []

******************************************************************************/
bdd_ptr minterm_vars[MAX_VAR_INDEX];
int minterm_vars_dim = 0;
int get_minterm_vars_dim(){return(minterm_vars_dim);}
void set_minterm_vars_dim(int j){minterm_vars_dim = j;}
void reset_minterm_vars_dim(){minterm_vars_dim = 0;}

/**Variable********************************************************************

  Synopsis    [The hash used to take care of already printed variable value.]

  Description [The hash used to take care of already printed variable
  value. It is used by <tt>print_state</tt>.]

  SeeAlso     [print_state]

******************************************************************************/
static hash_ptr print_hash;
void init_print_hash() { print_hash = new_assoc(); }
void insert_print_hash(node_ptr x, node_ptr y) { insert_assoc(print_hash, x, y);}
node_ptr lookup_print_hash(node_ptr x) {return(find_assoc(print_hash, x));}
void clear_print_hash() {clear_assoc(print_hash);}

/**Variable********************************************************************

  Synopsis    [The number of boolean state variables created.]

  Description [It is the number of boolean state variables created to
  encode symbolic state variables, current and next.]

  SeeAlso     []

******************************************************************************/
static int num_of_state_variables = 0; 
static void num_of_state_variables_set(int n) { num_of_state_variables = n; }
static int num_of_state_variables_get(void) { return(num_of_state_variables); }
static void num_of_state_variables_inc(void) { num_of_state_variables++; } 

/**Variable********************************************************************

  Synopsis    [The number of boolean input variables created.]

  Description [It is the number of boolean input state variables
  created to encode symbolic input variables, both current and next.]

  SeeAlso     []

******************************************************************************/
static int num_of_input_variables = 0;
static void num_of_input_variables_set(int n) { num_of_input_variables = n; }
static int num_of_input_variables_get(void) { return(num_of_input_variables); }
static void num_of_input_variables_inc(void) { num_of_input_variables++; }

/**Variable********************************************************************

  Synopsis    [The number of boolean variables created.]

  Description [It is the number of boolean variables created to encode
  both input and state variables. It is the sum of
  <tt>num_of_input_variables</tt> plus <tt>num_of_state_variables</tt>.]

  SeeAlso     []

******************************************************************************/
static int num_of_variables = 0;
static int num_of_variables_get(void) { return(num_of_variables); }
static void num_of_variables_set(int n) { num_of_variables = n; }
static void num_of_variables_inc(void) { num_of_variables++; }

/**Variable********************************************************************

  Synopsis    [The number of current state boolean variables created.]

  Description [It is the number of current state boolean variables created.]

  SeeAlso     []

******************************************************************************/
static int real_num_of_state_vars = 0;
static void real_num_of_state_vars_set(int n) { real_num_of_state_vars = n; }
static int real_num_of_state_vars_get(void) { return(real_num_of_state_vars);}
static void real_num_of_state_vars_inc(void) { real_num_of_state_vars++; }

/**Variable********************************************************************

  Synopsis    [The constant hash.]

  Description [This hash associates to an atom the corresponding ADD
  leaf if defined. Suppose to have a declaration of this kind:<br> 
  <pre>
  VAR 
    state : {idle, stopped}
  <pre>
  then in the constant hash for the atom <tt>idle</tt> there is the
  corresponding leaf ADD, i.e. the ADD whose value is the symbol
  <tt>idle</tt>. This hash is used by the evaluator.]

  SeeAlso     [eval]

******************************************************************************/
static hash_ptr constant_hash;
void init_constant_hash() { constant_hash = new_assoc(); }
void insert_constant_hash(node_ptr x, node_ptr y) { insert_assoc(constant_hash, x, y);}
node_ptr lookup_constant_hash(node_ptr x) {return(find_assoc(constant_hash, x));}
static assoc_retval constant_hash_free(char *key, char *data, char * arg) {
  add_ptr element = (add_ptr)data;

  if (element != (add_ptr)NULL) {
    add_free(dd_manager, element);
  }
  return(ASSOC_DELETE);
}

void clear_constant_hash() {clear_assoc_and_free_entries(constant_hash, constant_hash_free);}

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static add_ptr CompileEncodeVarRecur ARGS((node_ptr, node_ptr, int, int, boolean, boolean));
static void get_bdd_state_var ARGS((boolean flag));
static void get_bdd_input_var ARGS((boolean flag));
static void create_boolean_input_var ARGS((node_ptr, int, int));
static void create_boolean_state_var ARGS((node_ptr, int, int));
/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Initializes the data structure relative to BDD variables.]

  Description        [Initializes the data structure relative to BDD
  variables, i.e. the array used to store correspondence between
  variable names and bdd variables, and the array used to shift
  forward and backward during image computation.]

  SideEffects        []

******************************************************************************/
void Compile_EncodeVarsInit(void)
{
  int j;

  for (j = 0; j < MAX_VAR_INDEX; j++) {
    bool_variable_names[j] = Nil;
    variable_names[j] = Nil;
    current2next[j] = j;
    next2current[j] = j;
    minterm_vars[j] = (bdd_ptr)NULL;
  }
  minterm_vars_dim = 0;
}

/**Function********************************************************************

  Synopsis           [Encode a variable into BDD.]

  Description        [This function encodes the variable <tt>name</tt>,
  whose range is <tt>range</tt>, into BDD variables. The boolean
  variables created are grouped together, in such a way that the BDD
  dynamic reordering consider them as a single block. <tt>flag</tt> is
  used to indicate if the created variables have to be inserted in the
  array used to extract complete minterms from a BDD.]

  SideEffects        []

  SeeAlso            [CompileEncodeVarRecur]

******************************************************************************/
add_ptr Compile_EncodeVar(node_ptr name, node_ptr range, boolean is_input, boolean flag)
{
  int start_index, offset;
  add_ptr encoding;
  
  start_index = num_of_variables_get() + 1;
  encoding = CompileEncodeVarRecur(name, range, num_of_variables_get(), 0, is_input, flag);
  offset = num_of_variables_get() - start_index; /* offset <= 0 no variables created */
  if (offset > 0) {
    variable_names[start_index] = name;
    dd_new_var_block(dd_manager, start_index, offset + 1);
    if (opt_verbose_level_gt(options, 3)) {
      fprintf(nusmv_stderr, "  *****************************************************\n");
      fprintf(nusmv_stderr, "  Built block from var %d to var %d enclosed \n", start_index, start_index + offset);
      fprintf(nusmv_stderr, "  The number of BDD vars in this group is %d\n", offset + 1);
      fprintf(nusmv_stderr, "  *******************************************************\n");
    }
  } else {
    if (opt_verbose_level_gt(options, 3)) {
      fprintf(nusmv_stderr, "  The current variable has no associated values,\n");
      fprintf(nusmv_stderr, "  no BDD variables are created for it.\n\n");
    }
  }
  return(encoding);
}

/**Function********************************************************************

  Synopsis           [Encodes a scalar variables into boolean variables.]

  Description        [This function computes the ADD for representing the
  boolean encoding of a scalar variable with values listed in "range". If
  X has range {a, b, c}, the propositions (X == a), (X == b), (X == c)
  are encoded by means of two binary variables. E.g. <br>
  <pre>
              x1
            1/  \\0 
            x2   \\ 
           /  \\   \\
          a    c   b
  </pre>
  Where (x1 and x2) represents (X == a), while (x1 and not x2) represents (X
  == b), and (x1 = 0) represents (X == c).]

  SideEffects        [<tt>num_of_state_variables</tt>,
  <tt>num_of_input_variables</tt> are  modified depending the kind of
  variable we are encoding.]

  SeeAlso            []

******************************************************************************/
static add_ptr CompileEncodeVarRecur(node_ptr name, node_ptr range, int curlevel,
                                     int num, boolean is_input, boolean flag)
{
  if (range == Nil) internal_error("CompileEncodeVarRecur: range = Nil");

  /* Base case: We reached a leaf */
  if (cdr(range) == Nil) {
    node_ptr v = find_atom(car(range));
    add_ptr temp = (add_ptr)lookup_constant_hash(v);
    if (temp) {
      add_ref(temp);
      return(temp);
    }
    temp = add_leaf(dd_manager, v);
    /* DOT and ARRAY included to keep track of process names */
    if (v && ((node_get_type(v) == ATOM) || 
              (node_get_type(v) == DOT)  ||
              (node_get_type(v) == ARRAY))) {
      /* 
         We increment the reference count since we store the ADD in
         constant_hash
      */
      add_ref(temp);
      insert_constant_hash(v, (node_ptr)temp);
      insert_check_constant_hash(v, v);
    }
    return(temp);
  }

  /* Step case: */
  if (is_input == true) {
    curlevel += 2;
    if (curlevel > num_of_variables) {
      get_bdd_input_var(flag);
      create_boolean_input_var(name, num, curlevel - 1);
    }
    {
      /*
        The element list is decomposed and left and right are
        allocated in the bdd
      */
      add_ptr p0 = CompileEncodeVarRecur(name, even_elements(range), curlevel, num + 1, is_input, flag);
      add_ptr p1 = CompileEncodeVarRecur(name, odd_elements(range), curlevel, num + 1, is_input, flag);
      add_ptr result = add_build(dd_manager, curlevel - 1, p0, p1);

      add_free(dd_manager, p0);
      add_free(dd_manager, p1);
      return(result);
    }
  } else {
    curlevel += 2;
    if (curlevel > num_of_variables) {
      get_bdd_state_var(flag);
      create_boolean_state_var(name, num, curlevel - 1);
    }
    {
      /*
        The element list is decomposed and left and right are
        allocated in the bdd
      */
      add_ptr p0 = CompileEncodeVarRecur(name, even_elements(range), curlevel, num + 1, is_input, flag);
      add_ptr p1 = CompileEncodeVarRecur(name, odd_elements(range), curlevel, num + 1, is_input, flag);
      add_ptr result = add_build(dd_manager, curlevel - 1, p0, p1);

      add_free(dd_manager, p0);
      add_free(dd_manager, p1);
      return(result);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Return the list of variables corresponding
  to the current order of variables in the system.]

  Description        [It returns the list of variables corresponding
  to the current order of variables in the system.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Compile_GetOrdering(void)
{
  int cvl, max_level;
  int flag = 0;
  node_ptr current_ordering = Nil;

  max_level = dd_get_size(dd_manager);
  if (opt_verbose_level_gt(options, 5))
    fprintf(nusmv_stderr, "Number of variables: %d\n", max_level);
  for (cvl = 0; (cvl < max_level) ; cvl++) {
    int index = dd_get_var_at_level(dd_manager, cvl);
    node_ptr name = variable_names[index];

   if (opt_verbose_level_gt(options, 5))
     fprintf(nusmv_stderr,"level %d with variable %d\n", cvl, index);
   if (name == proc_selector_internal_vname) flag = 1;
   if (name != Nil) current_ordering = cons(name, current_ordering);
  }
  current_ordering = reverse(current_ordering);
  if (flag == 0) current_ordering = cons(proc_selector_internal_vname, current_ordering);
  return(current_ordering);
}


/**Function********************************************************************

  Synopsis           [Print a set of states.]

  Description        [Prints the states represented by a bdd.]

  SideEffects        [<tt>print_hash</tt> is reset.]

  SeeAlso            [print_reachable_states print_state]

******************************************************************************/
void print_states(bdd_ptr states)
{
  extern bdd_ptr minterm_vars[];
  int j;
  int array_of_states_size =  
    bdd_count_minterm(dd_manager, states, get_minterm_vars_dim());
  bdd_ptr *array_of_states = ALLOC(bdd_ptr, array_of_states_size);

  bdd_pick_all_terms(dd_manager, states, 
                     minterm_vars, get_minterm_vars_dim(),
                     array_of_states, array_of_states_size);
  
  clear_print_hash();
  for(j=0; j < array_of_states_size; j++){
    fprintf(nusmv_stdout, "------- State %4.d ------\n", j+1);
    set_indent_size(2);
    print_state(array_of_states[j], all_symbols, 0);
    reset_indent_size();
  }
  fprintf(nusmv_stdout, "-------------------------\n");
  clear_print_hash();

  for(j=0; j < array_of_states_size; j++)
    bdd_free(dd_manager,array_of_states[j]);
  FREE(array_of_states);
}

/**Function********************************************************************

  Synopsis           [Prints out the state represented by the given minterm.]

  Description        [Prints out the state represented by the given
  minterm. If <tt>changes_only</tt> is 1, than only state variables
  which assume a different value from the previous printed one are
  printed out.]

  SideEffects        [<tt>print_hash</tt> is modified.]

******************************************************************************/
void print_state(bdd_ptr s, node_ptr symbols, int changes_only)
{
  node_ptr proc_name;
  node_ptr process_selector_data;
  add_ptr add_state, tmp;
  add_ptr process_selector_enc;
  node_ptr los = symbols;

  add_state = bdd_to_add(dd_manager, s);
  process_selector_data = lookup_symbol_hash(proc_selector_internal_vname);
  if (process_selector_data != NULL) {
    process_selector_enc = (add_ptr)car(process_selector_data);
    tmp = add_if_then(dd_manager, add_state, process_selector_enc);
    proc_name = add_value(dd_manager, tmp);
    add_free(dd_manager, tmp);

    if ((proc_name != Nil) && (proc_name != sym_intern("main"))) {
      indent_node(nusmv_stdout, "[executing process ", proc_name, "]\n");
    }
  }
  while (los) {
    node_ptr cur_sym_value;
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);

    los = cdr(los);
    tmp = add_if_then(dd_manager, add_state, cur_sym_vals);
    cur_sym_value = add_value(dd_manager, tmp);
    add_free(dd_manager, tmp);
    add_free(dd_manager, cur_sym_vals);
    if (changes_only) {
      if (cur_sym_value == lookup_print_hash(cur_sym)) continue;
      insert_print_hash(cur_sym, cur_sym_value);
    }
    indent_node(nusmv_stdout, "", cur_sym, " = ");
    print_node(nusmv_stdout, cur_sym_value);
    fprintf(nusmv_stdout, "\n");
  }
  add_free(dd_manager, add_state);
}

/**Function********************************************************************

  Synopsis           [Prints out the symbolic names of boolean
  variables stored in a cube.]

  Description        [Given a cube of boolean BDD variables, this
  function prints out the symbolic names of the corresponding
  variables. The symbolic name of the variables to be printed out are
  listed in <tt>list_of_sym</tt>.]

  SideEffects        [None]

******************************************************************************/
void print_state_vars(DdManager * dd, bdd_ptr cube, node_ptr list_of_sym)
{
  node_ptr los = list_of_sym;
  add_ptr a_cube = bdd_to_add(dd, cube);

  fprintf(nusmv_stdout, "Current VARS:\n");
  while (los) {
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);
    add_ptr cur_sym_cube = add_support(dd, cur_sym_vals);
    add_ptr is_in = add_cube_diff(dd, a_cube, cur_sym_cube);

    los = cdr(los);
    if (is_in != a_cube) {
      indent_node(nusmv_stdout,"", cur_sym, " ");
    }
    add_free(dd, cur_sym_vals);
    add_free(dd, cur_sym_cube);
    add_free(dd, is_in);
  }
  fprintf(nusmv_stdout, "\nNext VARS:\n");  
  los = list_of_sym;
  while (los) {
    node_ptr cur_sym = car(los);
    add_ptr cur_sym_vals = eval(cur_sym, Nil);
    add_ptr next_cur_sym_vals = add_permute(dd, cur_sym_vals, current2next);
    add_ptr next_cur_sym_cube = add_support(dd, next_cur_sym_vals);
    add_ptr next_is_in = add_cube_diff(dd, a_cube, next_cur_sym_cube);

    los = cdr(los);
    if (next_is_in != a_cube) {
      indent_node(nusmv_stdout,"", find_node(NEXT, cur_sym, Nil), " ");
    }
    add_free(dd, cur_sym_vals);
    add_free(dd, next_cur_sym_vals);
    add_free(dd, next_cur_sym_cube);
    add_free(dd, next_is_in);
  }
  add_free(dd, a_cube);
  fprintf(nusmv_stdout,"\n");
}

/**Function********************************************************************

  Synopsis           [Return the number of states of a given BDD.]

  Description        [Return the number of minterms (i.e. states)
  represented by a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
double bdd_count_states(DdManager *dd, add_ptr fn) 
{
  return(bdd_count_minterm(dd, fn, real_num_of_state_vars));
}

/**Function********************************************************************

  Synopsis           [Return the number of states of a given ADD.]

  Description        [Return the number of minterms (i.e. states)
  represented by a ADD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
double add_count_states(DdManager *dd, bdd_ptr fn) 
{
  return(add_count_minterm(dd, fn, real_num_of_state_vars));
}

/**Function********************************************************************

  Synopsis           [Shift the given set from current variables to next ones.]

  Description        [Shift the given set from current variables to next ones.]

  SideEffects        []

  SeeAlso            [add_shift_backward]

******************************************************************************/
add_ptr add_shift_forward(DdManager * dd, add_ptr fn)
{ return(add_permute(dd, fn, current2next)); }

/**Function********************************************************************

  Synopsis           [Shift the given set from next variables to current ones.]

  Description        [Shift the given set from next variables to current ones.]

  SideEffects        []

  SeeAlso            [add_shift_forward]

******************************************************************************/
add_ptr add_shift_backward(DdManager * dd, add_ptr fn)
{ return(add_permute(dd, fn, next2current)); }



/**Function********************************************************************

  Synopsis           [Shift the given set from a subset of the current variables to next ones.]

  Description        [Shift the given set from a subset of the current variables to next ones.]

  SideEffects        []

  SeeAlso            [bdd_shift_backward, bdd_permute]

******************************************************************************/
bdd_ptr bdd_shift_set(DdManager * dd, bdd_ptr fn, int *variables)
{
  bdd_ptr result;

  result = bdd_permute(dd, fn, variables);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Shift the given set from current variables to next ones.]

  Description        [Shift the given set from current variables to next ones.]

  SideEffects        []

  SeeAlso            [bdd_shift_backward, bdd_permute]

******************************************************************************/
bdd_ptr bdd_shift_forward(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_permute(dd, fn, current2next);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Shift the given set from next variables to current ones.]

  Description        [Shift the given set from next variables to current ones.]

  SideEffects        []

  SeeAlso            [bdd_shift_forward, bdd_permute]

******************************************************************************/
bdd_ptr bdd_shift_backward(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_permute(dd, fn, next2current);
  return(result);
}
    

/**Function********************************************************************

  Synopsis           [Extracts a minterm from a given BDD.]

  Description        [Extracts a minterm from a given BDD.]

  SideEffects        []

  SeeAlso            [bdd_pick_one_minterm]

******************************************************************************/
bdd_ptr bdd_pick_one_state(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_pick_one_minterm(dd, fn, minterm_vars, minterm_vars_dim);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Extracts a random minterm from a given BDD.]

  Description        [Extracts a random minterm from a given BDD.]

  SideEffects        []

  SeeAlso            [bdd_pick_one_minterm_rand]

******************************************************************************/
bdd_ptr bdd_pick_one_state_rand(DdManager * dd, bdd_ptr fn)
{
  bdd_ptr result;

  result = bdd_pick_one_minterm_rand(dd, fn, minterm_vars, minterm_vars_dim);
  return(result);
}


/**Function********************************************************************

  Synopsis           [Creates the array necessary to extract minterms
  from a BDD.]

  Description        [Creates the array necessary to extract minterms
  from a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void make_minterm_vars(void)
{ 
  int i, k = 0;

  /* We need to extract only current state variables */
  for (i = 1; i < num_of_variables; i+=2) {
    minterm_vars[k++] = bdd_new_var_with_index(dd_manager, i);
  }
  minterm_vars_dim = k;
}

/**Function********************************************************************

  Synopsis           [Adds a variable to the sarray necessary to extract minterms
  from a BDD.]

  Description        [Adds a variable to the sarray necessary to extract minterms
  from a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void add_var_to_minterm_vars(int vn)
{ 
  if (vn < MAX_VAR_INDEX) {
    if (minterm_vars[minterm_vars_dim] != (bdd_ptr)NULL) {
      bdd_free(dd_manager, minterm_vars[minterm_vars_dim]);
    }
    minterm_vars[minterm_vars_dim++] = bdd_new_var_with_index(dd_manager, vn);
  }
  else {
    fprintf(nusmv_stderr, "Maximum number of BDD variuables reached.\n");
    nusmv_exit(1);
  }
}

/**Function********************************************************************

  Synopsis           [Free the array necessary to extract minterms.]
  from a BDD.]

  Description        [Free the array necessary to extract minterms
  from a BDD.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void free_minterm_vars(void)
{
  int i;

  for (i = 0; i < MAX_VAR_INDEX; i++) {
    if (minterm_vars[i] != (bdd_ptr)NULL) {
      bdd_free(dd_manager, minterm_vars[i]);
      minterm_vars[i] = (bdd_ptr)NULL;
    }
  }
}


/**Function********************************************************************

  Synopsis           [Reset the counters necessary to the encoder.]

  Description        [Reset the counters necessary to the encoder.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void reset_encode()
{
  num_of_variables_set(0);
  num_of_input_variables_set(0);
  num_of_state_variables_set(0);
  real_num_of_state_vars_set(0);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
node_ptr get_bool_variable_name(int i)
{
  nusmv_assert(i < MAX_VAR_INDEX);
  return(bool_variable_names[i]);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Adds a new boolean variable to the DD package.]

  Description        [Adds a new boolean variable to the DD manager.
  If it is a state variable, then also the "next" variable is created,
  and the data structure to perform substitution of "current" and "next"
  variables is filled in.]

  SideEffects        []

  SeeAlso            [get_bdd_input_var]

******************************************************************************/
static void get_bdd_state_var(boolean flag)
{
  add_ptr tmp_1;
  
  if (num_of_variables >= MAX_VAR_INDEX) error_too_many_vars();  

  if (opt_verbose_level_gt(options, 2))
    fprintf(nusmv_stderr, "  BDD variable %d, ", num_of_variables+1);
  tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
  add_and_accumulate(dd_manager, &state_variables_add, tmp_1);
  add_free(dd_manager, tmp_1);
  if (flag == true) {
    add_var_to_minterm_vars(num_of_variables);
  }
  if (num_of_variables >= MAX_VAR_INDEX) error_too_many_vars();  
  if (opt_verbose_level_gt(options, 2))
    fprintf(nusmv_stderr, " next BDD variable %d\n", num_of_variables+1);
  tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
  add_and_accumulate(dd_manager, &next_state_variables_add, tmp_1);
  add_free(dd_manager, tmp_1); 
  current2next[num_of_variables - 1] = num_of_variables;
  next2current[num_of_variables]     = num_of_variables - 1;
  num_of_state_variables += 2;
  real_num_of_state_vars_inc();
}

/**Function********************************************************************

  Synopsis           [Adds a new boolean variable to the DD package.]

  Description        [Adds a new boolean variable to the DD
  manager. This function is used to create the boolean variables
  needed to encode input variables.]

  SideEffects        []

  SeeAlso            [get_bdd_state_var]

******************************************************************************/
static void get_bdd_input_var(boolean flag)
{
  add_ptr tmp_1;
  
  if (num_of_variables >= MAX_VAR_INDEX) error_too_many_vars();
  if (opt_verbose_level_gt(options, 2))
    fprintf(nusmv_stderr, "  BDD input variable %d ", num_of_variables+1);
  tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
  add_and_accumulate(dd_manager, &input_variables_add, tmp_1);
  add_free(dd_manager, tmp_1);
  if (flag == true) {
    add_var_to_minterm_vars(num_of_variables);
  }
  if (num_of_variables >= MAX_VAR_INDEX) error_too_many_vars();
  if (opt_verbose_level_gt(options, 2))
    fprintf(nusmv_stderr, " next BDD variable %d\n", num_of_variables+1);
  tmp_1 = add_new_var_with_index(dd_manager, ++num_of_variables);
  add_and_accumulate(dd_manager, &next_input_variables_add, tmp_1);
  add_free(dd_manager, tmp_1); 

  current2next[num_of_variables - 1] = num_of_variables;
  next2current[num_of_variables]     = num_of_variables - 1;
  num_of_input_variables += 2;

  return;
}


/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
void create_boolean_state_var(node_ptr name, int num, int level)
{
  node_ptr num_i = find_node(NUMBER, (node_ptr)num, Nil);
  node_ptr name_i = find_node(DOT, name, num_i);
  node_ptr next_i = find_node(NEXT, name_i, Nil);
  add_ptr add_var_i = add_new_var_with_index(dd_manager, level);

  bool_variable_names[level]     = name_i;
  bool_variable_names[level + 1] = next_i;
  insert_symbol_hash(name_i, find_node(VAR, (node_ptr)add_var_i, boolean_type));
  state_boolean_variables = cons(name_i, state_boolean_variables);
  insert_boolean_variable(name_i);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
static void create_boolean_input_var(node_ptr name, int num, int level)
{
  node_ptr num_i = find_node(NUMBER, (node_ptr)num, Nil);
  node_ptr name_i = find_node(DOT, name, num_i);
  node_ptr next_i = find_node(NEXT, name_i, Nil);
  add_ptr add_var_i = add_new_var_with_index(dd_manager, level);

  bool_variable_names[level]     = name_i;
  bool_variable_names[level + 1] = next_i;
  insert_symbol_hash(name_i, find_node(VAR, (node_ptr)add_var_i, boolean_type));
  input_boolean_variables = cons(name_i, input_boolean_variables);
  insert_boolean_variable(name_i);
}

int is_next_variable(int index) {
  return (next2current[index] != index);
}

int is_current_variable(int index) {
  return (current2next[index] != index);
}

int is_input_variable(int index) {
  return (next2current[index] == index && current2next[index] == index);
}
