/**CFile***********************************************************************

  FileName    [compileDP.c]

  PackageName [compile]

  Synopsis    [Methods of DPDesc, DPList, DPTrans structures]

  Description [Methods of DPCluster, DPList, DPTrans structures for all
               disjunctive partitioning methods.]

  SeeAlso     []

  Author      [Emanuele Olivetti]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 
#include "ustring.h" 
#include "node.h" 

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef struct bounds {
  int top;
  int bottom;
} bounds;

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/\
static node_ptr extract ARGS((node_ptr sexp, node_ptr proc, node_ptr *bdd_vars, node_ptr *vars, int *is_identity));
static node_ptr get_bdd_variables ARGS((node_ptr var));
static CPList CPListBuildBasicProcRecur ARGS((CPList clusters, node_ptr sexp, node_ptr proc, node_ptr *bdd_vars, node_ptr *vars, add_ptr assumption));
static CPList CPListBuildElementaryProcBDDList ARGS((node_ptr var_list, node_ptr proc, int *top_ptr, int *bottom_ptr, node_ptr *bdd_vars, add_ptr assumption));
static boolean phase_of ARGS((node_ptr process));
static char *get_name ARGS((node_ptr node));
static CPList CPListBuildPartitioned ARGS((CPList cluster));
static CPList CPListBuildPartitionedMonolithic ARGS((CPList CL));
static CPList CPListBuildPartitionedThreshold ARGS((CPList CL));
static CPList CPListBuildPartitionedIwls95 ARGS((CPList CL));
static char *dup_string ARGS((char *str));
static int *SupportBuildList ARGS((node_ptr bdd_vars));
static bdd_ptr SupportBuildBDD ARGS((node_ptr bdd_vars));
static int *SupportDup ARGS((int *support));
static void SupportPrint ARGS((FILE *file, int *support, int detailed));

/**Function********************************************************************

  Synopsis           [Builds a transition relation partitioned by process]

  Description        [Builds a transition relation partitioned by process]

  SideEffects        []

******************************************************************************/
DPList DPListBuildProcs(node_ptr list_variables, 
		node_ptr procs_expr,
		Proc_Type type,
		bdd_ptr assumption)
{
  DPList result = DPListEmpty;
  int process_order = opt_process_order(options);

  {
    node_ptr p = procs_expr;
    node_ptr running = sym_intern(RUNNING_SYMBOL);

    while (p != Nil) { /* Loops over processes */
      node_ptr process_name = car(car(p));
      node_ptr process_running = eval_struct(running, process_name);
      boolean phase1 = phase_of(car(p));
      char *name = (char *)node2str(process_name);

      if(type == PT_Both || (phase1 && type == PT_Phase1) || (!phase1 && type == PT_Phase2))
      {
	CPList cluster_bdd;
	int top = -1, bottom = -1;
	// List of BDD variable affected by the current process
	node_ptr bdd_vars = Nil;

	/* converting variables list to bdd list */
	cluster_bdd = CPListBuildElementaryProcBDDList(
	    list_variables, process_running,
	    &top, &bottom, &bdd_vars, assumption);

	/* TODO: Top and bottom are NOT computed correctly */
	/* TODO: Top and bottom can be computed from the bdd_vars */
	/* it top or bottom is -1 means that there are no modified variables */
	/*
	if(top == -1 || bottom == -1)
	{
	  nusmv_assert(top == -1);
	  nusmv_assert(bottom == -1);

	  // ignore this transition relation because it is the identity
	  CPListFree(dd_manager, cluster_bdd);
	} else {
	*/
	{
	  CPList list = CPListBuildPartitioned(cluster_bdd);
	  CPListFree(dd_manager, cluster_bdd);
	  int *support = NULL;
	  bdd_ptr support_bdd = NULL;

	  if(opt_equality_reduced(options))
	  {
	    support = SupportBuildList(bdd_vars);
	    support_bdd = SupportBuildBDD(bdd_vars);
	  }

	  /* saves top and bottom information for later re-ordering */
	  if(process_order)
	  {
	    struct bounds *top_bottom;

	    top_bottom = (struct bounds *)malloc(sizeof(struct bounds));
	    top_bottom->top = top;
	    top_bottom->bottom = bottom;

	    {
	      DPDesc *desc = DPDescAlloc();
	      DPDescSetSupportBDD(desc, support_bdd);
	      DPDescSetSupportVector(desc, support);
	      DPDescSetCList(desc, list);
	      DPDescSetName(desc, name);
	      DPDescSetInfo(desc, top_bottom);

	      result = DPListCons(desc, result);
	    }
	  } else {
	    DPDesc *desc = DPDescAlloc();
	    DPDescSetSupportBDD(desc, support_bdd);
	    DPDescSetSupportVector(desc, support);
	    DPDescSetCList(desc, list);
	    DPDescSetName(desc, name);

	    result = DPListCons(desc, result);
	  }
	}
      }

      p = cdr(p);
    }
  }

  if(process_order)
  /* order the list of transition relations by their top value */
  {
    /* this will contain the ordered list */
    DPList tmp = DPListEmpty;
    /* length of the list */
    int len = DPListLength(result);
    /* is the cluster already been used */
    int used[len];
    /* pointer to the clusters */
    CPList elems[len];
    char *names[len];
    struct bounds *infos[len];
    int *supports[len];
    bdd_ptr support_bdds[len];

    /* initialize the cluster and used arrays */
    {
      DPList iter = result;
      int i;

      for(i = 0; i < len; i++)
      {
	DPDesc *desc = DPListGetItem(iter);
	used[i] = 0;
	elems[i] = DPDescGetCList(desc);
	names[i] = DPDescGetName(desc);
	infos[i] = DPDescGetInfo(desc);
	supports[i] = DPDescGetSupportVector(desc);
	support_bdds[i] = DPDescGetSupportBDD(desc);
	iter = DPListNext(iter);
      }
    }

    /* create the ordered list */
    {
      int i;

      for(i = 0; i < len; i++)
      {
	/* find the next element */
	int value = -1;
	int index = -1;

	{
	  int j;

	  for(j = 0; j < len; j++)
	  {
	    if(!used[j]) {
	      struct bounds *top_bottom = infos[j];

	      /* the next element to add depends on the process_order */
	      switch(process_order) {
		case 1:
		  if(top_bottom->top < value || value == -1)
		  {
		    value = top_bottom->top;
		    index = j;
		  }
		  break;

		case 2:
		  if(top_bottom->top > value || value == -1)
		  {
		    value = top_bottom->top;
		    index = j;
		  }
		  break;

		case 3:
		  if(top_bottom->bottom < value || value == -1)
		  {
		    value = top_bottom->bottom;
		    index = j;
		  }
		  break;

		case 4:
		  if(top_bottom->bottom > value || value == -1)
		  {
		    value = top_bottom->bottom;
		    index = j;
		  }
		  break;
	      }
	    }
	  }

	  if(opt_verbose_level_gt(options, 3))
	    switch(process_order) {
	      case 1:
		fprintf(nusmv_stderr, "top(%d) = %d\n",
		    len-1-DPListLength(tmp), value);
		break;

	      case 2:
		fprintf(nusmv_stderr, "top(%d) = %d\n",
		    len-1-DPListLength(tmp), value);
		break;

	      case 3:
		fprintf(nusmv_stderr, "bottom(%d) = %d\n",
		    len-1-DPListLength(tmp), value);
		break;

	      case 4:
		fprintf(nusmv_stderr, "bottom(%d) = %d\n",
		    len-1-DPListLength(tmp), value);
		break;
	    }

	  /* remember we used this entry already */
	  used[index] = 1;

	  /* adds the cluster at the head of the new list */
	  {
	    CPList l = CPListDup(elems[index]);
	    char *name = dup_string(names[index]);
	    int *support = SupportDup(supports[index]);
	    bdd_ptr support_bdd = bdd_dup(support_bdds[index]);

	    {
	      DPDesc *desc = DPDescAlloc();
	      DPDescSetSupportBDD(desc, support_bdd);
	      DPDescSetSupportVector(desc, support);
	      DPDescSetCList(desc, l);
	      DPDescSetName(desc, name);

	      tmp = DPListCons(desc, tmp);
	    }
	  }
	}
      }
    }

    DPListFree(dd_manager, result);
    result = tmp;
  }

  return (result);
}

void DPListPrintInfo(FILE *file, DPList DL)
{
  DPDesc *desc = DPListGetItem(DL);
  CPList list = DPDescGetCList(desc);
  char *name = DPDescGetName(desc);
  void *info = DPDescGetInfo(desc);

  nusmv_assert(info == NULL);

  fprintf(file, "Transition relation for process %s:\n", name);
  CPListPrintInfo(file, list);

  if(opt_equality_reduced(options))
  {
    int *support = DPDescGetSupportVector(desc);

    nusmv_assert(support != NULL);
    fprintf(file, "Transition relation support:\n");
    SupportPrint(file, support, opt_verbose_level_gt(options, 3));
  }
}

DPDesc *DPDescAlloc()
{
  DPDesc *desc;

  desc = ALLOC(DPDesc, 1);
  desc->support_bdd = NULL;
  desc->support_vector = NULL;
  desc->cl = CPListEmpty;
  desc->name = NULL;
  desc->info = NULL;

  return desc;
}

DPDesc *DPDescDup(DPDesc *oldD)
{
  DPDesc *desc;

  desc = DPDescAlloc();

  if(oldD != (DPDesc *)NULL) {
    {
      bdd_ptr tmp;

      tmp = DPDescGetSupportBDD(oldD);
      if (tmp != NULL)
	DPDescSetSupportBDD(desc, bdd_dup(tmp));
      else
	DPDescSetSupportBDD(desc, NULL);
    }
    {
      int *tmp;

      tmp = DPDescGetSupportVector(oldD);
      if(tmp != NULL)
	DPDescSetSupportVector(desc, SupportDup(tmp));
      else
	DPDescSetSupportVector(desc, NULL);
    }
    {
      CPList tmp;

      tmp = DPDescGetCList(oldD);
      if(tmp != CPListEmpty)
	DPDescSetCList(desc, CPListDup(tmp));
      else
	DPDescSetCList(desc, CPListEmpty);
    }
    {
      char *tmp;

      tmp = DPDescGetName(oldD);
      if(tmp != NULL)
	DPDescSetName(desc, strdup(tmp));
      else
	DPDescSetName(desc, NULL);
    }
  /* Warning: Infos are not duplicated! */
    {
      void *tmp;

      tmp = DPDescGetInfo(oldD);
      if(tmp != NULL)
	DPDescSetInfo(desc, tmp);
      else
	DPDescSetInfo(desc, NULL);
    }
  }

  return desc;
}

void DPDescFree(DdManager *dd, DPDesc *D)
{
  if(D != NULL) {
    if(D->support_bdd != NULL) bdd_free(dd, D->support_bdd);
    if(D->support_vector != NULL) FREE(D->support_vector);
    if(D->cl != CPListEmpty) CPListFree(dd, D->cl);
    if(D->name != NULL) free(D->name);
    if(D->info != NULL) {
      fprintf(nusmv_stderr, "Trying to deallocate a descriptor but not its Info.\n");
      abort();
    }
    FREE(D);
  }
}

void DPDescFreeAll(DdManager *dd, DPDesc *D, int (*freeinfo)(void *, void *))
{
  if(D != NULL) {
    if(D->support_bdd != NULL) bdd_free(dd, D->support_bdd);
    if(D->support_vector != NULL) FREE(D->support_vector);
    if(D->cl != CPListEmpty) CPListFree(dd, D->cl);
    if(D->name != NULL) free(D->name);
    if(D->info != NULL) (*freeinfo)(dd, D->info);
    FREE(D);
  }
}

bdd_ptr DPDescGetSupportBDD(DPDesc *D)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    return D->support_bdd;
  else
    return NULL;
}

void DPDescSetSupportBDD(DPDesc *D, bdd_ptr support_bdd)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    D->support_bdd = support_bdd;
}

int *DPDescGetSupportVector(DPDesc *D)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    return D->support_vector;
  else
    return NULL;
}

void DPDescSetSupportVector(DPDesc *D, int *support_vector)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    D->support_vector = support_vector;
}

CPList DPDescGetCList(DPDesc *D)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    return D->cl;
  else
    return NULL;
}

void DPDescSetCList(DPDesc *D, CPList CL)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    D->cl = CL;
}

char *DPDescGetName(DPDesc *D)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    return D->name;
  else
    return NULL;
}

void DPDescSetName(DPDesc *D, char *name)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    D->name = name;
}

void *DPDescGetInfo(DPDesc *D)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    return D->info;
  else
    return NULL;
}

void DPDescSetInfo(DPDesc *D, void *info)
{
  nusmv_assert(D != NULL);

  if(D != NULL)
    D->info = info;
}

void DPDescFreeInfo(DPDesc *D, int (*freeinfo)(void *, void *))
{
  nusmv_assert(D != NULL);

  if(D != NULL) {
    if(DPDescGetInfo(D) != NULL) {
      if(!((*freeinfo)(dd_manager, DPDescGetInfo(D)))) {
	fprintf(nusmv_stderr, "DPDesc Info deallocation had some problems.\n");
	nusmv_exit(1);
      }
      DPDescSetInfo(D, NULL);
    }
  }
}

DPDesc *DPListGetItem(DPList DL) {
  return ((DPDesc *)car(DL));
}

DPList DPListNext(DPList DL)
{
  return (DPList)cdr(DL);
}

DPList DPListCons(DPDesc *D, DPList DL)
{
  return((DPList)cons((node_ptr)D, DL));
}

DPList DPListReverse(DPList DL)
{
  return((DPList)reverse(DL));
}

int DPListLength(DPList DL)
{
  return llength(DL);
}

DPList DPListAlloc(DPDesc *D)
{
  return DPListCons(D, DPListEmpty);
}

void DPListFree(DdManager *dd, DPList DL)
{
  DPList l = DL;

  while (DPListIsNotEmpty(l)) {
    DPDesc *desc = DPListGetItem(l);
    DPDescFree(dd, desc);
    l = DPListNext(l);
  }
  free_list(DL); 
}

void DPListFreeAll(DdManager *dd, DPList DL, int (*freeinfo)(void *, void *))
{
  DPList l = DL;

  while (DPListIsNotEmpty(l)) {
    DPDesc *desc = DPListGetItem(l);
    DPDescFreeAll(dd, desc, freeinfo);
    l = DPListNext(l);
  }
  free_list(DL); 
}

int DPListIsNotEmpty(DPList DL)
{
  return (DL != DPListEmpty);
}

DPList DPListDup(DPList DL)
{
  DPList l = DL;
  DPList newDL = Nil;
  
  while (DPListIsNotEmpty(l)) {
    DPDesc *desc = DPListGetItem(l);
    DPDesc *newdesc = DPDescDup(desc);

    newDL = DPListCons(newdesc, newDL);
    l = DPListNext(l);
  }

  newDL=DPListReverse(newDL);
  return(newDL);
}

void DPListBuildSchedule(DPList DL, bdd_ptr vars)
{
  DPList l = DL;

  while(DPListIsNotEmpty(l)) {
    DPDesc *desc = DPListGetItem(l);
    bdd_ptr support = DPDescGetSupportBDD(desc);
    CPList list = DPDescGetCList(desc);

    CPListBuildSchedule(list, vars);

    while(CPListIsNotEmpty(list)) {
      CPCluster *c = CPListGetItem(list);
      bdd_ptr Ei = CPClusterGetEi(c);
      if(opt_equality_reduced(options))
	bdd_or_accumulate(dd_manager, &Ei, support);
      CPClusterSetEi(c, Ei);
      list = CPListNext(list);
    }
    l = DPListNext(l);
  }
}

DPTrans_Ptr DPTransAlloc()
{
  DPTrans_Ptr c = ALLOC(DPTrans, 1);

  if (c == (DPTrans_Ptr)NULL) {
    fprintf(nusmv_stderr, "DPTransAlloc: Unable to allocate memory.\n");
    nusmv_exit(1);
    return((DPTrans_Ptr)NULL);
  }
  
  DPTransSetForward(c, DPListEmpty);
  DPTransSetBackward(c, DPListEmpty);

  return(c);
}

void DPTransFree(DPTrans_Ptr c)
{
  nusmv_assert(c != (DPTrans_Ptr)NULL);

  if (c->forward_trans != DPListEmpty)
    DPListFree(dd_manager, c->forward_trans);

  if (c->backward_trans != DPListEmpty)
    DPListFree(dd_manager, c->backward_trans);

  FREE(c);
}

DPTrans_Ptr DPTransDup(DPTrans_Ptr c)
{
  DPList tmp_listf;
  DPList tmp_listb;
  DPTrans_Ptr cnew = DPTransAlloc();

  if (DPListIsNotEmpty(DPTransGetForward(c))) {
    tmp_listf = DPListDup(DPTransGetForward(c));
    DPTransSetForward(cnew, tmp_listf);
  }
  
  if (DPListIsNotEmpty(DPTransGetBackward(c))) {
    tmp_listb = DPListDup(DPTransGetBackward(c));
    DPTransSetBackward(cnew, tmp_listb);
  }
  return(cnew);
}

void DPTransSetBackward(DPTrans_Ptr c, DPList L)
{
  nusmv_assert(c != (DPTrans_Ptr)NULL);
  c->backward_trans = L;
}

DPList DPTransGetBackward(DPTrans_Ptr c)
{
  nusmv_assert(c != (DPTrans_Ptr)NULL);
  return(c->backward_trans);
}

void DPTransSetForward(DPTrans_Ptr c, DPList L)
{
  nusmv_assert(c != (DPTrans_Ptr)NULL);
  c->forward_trans = L;
}

DPList DPTransGetForward(DPTrans_Ptr c)
{
  nusmv_assert(c != (DPTrans_Ptr)NULL);
  return(c->forward_trans);
}

void DPTransPrintInfo(FILE *file, DPTrans_Ptr c)
{
  if (c != (DPTrans_Ptr)NULL){
    DPList f = DPTransGetForward(c);
    DPList b = DPTransGetBackward(c);
    int p = 0;

    fprintf(file, "Transition relation for each process:\n");
    while(DPListIsNotEmpty(f)) {
      fprintf(file, "Forward Transition Relation:\n");
      DPListPrintInfo(file, f);
      fprintf(file, "Backward Transition Relation:\n");
      DPListPrintInfo(file, b);
      f = DPListNext(f);
      b = DPListNext(b);
    }
  }
}

static char *dup_string(char *str)
{
  return (str == NULL ? NULL : strdup(str));
}

static int *SupportBuildList(node_ptr bdd_vars)
{
  int size = MAX_VAR_INDEX;
  int *support = (int *)ALLOC(int, size);
  int i;

  for(i = 0; i < size; i++)
    support[i] = i;

  {
    node_ptr iter;
    for(iter = bdd_vars; iter != Nil; iter = cdr(iter))
    {
      int idx = node_get_int(iter);

      /* TODO: This more elegantly accessing next2current from compileEncode */
      /* TODO: For now assumes there are no inputs 			     */
      /* TODO: I could map support[idx] to -1 and then check during          */
      /*       permutation that no node is shifted to -1                     */
      if(is_next_variable(idx))
	support[idx] = idx-1;
    }
  }

  return support;
}

static bdd_ptr SupportBuildBDD(node_ptr bdd_vars)
{
  bdd_ptr support_bdd = bdd_one(dd_manager);
  node_ptr iter;

  for(iter = bdd_vars; iter != Nil; iter = cdr(iter))
  {
    int idx = node_get_int(iter);

    /* TODO: This more elegantly accessing next2current from compileEncode */
    /* TODO: For now assumes there are no inputs 			     */
    /* TODO: I could map support[idx] to -1 and then check during          */
    /*       permutation that no node is shifted to -1                     */
    if(is_current_variable(idx))
    {
      bdd_ptr tmp = bdd_new_var_with_index(dd_manager, idx);
      bdd_and_accumulate(dd_manager, &support_bdd, tmp);
      /* This free might not be needed */
      /* bdd_free(dd_manager, tmp); */
      /* If I add it, I get a runtime error in CUDD */
    }
  }

  bdd_and_accumulate(dd_manager, &support_bdd, input_variables_bdd);

  return support_bdd;
}

static int *SupportDup(int *support)
{
  int size = MAX_VAR_INDEX;
  int *s = (int *)ALLOC(int, size);
  int i;

  for(i = 0; i < size; i++)
    s[i] = support[i];

  return s;
}

static void SupportPrint(FILE *file, int *support, int detailed)
{
  int size = MAX_VAR_INDEX;
  int i;

  if(detailed)
  {
    for(i = 0; i < size; i++)
      if(support[i] != i)
	fprintf(file, "  v%d => v%d\n", i, support[i]);
  } else {
    int cnt = 0;
    for(i = 0; i < size; i++)
      if(support[i] != i)
	cnt++;
    fprintf(file, "  %d variables in the support\n", cnt);
  }
}

/*---------------------------------------------------------------------------*/
/* Definitions of static functions                                           */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Extracts the next function for a single process.]

  Description        [Extracts the next function for a single process.]

  SideEffects        []

******************************************************************************/
static node_ptr extract(node_ptr sexp, node_ptr proc, node_ptr *bdd_vars, node_ptr *vars, int *is_identity) {
  nusmv_assert(node_get_type(sexp) == EQDEF);

  {
    node_ptr lhs = car(sexp);
    node_ptr rhs = cdr(sexp);

    nusmv_assert(node_get_type(rhs) == CASE);

    {
      node_ptr cur = rhs;
      node_ptr acc = Nil;

      while(cur != Nil) {
	node_ptr n;
	node_ptr p;

	if(node_get_type(caar(cur)) == TRUEEXP) {
	  p = Nil;
	} else {
	  p = caar(cur);
	}

	if(p == Nil || proc == p) {
	  /* The transition relation for this variable is the identity */
	  int identity = (car(lhs) == cdar(cur));

	  if(!identity) {
	    /* computes which BDD variables correspond to this variable */
	    node_ptr new_bdd_vars = get_bdd_variables(car(lhs));

	    /* added BDD variables to the list of modified BDD variables */
	    *bdd_vars = append(*bdd_vars, new_bdd_vars);
	    /* added this variable to the list of modified variables */
	    *vars = cons(car(lhs), *vars);
	  }

	  *is_identity = identity;
	  return new_node(EQDEF, lhs, cdar(cur));
	}

	cur = cdr(cur);
	if(cur != Nil && node_get_type(cur) == FALSEEXP) cur = Nil;
	if(cur != Nil && node_get_type(cur) == TRUEEXP) cur = Nil;
      }

      nusmv_assert(false);
      return Nil;
    }
  }
}

/**Function********************************************************************

  Synopsis           [Returns the list of indices of the bdd variables
  corresponding to a given variable.]

  Description        [Returns the list of indices of the bdd variables
  corresponding to a given variable.]

  SideEffects        []

******************************************************************************/
static node_ptr get_bdd_variables(node_ptr var)
{
  int n_bdd_vars = dd_get_size(dd_manager), i;
  node_ptr result = Nil;
  char *name = get_name(var);

  for(i = 0; i < n_bdd_vars; i++)
  {
    node_ptr bool_var = get_bool_variable_name(i);
    if(bool_var != Nil) {
      if(node_get_type(bool_var) == NEXT)
	bool_var = car(bool_var);
      {
	node_ptr bdd_var = car(bool_var);

	if(var == bdd_var) {
	  if(opt_verbose_level_gt(options, 3))
	    fprintf(nusmv_stderr, "BDD variable %d belongs to %s\n", i, name);
	  result = cons((node_ptr)i, result);
	}
      }
    }
  }
  free(name);

  return result;
}

/**Function********************************************************************

  Synopsis           [Recursively build a process portioned transition
  relation.]

  Description        [Recursively build a process portioned transition
  relation.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static CPList CPListBuildBasicProcRecur(CPList clusters,
		node_ptr sexp,
		node_ptr proc,
		node_ptr *bdd_vars,
		node_ptr *vars,
		add_ptr assumption)
{
  CPList result;

  yylineno = node_get_lineno(sexp);

  if(node_get_type(sexp) == EQDEF) {
    // true if the sexp is the identity
    int is_identity;

    // extract the part of the sexp corresponding to the given process
    // also sets the value of is_identity and finds out which bdd variables
    // are involved in the transition relation.
    sexp = extract(sexp, proc, bdd_vars, vars, &is_identity);

    if(opt_equality_reduced(options))
    {
      // if the sexp is the identity, do not add it to the list of the BDDs
      if(is_identity)
	result = clusters;
      else
	result = CPListBuildBasicRecur(clusters, sexp, assumption);
    }
    else
      result = CPListBuildBasicRecur(clusters, sexp, assumption);
  } else {
    switch(node_get_type(sexp)) {
      case AND:
	{
	  result = CPListBuildBasicProcRecur(clusters, car(sexp),
	      proc, bdd_vars, vars, assumption);
	  result = CPListBuildBasicProcRecur(result, cdr(sexp),
	      proc, bdd_vars, vars, assumption);
	  break;
	}
      default:
	result = CPListBuildBasicRecur(clusters, sexp, assumption);
    }
  }

  return(result);
}

/**Function********************************************************************

  Synopsis           [Builds a clustered transition relation given
  variables and assumptions]

  Description        [Builds a clustered transition relation given
  variables and assumptions. You obtain a list of small transition
  relations, one item for each variable]

  SideEffects        []

******************************************************************************/
static CPList CPListBuildElementaryProcBDDList(node_ptr var_list, 
    node_ptr proc,
    int *top_ptr,
    int *bottom_ptr,
    node_ptr *bdd_vars,
    add_ptr assumption)
{
  node_ptr l;
  CPList cluster_bdd = Nil;
  node_ptr vars = Nil;

  if (opt_verbose_level_gt(options, 5)) {
    fprintf(nusmv_stderr, "state_variables = ");
    print_list(nusmv_stderr, state_variables, 0);
    fprintf(nusmv_stderr, "\n");
    {
      node_ptr iter;
      int cnt = 0;
      for(iter = real_state_variables; iter != Nil; iter = cdr(iter), cnt++)
      {
	fprintf(nusmv_stderr, "-- %6d ------------------------------\n", cnt);
	add_ptr add = (add_ptr)car(iter);
	Cudd_PrintDebug(dd_manager, add, 4, 4);
	fprintf(nusmv_stderr, "----------------------------------------\n\n");
      }
    }
  }

  CPListBuildStart();
  for (l = var_list; l != Nil; l = cdr(l)) {
    node_ptr var_i = car(l);
    node_ptr next_i = lookup_variable_sexp_model_hash(var_i);

    if (opt_verbose_level_gt(options, 2)) {
      indent_node(nusmv_stderr, " computing next values for variable: ",
	  var_i, "\n");
    }
  
    cluster_bdd = CPListBuildBasicProcRecur(cluster_bdd,
	var_model_sexp_get_next(next_i),
	proc, bdd_vars, &vars,
	assumption);
  }
  CPListBuildStop();

  if (opt_verbose_level_gt(options, 2))
  {
    {
      node_ptr iter;

      fprintf(nusmv_stderr, "bdd_vars = (");
      for(iter = *bdd_vars; iter != Nil; iter = cdr(iter))
      {
	int index = node_get_int(iter);
	fprintf(nusmv_stderr, "%d", index);
	if(cdr(iter) != Nil)
	  fprintf(nusmv_stderr, ", ");
      }
      fprintf(nusmv_stderr, ")");
    }
    {
      node_ptr iter;

      fprintf(nusmv_stderr, " - vars = (");
      for(iter = vars; iter != Nil; iter = cdr(iter))
      {
	print_node(nusmv_stderr, iter);
	if(cdr(iter) != Nil)
	  fprintf(nusmv_stderr, ", ");
      }
      fprintf(nusmv_stderr, ")");
    }
    fprintf(nusmv_stderr, "\n");
  }

  if (opt_verbose_level_gt(options, 2))
  {
    node_ptr iter;
    int top = -1, bottom = -1;

    for(iter = *bdd_vars; iter != Nil; iter = cdr(iter))
    {
      int index = node_get_int(iter);
      if(top == -1 || top > index)
	top = index;
      if(bottom == -1 || bottom < index)
	bottom = index;
    }

    fprintf(nusmv_stderr, "top = %d, bottom = %d\n", top, bottom);

    if(top_ptr != NULL)
      *top_ptr = top;

    if(bottom_ptr != NULL)
      *bottom_ptr = bottom;
  }

  cluster_bdd = CPListReverse(cluster_bdd);
  return(cluster_bdd);
} 

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

******************************************************************************/
static boolean phase_of(node_ptr process)
{
  char *name = get_name(car(process));

  if(name != NULL)
  {
    int len = strlen(name);

    if(len > 3)
      if(strcmp(name + len - 3, "_po") == 0)
      {
	free(name);
	return true;
      }

    free(name);
  }

  return false;
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

******************************************************************************/
static char *get_name(node_ptr node)
{
  if(node == Nil) return strdup("");

  switch(node_get_type(node))
  {
    case DOT:
      {
	char *lhs = get_name(car(node));
	char *rhs = get_name(cdr(node));

	if(strcmp(lhs, "") == 0)
	{
	  free(lhs);
	  return rhs;
	}

	if(strcmp(rhs, "") == 0)
	{
	  free(rhs);
	  return lhs;
	}

	{
	  int l = strlen(lhs) + strlen(rhs) + 2;
	  char *name = (char *)malloc(l);

	  snprintf(name, l, "%s.%s", lhs, rhs);

	  free(lhs);
	  free(rhs);

	  return name;
	}
      }
      
    case ATOM:
      return strdup(str_get_text(node_get_lstring(node)));

    case NUMBER:
      {
	int n = (int)car(node);
	char *name = (char *)malloc(8);

	snprintf(name, 8, "%d", n);

	return name;
      }

    case ARRAY:
      {
	char *lhs = get_name(car(node));
	char *rhs = get_name(cdr(node));

	nusmv_assert(lhs != NULL);
	nusmv_assert(rhs != NULL);

	{
	  int l = strlen(lhs) + strlen(rhs) + 3;
	  char *name = (char *)malloc(l);

	  snprintf(name, l, "%s[%s]", lhs, rhs);

	  free(lhs);
	  free(rhs);

	  return name;
	}
      }

    default:
      nusmv_assert(false);
      break;
  }

  return NULL;
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

******************************************************************************/
static CPList CPListBuildPartitioned(CPList cluster)
{
  Partition_Method pm = get_partition_method(options);

  switch(pm){
    case Monolithic:
      return CPListBuildPartitionedMonolithic(cluster);

    case Threshold:
      return CPListBuildPartitionedThreshold(cluster);

    case Iwls95CP:
      return CPListBuildPartitionedIwls95(cluster);

    default: 
      {
	rpterr("CPListBuildPartitioned: Unknown partitioning method.\n");
	break;
      }
  }
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

******************************************************************************/
static CPList CPListBuildPartitionedMonolithic(CPList CL)
{
  CPCluster *C = CPClusterAlloc();
  bdd_ptr result_bdd = CPListBuildMonolithicBDDFromCPlist(CL);

  CPClusterSetTi(C, result_bdd);

  return CPListAlloc(C);
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

******************************************************************************/
static CPList CPListBuildPartitionedThreshold(CPList CL)
{
  CPList newCL = CPListEmpty;
  CPListBuildThreshold(dd_manager, CL, &newCL, 
      get_conj_part_threshold(options));

  return newCL;
}

/**Function********************************************************************

  Synopsis           []

  Description        []

  SideEffects        []

******************************************************************************/
static CPList CPListBuildPartitionedIwls95(CPList CL)
{
  nusmv_assert(0);
}
