/**CFile***********************************************************************

  FileName    [compileCPUtil.c]

  PackageName [compile]

  Synopsis    [Utilities for the CP data structures.]

  Description [Utilities for the CP data structures. Here are provided
  primitives to add a cluster to a cluster list, and routines to check
  the correctness of the clustered partioned transition relation.]

  SeeAlso     []

  Author      [Emanuele Olivetti and Marco Roveri]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 

static char rcsid[] UTIL_UNUSED = "$Id: compileCPUtil.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/**AutomaticEnd***************************************************************/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [Perform some test on transition relation]

  Description        [Check the equality with monolithic transition relation,
                      and check the quantification schedule.]

  SideEffects        []

  SeeAlso            [CPCheckMonolithic, CPCheckQuantificationSchedule]

******************************************************************************/
void compileCPCheckPartitionedTransition(CPTrans_Ptr transition, node_ptr list_variables, bdd_ptr assumption)
{
  CPList mono = CPListBuildMonolithic(list_variables, assumption);
  int i = 0;
  fprintf(nusmv_stderr,
	  "\nChecking equality Partitioned vs. Monolithic : \n");
  i = CPListCheckComparing(dd_manager, CPTransGetBackward(transition), mono);

  fprintf(nusmv_stderr,"\tBackward : %d\n",i);

  i = CPListCheckComparing(dd_manager, CPTransGetForward(transition), mono);
  fprintf(nusmv_stderr,"\tForward  : %d\n",i);

  fprintf(nusmv_stderr,"\nChecking Quantification Schedule : \n");
  i = CPCheckQuantificationSchedule(dd_manager, CPTransGetBackward(transition));
  fprintf(nusmv_stderr,"\tBackward : %d\n",i);

  i = CPCheckQuantificationSchedule(dd_manager, CPTransGetForward(transition));
  fprintf(nusmv_stderr,"\tForward  : %d\n",i);
}

/**Function********************************************************************

  Synopsis    [Given a list of clusters representing the relations this
               function checks if the schedule is correct.]

  Description [Let Ci and Ti be the ith cube and relation in the list.
               The schedule is correct iff<br>
  <ol>
  <li> For all Tj: j > i, S(Tj) and S(Ci) do not intersect, i.e., the
       variables which are quantified in Ci should not appear in the
       Tj for j>i.</li>
  <li> For any j, S(Ci) and S(Cj) do not intersect.</li>
  </ol><br>

  where S(T) is the set of support of the BDD T.
  Returns 1 if the schedule is correct, 0 otherwise.
  This function is implemented for checking the correctness of the
  clustering algorithm only.]

  SideEffects []

******************************************************************************/
int CPCheckQuantificationSchedule(DdManager *dd, CPList Cl_list)
{
  CPList Clist = Cl_list;
  
  while (CPListIsNotEmpty(Clist)) {
    CPCluster * Cj = CPListGetItem(Clist);
    CPList Nlist = CPListNext(Clist);

    while (CPListIsNotEmpty(Nlist)) {
      bdd_ptr intersection;
      CPCluster * Ci = CPListGetItem(Nlist);
      
      intersection = bdd_cube_diff(dd, CPClusterGetEi(Cj),
                                   bdd_support(dd, CPClusterGetTi(Ci)));

      bdd_free(dd, intersection);
      if (intersection != CPClusterGetEi(Cj))
        /* Violates the condition "a" */
        return(-1);
      intersection = bdd_cube_diff(dd, CPClusterGetEi(Cj), CPClusterGetEi(Ci));
      bdd_free(dd, intersection);
      if ((intersection != CPClusterGetEi(Cj)) &&
          (get_partition_method(options) == Iwls95CP))
        /* Violates the condition "b" */
        return(-2);
      Nlist = CPListNext(Nlist);
    } /* while (Nlist!= CPListEmpty) */
    Clist = CPListNext(Clist);
  } /* while (Clist != CPListEmpty) */
  return(1);
}

/**Function********************************************************************

  Synopsis           [Add the LTL tableau as a new BDD cluster in the transition relation.]

  Description        [Add the LTL tableau as a new BDD cluster in the transition relation.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
CPTrans_Ptr Compile_CPAddClusterAndFillSchedule(CPTrans_Ptr trans,
                                                bdd_ptr tableau_bdd)
{
  bdd_ptr vars;
  CPCluster * Ci;
  CPList fwd_clusters , bwd_clusters, old;
  CPTrans_Ptr result = CPTransAlloc();
  
  nusmv_assert(result != (CPTrans_Ptr)NULL);

  Ci = CPClusterAlloc();

  CPClusterSetTi(Ci, bdd_dup(tableau_bdd));

  vars = bdd_and(dd_manager, state_variables_bdd, input_variables_bdd); 
  old = CPTransGetForward(trans);
  fwd_clusters = CPListAddOneCluster(Ci, old, vars);
  CPTransSetForward(result, fwd_clusters);

  vars = bdd_and(dd_manager, next_state_variables_bdd,
                             next_input_variables_bdd);
  old = CPTransGetBackward(trans);
  bwd_clusters = CPListAddOneCluster(Ci, old, vars);
  CPTransSetBackward(result, bwd_clusters);

  CPClusterFree(dd_manager, Ci);
  return(result);
}

