#define NO_PRINT_STATES

/**CFile***********************************************************************

  FileName    [compileCP.c]

  PackageName [compile]

  Synopsis    [Methods of CPCluster, CPList, CPTrans structures]

  Description [Methods of CPCluster, CPList, CPTrans structures for all
               conjunctive partitioning methods.]

  SeeAlso     []

  Author      [Emanuele Olivetti]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h" 
#include "ustring.h" 
#include "node.h" 

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [Hash to keep track of already considered expressions.]

  Description [Hash to keep track of already considered expressions.]

  SeeAlso     []

******************************************************************************/
static st_table * CP_hash = (st_table*)NULL;

static void init_CP_hash() 
{
  CP_hash = st_init_table(st_ptrcmp, st_ptrhash);
  if (CP_hash == (st_table *)NULL) {
    fprintf(nusmv_stderr, "init_CP_hash: unable to allocate local hash.\n");
    nusmv_exit(1);
  }
}
static void quit_CP_hash()
{
  nusmv_assert(CP_hash != (st_table *)NULL);
  st_free_table(CP_hash);
}
static void insert_CP_hash(node_ptr el)
{
  if (st_add_direct(CP_hash, (char *)el, (char *)1) == ST_OUT_OF_MEM) {
    fprintf(nusmv_stderr, "insert_CP_hash: Unable to insert result in local hash.\n");
    nusmv_exit(1);
  }
}
static int query_CP_hash(node_ptr el)
{
  int v;
  int result = st_lookup(CP_hash, (char *)el, (char **)&v);

  return(result);
}


/**AutomaticStart*************************************************************/

/**AutomaticEnd***************************************************************/



/*-------------------------------------------------------------*/
/* CPCluster functions */
/*-------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Allocates and initializes memory for a cluster.]

  Description        [Allocates and initializes memory for a cluster.]

  SideEffects        []

******************************************************************************/
CPCluster * CPClusterAlloc(void)
{
  CPCluster * CI;

  CI = ALLOC(CPCluster, 1);
  CI->Ti          = (bdd_ptr)NULL;
  CI->Ei          = (bdd_ptr)NULL;
  CI->Info        = (void *)NULL;
  return(CI);
}

/**Function********************************************************************

  Synopsis          [Duplicates a cluster allocating necessary memory.]

  Description       [Duplicates a cluster allocating necessary
  memory. Infos are not duplicated.]

  SideEffects       []

******************************************************************************/
CPCluster * CPClusterDup(CPCluster * oldC)
{
  CPCluster * CI;
  CI = CPClusterAlloc();

  if (oldC != (CPCluster *)NULL) {
    bdd_ptr tmp;
    void * info;
    tmp = CPClusterGetTi(oldC);
    if (tmp != (bdd_ptr)NULL)
      CPClusterSetTi(CI, bdd_dup(tmp));
    else
      CPClusterSetTi(CI,(bdd_ptr)NULL);
    tmp = CPClusterGetEi(oldC);
    if (tmp != (bdd_ptr)NULL)
      CPClusterSetEi(CI, bdd_dup(tmp));
    else
      CPClusterSetEi(CI,(bdd_ptr)NULL);
    info = CPClusterGetInfo(oldC);
    if (info != (void *)NULL)
  /* Warning: Infos are not duplicated! */
      CPClusterSetInfo(CI, info);
    else
      CPClusterSetInfo(CI,(void *)(NULL));
  }
  return(CI);
}

/**Function********************************************************************

  Synopsis           [Frees the  memory allocated for CPCluster struct.]

  Description        [Frees the  memory allocated for CPCluster
  struct. It doesn't free the Info data pointend by the Info field.]

  SideEffects        []

******************************************************************************/
void CPClusterFree(DdManager *dd, CPCluster * CI)
{
  if (CI != (CPCluster *)NULL) {
    if (CI->Ti        != (bdd_ptr)NULL) bdd_free(dd, CI->Ti);
    if (CI->Ei        != (bdd_ptr)NULL) bdd_free(dd, CI->Ei);
    if (CI->Info      != (void *)NULL) {
      fprintf(nusmv_stderr, "Trying to deallocate a cluster but not its Info.\n");
      abort();
    }
    FREE(CI);
  }
}

/**Function********************************************************************

  Synopsis           [Frees the  memory allocated for CPCluster struct.]

  Description        [Frees the  memory allocated for CPCluster struct
  and the Info data, given a method.]

  SideEffects        []

******************************************************************************/
void CPClusterFreeAll(DdManager *dd, CPCluster * CI,
                      int (*freeinfo)(void *, void *))
{
  if (CI != (CPCluster *)NULL) {
    if (CI->Ti        != (bdd_ptr)NULL) bdd_free(dd, CI->Ti);
    if (CI->Ei        != (bdd_ptr)NULL) bdd_free(dd, CI->Ei);
    if (CI->Info      != (void *)NULL) (*freeinfo)(dd, CI->Info);
    FREE(CI);
  }
}

/**Function********************************************************************

  Synopsis           [Checks if two clusters are equal.]

  Description        [Checks if two clusters are equal. If it is the
  case then return 1 else return 0. <br>
  Notice that the check is performed only using the \"Ti\" field of the
  CPCluster structures.] 

  SideEffects        []

******************************************************************************/
int CPClusterEqual(CPCluster * A, CPCluster * B)
{
  nusmv_assert(A != (CPCluster *)NULL);
  nusmv_assert(B != (CPCluster *)NULL);

  return((CPClusterGetTi(A) == CPClusterGetTi(B)));
}


/**Function********************************************************************

  Synopsis           [Returns a pointer to the transition inside the cluster]

  Description        [Returns a pointer to the transition inside the cluster]

  SideEffects        []

******************************************************************************/
bdd_ptr CPClusterGetTi(CPCluster * CI)
{
  nusmv_assert(CI != (CPCluster *)NULL);
  
  if (CI != NULL)
    return(CI->Ti);
  else
    return((bdd_ptr)NULL);
}

/**Function********************************************************************

  Synopsis           [Sets the transition inside the cluster]

  Description        [Sets the transition inside the cluster]

  SideEffects        []

******************************************************************************/
void CPClusterSetTi(CPCluster * CI, bdd_ptr newTi)
{
  nusmv_assert(CI != (CPCluster *)NULL);

  if (CI != NULL)
    CI->Ti = newTi;
}

/**Function********************************************************************

  Synopsis           [Returns a pointer to the variables to be quantified]

  Description        [Returns a pointer to the variables to be
  quantified respect to the transition relation inside the cluster]

  SideEffects        []

******************************************************************************/
bdd_ptr CPClusterGetEi(CPCluster * CI)
{
  nusmv_assert(CI != (CPCluster *)NULL);

  if (CI != NULL)
    return(CI->Ei);
  else
    return((bdd_ptr)NULL);
}

/**Function********************************************************************

  Synopsis           [Sets the variables to be quantified inside the cluster]

  Description        [Sets the variables to be quantified inside the cluster]

  SideEffects        []

******************************************************************************/
void CPClusterSetEi(CPCluster * CI, bdd_ptr newEi)
{
  nusmv_assert(CI != (CPCluster *)NULL);

  if (CI != NULL)
    CI->Ei = newEi;
}

/**Function********************************************************************

  Synopsis           [Returns a pointer to the (optional) info of the cluster]

  Description        [Returns a pointer to the (optional) info of the cluster]

  SideEffects        []

******************************************************************************/
void * CPClusterGetInfo(CPCluster * CI)
{
  nusmv_assert(CI != (CPCluster *)NULL);

  if (CI != NULL)
    return(CI->Info);
  else
    return((void *)NULL);
}

/**Function********************************************************************

  Synopsis           [Sets the (optional) info inside the cluster]

  Description        [Sets the (optional) info inside the cluster]

  SideEffects        []

******************************************************************************/
void CPClusterSetInfo(CPCluster * CI, void * newInfo)
{
  nusmv_assert(CI != (CPCluster *)NULL);

  if (CI != NULL)
    CI->Info = newInfo;
}

/**Function********************************************************************

  Synopsis           [Frees the info of a cluster]

  Description        [Frees the info of a cluster given a proper fuction]

  SideEffects        []

******************************************************************************/
void CPClusterFreeInfo(CPCluster * CI, int (*freeinfo)(void *, void *))
{
  nusmv_assert(CI != (CPCluster *)NULL);

  if (CI != NULL) {
    if (CPClusterGetInfo(CI) != (void *)NULL) {
      if (!((*freeinfo)(dd_manager, CPClusterGetInfo(CI)))) {
	fprintf(nusmv_stderr, "CPCluster Info deallocation had some problems.\n");
	nusmv_exit(1);
	}
      /* Resets the info field. */
      CPClusterSetInfo(CI, (void *)NULL);
    }
  }
}
/*-------------------------------------------------------------*/
/* End of CPCluster's functions */
/*-------------------------------------------------------------*/

/*-------------------------------------------------------------*/
/* CPList functions */
/*-------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis [Returns the cluster held by the current position of the list]

  Description [Returns the cluster held by the current position of the list]

  SideEffects        []

******************************************************************************/
CPCluster * CPListGetItem(CPList CL)
{
  return((CPCluster *)car(CL));
}

/**Function********************************************************************

  Synopsis           [Returns the next position in the list]

  Description        [Returns the next position in the list]

  SideEffects        []

******************************************************************************/
CPList CPListNext(CPList CL)
{
  return((CPList)cdr(CL));
}


/**Function********************************************************************

  Synopsis           [Adds a cluster on top of a given list]

  Description        [Adds a cluster on top of a given list]

  SideEffects        []

******************************************************************************/
CPList CPListCons(CPCluster * C , CPList CL)
{
  return((CPList)cons((node_ptr)C, CL));
}

/**Function********************************************************************

  Synopsis           [Reverses a given list]

  Description        [Reverses a given list]

  SideEffects        [The argument is modified]

******************************************************************************/
CPList CPListReverse(CPList CL)
{
  return((CPList)reverse(CL));
}

/**Function********************************************************************

  Synopsis           [Returns the number of items in the list]

  Description        [Returns the number of items in the list]

  SideEffects        []

******************************************************************************/
int CPListLength(CPList CL)
{
  return(llength(CL));
}

/**Function********************************************************************

  Synopsis           [Creates a list with a given cluster]

  Description        [Creates a list with a given cluster]

  SideEffects        []

******************************************************************************/
CPList CPListAlloc(CPCluster * el)
{
  return(CPListCons(el , CPListEmpty));
}

/**Function********************************************************************

  Synopsis           [Frees the list of clusters and all its elements]

  Description        [Frees the list of clusters and all its
  elements. It doesn't free the Info field of the cluster.]

  SideEffects        []

******************************************************************************/
void CPListFree(DdManager *dd, CPList CL)
{
   CPList l = CL;

   while (CPListIsNotEmpty(l)) {
     CPCluster * c = CPListGetItem(l);

     CPClusterFree(dd, c);
     l = CPListNext(l);
   }
   free_list(CL); 
}


/**Function********************************************************************

  Synopsis           [Tests if a list is empty.]

  Description        [Tests if a list is empty.]

  SideEffects        []

******************************************************************************/
int CPListIsEmpty(CPList CL)
{
  return(CL == CPListEmpty);
}

/**Function********************************************************************

  Synopsis           [Tests if a list is empty.]

  Description        [Tests if a list is empty.]

  SideEffects        []

******************************************************************************/
int CPListIsNotEmpty(CPList CL)
{
  return(CL != CPListEmpty);
}


/**Function********************************************************************

  Synopsis           [Frees the list of clusters and all its elements]

  Description        [Frees the list of clusters and all its elements,
  Info field too, given a proper function.] 

  SideEffects        []

******************************************************************************/
void CPListFreeAll(DdManager *dd, CPList CL, int (*freeinfo)(void *, void *))
{
   CPList l = CL;

   while (CPListIsNotEmpty(l)) {
     CPCluster * c = CPListGetItem(l);

     CPClusterFreeAll(dd, c, freeinfo);
     l = CPListNext(l);
   }
   free_list(CL); 
}

/**Function********************************************************************

  Synopsis           [Frees the only Info field of each cluster]

  Description        [Frees the only Info field of each cluster in the list given a proper function, but not the cluster, nor the list.]

  SideEffects        []

******************************************************************************/
void CPListFreeInfo(CPList CL, int (*freeinfo)(void *, void *))
{
   CPList l = CL;

   while (CPListIsNotEmpty(l)) {
     CPCluster * c = CPListGetItem(l);

     CPClusterFreeInfo(c, freeinfo);
     l = CPListNext(l);
   }
}

/**Function********************************************************************

  Synopsis           [Dups a given list of clusters]

  Description        [Duplicates a given list of clusters and each
  cluster. The Info field of each cluster is not duplicated, but only
  initialized.]

  SideEffects        []

******************************************************************************/
CPList CPListDup(CPList CL)
{
  CPList l = CL;
  CPList newCL = Nil;
  
  while (CPListIsNotEmpty(l)) {
    CPCluster * c = CPListGetItem(l);
    CPCluster * newc = CPClusterDup(c);

    newCL = CPListCons(newc, newCL);
    l = CPListNext(l);
  }
  newCL=CPListReverse(newCL);
  return(newCL);
}


/**Function********************************************************************

  Synopsis           [Deletes every occurrence of a cluster in a list]

  Description        [Deletes every occurrence of a cluster in a list]

  SideEffects        []

******************************************************************************/
CPList CPListDeleteCluster(CPList source, CPCluster * el)
{
  CPList result = Nil;
  
  while (CPListIsNotEmpty(source)) {
    if (!CPClusterEqual(CPListGetItem(source), el))
      result = CPListCons(CPListGetItem(source), result);
    source = CPListNext(source);
  }
  return(result);
}

/**Function********************************************************************

  Synopsis [Adds one cluster and compute the variables to be quantified]

  Description [Adds one cluster and compute the variables to be quantified]

  SideEffects        []

******************************************************************************/
CPList CPListAddOneCluster(CPCluster * Ci, CPList CL, bdd_ptr vars)
{
  CPList new_clusters, l;
  bdd_ptr tmp, tmp2;
  CPCluster * newCi;
  
  newCi = CPClusterAlloc();
  CPClusterSetTi(newCi, bdd_dup(CPClusterGetTi(Ci)));

  new_clusters = CPListDup(CL);

  tmp = vars;
  l = new_clusters;
  while (CPListIsNotEmpty(l)) {
    CPCluster * Ci = CPListGetItem(l);

    tmp2 = bdd_cube_diff(dd_manager, tmp, CPClusterGetEi(Ci));
    bdd_free(dd_manager, tmp);
    tmp = tmp2;
    l = CPListNext(l);
  }
  CPClusterSetEi(newCi, tmp);
  new_clusters = CPListCons(newCi, new_clusters);
  return(new_clusters);  
}

/**Function********************************************************************

  Synopsis           [Compare two transition relations]

  Description        [Compare two transition relations computing the
  monolithic equivalents and comparing them.]

  SideEffects        []

******************************************************************************/
int CPListCheckComparing(DdManager *dd, CPList CL1, CPList CL2)
{
  int result = 0;
  bdd_ptr Acc1 = bdd_one(dd);
  bdd_ptr Acc2 = bdd_one(dd);

  while (CPListIsNotEmpty(CL1)) {
    CPCluster * Ci = CPListGetItem(CL1);

    bdd_and_accumulate(dd, &Acc1, CPClusterGetTi(Ci));
    CL1 = CPListNext(CL1);
  }

  while (CPListIsNotEmpty(CL2)) {
    CPCluster * Ci = CPListGetItem(CL2);

    bdd_and_accumulate(dd, &Acc2, CPClusterGetTi(Ci));
    CL2 = CPListNext(CL2);
  }

  result = (Acc1 == Acc2);
  bdd_free(dd, Acc1);
  bdd_free(dd, Acc2);
  return(result);
}

/**Function********************************************************************

  Synopsis [Builds the quantification schedule of a given cluster list]

  Description [Builds the quantification schedule of a given cluster list]

  SideEffects        []

******************************************************************************/
void CPListBuildSchedule(CPList CL, bdd_ptr vars)
{
  bdd_ptr tmp = CPListBuildScheduleRecur(CL, vars);

  bdd_free(dd_manager, tmp);
  return;
}

/**Function********************************************************************

  Synopsis           [Builds a clustered transition relation given
  variables and assumptions]

  Description        [Builds a clustered transition relation given
  variables and assumptions. You obtain a list of small transition
  relations, one item for each variable]

  SideEffects        []

******************************************************************************/
CPList CPListBuildElementaryBDDList(node_ptr var_list, add_ptr assumption)
{
  node_ptr l;
  CPList cluster_bdd = Nil;

  init_CP_hash();
  for (l = var_list; l != Nil; l = cdr(l)) {
    node_ptr var_i = car(l);
    node_ptr next_i = lookup_variable_sexp_model_hash(var_i);

    if (opt_verbose_level_gt(options, 2)) {
      indent_node(nusmv_stderr, " computing next values for variable: ",
                  var_i, "\n");
    }
    cluster_bdd = CPListBuildBasicRecur(cluster_bdd,
					var_model_sexp_get_next(next_i),
					assumption);
  }
  quit_CP_hash();
  cluster_bdd = CPListReverse(cluster_bdd);
  return(cluster_bdd);
} 

/**Function********************************************************************

  Synopsis           [Forms the clusters of relations based on BDD
  size heuristic.]

  Description        [The clusters are formed by taking the product in
  order. Once the BDD size of the current cluster reaches a threshold, a new
  cluster is created.]

  SideEffects        []

******************************************************************************/
void CPListBuildThreshold(DdManager *dd, CPList RelationList,
                          CPList * NewRelationList, int threshold)
{
  int flag;
  CPList List = RelationList;
  bdd_ptr cluster, tmp_cluster, relation;

  *NewRelationList = CPListEmpty;
  while (CPListIsNotEmpty(List)) {
    flag = 0;
    cluster = bdd_one(dd);
    do {
      CPCluster * Ci = CPListGetItem(List);

      relation = CPClusterGetTi(Ci);
      tmp_cluster = bdd_and(dd, cluster, relation);

      if (((bdd_size(dd, cluster) <= threshold) &&
           (bdd_size(dd, relation) <= threshold)) || (flag == 0)){
        bdd_free(dd, cluster);
        cluster = tmp_cluster;
        flag = 1;
        List = CPListNext(List);
      }
      else {
        bdd_free(dd, tmp_cluster);
        break;
      }
    } while (CPListIsNotEmpty(List)); /* do */
    {
      CPCluster * New_Ci = CPClusterAlloc();

      CPClusterSetTi(New_Ci, cluster);
      *NewRelationList = CPListCons(New_Ci, *NewRelationList);
    }
  } /* while (List != Nil) */
}

/**Function********************************************************************

  Synopsis           [Builds the monolithic transition relation BDD]

  Description        [Starting from the the list of variables builds
  the corresponding transition relation BDD. The result is restricted to
  the set <tt>assumption</tt> using the Coudert and Madre algorithm.]

  SideEffects        []

******************************************************************************/
bdd_ptr CPListBuildMonolithicBDD(node_ptr list_variables, bdd_ptr assumption)
{
  CPList l, cluster_bdd;
  bdd_ptr result_bdd = bdd_one(dd_manager);

  /* converting variables list to bdd list */
  cluster_bdd = CPListBuildElementaryBDDList(list_variables, assumption);

  /* building monolithic BDD ANDing bdd list entries. */
  ;
  for (l = cluster_bdd; CPListIsNotEmpty(l) ; l = CPListNext(l)) {
    bdd_ptr tmp = CPClusterGetTi(CPListGetItem(l));

    bdd_and_accumulate(dd_manager, &result_bdd, tmp);
  }
  CPListFree(dd_manager, cluster_bdd);
  return(result_bdd);
}

/**Function********************************************************************

  Synopsis           [Builds the monolithic transition relation BDD]

  Description        [Starting from the a partitioned transtion
  relation it builds the BDD corresponding to the monolithic
  transition relation.]

  SideEffects        []

******************************************************************************/
bdd_ptr CPListBuildMonolithicBDDFromCPlist(CPList CL)
{
  CPList L;
  bdd_ptr result = bdd_one(dd_manager);

  for (L = CL ; CPListIsNotEmpty(L) ; L = CPListNext(L) ) {
    bdd_and_accumulate(dd_manager, &result, CPClusterGetTi(CPListGetItem(L)));
  }
  return(result);
}

/**Function********************************************************************

  Synopsis [Builds the monolithic transition relation BDD in a CPList]

  Description [Builds the monolithic transition relation BDD in a
  CPList. Wrapper of CPListBuildMonolithicBDD(). ]

  SideEffects        []

******************************************************************************/
CPList CPListBuildMonolithic(node_ptr list_variables, bdd_ptr assumption)
{

  CPList result;
  bdd_ptr result_bdd;
  CPCluster * C;

  result_bdd = CPListBuildMonolithicBDD(list_variables, assumption);

  C = CPClusterAlloc();
  CPClusterSetTi(C, result_bdd);
  result = CPListAlloc(C);

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, "size of monolithic transition relation: %d BDD nodes\n", bdd_size(dd_manager, result_bdd));
  }

  return(result);
}

/**Function********************************************************************

  Synopsis           [Prints size of each cluster in a given clustered
  transition relation]

  Description        [Prints size of each cluster in a given clustered
  transition relation]

  SideEffects        []

******************************************************************************/
void CPListPrintInfo(FILE *file, CPList CL)
{
  int i = 0;
  CPList l;

  for(l = CL; CPListIsNotEmpty(l); l = CPListNext(l)) {
    bdd_ptr Ti = CPClusterGetTi(CPListGetItem(l));

    /*
    fprintf(file, "cluster %d\t:\t%d\n", ++i, bdd_size(dd_manager, Ti));
    */
    fprintf(file, "cluster %d\t:\t%d (%p)\n", ++i, bdd_size(dd_manager, Ti),
	CPClusterGetInfo(CPListGetItem(l)));
  }
}

/**Function********************************************************************

  Synopsis [Prints detailed information of a given transition relation]

  Description        [Prints detailed information of a given
  transition relation. This is an old version still used.]

  SideEffects        []

******************************************************************************/
void CPListPrintClustersInfo(FILE * fp, DdManager * dd,
                             CPList ClusterList, node_ptr var_list)
{
  CPList cl;
  int c_id = 0;
  
  for (cl = ClusterList; CPListIsNotEmpty(cl); cl = CPListNext(cl), c_id++) {
    CPCluster * Ci = CPListGetItem(cl);

    fprintf(fp, "############################################################\n");
    fprintf(fp, "Cluster #%d:\n", c_id);
    fprintf(fp, "Size of C_%d = %d\n", c_id, bdd_size(dd, CPClusterGetTi(Ci)));
    fprintf(fp, "\nSize of E_%d = %d Boolean Variables\n", c_id,
            bdd_size(dd, CPClusterGetEi(Ci)) - 1);
    fprintf(fp, "The variables in E_%d are:\n", c_id);
    (void) print_state_vars(dd, CPClusterGetEi(Ci), var_list);
    fprintf(fp, "\n############################################################\n");
  }
}

/**Function********************************************************************

  Synopsis [Prints detailed information of a given transition relation]

  Description [Prints detailed information of a given transition relation]

  SideEffects        []

******************************************************************************/
void CPListPrintDetailedInfo(FILE *file, CPList CL)
{
  CPListPrintClustersInfo(file, dd_manager, CL, state_variables);
}

/**Function********************************************************************

  Synopsis [Computes the cube of the set of support of all the clusters.]

  Description [Given a list of clusters, it computes their set of support.] 

  SideEffects        []

******************************************************************************/
bdd_ptr CPListComputeClustersCube(DdManager * dd, CPList set_of_clusters)
{
  CPList soc = set_of_clusters;
  
  bdd_ptr result = bdd_one(dd);
  while (CPListIsNotEmpty(soc)) {
    CPCluster * Cur_Cluster = CPListGetItem(soc);
    bdd_ptr Supp_Ti = bdd_support(dd, CPClusterGetTi(Cur_Cluster));

    bdd_and_accumulate(dd, &result, Supp_Ti);
    bdd_free(dd, Supp_Ti);
    soc = CPListNext(soc);
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [Recursively build the portioned transition relation.]

  Description        [Recursively build the portioned transition
  relation. When the size of the current cluster is greater than the
  threshold found in the options, than a new cluster is created. The
  partitions are simplified using the given <tt>assumption<tt>.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
CPList CPListBuildBasicRecur(CPList clusters, node_ptr sexp,
                             add_ptr assumption)
{
  CPList result;
  
  if ((sexp == Nil) || (query_CP_hash(sexp))) {
   /* if the cluster has already been considered, thus we avoid a
       further insertion */
    result = clusters;
  }
  else {
    yylineno = node_get_lineno(sexp);
    switch(node_get_type(sexp)) {
    case AND:
      {
        result = CPListBuildBasicRecur(clusters, car(sexp), assumption);
        result = CPListBuildBasicRecur(result, cdr(sexp), assumption);
        break;
      }
    default:
      {
        add_ptr z = eval(sexp, Nil);

        if (add_is_one(dd_manager, z)) {
          result = clusters;
        }
        else {
          CPCluster * Ci = CPClusterAlloc();
          add_ptr tmp = add_simplify_assuming(dd_manager, z, assumption);

          CPClusterSetTi(Ci, add_to_bdd(dd_manager, tmp));
          CPClusterSetEi(Ci, (bdd_ptr)NULL);
          add_free(dd_manager, tmp);
          result = CPListCons(Ci, clusters);
        }
        add_free(dd_manager, z);
        break;
      }
    }
  }
  /* We keep track that the current expression has been already
     consdered for clustering */
  insert_CP_hash(sexp);
  return(result);
}


/**Function********************************************************************

  Synopsis           [Computes the schedule for the given list of clusters]

  Description        [Computes recursively the schedule for the given
  list of clusters]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
bdd_ptr CPListBuildScheduleRecur(CPList l, bdd_ptr vars)
{
  CPCluster * C;
  bdd_ptr supp, tmp, result, T;

  if (!CPListIsNotEmpty(l)) return (bdd_ptr)bdd_one(dd_manager);

  /* the recursive step */
  supp = CPListBuildScheduleRecur(CPListNext(l), vars);

  /* computing current variables to quantify out */
  tmp = bdd_cube_diff(dd_manager, vars, supp);

  /* computing the variables to send to upper level */
  C = CPListGetItem(l);
  CPClusterSetEi(C, tmp);
  T = CPClusterGetTi(C);
  tmp = bdd_support(dd_manager, T);
  result = bdd_and(dd_manager, supp, tmp);

  /* freeing temporary bdds */
  bdd_free(dd_manager, tmp);
  bdd_free(dd_manager, supp);
  return(result);
}

/*-------------------------------------------------------------*/
/* End of CPList's functions */
/*-------------------------------------------------------------*/

/*-------------------------------------------------------------*/
/* CPTrans functions */
/*-------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Sets the backward transition relation]

  Description        [Sets the backward transition relation]

  SideEffects        []

******************************************************************************/
void CPTransSetBackward(CPTrans_Ptr c, CPList b)
{
  nusmv_assert(c != (CPTrans_Ptr)NULL);
  c->backward_trans = b;
}

/**Function********************************************************************

  Synopsis           [Returns the backward transition relation list]

  Description        [Returns the backward transition relation list]

  SideEffects        []

******************************************************************************/
CPList CPTransGetBackward(CPTrans_Ptr c)
{
  nusmv_assert(c != (CPTrans_Ptr)NULL);
  return(c->backward_trans);
}

/**Function********************************************************************

  Synopsis           [Sets the forward transition relation]

  Description        [Sets the forward transition relation]

  SideEffects        []

******************************************************************************/
void CPTransSetForward(CPTrans_Ptr c, CPList f)
{
  nusmv_assert(c != (CPTrans_Ptr)NULL);
  c->forward_trans = f;
}

/**Function********************************************************************

  Synopsis           [Returns the forward transition relation list]

  Description        [Returns the forward transition relation list]

  SideEffects        []

******************************************************************************/
CPList CPTransGetForward(CPTrans_Ptr c)
{
  nusmv_assert(c != (CPTrans_Ptr)NULL);
  return(c->forward_trans);
}

/**Function********************************************************************

  Synopsis [Allocates memory for a conjunctive partitioning transition]

  Description [Allocates memory for a conjunctive partitioning transition]

  SideEffects        []

******************************************************************************/
CPTrans_Ptr CPTransAlloc()
{
  CPTrans_Ptr c = ALLOC(CPTrans, 1);

  if (c == (CPTrans_Ptr)NULL) {
    fprintf(nusmv_stderr, "CPTransAlloc: Unable to allocate memory.\n");
    nusmv_exit(1);
    return((CPTrans_Ptr)NULL);
  }
  
  CPTransSetForward(c, CPListEmpty);
  CPTransSetBackward(c, CPListEmpty);
  return(c);
}

/**Function********************************************************************

  Synopsis           [Frees a CPTrans and all data inside]

  Description        [Frees a CPTrans and all data inside, but not the
  Info of single clusters.]

  SideEffects        []

******************************************************************************/
void CPTransFree(CPTrans_Ptr c)
{
  nusmv_assert(c != (CPTrans_Ptr)NULL);

  if (c->forward_trans != CPListEmpty)
    CPListFree(dd_manager, c->forward_trans);

  if (c->backward_trans != CPListEmpty)
    CPListFree(dd_manager, c->backward_trans);

  FREE(c);
}

/**Function********************************************************************

  Synopsis           [Frees a CPTrans and all data inside]

  Description        [Frees a CPTrans and all data inside and the Info
  of each cluster given a proper function]

  SideEffects        []

******************************************************************************/
void CPTransFreeAll(CPTrans_Ptr c, int (*freeinfo)(void *, void*))
{
  nusmv_assert(c != (CPTrans_Ptr)NULL);

  if (c->forward_trans != CPListEmpty)
    CPListFreeAll(dd_manager, c->forward_trans, freeinfo);

  if (c->backward_trans != CPListEmpty)
    CPListFreeAll(dd_manager, c->backward_trans, freeinfo);
  
  FREE(c);
}

/**Function********************************************************************

  Synopsis           [Duplicates a CPTrans and all data inside]

  Description        [Duplicates a CPTrans and all data inside, but
  not the info of each cluster]

  SideEffects        []

******************************************************************************/
CPTrans_Ptr CPTransDup(CPTrans_Ptr c)
{
  CPList tmp_listf;
  CPList tmp_listb;
  CPTrans_Ptr cnew = CPTransAlloc();

  if (CPListIsNotEmpty(CPTransGetForward(c))) {
    tmp_listf = CPListDup(CPTransGetForward(c));
    CPTransSetForward(cnew, tmp_listf);
  }
  
  if (CPListIsNotEmpty(CPTransGetBackward(c))) {
    tmp_listb = CPListDup(CPTransGetBackward(c));
    CPTransSetBackward(cnew, tmp_listb);
  }
  return(cnew);
}

/**Function********************************************************************

  Synopsis           [Prints short info associated to a CPTrans]

  Description        [Prints info about the size of each cluster in
  forward/backward transition relation]

  SideEffects        []

******************************************************************************/
void CPTransPrintInfo(FILE *file, CPTrans_Ptr c)
{
  if (c != (CPTrans_Ptr)NULL){
    fprintf(file, "Forward Partitioning Schedule BDD cluster size (#nodes):\n");
    CPListPrintInfo(file, CPTransGetForward(c));
    fprintf(file, "Backward Partitioning Schedule BDD cluster size (#nodes):\n");
    CPListPrintInfo(file, CPTransGetBackward(c));
  }
}

/**Function********************************************************************

  Synopsis           [Prints short detailed associated to a CPTrans]

  Description        [Prints short detailed associated to a CPTrans]

  SideEffects        []

******************************************************************************/
void CPTransPrintDetailedInfo(FILE *file, CPTrans_Ptr c)
{
  node_ptr Fwd_trans = CPTransGetForward(c);
  node_ptr Bwd_trans = CPTransGetBackward(c);

  fprintf(file, "\n########################################\n");
  fprintf(file, "# Forward clusters: \n");
  fprintf(file, "########################################\n");
  CPListPrintDetailedInfo(file, Fwd_trans);
  fprintf(file, "\n########################################\n");
  fprintf(file, "# Backward clusters: \n");
  fprintf(file, "########################################\n");
  CPListPrintDetailedInfo(file, Bwd_trans);
  fprintf(file, "\n");
}

/*-------------------------------------------------------------*/
/* End of CPTrans's functions */
/*-------------------------------------------------------------*/

void CPListBuildStart()
{
  init_CP_hash();
}

void CPListBuildStop()
{
  quit_CP_hash();
}
