/**CFile***********************************************************************

  FileName    [compileBEval.c]

  PackageName [compile]

  Synopsis    []

  Description [Conversion from scalar expressions to boolean expressions.]

  SeeAlso     [optional]

  Author      [Alessandro Cimatti and Marco Roveri]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "compileInt.h"

static char rcsid[] UTIL_UNUSED = "$Id: compileBEval.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define INT_VAR_PREFIX "__det_"

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************

  Synopsis    [Counter used to create determinization variables.]

  Description [Each time a new determinization variable is introduced
  by the <tt>mk_new_var</tt> primitive, the current value of
  <tt>det_counter</tt> is used, and then it is incremented.]

  SeeAlso     [mk_new_var]

******************************************************************************/
static int det_counter = 0;

/**Variable********************************************************************

  Synopsis    [Determinization variables]

  Description [List of boolean variables introduced for the
  determinization of expressions.]

  SeeAlso     [optional]

******************************************************************************/
node_ptr determinization_boolean_variables = Nil;

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static node_ptr expr2bexpr_recur ARGS((node_ptr, boolean, boolean));
static node_ptr scalar_atom2bexpr ARGS((node_ptr, boolean, boolean));
static node_ptr add2bexpr ARGS((add_ptr, boolean, boolean));
static node_ptr mk_true ARGS((void));
static node_ptr mk_false ARGS((void));
static node_ptr mk_ite  ARGS((node_ptr, node_ptr, node_ptr));
static node_ptr mk_next ARGS((node_ptr c));
static node_ptr mk_new_var ARGS((void));
static int range_check ARGS((node_ptr, node_ptr));
static node_ptr add2bexpr_recur ARGS((DdManager *, add_ptr, boolean, boolean, st_table *));

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Converts a scalar expression into a boolean expression.]

  Description        [Takes an scalar expression intended to evaluate
  to boolean, maps through booleans down to the atomic scalar
  propositions, builds the corresponding boolean function, and returns
  the resulting boolean expression.

  The conversion of atomic scalar proposition is currently performed
  by generating the corresponding ADD, and then printing it in terms
  of binary variables.]

  SideEffects        [None]

  SeeAlso            [detexpr2bexpr, expr2bexpr_recur]

******************************************************************************/
node_ptr expr2bexpr(node_ptr expr)
{
  node_ptr res;
  int temp = yylineno;

  if (expr == Nil) {
    return(Nil);
  }
  yylineno = node_get_lineno(expr);
  res = expr2bexpr_recur(expr, false, true);
  yylineno = temp;
  return(res);
}


/**Function********************************************************************

  Synopsis           [Converts a scalar expression into a boolean expression.]

  Description        [Takes an scalar expression intended to evaluate
  to boolean, maps through booleans down to the atomic scalar
  propositions, builds the corresponding boolean function, and returns
  the resulting boolean expression.

  The conversion of atomic scalar proposition is currently performed
  by generating the corresponding ADD, and then printing it in terms
  of binary variables.

  An error is returned is determinization variables are introduced in
  the booleanization process.]

  SideEffects        [None]

  SeeAlso            [expr2bexpr, expr2bexpr_recur]

******************************************************************************/
node_ptr detexpr2bexpr(node_ptr expr)
{
  node_ptr res;
  int temp = yylineno;

  if (expr == Nil) {
    return(Nil);
  }
  yylineno = node_get_lineno(expr);
  res = expr2bexpr_recur(expr, false, false);
  yylineno = temp;
  return(res);
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Support functions for expressions abstract syntax.]

  Description        [Build truth, falsity, conditional and next expressions.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static node_ptr mk_true() {return(find_node(TRUEEXP, Nil, Nil));}
static node_ptr mk_false() {return(find_node(FALSEEXP, Nil, Nil));}
static node_ptr mk_ite(node_ptr c, node_ptr t, node_ptr e)
{
  return(new_node(CASE, new_node(COLON, c, t), e));
}

static node_ptr mk_next(node_ptr c)
{
  return(find_node(NEXT, c, Nil));
}

/**Function********************************************************************

  Synopsis           [Converts a generic expression into a boolean expression.]

  Description        [Takes an expression intended to evaluate to boolean,
   maps through booleans down to the atomic scalar propositions,
   builds the corresponding boolean function,
   and returns the resulting boolean expression.

   The conversion of atomic scalar proposition is currently performed
   by generating the corresponding ADD, and then printing it in terms
   of binary variables.

   The introduction of determinization variables is allowed only if flag
   <tt>allow_nondet</tt> is set to true.]

  SideEffects        [None]

  SeeAlso            [expr2bexpr, detexpr2bexpr]

******************************************************************************/
static node_ptr expr2bexpr_recur(node_ptr expr, boolean in_next, boolean allow_nondet)
{
  if (expr == Nil)
    return(Nil);

  switch(node_get_type(expr)) {

  case TRUEEXP:
  case FALSEEXP:
    return(expr);

  case NUMBER: {
   if (find_atom(expr) == one_number) {
     return(find_node(TRUEEXP, Nil, Nil));
   } else if (find_atom(expr) == zero_number) {
     return(find_node(FALSEEXP, Nil, Nil));
   } else {
     rpterr("Number that can not be casted to boolean");
   }
  }

  case DOT:
  case ATOM:
  case ARRAY: {
    node_ptr name = eval_struct(expr, Nil);
    /*
       name can be
       - define (when name_info is CONTEXT)
       - variable (when name_info is VAR)
    */
    node_ptr name_info = lookup_symbol_hash(name);

    if (name_info == (node_ptr)NULL) {
      rpterr("Undefined symbol");
    } else if (node_get_type(name_info) == CONTEXT) {
      /* define (in right hand side thus must be boolean, recur to check) */
      node_ptr body = lookup_flatten_def_hash(name);

      if (body == (node_ptr)NULL) error_undefined(name);
      return(expr2bexpr_recur(body, in_next, allow_nondet));
    } else if (node_get_type(name_info) == VAR) {
      /* variable, must be boolean */
      if (range_check(cdr(name_info),boolean_type) == 0) {
  rpterr("Unexpected non boolean variable");
      } else {
        int index = add_index(dd_manager, (add_ptr)car(name_info));

        name = get_bool_variable_name(index);
  if (in_next == true) {
    return(find_node(NEXT, name, Nil));
  } else {
    return(name);
  }
      }
    } else {
      rpterr("Unexpected data structure");
    }
  }

  case NOT:
    return(find_node(NOT,
         expr2bexpr_recur(car(expr), in_next, allow_nondet),
         Nil));

  case CONS:
  case AND:
  case OR:
  case XOR:
  case XNOR:
  case IMPLIES:
  case IFF:
    return(find_node(node_get_type(expr),
         expr2bexpr_recur(car(expr), in_next, allow_nondet),
         expr2bexpr_recur(cdr(expr), in_next, allow_nondet)));

  case CASE:
    return(find_node(CASE,
                     find_node(COLON,
             expr2bexpr_recur(car(car(expr)), in_next, allow_nondet),
             expr2bexpr_recur(cdr(car(expr)), in_next, allow_nondet)),
         expr2bexpr_recur(cdr(expr), in_next, allow_nondet)));

  case NEXT: {
    nusmv_assert((in_next == false));
    return(expr2bexpr_recur(car(expr), true, allow_nondet));
  }

  case EQDEF:
    {
      node_ptr var_name;
      node_ptr var;
      node_ptr lhs = car(expr);
      node_ptr rhs = cdr(expr);

      if (node_get_type(lhs) == SMALLINIT) {
        var_name = car(lhs);
        var = car(lhs);
      }
      else if (node_get_type(lhs) == NEXT) {
        var_name = car(lhs);
        var = lhs;
      } else {
        var_name = lhs;
        var = lhs;
      }

      {
        node_ptr name = eval_struct(var_name, Nil);
        /* name must be a defined variable */
        node_ptr name_info = lookup_symbol_hash(name);

        if (name_info == (node_ptr)NULL) {
          rpterr("Undefined symbol");
        } else if (node_get_type(name_info) == VAR) {
          if (cdr(name_info) == boolean_type) {
            /* boolean variable, rhs will be determinized */
            int index = add_index(dd_manager, (add_ptr)car(name_info));

            name = get_bool_variable_name(index);
            if (node_get_type(lhs) == NEXT) {
              var = find_node(NEXT, name, Nil);
            }
            else {
              var = name;
            }
            return(find_node(IFF, var, expr2bexpr_recur(rhs, in_next, allow_nondet)));
          } else {
            /* scalar variable */
            nusmv_assert((in_next == false));
            return(scalar_atom2bexpr(expr, in_next, allow_nondet));
          }
        } else {
          rpterr("Unexpected data structure, variable was expected");
        }
      }
    }

  /* Predicates over scalar terms (guaranteed to return boolean) */
  case LT:
  case GT:
  case LE:
  case GE:
  /* Predicates over possibly scalar terms (guaranteed to return boolean) */
  case EQUAL:
  case NOTEQUAL:
  /* Function symbols over scalar terms
     Might return boolean, but check is needed
     Assumption: if boolean is returned, then the function is
     determinized by introducing a variable on the {0,1} leaves.
  */
  case PLUS:
  case MINUS:
  case TIMES:
  case DIVIDE:
  case MOD:
  case UNION:
  case SETIN: {
    return(scalar_atom2bexpr(expr, in_next, allow_nondet));
  }

  /* UNARY CTL/LTL OPERATORS */
  case EX:
  case AX:
  case EG:
  case AG:
  case EF:
  case AF:
  case OP_NEXT:
  case OP_PREC:
  case OP_NOTPRECNOT:
  case OP_FUTURE:
  case OP_ONCE:
  case OP_GLOBAL:
  case OP_HISTORICAL:
                  {
    return(find_node(node_get_type(expr),
         expr2bexpr_recur(car(expr), in_next, allow_nondet),
         Nil));
  }

  /* BINARY CTL/LTL OPERATORS */
  case EU:
  case AU:
  case MINU:
  case MAXU:
  case UNTIL:
  case RELEASES:
  case SINCE: 
  case TRIGGERED: {
    return(find_node(node_get_type(expr),
         expr2bexpr_recur(car(expr), in_next, allow_nondet),
         expr2bexpr_recur(cdr(expr), in_next, allow_nondet)));
  }

  /* BOUNDED TEMPORAL OPERATORS
     (cdr is range, no recursion needed) */
  case EBF:
  case ABF:
  case EBG:
  case ABG:
  case EBU:
  case ABU: {
    return(find_node(node_get_type(expr),
         expr2bexpr_recur(car(expr), in_next, allow_nondet),
         cdr(expr)));
  }

  case CONTEXT: {
    return(expr2bexpr_recur(cdr(expr), in_next, allow_nondet));
  }

  case TWODOTS: {
    rpterr("Unexpected TWODOTS node");
    return((node_ptr)NULL);
  }
  default: {
    internal_error("expr2bexpr_recur: type = %d\n", node_get_type(expr));
    return((node_ptr)NULL);
    }
  }
}

/**Function********************************************************************

  Synopsis           [Converts an atomic expression into the corresponding (boolean)
  expression.]

  Description        [Takes an atomic expression and converts it into
  a cooresponding boolean expression.

  The introduction of determinization variables is allowed only if flag
  <tt>allow_nondet</tt> is set to true.]

  SideEffects        [A new boolean variable can be introduced.]

  SeeAlso            [add2bexpr_recur, mk_new_var]

******************************************************************************/
static node_ptr scalar_atom2bexpr(node_ptr expr, boolean in_next, boolean allow_nondet)
{
  node_ptr res;
  int temp = yylineno;
  add_ptr bool_add = eval(expr, Nil);

  yylineno = node_get_lineno(expr);
  res = add2bexpr(bool_add, in_next, allow_nondet);
  add_free(dd_manager, bool_add);
  yylineno = temp;

  return(res);
}


/**Function********************************************************************

  Synopsis           [Converts a ADD into the corresponding (boolean)
  expression.]

  Description        [Takes an ADD with leaves 0, 1, or {0,1}.

   The case of {0,1} leaves is allowed only if flag
   <tt>allow_nondet</tt> is set to true.]

   Recurs down on the structure of the ADD, and maps each non terminal
   node into an if-then-else expression, maps 0 and 1 terminal nodes
   into true and false expressions, and maps {0,1} into a newly
   introduced variable to determinize the expression.]

  SideEffects        [A new boolean variable can be introduced.]

  SeeAlso            [add2bexpr_recur, mk_new_var]

******************************************************************************/
static node_ptr add2bexpr(add_ptr bool_add, boolean in_next, boolean allow_nondet)
{
  node_ptr result;
  int reordering = 0;
  dd_reorderingtype rt;

  st_table * lc = st_init_table(st_ptrcmp, st_ptrhash);
  if (lc == (st_table *)NULL) {
    fprintf(nusmv_stderr, "add2bexpr: unable to allocate local hash.\n");
    nusmv_exit(1);
  }

  /* If dynamic reordering is enabled, it is temporarily disabled */
  if ((reordering = dd_reordering_status(dd_manager, &rt))) {
    dd_autodyn_disable(dd_manager);
  }

  result = add2bexpr_recur(dd_manager, bool_add, in_next, allow_nondet, lc);

  /* If dynamic reordering was enabled, then it is enabled */
  if (reordering == 1) {
    dd_autodyn_enable(dd_manager, rt);
  }
  st_free_table(lc);

  return(result);
}

/**Function********************************************************************

  Synopsis           [Converts a ADD into the corresponding (boolean)
  expression.]

  Description        [Auxiliary function for add2bexpr.]

  SideEffects        [A new boolean variable can be introduced.]

  SeeAlso            [add2bexpr, range_check]

******************************************************************************/
static node_ptr add2bexpr_recur(
 DdManager *dd,
 add_ptr bool_add,
 boolean in_next,
 boolean allow_nondet,
 st_table * lc)
{
  node_ptr result;
  nusmv_assert((lc != (st_table *)NULL));
  nusmv_assert((dd != (DdManager *)NULL));
  nusmv_assert((bool_add != (add_ptr)NULL));

  /* base case */
  if (add_isleaf(bool_add)) {
    node_ptr leaf = add_get_leaf(dd, bool_add);
    if (add_is_one(dd_manager, bool_add)) return(mk_true());
    if (add_is_zero(dd_manager, bool_add)) return(mk_false());
    { /* It something different from true or false */
      if (range_check(leaf, boolean_type) == 1) {
        if (llength(leaf) == 1) {
          /* it is (cons (number 0) Nil) or (cons (number 1) Nil) and
             nothing must be created */
          if (car(leaf) == one_number) {
            return(mk_true());
          }
          else if (car(leaf) == zero_number) {
            return(mk_false());
          }
          else {
            fprintf(nusmv_stderr,
                    "add2bexpr_recur: Attempting to convert a non boolean set.\n");
            indent_node(nusmv_stderr, "value = ",
                        add_get_leaf(dd, bool_add), "\n");
            nusmv_exit(1);
            return((node_ptr)NULL);
          }
        }
        else {
          /* a new fresh variable is needed to determinize the result */
          if (allow_nondet) {
            result = mk_ite(mk_new_var(), mk_true(), mk_false());
          }
          else {
            fprintf(nusmv_stderr,
                    "add2bexpr_recur: Attempting to convert a nondeterministic expression.\n");
            nusmv_exit(1);
            return((node_ptr)NULL);
          }
        }
      } else {
        fprintf(nusmv_stderr,
                "add2bexpr_recur: Attempting to convert a non boolean set.\n");
        indent_node(nusmv_stderr, "value = ",
                    add_get_leaf(dd, bool_add), "\n");
        nusmv_exit(1);
        return((node_ptr)NULL);
      }
    }
  }
  else {
    /* step case */
    node_ptr t, e, var;
    int index;

    if (st_lookup(lc, (char *)bool_add, (char **)&result)) {
      return(result);
    }

    index = add_index(dd_manager, bool_add);

    t = add2bexpr_recur(dd, add_then(dd, bool_add), in_next, allow_nondet, lc);
    if (t == (node_ptr)NULL) return((node_ptr)NULL);
    e = add2bexpr_recur(dd, add_else(dd, bool_add), in_next, allow_nondet, lc);
    if (e == (node_ptr)NULL) return((node_ptr)NULL);

    var = get_bool_variable_name(index);
    if (var == Nil) {
      fprintf(nusmv_stderr,
              "add2bexpr_recur: No variable associated to BDD variable %d\n",
              index);

      return((node_ptr)NULL);
    }
    if (in_next == true) {
      var = mk_next(var);
    }
    result = mk_ite(var, t, e);
    if (result == (node_ptr)NULL) return((node_ptr)NULL);
    if (st_add_direct(lc, (char *)bool_add, (char *)result) == ST_OUT_OF_MEM) {
      fprintf(nusmv_stderr,
              "add2bexpr_recur: Unable to insert result in local hash.\n");
      return((node_ptr)NULL);
    }
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [Create a new (boolean) variable.]

  Description        [Creates a fresh variable name, creates the
  corresponding name expressions, stores it as input variable, creates
  the corresponding encoding, builds and stores the resulting variable
  node.]

  SideEffects        [A new dd variable is created.]

  SeeAlso            [INT_VAR_PREFIX, det_counter, Compile_EncodeVar]

******************************************************************************/
static node_ptr mk_new_var(void) {
  node_ptr nvar;
  char name[20];

  sprintf(name, "%s%d", INT_VAR_PREFIX, det_counter++);

  nvar = find_node(DOT, Nil, sym_intern(name));

  /* We add the new variable at the end of the ordering */
  /* notice that append performs side effect on the list like nconc in LISP */
  /* input_variables = append(input_variables, cons(nvar, Nil)); */
  determinization_boolean_variables = cons(nvar, determinization_boolean_variables);
  {
    add_ptr encoding = Compile_EncodeVar(nvar, boolean_type, true, false);

    insert_symbol_hash(nvar, new_node(VAR, (node_ptr)encoding, boolean_type));
  }
  return(expr2bexpr(nvar));
}

/**Function********************************************************************

  Synopsis           [Checks if the first argument is contained in the second.]

  Description        [Returns true if the first argument is contained in the
  set represented by the second, false otherwise. If the first
  argument is not a CONS, then it is considered to be a singleton.]

  SideEffects        [None]

  SeeAlso            [in_list]

******************************************************************************/
static int range_check(node_ptr s, node_ptr d) {
  if (d == Nil) return(0);

  if (node_get_type(s) == CONS) {
    while(s != Nil) {
      if (in_list(car(s), d) == 1) return(1);
      s = cdr(s);
    }
  }
  else {
    if (in_list(s, d) == 1) return(1);
  }
  return(0);
}

