/**CFile***********************************************************************

  FileName    [compileAffinity.c]

  PackageName [compile]

  Synopsis    [Affinity clustering optimized.]

  Description [Affinity clustering optimized.]

  SeeAlso     []

  Author      [Emanuele Olivetti]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/


#include "compileInt.h" 
#include "heap.h"

static char rcsid[] UTIL_UNUSED = "$Id: compileAffinity.c,v 1.1.1.1 2003/02/06 19:01:17 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [record of the list to be used with the heap]

  Description [This is the record of the list to be used with the heap, basic
  blocks of optimized affinity clustering.]

  SeeAlso     []

******************************************************************************/
struct list_entry_struct {
  boolean  flag; /* tells if the cluster exists */
  CPCluster * C; /* pointer to the element (cluster) of the original
                    cluster list */
};

typedef struct list_entry_struct list_entry;

#define LE_GetFlag(LE) ((list_entry *)(LE))->flag
#define LE_SetFlag(LE,f) ((list_entry *)(LE))->flag = f
#define LE_GetCluster(LE) ((list_entry *)(LE))->C
#define LE_SetCluster(LE,Cl) ((list_entry *)(LE))->C = Cl

/**Struct**********************************************************************

  Synopsis    [pair of pointers to the list.]

  Description [This is the record of the heap that point to the support list.]

  SeeAlso     []

******************************************************************************/
struct pair_struct {
  list_entry * C1;
  list_entry * C2;
};

typedef struct pair_struct pair;

#define Pair_GetC1(P) ((pair *)(P))->C1
#define Pair_SetC1(P,C) ((pair *)(P))->C1 = C
#define Pair_GetC2(P) ((pair *)(P))->C2
#define Pair_SetC2(P,C) ((pair *)P)->C2 = C

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Allocs a list_entry]

  Description        [Allocs a list_entry]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static list_entry * list_entryAlloc(void)
{
  list_entry * L_E;

  L_E = ALLOC(list_entry, 1);
  L_E->flag        = false;
  L_E->C           = (CPCluster *)NULL;
  return(L_E);
}

/**Function********************************************************************

  Synopsis           [Allocates a pair]

  Description        [Allocates a pair]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static pair * pairAlloc(void)
{
  pair * P;

  P = ALLOC(pair, 1);
  P->C1          = (list_entry *)NULL;
  P->C2          = (list_entry *)NULL;
  return(P);
}

/**Function********************************************************************

  Synopsis           [Compute the Affinity of two BDD clusters.]

  Description        [Compute the Affinity between two BDD clusters as 
  suggested by Moon, Hachtel, Somenzi in BBT paper. Affinity is the ratio
  between the number of shared variables and the number of the union of
  all variables (intersection/union)]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
#define MHS_AFFINITY_DEFINITION
#ifdef MHS_AFFINITY_DEFINITION
double compute_bdd_affinity(bdd_ptr a, bdd_ptr b)
{
  double result;
  bdd_ptr supp_a, supp_b, I , U;

  supp_a = bdd_support(dd_manager, a);
  supp_b = bdd_support(dd_manager, b);
  I = bdd_cube_intersection(dd_manager, supp_a, supp_b);
  U = bdd_cube_union(dd_manager, supp_a, supp_b);
  bdd_free(dd_manager, supp_a);
  bdd_free(dd_manager, supp_b);
  if (bdd_size(dd_manager, U) != 0) {
    result = ((double)bdd_size(dd_manager, I)) /
             ((double)bdd_size(dd_manager, U));
  }
  else {
    /* a,b are empty cubes!*/
    nusmv_assert(false);
  }
  bdd_free(dd_manager, I);
  bdd_free(dd_manager, U);
  return(result);
}
#else
/**Function********************************************************************

  Synopsis           [Compute the Affinity of two BDD clusters.]

  Description        [Compute the Affinity between two BDDs. This is
  an alternative definition to the one suggested by by Moon, Hachtel,
  Somenzi in BBT paper.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
double compute_bdd_affinity(bdd_ptr a, bdd_ptr b)
{
  double result;
  double s_a, s_b;
  bdd_ptr c;
  c = bdd_and(dd_manager, a, b);
  
  s_a = (double)bdd_size(dd_manager, a);
  s_b = (double)bdd_size(dd_manager, b);
  if ((s_a != 0) && (s_b != 0)) {
    result = ((double)bdd_size(dd_manager, c)) / (s_a + s_b);
  }
  else {
    /* a,b are empty bdd!*/
    nusmv_assert(false);
  }
  bdd_free(dd_manager, c);
  return(result);
}
#endif

/**Function********************************************************************

  Synopsis           [Add a new entry in support list and new pairs in heap.]

  Description        [Add a new entry in support list and new pairs in heap.
  Pairs with a dead cluster are skipped]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static node_ptr list_heap_add(CPCluster * C, node_ptr L, heap H)
{
  bdd_ptr T1;
  list_entry * New_entry;
  node_ptr List = L;

  New_entry = list_entryAlloc();
  LE_SetFlag(New_entry, true);
  LE_SetCluster(New_entry, C);
  
  T1 = CPClusterGetTi(C);
  while (List != Nil) {
    list_entry * l_e = (list_entry*)car(List);
 
    if (LE_GetFlag(l_e) == true) {
      pair * P = pairAlloc();
      bdd_ptr T2 = CPClusterGetTi(LE_GetCluster(l_e));
      double affinity = compute_bdd_affinity(T1, T2);

      Pair_SetC1(P, New_entry);
      Pair_SetC2(P, l_e);
      heap_add(H, affinity, P);
    }
    List = cdr(List);
  }

  L = cons((node_ptr)New_entry, L);
  return(L);
}

/**Function********************************************************************

  Synopsis           [Delete a cluster in support list.]

  Description        [Delete a cluster in support list.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static void list_del(list_entry * C)
{
  nusmv_assert(C != (list_entry *)NULL);
  LE_SetFlag(C, false);
  return;
}

/**Function********************************************************************

  Synopsis           [Copy over threshold clusters in result list or in support
  list & heap.]

  Description        [Copy over threshold clusters in result list or in support
  list & heap. It doesn't modify the input list.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static int affinity_move_clusters(CPList List, CPList * NewRelationList,
                                  int threshold, node_ptr * L, heap H)
{
  int n = 0;

  while (CPListIsNotEmpty(List)) {
    CPCluster * Ci = CPListGetItem(List);
    bdd_ptr Ti = CPClusterGetTi(Ci);

    if (bdd_size(dd_manager, Ti) > threshold) {
      CPCluster * New_Ci = CPClusterAlloc();

      CPClusterSetTi(New_Ci, bdd_dup(Ti));
      *NewRelationList = CPListCons(New_Ci, *NewRelationList);
    }
    else {
      *L = list_heap_add(CPListGetItem(List), *L, H);
      n++;
    }
    List = cdr(List);
  }
  return(n);
}

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [OPTIMIZED affinity clustering]

  Description        [OPTIMIZED affinity clustering]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void compileCPMakeAffinityClusters(DdManager * dd, CPList RelationList,
				   CPList * NewRelationList,
				   int threshold)
{
  int n;
  pair * P;
  node_ptr L = Nil;
  heap H = heap_create();
  CPList oldList = (CPList)NULL;

  oldList = CPListDup(RelationList);

  *NewRelationList = Nil;

  n = affinity_move_clusters(oldList, NewRelationList, threshold, &L, H);

  /* clustering */
  while (n > 1) {
    P = (pair*)heap_getmax(H);
    if ((LE_GetFlag(Pair_GetC1(P)) == true) &&
        (LE_GetFlag(Pair_GetC2(P)) == true)) {
      bdd_ptr T1 , T2 , New_T;
      CPCluster * New_C;

      T1 = CPClusterGetTi(LE_GetCluster(Pair_GetC1(P)));
      T2 = CPClusterGetTi(LE_GetCluster(Pair_GetC2(P)));
      New_T = bdd_and(dd, T1, T2);
      list_del(Pair_GetC1(P));
      list_del(Pair_GetC2(P));
	
      New_C = CPClusterAlloc();
      CPClusterSetTi(New_C, New_T);
	
      if (bdd_size(dd, New_T) > threshold) {
        *NewRelationList=CPListCons(New_C, *NewRelationList);
        n-=2;
      }
      else {
        L = list_heap_add(New_C, L, H);
        n--;
      }
    }
  }
  
  /* there can be a last one small cluster */
  if (n == 1) {
    node_ptr tmplist = L;

    while ((tmplist != Nil) && (LE_GetFlag(car(tmplist)) == false)) {
      tmplist = cdr(tmplist);
    }
    if (tmplist != Nil){
      bdd_ptr T;
      CPCluster * C;

      C = CPClusterAlloc();
      T = CPClusterGetTi(LE_GetCluster(car(tmplist)));
      CPClusterSetTi(C, bdd_dup(T));
      *NewRelationList=CPListCons(C, *NewRelationList);
    }
    else {
      fprintf(nusmv_stdout,"Affinity Optimized Inconsitency!!!\n");
      nusmv_assert(false);
    }
  }

  /* freeing no more used structures */
  {
    node_ptr tmplist = L;
    while (tmplist != Nil) {
      bdd_ptr T;

      T = CPClusterGetTi(LE_GetCluster(car(tmplist)));
      bdd_free(dd, T);
      tmplist = cdr(tmplist);
    }
    free_list(L);
    while (heap_isempty(H) == 0) {
      P = heap_getmax(H);
    }
    heap_destroy(H);
  }
}

