/**CHeaderFile*****************************************************************

  FileName    [compile.h]

  PackageName [compile]

  Synopsis    [Compilation of NuSMV input language into BDD.]

  Description [This package contains the compiler of NuSMV code into
  BDD. It works on a flattened/instantiated structure. Performs the
  checks on the parse tree and generates the encoding of scalar
  variables into BDDs. Then, the transition relation is created with
  different methods.]

  Author      [Marco Roveri, Emanuele Olivetti]

  Copyright   [
  This file is part of the ``compile'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

  Revision    [$Id: compile.h,v 1.9 2003/10/28 17:29:17 flerda Exp $]

******************************************************************************/
#ifndef _COMPILE
#define _COMPILE

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Data structure used to store the results of compilation.]

  Description [Data structure used to store the results of compilation.]

******************************************************************************/
struct cmp_struct {
  int      read_model;
  int      flatten_hierarchy;
  int      encode_variables;
  int      process_selector;
  int      build_frames;
  int      compile_check;
  int      build_init;
  int      build_model;
  int      build_flat_model;
  int      build_bool_model;
  int      bmc_setup;
  int      fairness_constraints;
  int      coi;
  int      build_model_setup;
  node_ptr init_expr;
  node_ptr invar_expr;
  node_ptr trans_expr;
  node_ptr procs_expr;
  node_ptr justice_expr;
  node_ptr compassion_expr;
  node_ptr spec_expr;
  node_ptr compute_expr;
  node_ptr ltlspec_expr;
  node_ptr invar_spec_expr;
};

/**Struct**********************************************************************

  Synopsis    [The parameters controlling the ordering of clusters.]

  Description [This structure contains the options to perform ordering
  of clusters in the IWLS95 partitioning method. <br>
  <code>clusterSize</code> is the threshold value used to create
  clusters. <code>W1</code>, <code>W2</code> etc are the weights used
  in the heuristic algorithms to order the clusters for early
  quantification. <code>verbosity</code> controls the amount of
  information printed during IWLS95 image computation.]

******************************************************************************/
struct Iwls95OptionStruct {
  int clusterSize;
  int verbosity; /* To choose if we want local verbosity in the image iwls computation */
  int W1;
  int W2; /* Weights attached to various parameters. */
  int W3; /* For details please refer to the paper */
  int W4; /* (insert reference of the paper IWLS95) */
};

/**Struct**********************************************************************

  Synopsis    [This general structure contains the information about a 
               particular cluster.]

  Description [There is one CPCluster per cluster. The
  various fields of this structure contain the information for
  image computation purposes.]

  SeeAlso     []

******************************************************************************/
struct CPCluster_struct {
  bdd_ptr Ti;           /* The current Cluster */
  bdd_ptr Ei;           /* Variables that can be existentially
			   quantified when Ti is multiplied in the
			   product */ 
  void * Info;          /* Additional Information */
};

typedef struct CPCluster_struct  CPCluster;
typedef struct CPCluster_struct *  CPCluster_Ptr;

/**Struct**********************************************************************

  Synopsis    [The Conjunctive partioned cluster list.]

  Description [The conjunctive partitioned cluster list.]

******************************************************************************/
typedef node_ptr CPList;

/**Struct**********************************************************************

  Synopsis    [Contains the descriptor for an element of the disjunctive
	       partition.]

  Description [Contains the descriptor for an element of the disjunctive
	       partition.]

  SeeAlso     []

******************************************************************************/
struct DPDesc_struct {
  bdd_ptr support_bdd;
  int *support_vector;
  CPList cl;
  char *name;
  void *info;
};

typedef struct DPDesc_struct  DPDesc;
typedef struct DPDesc_struct *DPDesc_Ptr;

/**Struct**********************************************************************

  Synopsis    [The disjunctive partitioned list.]

  Description [The disjunctive partitioned list.]

******************************************************************************/
typedef node_ptr DPList;


/**Macro***********************************************************************

  Synopsis     [The empty CP List]

  Description  [The empty CP List]

  SideEffects  [None]

******************************************************************************/
#define CPListEmpty (CPList)Nil

/**Macro***********************************************************************

  Synopsis     [The empty DP List]

  Description  [The empty DP List]

  SideEffects  [None]

******************************************************************************/
#define DPListEmpty (DPList)Nil

/**Struct**********************************************************************

  Synopsis    [The structure used to represent the transition relation.]

  Description [The structure used to represent the transition
  relation. The structure has two fields, one for the forward image
  and the other for the backward image. Each element is a cluster
  corresponding to two BDDs the first is the transition realtion the
  other is the cube of the BDD variable to quantify out in the
  corresponding image.]

******************************************************************************/
struct CPTrans_struct {
  CPList forward_trans;
  CPList backward_trans;
};

typedef struct CPTrans_struct CPTrans;
typedef struct CPTrans_struct * CPTrans_Ptr;


struct DPTrans_struct {
  DPList forward_trans;
  DPList backward_trans;
};

typedef struct DPTrans_struct DPTrans;
typedef struct DPTrans_struct * DPTrans_Ptr;


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
/**Enum************************************************************************

  Synopsis    [Enumerates the different types of a specification]

  Description [Enumerates the different types of a specification]

******************************************************************************/
typedef enum {ST_Notype, ST_Ctl, ST_Ltl, ST_Invar, ST_Compute} Spec_Type;
typedef enum {PT_Phase1, PT_Phase2, PT_Both} Proc_Type;

/**Enum************************************************************************

  Synopsis    [Enumerates the status of a specification]

  Description [Enumerates the status of a specification]

******************************************************************************/
typedef enum {SS_Nostatus, SS_Unchecked, SS_True, SS_False, SS_Wrong, SS_Number} Spec_Status;

/**Enum************************************************************************

  Synopsis    [Enumerates the different result types of a formula]

  Description [Enumerates the different result types of a formula]

******************************************************************************/
typedef enum {wff_NIL, wff_ERROR, wff_BOOLEAN, wff_NUMERIC, wff_SYMBOLIC} wff_result_type;

typedef struct Iwls95OptionStruct Iwls95OptionStruct_t;
typedef struct cmp_struct cmp_struct_rec;
typedef struct cmp_struct * cmp_struct_ptr;

typedef struct _Fsm_SexpRec    Fsm_SexpRec;
typedef struct _Fsm_SexpRec  * Fsm_SexpPtr;

typedef struct _Fsm_BddRec     Fsm_BddRec;
typedef struct _Fsm_BddRec   * Fsm_BddPtr;

#include "be.h" /* the generic boolean expressions interface */

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Macro***********************************************************************

  Synopsis     [The symbolic name of the input process selector variable.]

  Description  [This is the internal symbolic name of process selector
  variable. The range of this variable is the set of names of the
  instantiated processes.]

  SideEffects  []

******************************************************************************/
#define PROCESS_SELECTOR_VAR_NAME "_process_selector_"

/**Macro***********************************************************************

  Synopsis     [The "running" symbol.]

  Description  [The "running" symbol used to refer the internal
  variable "running" of  processes.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define RUNNING_SYMBOL "running"

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/
EXTERN void Compile_Init ARGS((void));
EXTERN void Compile_End  ARGS((void));
EXTERN void Compile_CheckProgram ARGS((node_ptr, node_ptr, node_ptr, node_ptr, node_ptr, node_ptr));

EXTERN int eval_num ARGS((node_ptr, node_ptr));
EXTERN add_ptr eval_constant ARGS((node_ptr expr, node_ptr context));
EXTERN node_ptr eval_struct ARGS((node_ptr, node_ptr));
EXTERN node_ptr eval_tree ARGS((node_ptr, node_ptr));
EXTERN node_ptr node_plus ARGS((node_ptr, node_ptr));
EXTERN node_ptr node_plus1 ARGS((node_ptr));
EXTERN add_ptr eval ARGS((node_ptr, node_ptr));
EXTERN add_ptr eval_simplify ARGS((node_ptr, node_ptr, add_ptr));
EXTERN node_ptr Compile_ReadOrder ARGS((const char *));
EXTERN void Compile_WriteOrder ARGS((const char *, int));
EXTERN node_ptr sym_intern ARGS((char *));
EXTERN void Compile_FlattenHierarchy ARGS((node_ptr, node_ptr, node_ptr *,\
                                           node_ptr *, node_ptr *, node_ptr *,\
                                           node_ptr *, node_ptr *, node_ptr *,\
                                           node_ptr *, node_ptr *, node_ptr *,\
					   node_ptr));
EXTERN void Compile_FlattenTableau ARGS((node_ptr, node_ptr, node_ptr *,\
					 node_ptr *, node_ptr *, node_ptr *,\
					 node_ptr *, node_ptr *, node_ptr *,\
					 node_ptr *, node_ptr *, node_ptr *,\
					 node_ptr));
EXTERN void Compile_EncodeVarsInit ARGS((void));
EXTERN void Compile_BuildVarsBdd ARGS((void));
EXTERN void build_real_state_variables ARGS((void));
EXTERN double bdd_count_states ARGS((DdManager *, bdd_ptr fn));
EXTERN double add_count_states ARGS((DdManager *, bdd_ptr fn)); 
EXTERN void build_proc_selector ARGS((node_ptr));
EXTERN void Compile_CompileInit ARGS((node_ptr, node_ptr));
EXTERN void Compile_CompileModel ARGS((node_ptr, node_ptr, node_ptr, add_ptr));
EXTERN void compute_fairness_constraints ARGS((Fsm_BddPtr, node_ptr));

EXTERN void reset_encode ARGS((void));

void print_state ARGS((bdd_ptr, node_ptr, int));
void print_states ARGS((bdd_ptr));
void init_print_hash ARGS((void));
void insert_print_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_print_hash ARGS((node_ptr));
void clear_print_hash ARGS((void));

void init_symbol_hash ARGS((void));
void insert_symbol_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_symbol_hash ARGS((node_ptr));
void clear_symbol_hash ARGS((void));

void init_param_hash ARGS((void));
void insert_param_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_param_hash ARGS((node_ptr));
void clear_param_hash ARGS((void));

void init_flatten_constant_hash ARGS((void));
void insert_flatten_constant_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_flatten_constant_hash ARGS((node_ptr));
void clear_flatten_constant_hash ARGS((void));

void init_flatten_def__hash ARGS((void));
void insert_flatten_def_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_flatten_def_hash ARGS((node_ptr));
void clear_flatten_def_hash ARGS((void));


void init_constant_hash ARGS((void));
void insert_constant_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_constant_hash ARGS((node_ptr));
void clear_constant_hash ARGS((void));

void init_value_hash ARGS((void));
void insert_value_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_value_hash ARGS((node_ptr));
void clear_value_hash ARGS((void));

void init_check_constant_hash ARGS((void));
void insert_check_constant_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_check_constant_hash ARGS((node_ptr));
void clear_check_constant_hash ARGS((void));

void init_variable_sexp_model_hash ARGS((void));
void insert_variable_sexp_model_hash  ARGS((node_ptr, node_ptr));
node_ptr lookup_variable_sexp_model_hash ARGS((node_ptr));
void clear_variable_sexp_model_hash ARGS((void));
void free_variable_sexp_model_hash_free ARGS((void));

void init_flatten_def_hash ARGS((void));
void insert_flatten_def_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_flatten_def_hash ARGS((node_ptr));
void clear_flatten_def_hash ARGS((void));

void init_wfftype_hash ARGS((void));
void insert_wfftype_hash ARGS((node_ptr, wff_result_type));
wff_result_type lookup_wfftype_hash ARGS((node_ptr));
void clear_wfftype_hash ARGS((void));

void init_assign_db_hash ARGS((void));
void insert_assign_db_hash ARGS((node_ptr, node_ptr));
node_ptr lookup_assign_db_hash ARGS((node_ptr));
void clear_assign_db_hash ARGS(());

void init_eval_hash ARGS((void));
void clear_eval_hash ARGS(());

void print_state_vars ARGS((DdManager *, bdd_ptr, node_ptr));
EXTERN void make_minterm_vars ARGS((void));
EXTERN void free_minterm_vars ARGS((void));
EXTERN void compilePrintBddModelStatistic ARGS((FILE *, Fsm_BddPtr));

EXTERN add_ptr add_shift_forward ARGS((DdManager * dd, add_ptr fn));
EXTERN add_ptr add_shift_backward ARGS((DdManager * dd, add_ptr fn));
EXTERN bdd_ptr bdd_shift_set ARGS((DdManager * dd, add_ptr fn, int *map));
EXTERN bdd_ptr bdd_shift_forward ARGS((DdManager * dd, bdd_ptr fn));
EXTERN bdd_ptr bdd_shift_backward ARGS((DdManager * dd, bdd_ptr fn));
EXTERN bdd_ptr bdd_pick_one_state ARGS((DdManager * dd, bdd_ptr fn));
EXTERN bdd_ptr bdd_pick_one_state_rand ARGS((DdManager * dd, bdd_ptr fn));
EXTERN cmp_struct_ptr cmp_struct_init ARGS((void));
EXTERN int cmp_struct_get_read_model ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_read_model ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_flatten_hrc ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_flatten_hrc ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_encode_variables ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_encode_variables ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_process_selector ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_process_selector ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_build_frames ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_build_frames ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_compile_check ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_compile_check ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_build_init ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_build_init ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_build_model ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_build_model ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_build_flat_model ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_build_flat_model ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_build_bool_model ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_build_bool_model ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_fairness ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_fairness ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_coi ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_coi ARGS((cmp_struct_ptr cmp));
EXTERN int cmp_struct_get_bmc_setup ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_bmc_setup ARGS((cmp_struct_ptr cmp));
EXTERN node_ptr cmp_struct_get_init ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_init ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_invar ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_invar ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_trans ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_trans ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_procs ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_procs ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_justice ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_justice ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_compassion ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_compassion ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_spec ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_spec ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_compute ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_compute ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_ltlspec ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_ltlspec ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN node_ptr cmp_struct_get_invar_spec ARGS((cmp_struct_ptr cmp));
EXTERN void cmp_struct_set_invar_spec ARGS((cmp_struct_ptr cmp, node_ptr n));
EXTERN Iwls95OptionStruct_t * Iwls95GetOptions ARGS((void));
EXTERN void Iwls95FreeClustersList ARGS((DdManager *, node_ptr));
EXTERN void init_module_hash ARGS((void));
EXTERN void insert_module_hash ARGS((node_ptr, node_ptr));
EXTERN node_ptr lookup_module_hash ARGS((node_ptr));
EXTERN void clear_module_hash ARGS((void));

EXTERN node_ptr Compile_FlattenSexp ARGS((node_ptr, node_ptr));
EXTERN node_ptr Compile_FlattenSexpExpandDefine ARGS((node_ptr, node_ptr));
EXTERN node_ptr Compile_FlattenProcess ARGS((node_ptr));
EXTERN int Compile_WriteFlatten ARGS((FILE *, cmp_struct_ptr));
EXTERN int Compile_WriteFlattenBool ARGS((FILE *, Fsm_SexpPtr, cmp_struct_ptr));
EXTERN node_ptr expr2bexpr ARGS((node_ptr));
EXTERN node_ptr detexpr2bexpr ARGS((node_ptr));
EXTERN void init_coi_hash ARGS((void));
EXTERN void clear_coi_hash ARGS((void));
EXTERN void init_define_dep_hash ARGS((void));
EXTERN void clear_define_dep_hash ARGS((void));
EXTERN void init_dependencies_hash ARGS((void));
EXTERN void clear_dependencies_hash ARGS((void));
EXTERN Set_t Formula_GetDependencies ARGS((node_ptr, node_ptr));
EXTERN Set_t Formulae_GetDependencies ARGS((node_ptr, node_ptr, node_ptr));
EXTERN Set_t ComputeCOI ARGS((Set_t));

EXTERN Fsm_SexpPtr Compile_FsmSexpAlloc ARGS((void));
EXTERN Fsm_SexpPtr Compile_FsmSexpDup ARGS((Fsm_SexpPtr));
EXTERN void Compile_FsmSexpFree ARGS((Fsm_SexpPtr));
EXTERN void Compile_FsmSexpSetInit ARGS((Fsm_SexpPtr, node_ptr));
EXTERN node_ptr Compile_FsmSexpGetInit ARGS((Fsm_SexpPtr));
EXTERN void Compile_FsmSexpSetInvar ARGS((Fsm_SexpPtr, node_ptr));
EXTERN node_ptr Compile_FsmSexpGetInvar ARGS((Fsm_SexpPtr));
EXTERN void Compile_FsmSexpSetTrans ARGS((Fsm_SexpPtr, node_ptr));
EXTERN node_ptr Compile_FsmSexpGetTrans ARGS((Fsm_SexpPtr));
EXTERN void Compile_FsmSexpSetJustice ARGS((Fsm_SexpPtr, node_ptr));
EXTERN node_ptr Compile_FsmSexpGetJustice ARGS((Fsm_SexpPtr));
EXTERN void Compile_FsmSexpSetCompassion ARGS((Fsm_SexpPtr, node_ptr));
EXTERN node_ptr Compile_FsmSexpGetCompassion ARGS((Fsm_SexpPtr));
EXTERN Fsm_BddPtr Compile_FsmBddAlloc ARGS((void));
EXTERN Fsm_BddPtr Compile_FsmBddDup ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddFree ARGS((Fsm_BddPtr));
EXTERN bdd_ptr Compile_FsmBddGetInit ARGS((Fsm_BddPtr));

EXTERN void Compile_FsmBddSetInit ARGS((Fsm_BddPtr, bdd_ptr));
EXTERN bdd_ptr Compile_FsmBddGetInvar ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetInvar ARGS((Fsm_BddPtr, bdd_ptr));

EXTERN CPTrans_Ptr Compile_FsmBddGetMonoTrans ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetMonoTrans ARGS((Fsm_BddPtr, CPTrans_Ptr));
EXTERN CPTrans_Ptr Compile_FsmBddGetIwls95CP ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetIwls95CP ARGS((Fsm_BddPtr, CPTrans_Ptr));
EXTERN CPTrans_Ptr Compile_FsmBddGetThreshold ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetThreshold ARGS((Fsm_BddPtr, CPTrans_Ptr));
EXTERN DPTrans_Ptr Compile_FsmBddGetSaturation ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetSaturation ARGS((Fsm_BddPtr, DPTrans_Ptr));
EXTERN DPTrans_Ptr Compile_FsmBddGetPhase1 ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetPhase1 ARGS((Fsm_BddPtr, DPTrans_Ptr));
EXTERN DPTrans_Ptr Compile_FsmBddGetPhase2 ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetPhase2 ARGS((Fsm_BddPtr, DPTrans_Ptr));

EXTERN node_ptr Compile_FsmBddGetJustice ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetJustice ARGS((Fsm_BddPtr, node_ptr));
EXTERN node_ptr Compile_FsmBddGetCompassion ARGS((Fsm_BddPtr));
EXTERN void Compile_FsmBddSetCompassion ARGS((Fsm_BddPtr, node_ptr));
EXTERN bdd_ptr Compile_FsmBddGetFairStates ARGS((Fsm_BddPtr));
EXTERN void  Compile_FsmBddSetFairStates ARGS((Fsm_BddPtr, bdd_ptr));

EXTERN Fsm_SexpPtr Compile_MakeFsmSexpr ARGS((node_ptr, node_ptr, node_ptr));
EXTERN Fsm_SexpPtr Compile_MakeFsmBExpr ARGS((node_ptr, node_ptr, node_ptr));
EXTERN Fsm_BddPtr Compile_MakeFsmBdd ARGS((node_ptr, node_ptr, node_ptr, add_ptr));

EXTERN CPTrans_Ptr Compile_BuildModelMonolithic ARGS((node_ptr, add_ptr));

EXTERN CPTrans_Ptr Compile_BuildModelThreshold ARGS((node_ptr, int, add_ptr));
EXTERN DPTrans_Ptr Compile_BuildModelSaturation ARGS((node_ptr, node_ptr, add_ptr));
EXTERN DPTrans_Ptr Compile_BuildModelPhase1 ARGS((node_ptr, node_ptr, add_ptr));
EXTERN DPTrans_Ptr Compile_BuildModelPhase2 ARGS((node_ptr, node_ptr, add_ptr));

EXTERN CPTrans_Ptr Compile_BuildModelIwls95CP ARGS((node_ptr, Iwls95OptionStruct_t *, add_ptr));

EXTERN void print_conjunctive_partition_info ARGS((FILE *, node_ptr));
EXTERN bdd_ptr Compile_BuildInitBdd ARGS((node_ptr));
EXTERN bdd_ptr Compile_BuildInvarBdd ARGS((node_ptr, add_ptr));

EXTERN CPTrans_Ptr Compile_CPAddClusterAndFillSchedule ARGS((CPTrans_Ptr trans, bdd_ptr tableau_bdd));

EXTERN int get_minterm_vars_dim ARGS((void));
EXTERN void set_minterm_vars_dim ARGS((int));
EXTERN void reset_minterm_vars_dim ARGS((void));
EXTERN node_ptr get_boolean_variable_list ARGS((void));
EXTERN void reset_boolean_variable_list   ARGS((void));
EXTERN void set_boolean_variable_list     ARGS((node_ptr));
EXTERN void insert_boolean_variable       ARGS((node_ptr b)); 
EXTERN void free_boolean_variable_list    ARGS((void));

EXTERN void Compile_InitializeBuildModel ARGS((void));

EXTERN void Compile_InitializeSexpBuildModel ARGS((node_ptr));

EXTERN void start_test ARGS((char *));
EXTERN void end_test ARGS((char *));

EXTERN int Compile_CheckWff ARGS((node_ptr, node_ptr));


/*--------------------------------------------------------------*/
/* New Api for Conjunctive Partitioning (E.Olivetti) */
/*--------------------------------------------------------------*/

/* CPCluster */
/* Low level */

EXTERN CPCluster * CPClusterAlloc ARGS((void));
EXTERN CPCluster * CPClusterDup ARGS((CPCluster * oldC));
EXTERN void CPClusterFree ARGS((DdManager *dd, CPCluster * CI));
EXTERN void CPClusterFreeAll ARGS((DdManager *dd, CPCluster * CI, int (*freeinfo)(void *, void *)));
EXTERN int CPClusterEqual ARGS((CPCluster * A, CPCluster * B));
EXTERN bdd_ptr CPClusterGetTi ARGS((CPCluster * CI));
EXTERN void CPClusterSetTi ARGS((CPCluster * CI, bdd_ptr newTi));
EXTERN bdd_ptr CPClusterGetEi ARGS((CPCluster * CI));
EXTERN void CPClusterSetEi ARGS((CPCluster * CI, bdd_ptr newEi));
EXTERN void * CPClusterGetInfo ARGS((CPCluster * CI));
EXTERN void CPClusterSetInfo ARGS((CPCluster * CI, void * newInfo));
EXTERN void CPClusterFreeInfo ARGS((CPCluster * CI, int (*freeinfo)(void *, void *)));

/* DPDesc */
EXTERN DPDesc *DPDescAlloc ARGS((void));
EXTERN DPDesc *DPDescDup ARGS((DPDesc *oldD));
EXTERN void DPDescFree ARGS((DdManager *dd, DPDesc *D));
EXTERN void DPDescFreeAll ARGS((DdManager *dd, DPDesc *D, int (*freeinfo)(void *, void *)));
EXTERN bdd_ptr DPDescGetSupportBDD ARGS((DPDesc *D));
EXTERN void DPDescSetSupportBDD ARGS((DPDesc *D, bdd_ptr support_bdd));
EXTERN int *DPDescGetSupportVector ARGS((DPDesc *D));
EXTERN void DPDescSetSupportVector ARGS((DPDesc *D, int *support_vector));
EXTERN CPList DPDescGetCList ARGS((DPDesc *D));
EXTERN void DPDescSetCList ARGS((DPDesc *D, CPList CL));
EXTERN char *DPDescGetName ARGS((DPDesc *D));
EXTERN void DPDescSetName ARGS((DPDesc *D, char *name));
EXTERN void *DPDescGetInfo ARGS((DPDesc *D));
EXTERN void DPDescSetInfo ARGS((DPDesc *D, void *info));
EXTERN void DPDescFreeInfo ARGS((DPDesc *D, int (*freeinfo)(void *, void *)));

/* DPList */
EXTERN DPDesc *DPListGetItem ARGS((DPList DL));
EXTERN DPList DPListNext ARGS((DPList DL));
EXTERN DPList DPListCons ARGS((DPDesc *D, DPList DL));
EXTERN DPList DPListReverse ARGS((DPList DL));
EXTERN int DPListLength ARGS((DPList DL));
EXTERN DPList DPListAlloc ARGS((DPDesc *D));
EXTERN void DPListFree ARGS((DdManager *dd, DPList DL));
EXTERN void DPListFreeAll ARGS((DdManager *dd, DPList DL, int (*freeinfo)(void *, void *)));
EXTERN void DPListFreeInfo ARGS((DPList DL, int (*freeinfo)(void *, void *)));
EXTERN DPList DPListDup ARGS((DPList DL));
EXTERN int DPListIsNotEmpty ARGS((DPList DL));

EXTERN void DPListBuildSchedule ARGS((DPList DL, bdd_ptr vars));

EXTERN DPList DPListBuildProcs ARGS((node_ptr list_variables, node_ptr procs_expr, Proc_Type type, bdd_ptr assumption));

/* CPList */
/* Low level */
EXTERN CPCluster * CPListGetItem ARGS((CPList CL));
EXTERN CPList CPListNext ARGS((CPList CL));
EXTERN CPList CPListCons ARGS((CPCluster * C, CPList CL));
EXTERN CPList CPListReverse ARGS((CPList CL));
EXTERN int CPListLength ARGS((CPList CL));
EXTERN CPList CPListAlloc ARGS((CPCluster * el));
EXTERN void CPListFree ARGS((DdManager *dd, CPList CL));
EXTERN void CPListFreeAll ARGS((DdManager *dd, CPList CL, int (*freeinfo)(void *, void *)));
EXTERN void CPListFreeInfo ARGS((CPList CL, int (*freeinfo)(void *, void *)));
EXTERN CPList CPListDup ARGS((CPList CL));
EXTERN CPList CPListDeleteCluster ARGS((CPList CL, CPCluster * el));
EXTERN int CPListIsNotEmpty ARGS((CPList CL));
/* Mid Level */
EXTERN CPList CPListAddOneCluster ARGS((CPCluster * el, CPList CL, bdd_ptr vars));
EXTERN int CPListCheckComparing ARGS((DdManager *dd, CPList CL1, CPList CL2));
EXTERN void CPListBuildSchedule ARGS((CPList CL, bdd_ptr vars));
/* High Level */
EXTERN void CPListBuildStart ARGS((void));
EXTERN void CPListBuildStop ARGS((void));
EXTERN CPList CPListBuildElementaryBDDList ARGS((node_ptr vars, add_ptr assumption));
EXTERN CPList CPListBuildBasicRecur ARGS((CPList clusters, node_ptr sexp, add_ptr assumption));
EXTERN bdd_ptr CPListBuildScheduleRecur ARGS((CPList l, bdd_ptr vars));
EXTERN void CPListBuildThreshold ARGS((DdManager *dd, CPList RelationList, CPList * NewRelationList, int threshold));
EXTERN bdd_ptr CPListBuildMonolithicBDD ARGS((node_ptr vl, bdd_ptr assumption));
EXTERN bdd_ptr CPListBuildMonolithicBDDFromCPlist ARGS((CPList CL));
EXTERN CPList CPListBuildMonolithic ARGS((node_ptr vl, bdd_ptr assumption));
EXTERN void CPListPrintInfo ARGS((FILE *file, CPList CL));
EXTERN void CPListPrintClustersInfo ARGS((FILE * fp, DdManager * dd,
					  CPList ClusterList, node_ptr var_list));
EXTERN bdd_ptr CPListComputeClustersCube ARGS((DdManager * dd, CPList set_of_clusters));

/* DPTrans */
EXTERN DPTrans_Ptr DPTransAlloc ARGS(());
EXTERN void DPTransFree ARGS((DPTrans_Ptr c));
EXTERN DPTrans_Ptr DPTransDup ARGS((DPTrans_Ptr c));
EXTERN void DPTransSetBackward ARGS((DPTrans_Ptr c, DPList L));
EXTERN DPList DPTransGetBackward ARGS((DPTrans_Ptr c));
EXTERN void DPTransSetForward ARGS((DPTrans_Ptr c, DPList L));
EXTERN DPList DPTransGetForward ARGS((DPTrans_Ptr c));
EXTERN void DPTransPrintInfo ARGS((FILE *file, DPTrans_Ptr c));

/* CPTrans */
/* Low level */
EXTERN CPTrans_Ptr CPTransAlloc ARGS(());
EXTERN void CPTransFree ARGS((CPTrans_Ptr c));
EXTERN CPTrans_Ptr CPTransDup ARGS((CPTrans_Ptr c));
EXTERN void CPTransSetBackward ARGS((CPTrans_Ptr c, CPList L));
EXTERN CPList CPTransGetBackward ARGS((CPTrans_Ptr c));
EXTERN void CPTransSetForward ARGS((CPTrans_Ptr c, CPList L));
EXTERN CPList CPTransGetForward ARGS((CPTrans_Ptr c));
/* High Level */
EXTERN void CPTransPrintInfo ARGS((FILE *file, CPTrans_Ptr c));
EXTERN void CPTransPrintDetailedInfo ARGS((FILE *file, CPTrans_Ptr c));

EXTERN CPTrans_Ptr Compile_BuildModelMonolithic_ ARGS((node_ptr , bdd_ptr));
EXTERN CPTrans_Ptr Compile_BuildModelThreshold_ ARGS((node_ptr, int, bdd_ptr));
EXTERN CPTrans_Ptr Compile_BuildModelIwls95CP_ ARGS((node_ptr, Iwls95OptionStruct_t *, add_ptr));

EXTERN int is_next_variable ARGS((int index));
EXTERN int is_current_variable ARGS((int index));
EXTERN int is_input_variable ARGS((int index));

#endif /* _COMPILE */
