/**CFile***********************************************************************

  FileName [bmcVarsMgr.c]

  PackageName [bmc]

  Synopsis [Variables Manager for the <tt>bmc</tt> package]

  Description []

  SeeAlso []

  Author [Alessandro Cimatti and Lorenzo Delana]

  Copyright [ This file is part of the ``bmc'' package of NuSMV
  version 2.  Copyright (C) 2000-2001 by ITC-irst and University of
  Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it> or
  email to <nusmv-users@irst.itc.it>.  Please report bugs to
  <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to
  <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcVarsMgr.h"
#include "bmcInt.h" /* for 'options' */

#include "be.h" 

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Variables Manager (core data structure)]

  Description [Variable Manager is a private structure owned by this module.
  In order to access to its members use the supplied methods, which give you
  access to the general part of the structure tied to rbc generic interface.
  The structural part specific of the current implementation (tables and other
  specific sub-structures) is not accessible from the outer space.]

  SeeAlso     [bmc_vars_init, bmc_vars_quit]

******************************************************************************/
typedef struct VarsMgr_TAG
{
  /* ----------------------------------------------------------------------- */
  /*    Accessible generic part, tied to rbc interface                       */
  /* ----------------------------------------------------------------------- */
  Be_Manager_ptr env;              /* the boolean expr manager */
  int num;                         /* the number of state variables */
  int maxtime;                     /* max time reached until now for the
				      variables environment */

  /* ----------------------------------------------------------------------- */
  /*    Private part specific of implementation                              */
  /* ----------------------------------------------------------------------- */
  hash_ptr statevar2curvar_table;   /* table for statevar->curvar
               conversion */
  node_ptr* curvar2statevar_table; /* table for curvar->statevar
               conversion */
  int* cnexp_subst_array;          /* table for current and next vars
              substitution */
} VarsMgr;


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static void Bmc_VarsMgr_ExtendMaxTime ARGS((VarsMgr_ptr vars_mgr,
					    const int new_maxtime));


/*---------------------------------------------------------------------------*/
/* Declaration of private functions                                          */
/*---------------------------------------------------------------------------*/

/* Following private wrappers assure a more generic access to the be part
   into Variables Manager structure: */
static void
set_env ARGS((VarsMgr_ptr vars_mgr, Be_Manager_ptr pNewBeManager));

static void set_num     ARGS((VarsMgr_ptr vars_mgr, const int iNewNum));
static void set_maxtime ARGS((VarsMgr_ptr vars_mgr, const int iMaxTime));

static void
scan_state_vars_bexp ARGS((VarsMgr_ptr vars_mgr, node_ptr state_vars_bexp));


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [<b>Returns</b> the
  <tt>number of variables currently handled by the Variables Manager</tt>]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
int Bmc_VarsMgr_GetNum(const VarsMgr_ptr vars_mgr) { return vars_mgr->num; }


/**Function********************************************************************

  Synopsis           [<b>Returns</b> the
  <tt>maximum time currently handled by the variable environment</tt>]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
int Bmc_VarsMgr_GetMaxTime(const VarsMgr_ptr vars_mgr)
{
  return vars_mgr->maxtime;
}


/**Function********************************************************************

  Synopsis           [<b>Returns</b> the
  <tt>Boolean Expressions manager</tt> contained into the variable manager]

  Description        [Warning: do not delete the returned instance of 
  Be_Manager class]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
Be_Manager_ptr Bmc_VarsMgr_GetBeMgr(const VarsMgr_ptr vars_mgr)
{
  return vars_mgr->env;
}


/**Function********************************************************************

  Synopsis           [Returns true if the given index is a state variable 
  index, false if the given index refers to a non-state variable]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
boolean Bmc_VarsMgr_IsStateIndex(const VarsMgr_ptr vars_mgr,
				 const int index, const int maxTime)
{
  if ( index >= (2 * Bmc_VarsMgr_GetNum(vars_mgr))
       && (index <  ((2 + 1 + maxTime) * Bmc_VarsMgr_GetNum(vars_mgr))) ) {
    return true;
  }
  else return false;
}


/**Function********************************************************************

  Synopsis           [<b>Shift</b> given <i>current</i> <b>expression at
  next time</b>]

  Description        [Warning: argument 'exp' must contain only variables in
  current time, otherwise results will be unpredictible, surely inconsistent]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_ShiftCurr2Next(const VarsMgr_ptr vars_mgr,
				  const be_ptr exp)
{
  return Be_ShiftVar( Bmc_VarsMgr_GetBeMgr(vars_mgr),
       exp,
       Bmc_VarsMgr_GetNum(vars_mgr) );
}


/**Function********************************************************************

  Synopsis           [<b>Shift</b> given <i>current, next</i> <b>expression at
  specified <tt>time</tt></b>]

  Description        []

  SideEffects        [extends variables environment and maxtime, if required]

  SeeAlso            []

******************************************************************************/
be_ptr
Bmc_VarsMgr_ShiftCurrNext2Time(const VarsMgr_ptr vars_mgr,
			       const be_ptr exp, const int time)
{
  nusmv_assert(time >= 0);

  /* extends maxtime, if required */
  Bmc_VarsMgr_ExtendMaxTime(vars_mgr, time + 1);

  /* return exp with variables shifted at given time */
  return Be_ShiftVar( Bmc_VarsMgr_GetBeMgr(vars_mgr),
       exp,
       Bmc_VarsMgr_Time2AbsIndex(vars_mgr, time) );
}


/**Function********************************************************************

  Synopsis           [<b>Shift</b> given <i>current, next</i> <b>expression at
  specified <tt>ctime</tt> (current vars at time) and <tt>ntime</tt> (next
  vars at time)</b>]

  Description        []

  SideEffects        [extends variables environment and maxtime, if required]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_ShiftCurrNext2Times(const VarsMgr_ptr vars_mgr,
				       const be_ptr exp,
				       const int ctime, const int ntime)
{
  int i;
  int firstvar_at_ctime, firstvar_at_ntime;

  /* lazy evaluation */
  if ( Be_IsTrue(Bmc_VarsMgr_GetBeMgr(vars_mgr), exp) ||
       Be_IsFalse(Bmc_VarsMgr_GetBeMgr(vars_mgr), exp) ) {
    return exp;
  }

  /* extends maxtime, if required */
  Bmc_VarsMgr_ExtendMaxTime(vars_mgr, max(ctime, ntime));

  firstvar_at_ctime = Bmc_VarsMgr_Time2AbsIndex(vars_mgr, ctime);
  firstvar_at_ntime = Bmc_VarsMgr_Time2AbsIndex(vars_mgr, ntime);

  for (i = 0;
       i <= (Bmc_VarsMgr_GetNum(vars_mgr) - 1);
       i++) {
    /* setup substitution for ith-curvar in the array for substitution,
       with ith-timedvar at time ctime */
    vars_mgr->cnexp_subst_array [i] = firstvar_at_ctime + i;

    /* setup substitution for ith-nextvar in the array for substitution,
       with ith-timedvar at time ntime */
    vars_mgr->cnexp_subst_array [i + Bmc_VarsMgr_GetNum(vars_mgr)] =
      firstvar_at_ntime + i;
  }

  /* return given exp with symbolic (current, next) variables substituited
     with correspondent timed variables at times ctime and ntime, resp. */
  {
    return Be_VarSubst(Bmc_VarsMgr_GetBeMgr(vars_mgr),
          exp,
          vars_mgr->cnexp_subst_array);
  }
}


/**Function********************************************************************

  Synopsis           [<b>Make an AND interval</b> of given expression using
  <b>range \[<tt>from</tt>, <tt>to</tt>\]</b>]

  Description        []

  SideEffects        [be hash may change]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_MkAndCurrNextInterval(const VarsMgr_ptr vars_mgr,
					 const be_ptr exp,
					 const int from, const int to)
{
  /* We accept the cases (from <= to) and (from == to - 1).
     The latter may exist when the unrolling is performed at high level */
  nusmv_assert(from <= to+1);

  if (from > to) {
    /* ends the recursion */
    return Be_Truth(Bmc_VarsMgr_GetBeMgr(vars_mgr));
  }
  else {
    return Be_And( Bmc_VarsMgr_GetBeMgr(vars_mgr),
          Bmc_VarsMgr_MkAndCurrNextInterval(vars_mgr, exp,
              from, to-1),
          Bmc_VarsMgr_ShiftCurrNext2Time(vars_mgr, exp, to) );
  }
}


/**Function********************************************************************

  Synopsis           [<b>Make an OR interval</b> of given expression using
  <b>range \[<tt>from</tt>, <tt>to</tt>\]</b>]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_MkOrCurrNextInterval(const VarsMgr_ptr vars_mgr,
					const be_ptr exp,
					const int from, const int to)
{
  /* We accept the cases (from <= to) and (from == to - 1).
     The latter may exist when the unrolling is performed at high level */
  nusmv_assert(from <= to+1);

  if (from > to) {
    /* ends the recursion */
    return Be_Falsity(Bmc_VarsMgr_GetBeMgr(vars_mgr));
  }
  else {
    return Be_Or( Bmc_VarsMgr_GetBeMgr(vars_mgr),
         Bmc_VarsMgr_ShiftCurrNext2Time(vars_mgr, exp, from),
         Bmc_VarsMgr_MkOrCurrNextInterval(vars_mgr, exp,
                  from + 1, to) );
  }
}


/**Function********************************************************************

  Synopsis           [Search the association from given state variable and
  its relative be representation, and returns the BE form]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_State2Curr(const VarsMgr_ptr vars_mgr,
			      const node_ptr statevar)
{
  be_ptr curvar = (be_ptr) find_assoc(vars_mgr->statevar2curvar_table,
          statevar);

  nusmv_assert(curvar != (be_ptr) NULL);

  return curvar;
}


/**Function********************************************************************

  Synopsis           [Given the relative index returns the state variable
  which addresses]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr Bmc_VarsMgr_Index2State(const VarsMgr_ptr vars_mgr, const int i)
{
  nusmv_assert(i < Bmc_VarsMgr_GetNum(vars_mgr));
  return (vars_mgr->curvar2statevar_table)[i];
}


/**Function********************************************************************

  Synopsis           [Given a be representation of a variable, returns the
  relative state variable]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
node_ptr Bmc_VarsMgr_Curr2State(const VarsMgr_ptr vars_mgr,
				const be_ptr curvar)
{
  int curvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr), curvar);

  nusmv_assert((curvar_index >= 0)
         && (curvar_index < Bmc_VarsMgr_GetNum(vars_mgr)));

  return Bmc_VarsMgr_Index2State(vars_mgr, curvar_index);
}


/**Function********************************************************************

  Synopsis           [As Bmc_VarsMgr_Index2State but in terms of be
  representation]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Index2Curr(const VarsMgr_ptr vars_mgr, const int i)
{
  nusmv_assert((i >= 0) && (i < Bmc_VarsMgr_GetNum(vars_mgr)));

  return Be_Index2Var(Bmc_VarsMgr_GetBeMgr(vars_mgr), i);
}


/**Function********************************************************************

  Synopsis           [As Bmc_VarsMgr_Index2Curr but relatively to next state
  variables block]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Index2Next(const VarsMgr_ptr vars_mgr, const int i)
{
  nusmv_assert((i >= 0) && (i < Bmc_VarsMgr_GetNum(vars_mgr)));

  return Be_Index2Var( Bmc_VarsMgr_GetBeMgr(vars_mgr),
          Bmc_VarsMgr_GetNum(vars_mgr) + i );
}


/**Function********************************************************************

  Synopsis           [As Bmc_VarsMgr_Index2Next but relatively to 'time' indexed
  state variables block]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Index2Timed(const VarsMgr_ptr vars_mgr,
			       const int i, const int time)
{
  /* extends maxtime, if required */
  Bmc_VarsMgr_ExtendMaxTime(vars_mgr, time + 1);

  nusmv_assert((i >= 0) && (i < Bmc_VarsMgr_GetNum(vars_mgr)));
  nusmv_assert(time <= Bmc_VarsMgr_GetMaxTime(vars_mgr));

  return Be_Index2Var( Bmc_VarsMgr_GetBeMgr(vars_mgr),
        (Bmc_VarsMgr_GetNum(vars_mgr) * (2 + time)) + i );
}


/**Function********************************************************************

  Synopsis           [Given a be current variable, returns the be variable
  corrisponding to the next variables block]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Curr2Next(const VarsMgr_ptr vars_mgr, const be_ptr curvar)
{
  int curvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr), curvar);

  nusmv_assert((curvar_index >= 0)
         && (curvar_index < Bmc_VarsMgr_GetNum(vars_mgr)));

  return Bmc_VarsMgr_Index2Next(vars_mgr, curvar_index);
}


/**Function********************************************************************

  Synopsis           [As Bmc_Curr2Next but current variable must be
  supplied in state variable form instead of BE form ]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_State2Next(const VarsMgr_ptr vars_mgr,
			      const node_ptr statevar)
{
  return Bmc_VarsMgr_Curr2Next(vars_mgr,
            Bmc_VarsMgr_State2Curr(vars_mgr, statevar));
}


/**Function********************************************************************

  Synopsis           [Given a be next variable, returns the be variable
  corrisponding to the 'time' index variables block]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
be_ptr
Bmc_VarsMgr_Next2Timed(const VarsMgr_ptr vars_mgr,
		       const be_ptr nextvar, const int time)
{
  int nextvar_index;

  nextvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr),
          nextvar);

  /* extends maxtime, if required */
  Bmc_VarsMgr_ExtendMaxTime(vars_mgr, time + 1);

  nusmv_assert((nextvar_index >= Bmc_VarsMgr_GetNum(vars_mgr)) &&
         (nextvar_index < (2 * Bmc_VarsMgr_GetNum(vars_mgr))));

  return Bmc_VarsMgr_Index2Timed(vars_mgr,
         nextvar_index - Bmc_VarsMgr_GetNum(vars_mgr),
         time);
}


/**Function********************************************************************

  Synopsis           [Given a be current variable, returns the be variable
  corrisponding to the 'time' index variables block]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr
Bmc_VarsMgr_Curr2Timed(const VarsMgr_ptr vars_mgr,
		       const be_ptr curvar, const int time)
{
  int curvar_index;

  /* extends maxtime, if required */
  Bmc_VarsMgr_ExtendMaxTime(vars_mgr, time + 1);

  curvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr), curvar);

  nusmv_assert((curvar_index >= 0)
         && (curvar_index < Bmc_VarsMgr_GetNum(vars_mgr)));

  return Bmc_VarsMgr_Index2Timed(vars_mgr, curvar_index, time);
}


/**Function********************************************************************

  Synopsis           [As Bmc_Curr2Timed but current variable must be
  supplied in state variable form instead of BE form]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr
Bmc_VarsMgr_State2Timed(const VarsMgr_ptr vars_mgr,
			const node_ptr statevar, const int time)
{
  /* extends maxtime, if required */
  Bmc_VarsMgr_ExtendMaxTime(vars_mgr, time + 1);

  return Bmc_VarsMgr_Curr2Timed(vars_mgr,
        Bmc_VarsMgr_State2Curr(vars_mgr, statevar),
        time);
}


/**Function********************************************************************

  Synopsis           [Given a variable which belongs to a timed block of
  variables, returns the corrisponding current variable in BE form]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Timed2Curr(const VarsMgr_ptr vars_mgr,
			      const be_ptr timedvar)
{
  int timedvar_index;

  timedvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr),
           timedvar);

  nusmv_assert((timedvar_index >= (2 * Bmc_VarsMgr_GetNum(vars_mgr)))
         && (timedvar_index
       < ((2 + 1 + Bmc_VarsMgr_GetMaxTime(vars_mgr))
          * Bmc_VarsMgr_GetNum(vars_mgr))));

  return Bmc_VarsMgr_Index2Curr(vars_mgr,
        timedvar_index % Bmc_VarsMgr_GetNum(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [Given a variable which belongs to a timed block of
  variables, returns the corrisponding state variable]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Bmc_VarsMgr_Timed2State(const VarsMgr_ptr vars_mgr,
				 const be_ptr timedvar)
{
  int timedvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr),
               timedvar);

  nusmv_assert((timedvar_index >= (2 * Bmc_VarsMgr_GetNum(vars_mgr)))
         && (timedvar_index < ((2 + 1 + Bmc_VarsMgr_GetMaxTime(vars_mgr))
             * Bmc_VarsMgr_GetNum(vars_mgr))));

  return Bmc_VarsMgr_Index2State(vars_mgr, timedvar_index
                   % Bmc_VarsMgr_GetNum(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [Given a variable which belongs to the next block of
  variables, returns the corrisponding state variable]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Bmc_VarsMgr_Next2State(const VarsMgr_ptr vars_mgr,
				const be_ptr nextvar)
{
  int nextvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr),
              nextvar);

  nusmv_assert((nextvar_index >= Bmc_VarsMgr_GetNum(vars_mgr))
         && (nextvar_index < (2 * Bmc_VarsMgr_GetNum(vars_mgr))));

  return Bmc_VarsMgr_Index2State(vars_mgr,
         nextvar_index - Bmc_VarsMgr_GetNum(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [Given a variable which belongs to a timed block of
  variables, returns the corrisponding variable in BE form which belongs to
  the next block of variables]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Timed2Next(const VarsMgr_ptr vars_mgr,
			      const be_ptr timedvar)
{
  int timedvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr),
               timedvar);

  nusmv_assert((timedvar_index >= (2 * Bmc_VarsMgr_GetNum(vars_mgr)))
         && (timedvar_index
       < ((2 + 1 + Bmc_VarsMgr_GetMaxTime(vars_mgr))
          * Bmc_VarsMgr_GetNum(vars_mgr))));

  return Bmc_VarsMgr_Index2Next(vars_mgr,
        timedvar_index % Bmc_VarsMgr_GetNum(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [Given a variable which belongs to the next block of
  variables, returns the corrisponding current variable in BE form]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_VarsMgr_Next2Curr(const VarsMgr_ptr vars_mgr,
			     const be_ptr nextvar)
{
  int nextvar_index = Be_Var2Index(Bmc_VarsMgr_GetBeMgr(vars_mgr),
              nextvar);

  nusmv_assert((nextvar_index >= Bmc_VarsMgr_GetNum(vars_mgr))
         && (nextvar_index < (2 * Bmc_VarsMgr_GetNum(vars_mgr))));

  return Bmc_VarsMgr_Index2Curr(vars_mgr,
        nextvar_index % Bmc_VarsMgr_GetNum(vars_mgr));
}


/**Function********************************************************************

  Synopsis           [Returns the first variable in BE form which belongs to
  the given block of variables indexed by 'time']

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Bmc_VarsMgr_Time2AbsIndex(const VarsMgr_ptr vars_mgr,
			      const int time)
{
  return Bmc_VarsMgr_GetNum(vars_mgr) * (2 + time);
}


/**Function********************************************************************

  Synopsis           [Given a variable index returns the corresponding time]

  Description        [A current variable index returns 0, a next variable
  returns 1, a timed variable returns its time (p0 returns 0, p1 returns 1,
  and so on)]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Bmc_VarsMgr_AbsIndex2Time(const VarsMgr_ptr vars_mgr, const int varIdx)
{
  int iTime = -1;

  nusmv_assert( (varIdx >= 0) &&
    (varIdx < ((2+1+Bmc_VarsMgr_GetMaxTime(vars_mgr))
         * Bmc_VarsMgr_GetNum(vars_mgr))) );

  if (varIdx < Bmc_VarsMgr_GetNum(vars_mgr)) {
    iTime=0; /* current variable */
  }
  else {
    if (varIdx < (2 * Bmc_VarsMgr_GetNum(vars_mgr))) {
      iTime = 1; /* next variable */
    }
    else {
      /* timed variable */
      iTime = (varIdx / Bmc_VarsMgr_GetNum(vars_mgr)) - 2;
    }
  }

  nusmv_assert( (iTime>=0) && (iTime <= Bmc_VarsMgr_GetMaxTime(vars_mgr)) );

  return iTime;
}


/**Function********************************************************************

  Synopsis           [Given the index of a timed variable, returns the index
  of the corresponding state variable]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Bmc_VarsMgr_AbsIndex2Index(const VarsMgr_ptr vars_mgr, const int varIdx)
{
  nusmv_assert( (varIdx >= 0) &&
     (varIdx < ((2+1+Bmc_VarsMgr_GetMaxTime(vars_mgr))
          * Bmc_VarsMgr_GetNum(vars_mgr))) );

  return varIdx % Bmc_VarsMgr_GetNum(vars_mgr);
}


/**Function********************************************************************

  Synopsis           [Given the index of a state variable and the time,
  returns the index of the corresponding timed variable]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Bmc_VarsMgr_Index2AbsIndex(const VarsMgr_ptr vars_mgr,
			       const int idx, const int time)
{
  return Bmc_VarsMgr_Time2AbsIndex(vars_mgr, time) + idx;
}


/**Function********************************************************************

  Synopsis           [<b><i>Initialize</i> <tt>BMC</tt> Variables Manager</b>]

  Description        []

  SideEffects        [Alloc <tt>variables manager (data container)</tt><br>
  Follow structures are allocated in the
  <tt>variables manager (data container)</tt>:<br>
  - variables <tt>environment</tt><br>
  - variables <tt>state vars->current vars</tt> conversion table<br>
  - variables <tt>current vars->state vars</tt> conversion table<br>
  - variables <tt>substitution array</tt>]

  SeeAlso            [scan_state_vars_bexp]

******************************************************************************/
VarsMgr_ptr Bmc_VarsMgr_Create(node_ptr state_vars_bexp)
{

  VarsMgr_ptr vars_mgr = (VarsMgr_ptr) NULL;
  Be_Manager_ptr be_mgr; 

  /* Allocates the variables manager core data container */
  vars_mgr = ALLOC(VarsMgr, 1);
  nusmv_assert(vars_mgr != NULL);

  /* determine the number of state variables */
  set_num(vars_mgr, llength(state_vars_bexp));

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, "State variables in the variables manager: %d\n",
	    Bmc_VarsMgr_GetNum(vars_mgr)+1);
  }

  /* allocate variables for the two (current and next) frames
     used in initial model construction, and for the zero time frame.
     The variables environment is the manager for the be package */
  be_mgr = Be_RbcManager_Create((2 + 1) * Bmc_VarsMgr_GetNum(vars_mgr));
  set_env(vars_mgr, be_mgr);

  /* setup maxtime indicator in the variables manager */
  set_maxtime(vars_mgr, 0);

  /* allocate the instance for the statevar->curvar table */
  vars_mgr->statevar2curvar_table = new_assoc();

  /* allocate the instance for the curvar->statevar table */
  vars_mgr->curvar2statevar_table = ALLOC(node_ptr,
            Bmc_VarsMgr_GetNum(vars_mgr));
  nusmv_assert(vars_mgr->curvar2statevar_table != NULL);

  /* scan state variables and update be <-> bexp tables */
  scan_state_vars_bexp(vars_mgr, state_vars_bexp);

  /* alloc the array for variables substitution */
  if (opt_verbose_level_gt(options, 1)) {
  fprintf( nusmv_stderr, "Allocating cnexp_subst_array: %d locations\n",
    2 * (Bmc_VarsMgr_GetNum(vars_mgr)) );
  }

  vars_mgr->cnexp_subst_array = ALLOC(int, 2 * Bmc_VarsMgr_GetNum(vars_mgr));
  nusmv_assert(vars_mgr->cnexp_subst_array != NULL);

  return vars_mgr;
}


/**Function********************************************************************

  Synopsis           [<b><i>Quit</i> from <tt>BMC</tt> Variables Manager</b>]

  Description        []

  SideEffects        [Follow structures are freed in the <tt>variables
  manager (data container)</tt>:<br>
  - variables <tt>substitution array</tt>
  - variables <tt>current vars->state vars</tt> conversion table<br>
  - variables <tt>state vars->current vars</tt> conversion table<br>
  - variables <tt>environment</tt><br>
  Free <tt>variables manager (data container)</tt><br>]

  SeeAlso            [bmc_vars_init]

******************************************************************************/
void Bmc_VarsMgr_Delete(VarsMgr_ptr* vars_mgr_ref)
{
  nusmv_assert(vars_mgr_ref != NULL);
  if(*vars_mgr_ref == NULL) return;


  /* free the array for substitution */
  FREE((*vars_mgr_ref)->cnexp_subst_array);

  /* free the instance for the curvar->statevar table */
  FREE((*vars_mgr_ref)->curvar2statevar_table);

  /* free the instance for the statevar->curvar table */
  clear_assoc((*vars_mgr_ref)->statevar2curvar_table);

  /* free the variables environment */
  Be_RbcManager_Delete(Bmc_VarsMgr_GetBeMgr(*vars_mgr_ref));

  /* free the variables manager core data container */
  FREE(*vars_mgr_ref);
  *vars_mgr_ref = NULL;
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [<b>Extends</b> <tt>maxtime</tt> at given
  <tt>new_maxtime</tt>]

  Description        []

  SideEffects        [be hash may change]

  SeeAlso            []

******************************************************************************/
static void Bmc_VarsMgr_ExtendMaxTime(VarsMgr_ptr vars_mgr, int new_maxtime)
{
  nusmv_assert(new_maxtime >= 0);

  /* if is really necessary to do the maxtime extension... */
  if (new_maxtime > Bmc_VarsMgr_GetMaxTime(vars_mgr)) {
    Be_RbcManager_Reserve( Bmc_VarsMgr_GetBeMgr(vars_mgr),
			   /* reserve space for 2 time (curvars, nextvars) +
			      1 time (zero timed vars) +
			      new_maxtime time (timed vars until new_maxtime) */
			   (2 + 1 + new_maxtime)
			   * Bmc_VarsMgr_GetNum(vars_mgr) );

    /* keeps maxtime change */
    set_maxtime(vars_mgr, new_maxtime);
  }
}


/**Function********************************************************************

  Synopsis           [<b>Stores</b> a new <tt>variable environment
  data structure</tt>]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void set_env(VarsMgr_ptr vars_mgr, const Be_Manager_ptr pNewEnv)
{
  vars_mgr->env = pNewEnv;
}


/**Function********************************************************************

  Synopsis           [<b>Sets</b> the
  <tt>number of variables handled by the Variables Manager</tt>]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void set_num(VarsMgr_ptr vars_mgr, int iNewNum)
{
  vars_mgr->num = iNewNum;
}


/**Function********************************************************************

  Synopsis           [<b>Sets</b> the
  <tt>maximum time currently handled by the variable environment</tt>]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void set_maxtime(VarsMgr_ptr vars_mgr, int iNewMaxTime)
{
  vars_mgr->maxtime = iNewMaxTime;
}


/**Function********************************************************************

   Synopsis           [Builds all internal structures used in order to
   perform searches and conversion from different variable representation
   forms]

   Description        []

   SideEffects        [...]

   SeeAlso            [bmc_vars_init]

******************************************************************************/
static void scan_state_vars_bexp(VarsMgr_ptr vars_mgr,
				 node_ptr state_vars_bexp)
{
  int i;
  node_ptr ith_state_var_bexp = NULL;

  /* check for the following loop: */
  nusmv_assert( llength(state_vars_bexp) >= Bmc_VarsMgr_GetNum(vars_mgr) );

  /* loop over each state variable... */
  for (i = 0; i < Bmc_VarsMgr_GetNum(vars_mgr); i++) {
    /* get ith state variable bexp */
    ith_state_var_bexp = car(state_vars_bexp);

    /* update statevar->curvar table for associate ith state variable with
       ith current variable, that are in the variables environment */
    insert_assoc(vars_mgr->statevar2curvar_table,
      ith_state_var_bexp,
      (node_ptr) Be_Index2Var(Bmc_VarsMgr_GetBeMgr(vars_mgr),
                i));

    /* update curvar->statevar table for associate ith current variable,
       that are in the variables environment, with ith state variable */
    vars_mgr->curvar2statevar_table[i] = ith_state_var_bexp;

    if (opt_verbose_level_gt(options, 1)) {
      fprintf(nusmv_stderr, "[%3d] Boolean variable ", i);
      print_node(nusmv_stderr, car(state_vars_bexp));
      fprintf(nusmv_stderr, "\n");
    }

    /* to the next state vars */
    state_vars_bexp = cdr(state_vars_bexp);
  } /* for loop */
}

/**AutomaticEnd***************************************************************/
