/**CFile***********************************************************************

  FileName    [bmcTableau.c]

  PackageName [bmc]

  Synopsis    [Bmc.Tableau module]

  Description [This module contains all the tableau-related operations]

  SeeAlso     [bmcModel.c, bmcConv.c, bmcVarMgr.c bmcTableau.c,
               bmcTableauLTLformula.c,
               bmcTableauPLTLformula.c, bmcGen.c]

  Author      [Marco Benedetti, Roberto Cavada]

  Copyright   [
  This file is part of the ``bmc'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst and University of Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcTableau.h"
#include "bmcUtils.h"
#include "bmcModel.h"
#include "grammar.h"

#include "bmcInt.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static be_ptr
Bmc_Tableau_GetLoopCondition ARGS((const VarsMgr_ptr vars_mgr,
                                   const int k, const int l));

static be_ptr
Bmc_Tableau_GetAllLoopsDisjunction ARGS((const VarsMgr_ptr vars_mgr, const int k));


/* Tableaux for LTL: */
static be_ptr
Bmc_TableauLTL_GetNoLoop ARGS((const Bmc_Fsm_ptr be_fsm,
                               const node_ptr ltl_wff, const int k));
static be_ptr
Bmc_TableauLTL_GetSingleLoop ARGS((const Bmc_Fsm_ptr be_fsm,
                                   const node_ptr ltl_wff,
                                   const int k, const int l));
static be_ptr
Bmc_TableauLTL_GetAllLoops ARGS((const Bmc_Fsm_ptr be_fsm,
                                 const node_ptr ltl_wff,
                                 const int k, const int l));
static be_ptr
Bmc_TableauLTL_GetAllLoopsDepth1 ARGS((const Bmc_Fsm_ptr be_fsm,
                                       const node_ptr ltl_wff, const int k));


/* Tableaux for PLTL: */
static be_ptr
Bmc_TableauPLTL_GetNoLoop ARGS((const Bmc_Fsm_ptr be_fsm,
                                const node_ptr ltl_wff, const int k));
static be_ptr
Bmc_TableauPLTL_GetSingleLoop ARGS((const Bmc_Fsm_ptr be_fsm,
                                    const node_ptr ltl_wff,
				    const int k, const int l));
static be_ptr
Bmc_TableauPLTL_GetAllLoops ARGS((const Bmc_Fsm_ptr be_fsm,
                                  const node_ptr ltl_wff, 
				  const int k, const int l));
static be_ptr
Bmc_TableauPLTL_GetAllLoopsDepth1 ARGS((const Bmc_Fsm_ptr be_fsm,
                                        const node_ptr ltl_wff, const int k));


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [Builds tableau without loop at time zero, taking into
  account of fairness]

  Description        [Fairness evaluate to true if there are not fairness
  in the model, otherwise them evaluate to false because of no loop]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_Tableau_GetNoLoop(const Bmc_Fsm_ptr be_fsm,
			     const node_ptr ltl_wff, const int k)
{
  return (isPureFuture(ltl_wff) && !opt_bmc_force_pltl_tableau(options)) ?
        Bmc_TableauLTL_GetNoLoop (be_fsm, ltl_wff, k) :
        Bmc_TableauPLTL_GetNoLoop(be_fsm, ltl_wff, k) ;
}


/**Function********************************************************************

  Synopsis           [Builds tableau for all possible loops in \[l, k\], in
  the particular case in which depth is 1. This function takes into account
  of fairness]

  Description        [Builds the tableau in the case depth==1 as suggested
  by R. Sebastiani]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_Tableau_GetSingleLoop (const Bmc_Fsm_ptr be_fsm,
				  const node_ptr ltl_wff, const int k, const int l)
{
  return (isPureFuture(ltl_wff) && !opt_bmc_force_pltl_tableau(options)) ?
        Bmc_TableauLTL_GetSingleLoop  (be_fsm,ltl_wff,k,l) :
        Bmc_TableauPLTL_GetSingleLoop (be_fsm,ltl_wff,k,l) ;
}


/**Function********************************************************************

  Synopsis           [Builds tableau for all possible loops in \[l, k\[,
  taking into account of fairness]

  Description        [Each tableau takes into account of fairnesses relative
  to its step. All tableau are collected together into a disjunctive form.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_Tableau_GetAllLoops(const Bmc_Fsm_ptr be_fsm,
			       const node_ptr ltl_wff, const int k, const int l)
{
  return (isPureFuture(ltl_wff) && !opt_bmc_force_pltl_tableau(options)) ?
        Bmc_TableauLTL_GetAllLoops (be_fsm, ltl_wff, k, l) :
        Bmc_TableauPLTL_GetAllLoops(be_fsm, ltl_wff, k, l) ;
}


/**Function********************************************************************

  Synopsis           [Builds tableau for all possible loops in \[l, k\], in
  the particular case in which depth is 1. This function takes into account
  of fairness]

  Description        [Builds the tableau in the case depth==1 as suggested
  by R. Sebastiani]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_Tableau_GetAllLoopsDepth1(const Bmc_Fsm_ptr be_fsm,
				     const node_ptr ltl_wff, const int k)
{
  return (isPureFuture(ltl_wff) && !opt_bmc_force_pltl_tableau(options)) ?
        Bmc_TableauLTL_GetAllLoopsDepth1 (be_fsm, ltl_wff, k) :
        Bmc_TableauPLTL_GetAllLoopsDepth1(be_fsm, ltl_wff, k) ;
}



/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/* ====================================================================== */
/*                       Tableaux for LTL :                               */
/* ====================================================================== */

/**Function********************************************************************

  Synopsis           [Builds the tableau at time zero. Loop is allowed,
  fairness are taken into account]

  Description        []

  SideEffects        []

  SeeAlso            [BmcInt_Tableau_GetAtTime]

******************************************************************************/
static be_ptr
Bmc_TableauLTL_GetSingleLoopWithFairness(const Bmc_Fsm_ptr be_fsm,
					 const node_ptr ltl_wff,
					 const int k, const int l)
{
  be_ptr fairness = Bmc_Model_GetFairness(be_fsm, k, l);
  be_ptr tableau = BmcInt_Tableau_GetAtTime(Bmc_Fsm_GetVarsManager(be_fsm),
					    ltl_wff, 0, k, l);
  return Be_And(Bmc_VarsMgr_GetBeMgr(Bmc_Fsm_GetVarsManager(be_fsm)),
		tableau, fairness);
}


/**Function********************************************************************

  Synopsis           [Builds the tableau at time zero. Loop is allowed,
  fairness are taken into account]

  Description        []

  SideEffects        []

  SeeAlso            [BmcInt_Tableau_GetAtTime]

******************************************************************************/
static be_ptr
Bmc_TableauLTL_GetSingleLoop (const Bmc_Fsm_ptr be_fsm,
			      const node_ptr ltl_wff, const int k, const int l)
{
  be_ptr loopback = Bmc_Tableau_GetLoopCondition(Bmc_Fsm_GetVarsManager(be_fsm), 
						 k, l);

  be_ptr tableau  = Bmc_TableauLTL_GetSingleLoopWithFairness(be_fsm, ltl_wff, 
							     k, l);

  return Be_And(Bmc_VarsMgr_GetBeMgr(Bmc_Fsm_GetVarsManager(be_fsm)), 
		loopback, tableau);
}


/**Function********************************************************************

  Synopsis           [Builds tableau without loop at time zero, taking into
  account of fairness]

  Description        [Fairness evaluate to true if there are not fairness
  in the model, otherwise them evaluate to false because of no loop]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauLTL_GetNoLoop(const Bmc_Fsm_ptr be_fsm,
                         const node_ptr ltl_wff, const int k)
{
  VarsMgr_ptr vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  be_ptr fairness_k = Bmc_Model_GetFairness(be_fsm, k, Bmc_Utils_GetNoLoopback());

  /* lazy evaluation: */
  if ( Be_IsFalse(Bmc_VarsMgr_GetBeMgr(vars_mgr), fairness_k) ) {
    return fairness_k;
  }
  else {
    be_ptr tableau_k = BmcInt_Tableau_GetAtTime(vars_mgr, ltl_wff, 0, k,
						Bmc_Utils_GetNoLoopback());
    return Be_And(Bmc_VarsMgr_GetBeMgr(vars_mgr), fairness_k, tableau_k);
  }
}


/**Function********************************************************************

  Synopsis           [Builds tableau for all possible loops in \[l, k\[,
  taking into account of fairness]

  Description        [Each tableau takes into account of fairnesses relative
  to its step. All tableau are collected together into a disjunctive form.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauLTL_GetAllLoops(const Bmc_Fsm_ptr be_fsm,
			   const node_ptr ltl_wff, const int k, const int l)
{
  VarsMgr_ptr vars_mgr  = Bmc_Fsm_GetVarsManager(be_fsm);
  Be_Manager_ptr be_mgr = Bmc_VarsMgr_GetBeMgr(vars_mgr);
  be_ptr result = Be_Falsity(be_mgr);
  int j;

  /* asserts on l, k compatibility */
  nusmv_assert(!Bmc_Utils_IsNoLoopback(l));
  nusmv_assert(l < k);

  for (j = l; j < k; ++j) {
    be_ptr tableau_k_j = 
      Bmc_TableauLTL_GetSingleLoopWithFairness(be_fsm, ltl_wff, k, j);
    be_ptr loop_k_j = Bmc_Tableau_GetLoopCondition(vars_mgr, k, j);

    /* tableau + loopback + fairness at step k with loop at j = k */
    be_ptr tableau_LP_k_j = Be_And(be_mgr, tableau_k_j, loop_k_j);
    
    /* Accumulates the result: */
    result = Be_Or(be_mgr, result, tableau_LP_k_j);

    if (Be_IsTrue(be_mgr, result))  break; /* laziness: */
  }

  return result;
}


/**Function********************************************************************

  Synopsis           [Builds tableau for all possible loops in \[l, k\], in
  the particular case in which depth is 1. This function takes into account
  of fairness]

  Description        [Builds the tableau in the case depth==1 as suggested
  by R. Sebastiani]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauLTL_GetAllLoopsDepth1(const Bmc_Fsm_ptr be_fsm,
				 const node_ptr ltl_wff, const int k)
{
  be_ptr result=NULL;

  VarsMgr_ptr vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);

  switch (node_get_type(ltl_wff)) {
  case OP_FUTURE:
  case UNTIL:
    result = Be_Falsity(Bmc_VarsMgr_GetBeMgr(vars_mgr));
    break;

  case OP_NEXT:
    if (k==0) {
      result = 
	Be_And( Bmc_VarsMgr_GetBeMgr(vars_mgr),
		Bmc_TableauLTL_GetSingleLoopWithFairness(be_fsm, car(ltl_wff), 
							 k, 0),
		Bmc_Tableau_GetLoopCondition(vars_mgr, k, 0) );
    }
    else result = Be_Falsity(Bmc_VarsMgr_GetBeMgr(vars_mgr));
    break;

  case RELEASES:                        /* RELEASES     */
    result = Be_And( Bmc_VarsMgr_GetBeMgr(vars_mgr),
		     Bmc_Tableau_GetAllLoopsDisjunction(vars_mgr, k),
		     bmc_tableauGetGloballyAtTime(vars_mgr,
						  cdr(ltl_wff), 0, k, 0) );
    break;

  case TRUEEXP:
  case FALSEEXP:
  case DOT:
  case NOT:
    internal_error("Unexpected formula with depth zero had been found.\n");

  default:
    result = Be_And( Bmc_VarsMgr_GetBeMgr(vars_mgr),
		     Bmc_Tableau_GetAllLoopsDisjunction(vars_mgr, k),
		     BmcInt_Tableau_GetAtTime(vars_mgr, ltl_wff, 0, k, 0) );
  }

  nusmv_assert(result != NULL); /* result must exist */
  return result;
}



/* ====================================================================== */
/*                       Tableaux for PLTL :                              */
/* ====================================================================== */

/**Function********************************************************************

  Synopsis           [Returns the tableau for a PLTL formula on a bounded path
                      of length k, reasoning on fairness conditions as well.  ]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauPLTL_GetNoLoop(const Bmc_Fsm_ptr be_fsm,
                          const node_ptr ltl_wff, const int k)
{
 int noLoopBack         = Bmc_Utils_GetNoLoopback();

 VarsMgr_ptr vars_mgr   = Bmc_Fsm_GetVarsManager(be_fsm);

 Be_Manager_ptr be_mgr = Bmc_VarsMgr_GetBeMgr(vars_mgr);

 be_ptr   fairness_k   = Bmc_Model_GetFairness(be_fsm, k, noLoopBack);

 be_ptr    tableau_k   = Be_IsFalse(be_mgr, fairness_k) ?
   Be_Falsity(be_mgr) :
   Bmc_TableauPLTL_GetTableau(vars_mgr, ltl_wff, k, noLoopBack);

 return tableau_k;
}


/**Function********************************************************************

  Synopsis           [Returns the tableau for a PLTL formula on a (k,l)-loop,
                      conjuncted with both fairness conditions and the loop
                      condition on time steps k and l.]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauPLTL_GetSingleLoop (const Bmc_Fsm_ptr be_fsm,
                               const node_ptr ltl_wff, 
			       const int k, const int l)
{
  VarsMgr_ptr   vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  Be_Manager_ptr be_mgr = Bmc_VarsMgr_GetBeMgr(vars_mgr);

  be_ptr loopback = Bmc_Tableau_GetLoopCondition(vars_mgr, k, l);
  be_ptr fairness = Bmc_Model_GetFairness(be_fsm, k, l);
  be_ptr  tableau = Bmc_TableauPLTL_GetTableau(vars_mgr, ltl_wff, k, l);

  return Be_And(be_mgr, loopback, Be_And(be_mgr,fairness,tableau));
}


/**Function********************************************************************

  Synopsis           [Returns the conjunction of the single-loop tableaux for
                      all possible (k,l)-loops for a fixed k. Each single-loop
                      tableau takes into account of both fairness constraints
                      and loop condition.]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauPLTL_GetAllLoops(const Bmc_Fsm_ptr be_fsm,
                            const node_ptr ltl_wff,
                            const int k, const int startFromL)
{
  int l;

  VarsMgr_ptr   vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  Be_Manager_ptr be_mgr = Bmc_VarsMgr_GetBeMgr(vars_mgr);
  be_ptr result = Be_Falsity(be_mgr);


  for (l=startFromL; l<k; l++) {
    be_ptr tableau_k_l = Bmc_TableauPLTL_GetSingleLoop(be_fsm, ltl_wff, k, l);

    if (!Be_IsFalse(be_mgr, tableau_k_l)) {
      result = Be_Or(be_mgr, result, tableau_k_l);
    }
  }

 return result;
}


/**Function********************************************************************

  Synopsis           [Builds tableau for all possible (k,l)-loops for a
                      fixed k, in the particular case depth==1.
                      This function takes into account of fairness.]

  Description        [Builds the tableau in the case depth==1 as suggested
                      by R. Sebastiani.]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
static be_ptr
Bmc_TableauPLTL_GetAllLoopsDepth1(const Bmc_Fsm_ptr be_fsm,
                                  const node_ptr pltl_wff, const int k)
{
   be_ptr result;
   VarsMgr_ptr   vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
   Be_Manager_ptr  beMgr = Bmc_VarsMgr_GetBeMgr(vars_mgr);
   int nodeType  = node_get_type(pltl_wff);

   switch (nodeType) {

   /* A formula "Fg" (or "fUg") with depth one is such that g is a purely
      propositional formula.
      As long as g is propositional, the whole formula is not satisfiable,
      provided it already failed to be satisfied in the "no-loop" case. In
      fact, g does not exhibits newer behaviours throught loops, so it
      generates equivalent (admittedly unsatisfiable) formulas on every
      loop path.                                                             */
   case    UNTIL:
   case OP_FUTURE:
     result = Be_Falsity(beMgr);
     break;

   /* A formula "Xg" with depth one is such that g is a purely propositional
      formula. The standard one-loop function can be used to check the k=1
      case. Conversely, when k>1 we are ensured by the underlying BMC mechanism
      that the case k=1 has already been considered and that it generated an
      unsatisfiable formula. As a consequence of the propositional nature of g
      we can immediately conclude that every k-loop with k>1 generates an
      unsatisfiable formula as well. */
   case OP_NEXT:
     result = (k>1)? Be_Falsity(beMgr) : 
       Bmc_TableauPLTL_GetSingleLoop(be_fsm,pltl_wff,k,0);
     break;

   /* A formula "fRg" with depth one is such that g is a purely propositional
      formula, which has to hold at the time instant when "fRg" is evaluated,
      due to the semantics of "fRg". So, "g" has to hold everywhere, and it
      suffices to check for the validity of this condition.                  */

   case RELEASES:
     result = 
       Be_And(beMgr,
	      Bmc_Tableau_GetAllLoopsDisjunction(vars_mgr, k),
	      Bmc_TableauPLTL_GetAllTimeTableau(vars_mgr, cdr(pltl_wff), k));

   /* It can be proved that any depth-one PLTL formula is such that its
      tableau on (k-l)-loops does not depend on the value of l. So, the tableau
      itself is a common factor w.r.t. the OR-AND formula representing the set
      of all possible loopbacks, provided k is fixed. Thank to the distributive
      property of the logical AND w.r.t the OR, we are allowed to build the
      tableau once, and then put it together with the disjunction of all the
      loop conditions on k-loops.                                            */
   default:
     result = Be_And(beMgr,
		     Bmc_Tableau_GetAllLoopsDisjunction(vars_mgr, k),
		     Bmc_TableauPLTL_GetTableau(vars_mgr,pltl_wff, k, 0));
   }
   nusmv_assert(result != NULL);

   return result;
}


/**Function********************************************************************

  Synopsis           [Builds a tableau that constraints state k to be equal to
                      state l. This is the condition for a path of length (k+1)
                      to represent a (k-l)loop (new semantics).]

  Description        [State l and state k are forced to represent the same
                      state by conjuncting the if-and-only-if conditions
                      {Vil<->Vik} between Vil (variable i at time l) and Vik
                      (variable i at time k) for each variable Vi.]

  SideEffects        []

  SeeAlso            [Bmc_Tableau_GetAllLoopsDisjunction]

******************************************************************************/
static be_ptr
Bmc_Tableau_GetLoopCondition(const VarsMgr_ptr vars_mgr, const int k, const int l)
{
 be_ptr tableau_iff_constraints = Be_Truth(Bmc_VarsMgr_GetBeMgr(vars_mgr));

 int v;
 int numVariables = Bmc_VarsMgr_GetNum(vars_mgr);

 nusmv_assert(l < k);

 for (v=0; v<numVariables; v++) {
   tableau_iff_constraints =
     Be_And( Bmc_VarsMgr_GetBeMgr(vars_mgr), tableau_iff_constraints,
	     Be_Iff(Bmc_VarsMgr_GetBeMgr(vars_mgr),
		    Bmc_VarsMgr_Index2Timed(vars_mgr, v, l),
		    Bmc_VarsMgr_Index2Timed(vars_mgr, v, k)) );
 }

 return tableau_iff_constraints;
}


/**Function********************************************************************

  Synopsis           [Builds the disjunction of all the loops conditions
                      for (k-l)-loops with l in \[0, k\[]

  Description        [Builds a formula which is a disjunction over all the
                      loop conditions on k-loops, with l=0,1,...,k-1.]

  SideEffects        []

  SeeAlso            [Bmc_Tableau_GetLoopCondition]

******************************************************************************/
static be_ptr
Bmc_Tableau_GetAllLoopsDisjunction(const VarsMgr_ptr vars_mgr, const int k)
{
  be_ptr result = Be_Falsity(Bmc_VarsMgr_GetBeMgr(vars_mgr));

  int l;
  for (l=0; l<k; l++) {
    result = Be_Or(Bmc_VarsMgr_GetBeMgr(vars_mgr),
		   Bmc_Tableau_GetLoopCondition(vars_mgr, k, l),
		   result);
  }

  return result;
}


/**Function********************************************************************

  Synopsis           [ Checks wether a formula contains only future operators ]

  Description        []

  SideEffects        []

  SeeAlso            []
******************************************************************************/
boolean isPureFuture(const node_ptr pltl_wff)
{
 int nodeType = node_get_type(pltl_wff);

 if (isVariable(nodeType) || isConstantExpr(nodeType))  return true;
 else if (isPastOp(nodeType))  return false;
 else if (isBinaryOp(nodeType)) {
   return isPureFuture(car(pltl_wff)) && isPureFuture(cdr(pltl_wff));
 }
 else  return isPureFuture(car(pltl_wff));
}
