/**CFile***********************************************************************

  FileName    [bmcSat.c]

  PackageName [bmc]

  Synopsis    [The bmc-sat gateway implementation file]

  Description []

  SeeAlso     [the sat package]

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``bmc'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst and University of Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcSat.h"

#include "bmcInt.h" /* for 'options' */

#include "be.h"
#include "sim.h"
#include "mc.h" /* for 'eval_spec' */
#include "grammar.h" /* for constants */

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/**Struct*********************************************************************

  Synopsis           [Private declaration for class Bmc_SatTrace]
  Description        []
  SeeAlso            []

******************************************************************************/
typedef struct Bmc_SatTrace_TAG {
  node_ptr smvModel; /* the symbolic model */
  Sat_SatResult_ptr satResult; /* the original sover outcome */
  Bmc_Problem_ptr bmcProb; /* the original problem */
  int smvModel_k; /* the trace length (for lazy evaluation) */
  int cnfMaxVarIdx;  /* the maximum CNF variable index (for the explainer) */ 

} Bmc_SatTrace;


/**Struct*********************************************************************

  Synopsis           [Private declaration for class Bmc_Problem]
  Description        []
  SeeAlso            []

******************************************************************************/
typedef struct Bmc_Problem_TAG {
  be_ptr beProb;        /* the "original" problem, in be format */
  Be_Cnf_ptr cnfProb;   /* the corresponding CNF problem */
} Bmc_Problem;



/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static node_ptr
bmc_sat_trace_prop2symb ARGS((const Bmc_SatTrace_ptr self, 
			      const VarsMgr_ptr vars_mgr)); 

static void 
bmc_sat_trace_calculateSymbModel ARGS((const Bmc_SatTrace_ptr self,  
				       const VarsMgr_ptr vars_mgr, 
				       const int k));

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Calls the specified solver with the given problem, 
  and returns a sat result if there is any]

  Description        [The returned result MUST be deleted by the 
  calling function.]

  SideEffects        [A Sat_SatResult_ptr will be created, 
  and the calling function must delete it.]

  SeeAlso            [Sat_SatSolver, Sat_SatResult_ptr]

******************************************************************************/
Sat_SatResult_ptr Bmc_Sat_SolveProblem(const VarsMgr_ptr vars_mgr, 
				       Sat_SatSolver_ptr solver, 
				       const Bmc_Problem_ptr bmcProb, 
				       const int k)
{
  Sat_SatResult_ptr result = Sat_SatResult_Create();

  be_ptr beProb = Bmc_Problem_GetBeProblem(bmcProb);
  if (Be_IsTrue(Bmc_VarsMgr_GetBeMgr(vars_mgr), beProb)) {
    Sat_SatResult_SetSatisfiable(result);
    
    if (opt_verbose_level_gt(options, 0)) {    
      fprintf(nusmv_stderr, 
	 "The given problem has been evaluated as the constant value true.\n");
    }
  }
  else if (Be_IsFalse(Bmc_VarsMgr_GetBeMgr(vars_mgr), beProb)) {
    Sat_SatResult_SetUnsatisfiable(result);

    if (opt_verbose_level_gt(options, 0)) {    
      fprintf(nusmv_stdout, 
        "The given problem has been evaluated as the constant value false.\n");
    }
  }
  else {
    /*************************************************************
     * CALL SIM SOLVER INTERNALLY 
     * Using default parameters, which are hidden here
     *************************************************************/
    Sat_SatSolver_Solve(solver, Bmc_Problem_GetCnf(bmcProb), result);

    /*************************************************************
     * PROCESS SOLVER RESULTS
     *************************************************************/
    if (Sat_SatResult_IsError(result)) {
      fprintf( nusmv_stdout, "Solver: UNRECOVERABLE INTERNAL ERROR\n" );
    }
    else if (Sat_SatResult_IsUnsatisfiable(result)) {
      if (opt_verbose_level_gt(options, 0)) {
	fprintf( nusmv_stdout, "Solver: given problem is unsatisfiable\n" );
      }
    }
    else {
      /* problem here is satisfiable */
      nusmv_assert(Sat_SatResult_IsSatisfiable(result));
      if (opt_verbose_level_gt(options, 0)) {
	fprintf( nusmv_stdout, "Solver: given problem is satisfiable\n" );
      }
    }
  }

  return result;
}


/**Function********************************************************************

  Synopsis           [Calls the specified solver with the given invar problem, 
  and returns a sat result if there is any]

  Description        []

  SideEffects        [result will change]
  
  SeeAlso            []

******************************************************************************/
Sat_SatResult_ptr Bmc_Sat_SolveInvarProblem(const VarsMgr_ptr vars_mgr, 
					    Sat_SatSolver_ptr solver, 
					    const Bmc_Problem_ptr bmcProb)
{
  return Bmc_Sat_SolveProblem(vars_mgr, solver, bmcProb, 1);
}



/* ====================================================================== */
/* Bmc_Problem class methods:                                             */
/* ====================================================================== */


/**Function********************************************************************

  Synopsis           [Class Bmc_Problem constructor]

  Description        []

  SideEffects        []
  
  SeeAlso            [Bmc_Problem_Delete]

******************************************************************************/
Bmc_Problem_ptr Bmc_Problem_Create(const VarsMgr_ptr vars_mgr, 
				   const be_ptr prob)
{
  Bmc_Problem_ptr self = ALLOC(Bmc_Problem, 1);
  nusmv_assert(self != NULL);

  self->beProb = prob; /* keeps memory of the original be problem */

  /* performs the cnf conversion: */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "\nConverting the BE problem into CNF problem...\n");
  }
  
  self->cnfProb = Be_ConvertToCnf(Bmc_VarsMgr_GetBeMgr(vars_mgr), 
				  self->beProb);

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, " Conversion returned maximum variable index = %d\n",
	    Be_Cnf_GetMaxVarIndex(self->cnfProb));
    fprintf(nusmv_stderr, " Length of list of clauses = %d\n", 
	    Be_Cnf_GetClausesNumber(self->cnfProb));
    fprintf(nusmv_stderr, " Length of list of variables = %d\n", 
	    Be_Cnf_GetVarsNumber(self->cnfProb));
  }
  
  return self;
}


/**Function********************************************************************

  Synopsis           [Bmc_Problem destructor]

  Description        [Call when you no longer need the Bmc_Problem instance.]

  SideEffects        []
  
  SeeAlso            [Bmc_Problem_Create]

******************************************************************************/
void Bmc_Problem_Delete(Bmc_Problem_ptr self)
{
  nusmv_assert(self != NULL);
  Be_Cnf_Delete(self->cnfProb);
  FREE(self);
}


/**Function********************************************************************

  Synopsis           [Accessor method to retrieve the original problem in 
  BE format]

  Description        [The returned be is the same you used during instance 
  construction, by passing it to Bmc_Problem_Create]

  SideEffects        []
  
  SeeAlso            []

******************************************************************************/
be_ptr Bmc_Problem_GetBeProblem(const Bmc_Problem_ptr self)
{
  return self->beProb;
}


/**Function********************************************************************

  Synopsis           [Returns the problem in CNF format]

  Description        [You do not need to delete the returned 
  Be_Cnf instance. If you delete the Bmc_Problem instance had returned 
  a Be_Cnf instance by calling this method, also the returned Be_Cnf instance 
  will become invalid.]

  SideEffects        []
  
  SeeAlso            []

******************************************************************************/
Be_Cnf_ptr Bmc_Problem_GetCnf(const Bmc_Problem_ptr self)
{
  return self->cnfProb;
}



/* ====================================================================== */
/* Bmc_SatTrace class methods:                                            */
/* ====================================================================== */

/**Function********************************************************************

  Synopsis           [Class Bmc_SatTrace constructor]

  Description        [The method gets a satisfiable Sat_SatResult instance.
  The ownership of both arguments bmcProb and satResult remanins to the 
  caller (i.e. the Bmc_SatTrace destroyer does not delete also these 
  instances.)]

  SideEffects        []
  
  SeeAlso            [Bmc_SatTrace_Delete]

******************************************************************************/
Bmc_SatTrace_ptr Bmc_SatTrace_Create(Bmc_Problem_ptr bmcProb, 
				     Sat_SatResult_ptr satResult)
{
  Bmc_SatTrace_ptr self = ALLOC(Bmc_SatTrace, 1);
  nusmv_assert(self != NULL);

  /* are you trying to call me with a non-satisfiable problem? */
  nusmv_assert(Sat_SatResult_IsSatisfiable(satResult)); 
  
  self->smvModel     = new_list(); 
  self->satResult    = satResult;
  self->bmcProb      = bmcProb;
  self->smvModel_k   = 0;
  self->cnfMaxVarIdx = 0;
  
  return self;
}


/**Function********************************************************************

  Synopsis           [Class Bmc_SatTrace destructor]

  Description        [Call when you no longer need the given instance]

  SideEffects        [self will be invalidated]
  
  SeeAlso            []

******************************************************************************/
void Bmc_SatTrace_Delete(Bmc_SatTrace_ptr* self_ref) 
{
  free_list((*self_ref)->smvModel);
  FREE(*self_ref);
  *self_ref = NULL;
}


/**Function********************************************************************

  Synopsis           [Returns the bdd-based symbolic model]

  Description        [Returns the symbolic counterexample with max length k. 
  The VarMgr instance is required by the conversion routine. 
  Conversion is performed with memoizing, so you can call efficiently call 
  this method more than one time.]

  SideEffects        [self's internal state can change]
  
  SeeAlso            []

******************************************************************************/
node_ptr Bmc_SatTrace_GetSymbModel(const Bmc_SatTrace_ptr self,  
				   const VarsMgr_ptr vars_mgr, 
				   const int k)
{
  bmc_sat_trace_calculateSymbModel(self, vars_mgr, k);
  return self->smvModel;
}


/**Function********************************************************************

  Synopsis           [As Bmc_SatTrace_GetSymbModel, but for invariants]

  Description        []

  SideEffects        []
  
  SeeAlso            []

******************************************************************************/
node_ptr Bmc_SatTrace_GetSymbModelInvar(const Bmc_SatTrace_ptr self,  
					const VarsMgr_ptr vars_mgr)
{
  bmc_sat_trace_calculateSymbModel(self, vars_mgr, 1);
  return self->smvModel;
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Private service for class Bmc_SatTrace methods]

  Description        []

  SideEffects        []
  
  SeeAlso            []

******************************************************************************/
static void 
bmc_sat_trace_calculateSymbModel(const Bmc_SatTrace_ptr self,  
				 const VarsMgr_ptr vars_mgr, 
				 const int k)
{
  be_ptr beProb = Bmc_Problem_GetBeProblem(self->bmcProb); 
  
  /* lazy evaluation on the original problem: */
  if (Be_IsTrue(Bmc_VarsMgr_GetBeMgr(vars_mgr), beProb) 
      || Be_IsFalse(Bmc_VarsMgr_GetBeMgr(vars_mgr), beProb)) {
    /* The counterexample explainer can seem smarter 
       (it can say "Is it a trivial problem?") */ 
    free_list(self->smvModel);
    self->smvModel = new_list(); 
  }
  else {
    /* lazy evaluation on already performed conversions: */
    if (is_list_empty(self->smvModel) || (self->smvModel_k != k)) {
      self->smvModel_k = k;
      free_list(self->smvModel);
      self->smvModel = bmc_sat_trace_prop2symb(self, vars_mgr);
      self->cnfMaxVarIdx = 
	Be_Cnf_GetMaxVarIndex(Bmc_Problem_GetCnf(self->bmcProb));
    }  
  }
}



/* 
*/
/**Function********************************************************************

  Synopsis           [Private service for Bmc_SatTrace methods]

  Description        [Example of mapping of STATE VARIABLES and BE VARIABLES: 
  <PRE>
    ----------------------------------------------------------
    VAR        x    y    x'   y'   x0   y0   x1   y1   x2   y2
    BE index  0    1    2    3    4    5    6    7    8    9  
    CNF index  #    1    2    3    4    5    6    7    8    9 
    Time         -2    |   -1   |     0   |     1   |     2   |
    Varindex   0    1  | 0    1 |  0    1 |  0    1 |  0    1 |
    </PRE>
    ]

  SideEffects        []
  
  SeeAlso            []

******************************************************************************/
static node_ptr
bmc_sat_trace_prop2symb(const Bmc_SatTrace_ptr self, 
			const VarsMgr_ptr vars_mgr) 
{
  int stateVarsNum = Bmc_VarsMgr_GetNum(vars_mgr);
  lsGen genLit;
  int lit;
  int** model; 
  int time;
  int varindex;
  node_ptr trace = new_list(); /* variable for trace, list of bdd states */

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, "The propositional to symbolic model converter says that \
\n the input CNF model is the following:\n (");
    genLit = lsStart(Sat_SatResult_GetPropModel(self->satResult));
    while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK) {
      fprintf(nusmv_stderr, "%d ", lit);
    }
    lsFinish(genLit);
    fprintf(nusmv_stderr, " )\n");
  }
  
  if (opt_verbose_level_gt(options, 0)) {
    fprintf( nusmv_stderr, 
	     "Converting propositional model into symbolic model\n" );
  }

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(nusmv_stderr, "Number of state variables: %d, ", stateVarsNum);
    fprintf(nusmv_stderr, "Time steps from 0 to %d\n", self->smvModel_k);
  }

  /* allocates memory for model, and initialize all locations
     (default value for any unassigned literal is 'false') */
  model = ALLOC(int*, (self->smvModel_k + 1));
  nusmv_assert(model != NULL);

  for(time = 0; time <= self->smvModel_k; ++time) {
    model[time] = ALLOC(int, stateVarsNum);
    nusmv_assert(model[time] != NULL);
    /* resets the allocated array: */
    for (varindex = 0; varindex < stateVarsNum; ++varindex) {
      model[time][varindex] = 0;
    }
  }

  /* scan list of literals */
  genLit = lsStart(Sat_SatResult_GetPropModel(self->satResult));
  while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK) {
    int bevar_idx = abs(lit);

    if (Bmc_VarsMgr_IsStateIndex(vars_mgr, bevar_idx, 
				 self->smvModel_k)) {
      /* here only state variables indexes are allowed */
      time = Bmc_VarsMgr_AbsIndex2Time(vars_mgr, bevar_idx); 
      varindex = Bmc_VarsMgr_AbsIndex2Index(vars_mgr, bevar_idx);
      model[time][varindex] = (lit > 0) ? 1 : 0;

      if (opt_verbose_level_gt(options, 1)) {
	fprintf(nusmv_stderr, "Literal %5d: ", lit);
	fprintf( nusmv_stderr, "model[%d][%d] = % d      ", time, varindex,
		 model[time][varindex] );
	print_node(nusmv_stderr, 
		   Bmc_VarsMgr_Index2State(vars_mgr, varindex));
	fprintf(nusmv_stderr, " == %d", model[time][varindex]);
	fprintf(nusmv_stderr, "\n");
      }
    }
  }

  lsFinish(genLit);

  if (opt_verbose_level_gt(options, 1)) {
	fprintf(nusmv_stderr, "\n");
  }

  /*************************************************************
   * BUILD COUNTEREXAMPLE AS SEQUENCE OF BDDs
   *************************************************************/
  for(time = 0; time <= self->smvModel_k; ++time) {
    node_ptr state_exp = new_node(TRUEEXP, Nil, Nil);
    
    if (opt_verbose_level_gt(options, 1)) {
      fprintf(nusmv_stderr, " --------------- Time %d --------------- \n", 
	      time);
    }
    
    for(varindex = 0; varindex < stateVarsNum; ++varindex) { 
      node_ptr var_exp;
      
      if(model[time][varindex] != 0) {
	var_exp = Bmc_VarsMgr_Index2State(vars_mgr, varindex);
      }
      else {
	var_exp = new_node(NOT, 
			   Bmc_VarsMgr_Index2State(vars_mgr, varindex), 
			   Nil);
      }
      state_exp = new_node(AND, var_exp, state_exp);
    } /* for cycle */
    
    {
      bdd_ptr state_bdd = (bdd_ptr) eval_spec( Prop_MasterGetBddFsm(), 
					       state_exp, Nil );
      if (opt_verbose_level_gt(options, 1)) {
	print_state(state_bdd, all_symbols, 0);
      }
      trace = cons((node_ptr)state_bdd, trace);
    }
  } /* end of for time */

  /* destroys previously allocated model array: */
  for(time = 0; time <= self->smvModel_k; ++time) {
    FREE(model[time]);
  }
  FREE(model);

  return reverse(trace);
}

