/**CFile***********************************************************************

  FileName    [bmcDimacs.c]

  PackageName [bmc]

  Synopsis    [Dimacs format dumping functionalities]

  Description [This module supplies services that dump a Bmc problem 
  into a file, in DIMACS format.]

  SeeAlso     []

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``bmc'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst and University of Trento. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcDimacs.h"
#include "bmcInt.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int 
bmc_dimacs_openDimacsFile ARGS((const char* filename, FILE** file_ref));


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Opens a new file named filename, than dumps the given 
  invar problem in DIMACS format]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Bmc_Dimacs_DumpInvarProblemFilename(const VarsMgr_ptr vars_mgr, 
					const Bmc_Problem_ptr bmcProb,
					const char* filename)
{
  FILE* file;
  int ret = bmc_dimacs_openDimacsFile(filename, &file);

  if (ret == 0) Bmc_Dimacs_DumpInvarProblem(vars_mgr, bmcProb, file);
  return ret;
}


/**Function********************************************************************

  Synopsis           [Opens a new file named filename, than dumps the given 
  LTL problem in DIMACS format]

  Description        []

  SideEffects        []

  SeeAlso            []

******************************************************************************/
int Bmc_Dimacs_DumpProblemFilename(const VarsMgr_ptr vars_mgr, 
				   const Bmc_Problem_ptr bmcProb,
				   const char* filename,  
				   const int k)
{
  FILE* file;
  int ret = bmc_dimacs_openDimacsFile(filename, &file);

  if (ret == 0) Bmc_Dimacs_DumpProblem(vars_mgr, bmcProb, k, file);
  return ret;
}


/**Function********************************************************************

  Synopsis           [Dumps the given invar problem in the given file]

  Description        [dimacsfile must be writable]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Bmc_Dimacs_DumpInvarProblem(const VarsMgr_ptr vars_mgr, 
				 const Bmc_Problem_ptr bmcProb,
				 FILE* dimacsfile)
{
  Bmc_Dimacs_DumpProblem(vars_mgr, bmcProb, 1, dimacsfile);
}


/**Function********************************************************************

  Synopsis           [Dumps the given LTL problem in the given file]

  Description        [dimacsfile must be writable]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void Bmc_Dimacs_DumpProblem(const VarsMgr_ptr vars_mgr, 
			    const Bmc_Problem_ptr bmcProb,
			    const int k, 
			    FILE* dimacsfile)
{
  int time;
  nusmv_assert(dimacsfile != NULL); 

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(stderr, "Starting dump of the CNF problem into DIMACS file... (problem length is %d)\n", k);
  }  

  /* Writes the readable mapping table as a comment: */
  fprintf(dimacsfile, "c BMC problem generated by NuSMV\n");
  fprintf(dimacsfile, "c Time steps from 0 to %d, %d State Variables\n",
	  k, Bmc_VarsMgr_GetNum(vars_mgr));
  fprintf(dimacsfile, "c Model to Dimacs Conversion Table\n");

  for (time = 0; time <= k; ++time) {
    int varindex;
    fprintf(dimacsfile, "c \nc @@@@@ Time %d\n", time);
    for (varindex = 0; varindex < Bmc_VarsMgr_GetNum(vars_mgr); ++varindex) {
      fprintf(dimacsfile, "c CNF variable %d => Time %d, Model Variable ",
	      Bmc_VarsMgr_Index2AbsIndex(vars_mgr, varindex, time), time);
      print_node(dimacsfile, Bmc_VarsMgr_Index2State(vars_mgr, varindex));
      fprintf(dimacsfile, "\n");
    }
  }
  fprintf(dimacsfile, "c \n");

  /* Actually writes the dimacs data: */
  {
    Be_Cnf_ptr cnf = Bmc_Problem_GetCnf(bmcProb);
    lsList cl = LS_NIL;  
    lsGen  genCl, genLit;
    int    lit = 0;
  
    fprintf(dimacsfile, "c Beginning of the DIMACS dumping\n");
    /* Prints the model variables as a "special" comment line: */
    fprintf(dimacsfile, "c model %d\n", Be_Cnf_GetVarsNumber(cnf));
    fprintf(dimacsfile, "c ");

    genLit = lsStart(Be_Cnf_GetVarsList(cnf));
    while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK) {
      fprintf(dimacsfile, "%d ", lit);
    }
    lsFinish(genLit);
    fprintf(dimacsfile, "0\n");    

    /* Prints the problem header. */
    fprintf(dimacsfile, "p cnf %d %d\n", 
	    Be_Cnf_GetMaxVarIndex(cnf), 
	    Be_Cnf_GetClausesNumber(cnf));

    /* Prints the clauses. */
    genCl = lsStart(Be_Cnf_GetClausesList(cnf));
    while (lsNext(genCl, (lsGeneric*) &cl, LS_NH) == LS_OK) {
      genLit = lsStart(cl);
      while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK) {
	fprintf(dimacsfile, "%d ", lit);
      }
      lsFinish(genLit);
      fprintf(dimacsfile, "0\n");
    }
    lsFinish(genCl);

    fprintf(dimacsfile, "c End of dimacs dumping\n");
  }

  if (opt_verbose_level_gt(options, 1)) {
    fprintf(stderr, "End of dump.\n");
  }  
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Open a file named filename and returns its descriptor]

  Description        [The file is opened with the writable attribute. The 
  file descriptor is returned via the file_ref parameter. Returns 0 if the 
  function succedeed, otherwise the function prints out a warning in the 
  standard output and returns 1.]

  SideEffects        [file_ref will change]

  SeeAlso            []

******************************************************************************/
static int bmc_dimacs_openDimacsFile(const char* filename, FILE** file_ref)
{
  int ret = 0;
  if (opt_verbose_level_gt(options, 1)) {
    fprintf(stderr, "\nOpening file '%s' for writing...\n", filename);
  }  
  
  *file_ref = fopen(filename, "w");
  if ((*file_ref) == NULL)  {
    fprintf(nusmv_stdout, 
	    "\n*************    WARNING    *************\
 \n An error has occured when writing the file \"%s\".\
 \n DIMACS dumping is not allowed.\
 \n*************  END WARNING  *************\n\n", filename);
    ret = 1;
  }
  
  return ret;
}
