/**CFile***********************************************************************

  FileName    [bmcConv.c]

  PackageName [bmc]

  Synopsis    [Convertion function of BE to corresponding BDD boolean 
  expression, and viceversa]

  Description [This implementation is still depedent on the rbc package]

  SeeAlso     []

  Author      [Alessandro Cimatti and Lorenzo Delana]

  Copyright   [
  This file is part of the ``bmc'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst and University of Trento. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcConv.h"

#include "grammar.h" /* for NUMBER et similia */
#include "rbcInt.h" /* for RBCVAR et similia */ 
#include "array.h"

#include "bmcWff.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Be2bexpDfsData]

  Description [The data structure used for DFS traversal of RBC]

  SeeAlso     []

******************************************************************************/
typedef struct Be2bexpDfsData_TAG {
  VarsMgr_ptr vars_mgr; 
  array_t* stack;
  int head;
} Be2bexpDfsData;


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/* Primitives for stack used in BE traversal */
static void Be2bexpDfsData_push ARGS((Be2bexpDfsData*, node_ptr));
static node_ptr Be2bexpDfsData_head ARGS((Be2bexpDfsData*));
static node_ptr Be2bexpDfsData_pop  ARGS((Be2bexpDfsData*));

/* BE traversal functions */
static int  Be2bexp_Set   ARGS((be_ptr be, char* Be2bexpData, int sign));
static void Be2bexp_First ARGS((be_ptr be, char* Be2bexpData, int sign));
static void Be2bexp_Back  ARGS((be_ptr be, char* Be2bexpData, int sign));
static void Be2bexp_Last  ARGS((be_ptr be, char* Be2bexpData, int sign));

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Given a be, constructs the corresponding boolean
  expression]

  Description        [Descends the structure of the BE with dag-level 
  primitives. Uses the vars manager to perform all time-related operations. ]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
node_ptr Bmc_Conv_Be2Bexp(VarsMgr_ptr vars_mgr, be_ptr be)
{
  Dag_DfsFunctions_t Be2bexpFunctions;
  Be2bexpDfsData Be2bexpData;
  node_ptr ret;

  /* Cleaning the user fields. */
  Dag_Dfs(be, &dag_DfsClean, NIL(char));

  /* Setting up the DFS functions. */
  Be2bexpFunctions.Set        = Be2bexp_Set;
  Be2bexpFunctions.FirstVisit = Be2bexp_First;
  Be2bexpFunctions.BackVisit  = Be2bexp_Back;
  Be2bexpFunctions.LastVisit  = Be2bexp_Last;
 
  /* Setting up the DFS data. */
  /* :TODO??: optimizations on the quantity of nodes */
  Be2bexpData.vars_mgr = vars_mgr;  
  Be2bexpData.stack = array_alloc(node_ptr, 10);
  Be2bexpData.head = -1;

  /* Calling DFS on f. */
  Dag_Dfs(be, &Be2bexpFunctions, (char*)(&Be2bexpData));
  ret = Be2bexpDfsData_head(&Be2bexpData);

  array_free(Be2bexpData.stack);

  return ret;
}



/**Function********************************************************************

  Synopsis           [<b>Converts</b> given <b>boolean expression</b> into
  correspondent <b>reduced boolean circuit</b>]

  Description        [Uses the vars manager to perform all 
  time-related operations.]

  SideEffects        [be hash may change]

  SeeAlso            []

******************************************************************************/
be_ptr Bmc_Conv_Bexp2Be(VarsMgr_ptr vars_mgr, node_ptr bexp)
{
  /* if given expression is Nil, return truth be */
  /* RC: Nil value can be used in AND sequences, so a true value must
     be returned */
  if (bexp == Nil) {
    return Be_Truth(Bmc_VarsMgr_GetBeMgr(vars_mgr));
  }

  switch (node_get_type(bexp)) {
  case TRUEEXP:
    return Be_Truth(Bmc_VarsMgr_GetBeMgr(vars_mgr));

  case FALSEEXP:
    return Be_Falsity(Bmc_VarsMgr_GetBeMgr(vars_mgr));

  case NEXT:    
    /* converts NEXT arg (a state variable) into correspondent next variable
       and return this be variable */
    return Bmc_VarsMgr_State2Next(vars_mgr, car(bexp));

  case NOT:     
    /* NOT arg is converted into an be, and its negation is returned */
    return Be_Not( Bmc_VarsMgr_GetBeMgr(vars_mgr), 
		   Bmc_Conv_Bexp2Be(vars_mgr, car(bexp)) );

  case CONS: 
  case AND:  
    return Be_And( Bmc_VarsMgr_GetBeMgr(vars_mgr),  
		   Bmc_Conv_Bexp2Be(vars_mgr, car(bexp)),
		   Bmc_Conv_Bexp2Be(vars_mgr, cdr(bexp)) );

  case OR:   
    return Be_Or( Bmc_VarsMgr_GetBeMgr(vars_mgr), 
		  Bmc_Conv_Bexp2Be(vars_mgr, car(bexp)),
		  Bmc_Conv_Bexp2Be(vars_mgr, cdr(bexp)) );

  case IFF:  
    /* converts IFF args into two BEs, and return an IFF BE with converted
       BEs as childs */
    return Be_Iff( Bmc_VarsMgr_GetBeMgr(vars_mgr), 
		   Bmc_Conv_Bexp2Be(vars_mgr, car(bexp)),
		   Bmc_Conv_Bexp2Be(vars_mgr, cdr(bexp)) );

  case IMPLIES:
    /* convert IMPLIES args into two BEs, and return the IMPLIES BE with
       converted BEs as childs */
    return Be_Implies( Bmc_VarsMgr_GetBeMgr(vars_mgr),
		       Bmc_Conv_Bexp2Be(vars_mgr, car(bexp)),
		       Bmc_Conv_Bexp2Be(vars_mgr, cdr(bexp)) );

  case EQUAL:  
  case ASSIGN: 
    /* converts EQUAL and ASSIGN args into two BEs, and return an IFF BE
       with converted BEs as childs */
    return Be_Iff( Bmc_VarsMgr_GetBeMgr(vars_mgr),
		   Bmc_Conv_Bexp2Be(vars_mgr, car(bexp)),
		   Bmc_Conv_Bexp2Be(vars_mgr, cdr(bexp)) );
    
  case CASE:   
    /* converts "if"-"then"-"else" args of CASE into three BEs, and return
       a BE If-Then-Else with converted BEs as childs */
    return Be_Ite( Bmc_VarsMgr_GetBeMgr(vars_mgr), 
		   Bmc_Conv_Bexp2Be(vars_mgr, caar(bexp)),
		   Bmc_Conv_Bexp2Be(vars_mgr, cdar(bexp)),
		   Bmc_Conv_Bexp2Be(vars_mgr, cdr(bexp)) );

  case DOT:                              /* VARIABLE */
    return(Bmc_VarsMgr_State2Curr(vars_mgr, bexp));

  case ARRAY:                            /* VARIABLE */
  case ATOM:                             /* VARIABLE */
    internal_error("Not DOT node as variable has been found!\n");
    nusmv_assert(false); /* execution should never be here */

  case NUMBER:
  default:     
    internal_error("Bmc_Conv_Bexp2Be: Unexpected case value. Node type = %d\n",
		   node_get_type(bexp));
    nusmv_assert(false); /* execution should never be here */
  }
}


/**Function********************************************************************

  Synopsis           [<b>Converts</b> given <b>boolean expressions list </b> 
  into correspondent <b>reduced boolean circuits list</b>]

  Description        []

  SideEffects        [be hash may change]

  SeeAlso            []

******************************************************************************/
node_ptr Bmc_Conv_BexpList2BeList(VarsMgr_ptr vars_mgr, node_ptr bexp_list)
{
  if (bexp_list == Nil) 
  {
    return(Nil);
  }
  else 
  {
    return cons( (node_ptr)Bmc_Conv_Bexp2Be(vars_mgr, car(bexp_list)),
		 Bmc_Conv_BexpList2BeList(vars_mgr, cdr(bexp_list)) );
  }
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Sets a node into the stack]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void Be2bexpDfsData_push(Be2bexpDfsData* data, 
				  node_ptr value)
{
  (data -> head) ++;

  array_insert(node_ptr, data -> stack, data -> head, value);
}

/**Function********************************************************************

  Synopsis           [Be2bexpDfsData_head]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static node_ptr Be2bexpDfsData_head(Be2bexpDfsData* data)
{
  /* there is space available into the stack: */
  nusmv_assert((data -> head) != (-1));

  return (node_ptr) array_fetch(node_ptr, data->stack, data->head);
}

/**Function********************************************************************

  Synopsis           [Be2bexpDfsData_pop]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static node_ptr Be2bexpDfsData_pop(Be2bexpDfsData* data)
{
  node_ptr value = (node_ptr)Be2bexpDfsData_head(data);

  (data->head) --;

  return(value);
}

/**Function********************************************************************

  Synopsis           [Be2bexpSet]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static int Be2bexp_Set(be_ptr be, char* Be2bexpData, int sign)
{
  return -1;
}

/**Function********************************************************************

  Synopsis           [Be2bexpFirst]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void Be2bexp_First(be_ptr be, char* Be2bexpData, int sign)
{
  return;
}

/**Function********************************************************************

  Synopsis           [Be2bexp_Back]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void Be2bexp_Back(be_ptr be, char* Be2bexpData, int sign)
{
  return; 
}

/**Function********************************************************************

  Synopsis           [Be2bexp_Last]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void Be2bexp_Last(be_ptr be, char* Be2bexpData, int sign)
{
  int identifier = 0;
  int time, var_id;
  node_ptr left, right;
  Be_Manager_ptr be_mgr; 
  Rbc_t* rbc;

  VarsMgr_ptr vars_mgr = ((Be2bexpDfsData*)Be2bexpData)->vars_mgr;
  nusmv_assert(vars_mgr != NULL);

  /* :WARNING: This code strongly depends on the RBC structure */
  be_mgr = Bmc_VarsMgr_GetBeMgr(vars_mgr); 
  rbc = (Rbc_t*) Be_Manager_Be2Spec(be_mgr, be);

  switch (rbc->symbol) {
  case RBCTOP:
    if (sign == RBC_FALSE) {
      Be2bexpDfsData_push((Be2bexpDfsData*)Be2bexpData, Bmc_Wff_MkFalsity());
    }
    else {
      Be2bexpDfsData_push((Be2bexpDfsData*)Be2bexpData, Bmc_Wff_MkTruth());
    }
    break;    

  case RBCVAR:
    /* substitute the variable index, in the stack, with its correspondent
       state variable */
    
    time = Bmc_VarsMgr_AbsIndex2Time(vars_mgr, (int)(rbc->data));
    var_id = Bmc_VarsMgr_AbsIndex2Index(vars_mgr, (int)(rbc->data));

    if (sign == RBC_FALSE) {
      Be2bexpDfsData_push(
	     (Be2bexpDfsData*)Be2bexpData,
	     Bmc_Wff_MkNot( Bmc_Wff_MkXopNext(Bmc_VarsMgr_Index2State(vars_mgr, 
								      var_id), 
					      time) )
	     );
    }
    else {
      Be2bexpDfsData_push( (Be2bexpDfsData*)Be2bexpData,
			    Bmc_Wff_MkXopNext(Bmc_VarsMgr_Index2State(vars_mgr, 
								      var_id), 
					      time) );
    }
    break;
  case RBCAND:
  case RBCIFF:
    /* get the left bexp from the stack */
    right = Be2bexpDfsData_pop((Be2bexpDfsData*) Be2bexpData);

    /* get the right bexp from the stack */
    left = Be2bexpDfsData_pop((Be2bexpDfsData*) Be2bexpData);

    switch (rbc->symbol) {
    case RBCAND:
      identifier = AND;
      break;
    case RBCIFF:
      identifier = IFF;
      break;
    }

    if (sign == RBC_FALSE) {
      Be2bexpDfsData_push( (Be2bexpDfsData*)Be2bexpData,
			   Bmc_Wff_MkNot(find_node(identifier, left, right)) );
    }
    else {
      Be2bexpDfsData_push( (Be2bexpDfsData*)Be2bexpData,
			    find_node(identifier, left, right) );
    }
    break;
    
  default:
    /* execution should never be here: */
    internal_error("rbc->symbol had an invalid value: %d\n", rbc->symbol);
  }
}
