/**CFile***********************************************************************

  FileName    [bmcCmd.c]

  PackageName [bmc]

  Synopsis    [Bmc.Cmd module]

  Description [This module contains all the bmc commands implementation. 
  Options parsing and checking is performed here, than the high-level Bmc 
  layer is called]

  SeeAlso     [bmcPkg.c, bmcBmc.c]

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``bmc'' package of NuSMV version 2. 
  Copyright (C) 2000-2001 by ITC-irst and University of Trento. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcCmd.h"

#include "bmcInt.h"
#include "bmcBmc.h"
#include "bmcPkg.h"
#include "bmcUtils.h"

/* ---------------------------------------------------------------------- */
/* to be able to include 'prop.h'! */
#include "util.h"
#include "ucmd.h"
#include "array.h"
#include "node.h"
#include "dd.h"
#include "grammar.h"
#include "rbc.h"
#include "set.h"
#include "compile.h"
#include "opt.h"
#include "set.h"
#include "rbc.h"
#include "mc.h"
#include "ltl.h"
#include "cmd.h"
#include "prop.h"
/* ---------------------------------------------------------------------- */


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int UsageBmcSetup    ARGS((void));
static int UsageBmcSimulate ARGS((void));

static int UsageBmcCheckLtlSpec       ARGS((void));
static int UsageBmcCheckLtlSpecOnePb  ARGS((void));
static int UsageBmcGenLtlSpec         ARGS((void));
static int UsageBmcGenLtlSpecOnePb    ARGS((void));
static int UsageBmcCheckInvar         ARGS((void));
static int UsageBmcGenInvar           ARGS((void));

static outcome bmc_CommandsWith_k_l_OptionsHandling 
ARGS((const int argc, const char** argv, 
      Prop_Ptr* pProperty,
      Prop_Type type, 
      int* pb_length, 
      int* user_loop, 
      boolean* dimacsSpecified, 
      char* dimacsFilename, 
      const size_t dimacsFilename_len));


static outcome bmc_CommandsWithout_k_l_OptionsHandling
ARGS((const int argc, const char** argv, 
      Prop_Ptr* pProperty, 
      Prop_Type type, 
      boolean* dimacsSpecified, 
      char* dimacsFilename, 
      const size_t dimacsFilename_len));

/* error messages: */
static void bmcErrorInvalidProperty ARGS((const char* szFormula));
static void bmcErrorInvalidNumber ARGS((const char* szNumber));
static void bmcErrorInvalid_k_l ARGS((const int k, const int l));
static void bmcErrorPropertyAlreadySpecified ARGS((void));


/**AutomaticEnd***************************************************************/



/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [Initializes the bmc sub-system, and builds the model in
  a Boolean Expression format]

  Description        []

  SideEffects        [Overall the bmc system]

  SeeAlso            []

  CommandName        [bmc_setup] 	   

  CommandSynopsis    [Builds the model in a Boolean Epression format.]  

  CommandArguments   [\[-h\]]  

  CommandDescription [You must call this command before use any other 
  bmc-related command. Only one call per session is required.]  

******************************************************************************/
int Bmc_CommandBmcSetup(const int argc, const char** argv)
{
  /* processes the command options */ 
  int c;

  util_getopt_reset();
  while ((c = util_getopt((int)argc, (char**)argv, "h")) != EOF) {
    switch (c) {
    case 'h': return UsageBmcSetup();
    }
  }
  if (argc != 1) return UsageBmcSetup();

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
            "A model must be read before. Use the \"read_model\" command.\n");
    return 1;
  }

  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
            "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"encode_variables\" command.\n");
    return 1;
  }

  if (cmp_struct_get_build_bool_model(cmps) == 0) {
    fprintf(nusmv_stderr, "The model must be built before.\n");
    if (opt_verbose_level_gt(options, 1)) {
      fprintf(nusmv_stderr, "Use the \"build_bool_model\" command.\n");
    }
    return 1;
  }

  if (cmp_struct_get_bmc_setup(cmps)) {
    fprintf (nusmv_stderr, "A call to bmc_setup has already been done.\n");
    return 1;
  }

  /* This function does the actual work: */
  Bmc_Init();

  cmp_struct_set_bmc_setup(cmps);
  return 0;
}


/**Function********************************************************************

  Synopsis           [Usage string for Bmc_CommandBmcSetup]

  Description        []

  SideEffects        []

  SeeAlso            [Bmc_CommandBmcSetup]
******************************************************************************/
static int UsageBmcSetup (void)
{
  fprintf (nusmv_stderr, "usage: bmc_setup [-h]\n");
  fprintf (nusmv_stderr, "  -h \t\tPrints the command usage.\n");
  fprintf (nusmv_stderr, "   \t\tStarts up the BMC module and generates a Boolean Expression\n" 
	   "\t\twhich represents the model.\n");

  return 1;
}


/**Function********************************************************************

  Synopsis           [Bmc_CommandBmcSimulate generates a trace of the problem 
  represented from the simple path from 0 (zero) to k]

  Description        [Bmc_CommandBmcSimulate does not require a specification 
  to build the problem, because only the model is used to build it.]

  SideEffects        [None]

  SeeAlso            []

  CommandName        [bmc_simulate] 	   

  CommandSynopsis    [Generates a trace of the model from 0 (zero) to k]  

  CommandArguments   [\[-h | -k <length>\]]  

  CommandDescription [bmc_simulate does not require a specification 
  to build the problem, because only the model is used to build it. 
  The problem length is represented by the <i>-k</i> command parameter, 
  or by its default value stored in the environment variable 
  <i>bmc_length</i>.<BR>
  Command options:<p>
  <dl>
    <dt> <tt>-k <i>length</i></tt>
       <dd> <i>length</i> is the length of the generated simulation. <BR>
  </dl>
  ]  

******************************************************************************/
int Bmc_CommandBmcSimulate(const int argc, const char** argv)
{
  int c; /* for command line options */
  int k = get_bmc_pb_length(options);
 
  /* process command options */ 
  util_getopt_reset();
  while((c = util_getopt((int)argc, (char**)argv, "hk:")) != EOF) {
    switch (c) {
    case 'h':
      return UsageBmcSimulate();
      
    case 'k':
      {
	char* strNumber = util_strsav(util_optarg);
	if (util_str2int(strNumber, &k) != 0) {
 	  bmcErrorInvalidNumber(strNumber);
	  return 1;
	}
	if (k<0) {
 	  bmcErrorInvalidNumber(strNumber);
	  return 1;
	}
	break;  
      }
    } /* switch stmt */
  } /* while */ 
  
  if (argc != util_optind) {
    return UsageBmcSimulate();
  }
    
  /* make sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    
    return 1;
  }

  /* make sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }
  
  /* This function does the actual work: */
  Bmc_Simulate(Prop_MasterGetBeFsm(), k);

  return 0;
}


/**Function********************************************************************

  Synopsis           [Usage string for UsageBmcSimulate]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandBmcSimulate]

******************************************************************************/
static int UsageBmcSimulate (void)
{
  fprintf (nusmv_stderr, "usage: bmc_simulate [-h] [-k length]\n");
  fprintf (nusmv_stderr, "  -h \t\tPrints the command usage.\n");
  fprintf (nusmv_stderr, "  -k <length> \tSpecifies the simulation length\n" \
           "\t\tto be used when generating the simulated problem.\n");
  fprintf (nusmv_stderr, "\tGenerates a k-steps simulation using Bounded Model Checking.\n" \
	   "\tYou can specify k also by setting the variable bmc_length.\n");
  return 1;
}


/**Function********************************************************************

  Synopsis           [Checks the given LTL specification, or all LTL 
  specifications in the properties database if no formula is given]

  Description        [After command line processing this function calls
  the Bmc_GenSolveLtl to generate and solve all problems from 0 to k.
  Parameters are the maximum length and the loopback values.]

  SideEffects        [Properties database may change]

  SeeAlso            [Bmc_CommandCheckLtlSpecBmcOnePb, Bmc_GenSolveLtl]

  CommandName        [check_ltlspec_bmc] 	   

  CommandSynopsis    [Checks the given LTL specification, or all LTL 
  specifications if no formula is given. Checking parameters are the maximum 
  length and the loopback values]  

  CommandArguments   [\[-h | -n idx | -p "formula" \[IN context\]\] 
  \[-k max_length\] \[-l loopback\] \[-o filename\]]  

  CommandDescription [
  This command generates one or more problems, and calls 
  SAT solver for each one. Each problem is related to a specific problem 
  bound, which increases from zero (0) to the given maximum problem 
  length. Here "<i>length</i>" is the bound of the problem that system 
  is going to generate and/or solve. <BR>
  In this context the maximum problem bound is represented by the 
  <i>-k</i> command parameter, or by its default value stored in the 
  environment variable <i>bmc_length</i>.<BR>
  The single generated problem also depends on the "<i>loopback</i>"
  parameter you can explicitly specify by the <i>-l</i> option, or by its
  default value stored in the environment variable <i>bmc_loopback</i>. <BR>
  The property to be checked may be specified using the <i>-n idx</i> or  
  the <i>-p "formula"</i> options. <BR>
  If you need to generate a dimacs dump file of all generated problems, you 
  must use the option <i>-o "filename"</i>. <BR>  
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-n <i>index</i></tt>
       <dd> <i>index</i> is the numeric index of a valid LTL specification 
       formula actually located in the properties database. <BR>
    <dt> <tt>-p "formula" \[IN context\]</tt>
       <dd> Checks the <tt>formula</tt> specified on the command-line. <BR>
            <tt>context</tt> is the module instance name which the variables
            in <tt>formula</tt> must be evaluated in.
    <dt> <tt>-k <i>max_length</i></tt>
       <dd> <i>max_length</i> is the maximum problem bound must be reached. 
       Only natural number are valid values for this option. If no value 
       is given the environment variable <i>bmc_length</i> is considered 
       instead. 
    <dt> <tt>-l <i>loopback</i></tt>
       <dd> <i>loopback</i> value may be: <BR>
       - a natural number in (0, <i>max_length-1</i>). Positive sign ('+') can 
       be also used as prefix of the number. Any invalid combination of length
       and loopback will be skipped during the generation/solving process.<BR>
       - a negative number in (-1, -<i>bmc_length</i>). In this case 
       <i>loopback</i> is considered a value relative to <i>max_length</i>. 
       Any invalid combination of length and loopback will be skipped 
       during the generation/solving process.<BR>
       - the symbol 'X', which means "no loopback" <BR>
       - the symbol '*', which means "all possible loopback from zero to 
       <i>length-1</i>"
    <dt> <tt>-o <i>filename</i></tt>
       <dd> <i>filename</i> is the name of the dumped dimacs file. 
       It may contain special symbols which will be macro-expanded to form 
       the real file name. Possible symbols are: <BR>
       - @F: model name with path part <BR>
       - @f: model name without path part <BR>
       - @k: current problem bound <BR>
       - @l: current loopback value <BR>
       - @n: index of the currently processed formula in the properties 
       database <BR>
       - @@: the '@' character
  </dl>
  ]  

******************************************************************************/
int Bmc_CommandCheckLtlSpecBmc(const int argc, const char** argv)
{
  Prop_Ptr ltlprop=(Prop_Ptr)NULL;   /* The property being processed */
  outcome opt_handling_res; 
  boolean bCreateDimacs = false; /* create dimacs only if -o opt is specified */ 
  int k = get_bmc_pb_length(options);
  char dimacsFilename[DIMAX_FILENAME_MAXLEN]; 
  int relative_loop = 
    Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options), NULL);

  strncpy(dimacsFilename, get_bmc_dimacs_filename(options), 
	  DIMAX_FILENAME_MAXLEN); /* default value */

  /* ----------------------------------------------------------------------- */
  /* Options handling: */
  opt_handling_res = 
    bmc_CommandsWith_k_l_OptionsHandling(argc, argv, 
					 &ltlprop, Prop_Ltl,
					 &k, &relative_loop, 
					 &bCreateDimacs, dimacsFilename, 
					 sizeof(dimacsFilename)-1);
							  
  if (opt_handling_res == SUCCESS_REQUIRED_HELP) {
    return UsageBmcCheckLtlSpec();
  }
  if (opt_handling_res != SUCCESS)  return 1;
  /* ----------------------------------------------------------------------- */


  /* makes sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  /* makes sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }

  /* prepares the list of properties if no property was selected: */
  if (ltlprop == NIL_PTR(Prop_Ptr)) {
    lsList props = Prop_Db_GetPropsOfType(Prop_Ltl);
    lsGen  iterator; 
    Prop_Ptr prop;

    nusmv_assert(props != LS_NIL);

    lsForEachItem(props, iterator, prop) {
      Bmc_GenSolveLtl((Prop_Ptr)prop, k, relative_loop, true, 
		      (bCreateDimacs) ? GEN_DIMACS_SOLVE : SOLVE, 
		      dimacsFilename);
    }

    lsDestroy(props, NULL); /* the list is no longer needed */
  }
  else {
    /* its time to solve (a single property): */
    Bmc_GenSolveLtl(ltlprop, k, relative_loop, true, 
		    (bCreateDimacs) ? GEN_DIMACS_SOLVE : SOLVE, 
		    dimacsFilename);
  }

  return 0; 
}


/**Function********************************************************************

  Synopsis           [Usage string for command check_ltlspec_bmc]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandCheckLtlSpecBmc]

******************************************************************************/
static int UsageBmcCheckLtlSpec(void)
{
  fprintf(nusmv_stderr, "\nUsage: check_ltlspec_bmc [-h | -n idx | -p \"formula\"] [-k max_length] [-l loopback]\n\t\t\t [-o <filename>]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage\n");
  fprintf(nusmv_stderr, "  -n idx\tChecks the LTL property specified with <idx>\n");
  fprintf(nusmv_stderr, "  -p \"formula\"\tChecks the specified LTL property\n");
  fprintf(nusmv_stderr, "\t\tIf no property is specified, checks all LTL properties.\n"); 
  fprintf(nusmv_stderr, "  -k max_length\tChecks the property using <max_length> value instead of using the\n\t\tvariable <bmc_length> value\n");
  fprintf(nusmv_stderr, "  -l loopback\tChecks the property using <loopback> value instead of using the\n\t\tvariable <bmc_loopback> value\n");
  fprintf(nusmv_stderr, "  -o filename\tGenerates dimacs output file too. <filename> may contain patterns\n\n"); 

  return 1;
}



/**Function********************************************************************

  Synopsis           [Checks the given LTL specification, or all LTL 
  specifications if no formula is given. Checking parameters are the problem 
  bound and the loopback values]

  Description        [After command line processing this function calls 
  the Bmc_GenSolveLtl which generates and solve the singleton 
  problem with bound k and loopback l. <BR> 
  ] 

  SideEffects        [Property database may change]

  SeeAlso            [Bmc_CommandCheckLtlSpecBmc, Bmc_GenSolveLtl]

  CommandName        [check_ltlspec_bmc_onepb] 	   

  CommandSynopsis    [Checks the given LTL specification, or all LTL 
  specifications if no formula is given. Checking parameters are the single
  problem bound and the loopback values]  

  CommandArguments   [\[-h | -n idx | -p "formula" \[IN context\]\] 
  \[-k length\] \[-l loopback\] \[-o filename\]]  

  CommandDescription [As command check_ltlspec_bmc but it produces only one 
  single problem with fixed bound and loopback values, with no iteration
  of the problem bound from zero to max_length. <BR>  
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-n <i>index</i></tt>
       <dd> <i>index</i> is the numeric index of a valid LTL specification 
       formula actually located in the properties database. <BR>
       The validity of <i>index</i> value is checked out by the system.
    <dt> <tt>-p "formula \[IN context\]"</tt>
       <dd> Checks the <tt>formula</tt> specified on the command-line. <BR>
            <tt>context</tt> is the module instance name which the variables
            in <tt>formula</tt> must be evaluated in.
    <dt> <tt>-k <i>length</i></tt>
       <dd> <i>length</i> is the problem bound used when generating the 
       single problem. Only natural number are valid values for this option. 
       If no value is given the environment variable <i>bmc_length</i> is 
       considered instead. 
    <dt> <tt>-l <i>loopback</i></tt>
       <dd> <i>loopback</i> value may be: <BR>
       - a natural number in (0, <i>max_length-1</i>). Positive sign ('+') can 
       be also used as prefix of the number. Any invalid combination of length 
       and loopback will be skipped during the generation/solving process.<BR>
       - a negative number in (-1, -<i>bmc_length</i>). In this case 
       <i>loopback</i> is considered a value relative to <i>length</i>. 
       Any invalid combination of length and loopback will be skipped 
       during the generation/solving process.<BR>
       - the symbol 'X', which means "no loopback" <BR>
       - the symbol '*', which means "all possible loopback from zero to 
       <i>length-1</i>"
    <dt> <tt>-o <i>filename</i></tt>
       <dd> <i>filename</i> is the name of the dumped dimacs file. 
       It may contain special symbols which will be macro-expanded to form 
       the real file name. Possible symbols are: <BR>
       - @F: model name with path part <BR>
       - @f: model name without path part <BR>
       - @k: current problem bound <BR>
       - @l: current loopback value <BR>
       - @n: index of the currently processed formula in the properties 
       database <BR>
       - @@: the '@' character
  </dl>]  

******************************************************************************/
int Bmc_CommandCheckLtlSpecBmcOnePb(const int argc, const char** argv)
{
  Prop_Ptr ltlprop=(Prop_Ptr)NULL;   /* The property being processed */
  outcome opt_handling_res; 
  boolean bCreateDimacs = false; /* create dimacs only if -o opt is specified */ 
  int k = get_bmc_pb_length(options);
  char dimacsFilename[DIMAX_FILENAME_MAXLEN]; 
  int relative_loop =
    Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options), NULL);

  strncpy(dimacsFilename, get_bmc_dimacs_filename(options), 
	  DIMAX_FILENAME_MAXLEN); /* default value */
  
  /* ----------------------------------------------------------------------- */
  /* Options handling: */
  opt_handling_res = 
    bmc_CommandsWith_k_l_OptionsHandling(argc, argv, 
					 &ltlprop, Prop_Ltl, 
					 &k, &relative_loop, 
					 &bCreateDimacs, dimacsFilename, 
					 sizeof(dimacsFilename)-1);

  if (opt_handling_res == SUCCESS_REQUIRED_HELP) {
    return UsageBmcCheckLtlSpecOnePb();
  }
  if (opt_handling_res != SUCCESS) return 1;
  /* ----------------------------------------------------------------------- */


  /* make sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  /* make sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }

  /* prepare the list of properties if no property was selected: */
  if (ltlprop == NIL_PTR(Prop_Ptr)) {
    lsList props = Prop_Db_GetPropsOfType(Prop_Ltl);
    lsGen  iterator; 
    Prop_Ptr prop;

    nusmv_assert(props != LS_NIL);

    lsForEachItem(props, iterator, prop) {
      Bmc_GenSolveLtl((Prop_Ptr)prop, k, relative_loop, 
		      false, /* do not iterate k */ 
		      (bCreateDimacs) ? GEN_DIMACS_SOLVE : SOLVE, 
		      dimacsFilename);
    }

    lsDestroy(props, NULL); /* the list is no longer needed */
  }
  else {
    /* its time to solve: */
    Bmc_GenSolveLtl(ltlprop, k, relative_loop, 
		    false, /* do not iterate k */
		    (bCreateDimacs) ? GEN_DIMACS_SOLVE : SOLVE,
		    dimacsFilename);
  }

  return 0; 
}


/**Function********************************************************************

  Synopsis           [Usage string for command check_ltlspec_bmc_onepb]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandCheckLtlSpecBmcOnePb]

******************************************************************************/
static int UsageBmcCheckLtlSpecOnePb(void)
{
  fprintf(nusmv_stderr, "\nUsage: check_ltlspec_bmc_onepb [-h | -n idx | -p \"formula\"] [-k length] [-l loopback]\n\t\t\t [-o filename]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage\n");
  fprintf(nusmv_stderr, "  -n idx\tChecks the LTL property specified with <idx>\n");
  fprintf(nusmv_stderr, "  -p \"formula\"\tChecks the specified LTL property\n");
  fprintf(nusmv_stderr, "\t\tIf no property is specified, checks all LTL properties.\n"); 
  fprintf(nusmv_stderr, "  -k length\tChecks the property using <length> value instead of using the\n\t\tvariable <bmc_length> value\n");
  fprintf(nusmv_stderr, "  -l loopback\tChecks the property using <loopback> value instead of using the\n\t\tvariable <bmc_loopback> value\n");
  fprintf(nusmv_stderr, "  -o filename\tGenerates dimacs output file too. <filename> may contain patterns\n\n"); 

  return 1;
}


/**Function********************************************************************

  Synopsis           [Generates only one problem with fixed bound and 
  loopback, and dumps the problem to a dimacs file. The single problem 
  is dumped for the given LTL specification, or for all LTL 
  specifications if no formula is given]

  Description        [After command line processing it calls
  the function Bmc_GenSolveLtl to generate and dump the single problem.]

  SideEffects        [Property database may change]

  SeeAlso            [Bmc_CommandGenLtlSpecBmc, Bmc_GenSolveLtl]

  CommandName        [gen_ltlspec_bmc_onepb] 	   

  CommandSynopsis    [Dumps into one dimacs file the problem generated for
  the given LTL specification, or for all LTL specifications if no formula 
  is explicitly given. 
  Generation and dumping parameters are the problem bound and the loopback 
  values]  

  CommandArguments   [\[-h | -n idx | -p "formula" \[IN context\]\] \[-k length\] 
  \[-l loopback\] \[-o filename\]]  

  CommandDescription [ As the <i>gen_ltlspec_bmc</i> command, but it generates
  and dumps only one problem given its bound and loopback. <BR>
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-n <i>index</i></tt>
       <dd> <i>index</i> is the numeric index of a valid LTL specification 
       formula actually located in the properties database. <BR>
       The validity of <i>index</i> value is checked out by the system.
    <dt> <tt>-p "formula \[IN context\]"</tt>
       <dd> Checks the <tt>formula</tt> specified on the command-line. <BR>
            <tt>context</tt> is the module instance name which the variables
            in <tt>formula</tt> must be evaluated in.
    <dt> <tt>-k <i>length</i></tt>
       <dd> <i>length</i> is the single problem bound used to generate and 
       dump it. Only natural number are valid values for this option. 
       If no value is given the environment variable <i>bmc_length</i> 
       is considered instead. 
    <dt> <tt>-l <i>loopback</i></tt>
       <dd> <i>loopback</i> value may be: <BR>
       - a natural number in (0, <i>length-1</i>). Positive sign ('+') can 
       be also used as prefix of the number. Any invalid combination of length 
       and loopback will be skipped during the generation and dumping 
       process.<BR>
       - a negative number in (-1, -<i>length</i>). 
       Any invalid combination of length and loopback will be skipped during 
       the generation process.<BR>
       - the symbol 'X', which means "no loopback" <BR>
       - the symbol '*', which means "all possible loopback from zero to 
       <i>length-1</i>"
    <dt> <tt>-o <i>filename</i></tt>
       <dd> <i>filename</i> is the name of the dumped dimacs file. If this
       options is not specified, variable <i>bmc_dimacs_filename</i> will be
       considered. The file name string may contain special symbols which 
       will be macro-expanded to form the real file name. 
       Possible symbols are: <BR>
       - @F: model name with path part <BR>
       - @f: model name without path part <BR>
       - @k: current problem bound <BR>
       - @l: current loopback value <BR>
       - @n: index of the currently processed formula in the properties 
       database <BR>
       - @@: the '@' character
  </dl>]  

******************************************************************************/
int Bmc_CommandGenLtlSpecBmcOnePb(const int argc, const char** argv)
{
  Prop_Ptr ltlprop=(Prop_Ptr)NULL;   /* The property being processed */
  outcome opt_handling_res; 
  boolean bCreateDimacs = true; /* create dimacs in any case */
  int k = get_bmc_pb_length(options);
  char dimacsFilename[DIMAX_FILENAME_MAXLEN]; 
  int relative_loop = 
    Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options), NULL);

  strncpy(dimacsFilename, get_bmc_dimacs_filename(options), 
	  DIMAX_FILENAME_MAXLEN); /* default value */

  /* ----------------------------------------------------------------------- */
  /* Options handling: */
  opt_handling_res = 
    bmc_CommandsWith_k_l_OptionsHandling(argc, argv, 
					 &ltlprop, Prop_Ltl,
					 &k, &relative_loop, 
					 &bCreateDimacs, dimacsFilename, 
					 sizeof(dimacsFilename)-1);

  if (opt_handling_res == SUCCESS_REQUIRED_HELP) {
    return UsageBmcGenLtlSpecOnePb();
  }
  if (opt_handling_res != SUCCESS) return 1;
  /* ----------------------------------------------------------------------- */


  /* make sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  /* make sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }

  /* prepare the list of properties if no property was selected: */
  if (ltlprop == NIL_PTR(Prop_Ptr)) {
    lsList props = Prop_Db_GetPropsOfType(Prop_Ltl);
    lsGen  iterator; 
    Prop_Ptr prop;

    nusmv_assert(props != LS_NIL);

    lsForEachItem(props, iterator, prop) {
      Bmc_GenSolveLtl((Prop_Ptr)prop, k, relative_loop, 
		      false, /* do not iterate k */
		      GEN_DIMACS, dimacsFilename);
    }

    lsDestroy(props, NULL); /* the list is no longer needed */
  }
  else {
    /* its time to solve: */
    Bmc_GenSolveLtl(ltlprop, k, relative_loop, 
		    false, /* do not iterate k */
		    GEN_DIMACS, dimacsFilename);
  }

  return 0; 
}


/**Function********************************************************************

  Synopsis           [Usage string for command gen_ltlspec_bmc_onepb]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandGenLtlSpecBmcOnePb]

******************************************************************************/
static int UsageBmcGenLtlSpecOnePb(void)
{
  fprintf(nusmv_stderr, "\nUsage: gen_ltlspec_bmc_onepb [-h | -n idx | -p \"formula\"] [-k length] [-l loopback]\n\t\t\t [-o filename]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage\n");
  fprintf(nusmv_stderr, "  -n idx\tChecks the LTL property specified with <idx>\n");
  fprintf(nusmv_stderr, "  -p \"formula\"\tChecks the specified LTL property\n");
  fprintf(nusmv_stderr, "\t\tIf no property is specified, checks all LTL properties.\n"); 
  fprintf(nusmv_stderr, "  -k length\tChecks the property using <length> value instead of using the\n\t\tvariable <bmc_length> value\n");
  fprintf(nusmv_stderr, "  -l loopback\tChecks the property using <loopback> value instead of using the\n\t\tvariable <bmc_loopback> value\n");
  fprintf(nusmv_stderr, "  -o filename\tUses <filename> as dimacs file instead of \"bmc_dimacs_filename\"\t\t variable. <filename> may contain patterns\n\n"); 

  return 1;
}


/**Function********************************************************************

  Synopsis           [Generates length_max+1 problems iterating the problem 
  bound from zero to length_max, and dumps each problem to a dimacs file]

  Description        [Each problem is dumped for the given LTL specification, 
  or for all LTL specifications if no formula is given. 
  Generation parameters are the maximum bound and the loopback values. <BR>
  After command line processing it calls the function Bmc_GenSolveLtl 
  to generate and dump all problems from zero to k.]

  SideEffects        [Property database may change]

  SeeAlso            [Bmc_CommandGenLtlSpecBmcOnePb, Bmc_GenSolveLtl]

  CommandName        [gen_ltlspec_bmc] 	   

  CommandSynopsis    [Dumps into one or more dimacs files the given LTL 
  specification, or all LTL specifications if no formula is given. 
  Generation and dumping parameters are the maximum bound and the loopback 
  values]  

  CommandArguments   [\[-h | -n idx | -p "formula" \[IN context\]\] 
  \[-k max_length\] \[-l loopback\] \[-o filename\]]  

  CommandDescription [  This command generates one or more problems, and  
  dumps each problem into a dimacs file. Each problem is related to a specific
  problem bound, which increases from zero (0) to the given maximum problem 
  bound. In this short description "<i>length</i>" is the bound of the 
  problem that system is going to dump out. <BR>
  In this context the maximum problem bound is represented by the 
  <i>max_length</i> parameter, or by its default value stored in the 
  environment variable <i>bmc_length</i>.<BR>
  Each dumped problem also depends on the loopback you can explicitly 
  specify by the <i>-l</i> option, or by its default value stored in the 
  environment variable <i>bmc_loopback</i>. <BR>
  The property to be checked may be specified using the <i>-n idx</i> or  
  the <i>-p "formula"</i> options. <BR>
  You may specify dimacs file name by using the option <i>-o "filename"</i>, 
  otherwise the default value stored in the environment variable 
  <i>bmc_dimacs_filename</i> will be considered.<BR>  
  <p>
  Command options:<p>
  <dl>
    <dt> <tt>-n <i>index</i></tt>
       <dd> <i>index</i> is the numeric index of a valid LTL specification 
       formula actually located in the properties database. <BR>
       The validity of <i>index</i> value is checked out by the system.
    <dt> <tt>-p "formula \[IN context\]"</tt>
       <dd> Checks the <tt>formula</tt> specified on the command-line. <BR>
            <tt>context</tt> is the module instance name which the variables
            in <tt>formula</tt> must be evaluated in.
    <dt> <tt>-k <i>max_length</i></tt>
       <dd> <i>max_length</i> is the maximum problem bound used when 
       increasing problem bound starting from zero. Only natural number are
       valid values for this option. If no value is given the environment 
       variable <i>bmc_length</i> value is considered instead. 
    <dt> <tt>-l <i>loopback</i></tt>
       <dd> <i>loopback</i> value may be: <BR>
       - a natural number in (0, <i>max_length-1</i>). Positive sign ('+') can 
       be also used as prefix of the number. Any invalid combination of bound 
       and loopback will be skipped during the generation and 
       dumping process.<BR>
       - a negative number in (-1, -<i>bmc_length</i>). In this case 
       <i>loopback</i> is considered a value relative to <i>max_length</i>. 
       Any invalid combination of bound and loopback will be skipped during 
       the generation process.<BR>
       - the symbol 'X', which means "no loopback" <BR>
       - the symbol '*', which means "all possible loopback from zero to 
       <i>length-1</i>"
    <dt> <tt>-o <i>filename</i></tt>
       <dd> <i>filename</i> is the name of dumped dimacs files. If this
       options is not specified, variable <i>bmc_dimacs_filename</i> will be
       considered. The file name string may contain special symbols which 
       will be macro-expanded to form the real file name. 
       Possible symbols are: <BR>
       - @F: model name with path part <BR>
       - @f: model name without path part <BR>
       - @k: current problem bound <BR>
       - @l: current loopback value <BR>
       - @n: index of the currently processed formula in the properties 
       database <BR>
       - @@: the '@' character
  </dl>]  
   

******************************************************************************/
int Bmc_CommandGenLtlSpecBmc(const int argc, const char** argv)
{
  Prop_Ptr ltlprop=(Prop_Ptr)NULL;   /* The property being processed */
  outcome opt_handling_res; 
  boolean bCreateDimacs = true; /* create dimacs in any case */
  int k = get_bmc_pb_length(options);
  char dimacsFilename[DIMAX_FILENAME_MAXLEN]; 
  int relative_loop = 
    Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options), NULL);

  strncpy(dimacsFilename, get_bmc_dimacs_filename(options), 
	  DIMAX_FILENAME_MAXLEN); /* default value */

  /* ----------------------------------------------------------------------- */
  /* Options handling: */
  opt_handling_res = 
    bmc_CommandsWith_k_l_OptionsHandling(argc, argv, 
					 &ltlprop, Prop_Ltl,
					 &k, &relative_loop, 
					 &bCreateDimacs, dimacsFilename, 
					 sizeof(dimacsFilename)-1);

  if (opt_handling_res == SUCCESS_REQUIRED_HELP) {
    return UsageBmcGenLtlSpec();
  }
  if (opt_handling_res != SUCCESS) return 1;
  /* ----------------------------------------------------------------------- */


  /* make sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  /* make sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }

  /* prepare the list of properties if no property was selected: */
  if (ltlprop == NIL_PTR(Prop_Ptr)) {
    lsList props = Prop_Db_GetPropsOfType(Prop_Ltl);
    lsGen  iterator; 
    Prop_Ptr prop;

    nusmv_assert(props != LS_NIL);

    lsForEachItem(props, iterator, prop) {
      Bmc_GenSolveLtl((Prop_Ptr)prop, k, relative_loop, 
		      true, /* iterate on k */
		      GEN_DIMACS, dimacsFilename);
    }

    lsDestroy(props, NULL); /* the list is no longer needed */
  }
  else {
    /* its time to solve: */
    Bmc_GenSolveLtl(ltlprop, k, relative_loop, 
		    true, /* iterate on k */
		    GEN_DIMACS, dimacsFilename);
  }

  return 0; 
}

/**Function********************************************************************

  Synopsis           [Usage string for command gen_ltlspec_bmc]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandGenLtlSpecBmc]

******************************************************************************/
static int UsageBmcGenLtlSpec(void)
{
  fprintf(nusmv_stderr, "\nUsage: gen_ltlspec_bmc [-h | -n idx | -p \"formula\"] [-k max_length] [-l loopback]\n\t\t\t [-o filename]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage\n");
  fprintf(nusmv_stderr, "  -n idx\tChecks the LTL property specified with <idx>\n");
  fprintf(nusmv_stderr, "  -p \"formula\"\tChecks the specified LTL property\n");
  fprintf(nusmv_stderr, "\t\tIf no property is specified, checks all LTL properties.\n"); 
  fprintf(nusmv_stderr, "  -k max_length\tChecks the property using <max_length> value instead of using the\n\t\tvariable <bmc_length> value\n");
  fprintf(nusmv_stderr, "  -l loopback\tChecks the property using <loopback> value instead of using the\n\t\tvariable <bmc_loopback> value\n");
  fprintf(nusmv_stderr, "  -o filename\tUses <filename> as dimacs file instead of using the\n\t\t\"bmc_dimacs_filename\" variable. <filename> may contain patterns\n\n"); 

  return 1;
}


/**Function********************************************************************

  Synopsis           [Generates and dumps the problem for the given 
  invariant or for all invariants if no formula is given. SAT solver is not
  invoked.]

  Description        [After command line processing calls Bmc_GenSolveInvar 
  to dump the generated invariant problem. 
  If you specify the <i>-o "filename"</i> option a dimacs file named  
  "filename" will be created, otherwise the environment variable 
  <i>bmc_invar_dimacs_filename</i> value will be considered.]

  SideEffects        [Property database may change]

  SeeAlso            [Bmc_GenSolveInvar]

  CommandName        [gen_invar_bmc] 	   

  CommandSynopsis    [Generates the given invariant, or all 
  invariants if no formula is given]  

  CommandArguments   [\[-h | -n idx | -p "formula" \[IN context\]\] 
  \[-o filename\]]  

  CommandDescription [<p>
  Command options:<p>
  <dl>
    <dt> <tt>-n <i>index</i></tt>
       <dd> <i>index</i> is the numeric index of a valid INVAR specification 
       formula actually located in the properties database. <BR>
       The validity of <i>index</i> value is checked out by the system.
    <dt> <tt>-p "formula" \[IN context\]</tt>
       <dd> Checks the <tt>formula</tt> specified on the command-line. <BR>
            <tt>context</tt> is the module instance name which the variables
            in <tt>formula</tt> must be evaluated in.
    <dt> <tt>-o <i>filename</i></tt>
       <dd> <i>filename</i> is the name of the dumped dimacs file. If you 
       do not use this option the dimacs file name is taken from the 
       environment variable <i>bmc_invar_dimacs_filename</i>. <BR> 
       File name may contain special symbols which will be macro-expanded 
       to form the real dimacs file name. Possible symbols are: <BR>
       - @F: model name with path part <BR>
       - @f: model name without path part <BR>
       - @n: index of the currently processed formula in the properties 
       database <BR>
       - @@: the '@' character
  </dl>]  

******************************************************************************/
int Bmc_CommandGenInvarBmc(const int argc, const char** argv)
{
  Prop_Ptr invarprop=(Prop_Ptr)NULL;   /* The property being processed */
  outcome opt_handling_res; 
  boolean bCreateDimacs = true; /* create dimacs in any case */
  char dimacsFilename[DIMAX_FILENAME_MAXLEN]; 

  strncpy(dimacsFilename, get_bmc_invar_dimacs_filename(options), 
	  DIMAX_FILENAME_MAXLEN); /* default value */

  /* ----------------------------------------------------------------------- */
  /* Options handling: */
  opt_handling_res = 
    bmc_CommandsWithout_k_l_OptionsHandling(argc, argv, &invarprop, Prop_Invar,
					    &bCreateDimacs, dimacsFilename, 
					    sizeof(dimacsFilename)-1);

  if (opt_handling_res == SUCCESS_REQUIRED_HELP) {
    return UsageBmcGenInvar();
  }
  if (opt_handling_res != SUCCESS)  return 1;
  /* ----------------------------------------------------------------------- */


  /* make sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  /* make sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }

  /* prepare the list of properties if no property was selected: */
  if (invarprop == NIL_PTR(Prop_Ptr)) {
    lsList props = Prop_Db_GetPropsOfType(Prop_Invar);
    lsGen  iterator; 
    Prop_Ptr prop;

    nusmv_assert(props != LS_NIL);

    lsForEachItem(props, iterator, prop) {
      Bmc_GenSolveInvar((Prop_Ptr)prop, 
			GEN_DIMACS, dimacsFilename);
    }

    lsDestroy(props, NULL); /* the list is no longer needed */
  }
  else {
    /* its time to generate dimacs: */
    Bmc_GenSolveInvar(invarprop, GEN_DIMACS, dimacsFilename);
  }

  return 0; 
}


/**Function********************************************************************

  Synopsis           [Usage string for command gen_invar_bmc]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandGenInvarBmc]

******************************************************************************/
static int UsageBmcGenInvar(void)
{
  fprintf(nusmv_stderr, "\nUsage: gen_invar_bmc [-h | -n idx | -p \"formula\"] [-o filename]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage\n");
  fprintf(nusmv_stderr, "  -n idx\tChecks the INVAR property specified with <idx>\n");
  fprintf(nusmv_stderr, "  -p \"formula\"\tChecks the specified INVAR propositional property\n");
  fprintf(nusmv_stderr, "\t\tIf no property is specified, checks all INVAR properties.\n"); 
  fprintf(nusmv_stderr, "  -o filename\tUses <filename> as dimacs file instead of using the\n\t\t\"bmc_invar_dimacs_filename\" variable. <filename> may contain patterns\n\n"); 

  return 1;
}


/**Function********************************************************************

  Synopsis           [Generates and solve the given invariant, or all 
  invariants if no formula is given]

  Description        [After command line processing calls Bmc_GenSolveInvar 
  to solve and eventually dump the generated invariant problem. If you specify
  the <i>-o "filename"</i> option a dimacs file will be generated, otherwise
  no dimacs dump will be performed]

  SideEffects        [Property database may change]

  SeeAlso            [Bmc_GenSolveInvar]

  CommandName        [check_invar_bmc] 	   

  CommandSynopsis    [Generates and solve the given invariant, or all 
  invariants if no formula is given]  

  CommandArguments   [\[-h | -n idx | -p "formula" \[IN context\]\] 
  \[-o filename\]]  

  CommandDescription [<p>
  Command options:<p>
  <dl>
    <dt> <tt>-n <i>index</i></tt>
       <dd> <i>index</i> is the numeric index of a valid INVAR specification 
       formula actually located in the properties database. <BR>
       The validity of <i>index</i> value is checked out by the system.
    <dt> <tt>-p "formula \[IN context\]"</tt>
       <dd> Checks the <tt>formula</tt> specified on the command-line. <BR>
            <tt>context</tt> is the module instance name which the variables
            in <tt>formula</tt> must be evaluated in.
    <dt> <tt>-o <i>filename</i></tt>
       <dd> <i>filename</i> is the name of the dumped dimacs file. 
       It may contain special symbols which will be macro-expanded to form 
       the real file name. Possible symbols are: <BR>
       - @F: model name with path part <BR>
       - @f: model name without path part <BR>
       - @n: index of the currently processed formula in the properties 
       database <BR>
       - @@: the '@' character
  </dl>]  

******************************************************************************/
int Bmc_CommandCheckInvarBmc(const int argc, const char** argv)
{
  Prop_Ptr invarprop=(Prop_Ptr)NULL;   /* The property being processed */
  outcome opt_handling_res; 
  boolean bCreateDimacs = false; /* create dimacs in any case */
  char dimacsFilename[DIMAX_FILENAME_MAXLEN]; 

  strncpy(dimacsFilename, get_bmc_invar_dimacs_filename(options), 
	  DIMAX_FILENAME_MAXLEN); /* default value */

  /* ----------------------------------------------------------------------- */
  /* Options handling: */
  opt_handling_res = 
    bmc_CommandsWithout_k_l_OptionsHandling(argc, argv, &invarprop, Prop_Invar,
					    &bCreateDimacs, dimacsFilename, 
					    sizeof(dimacsFilename)-1);

  if (opt_handling_res == SUCCESS_REQUIRED_HELP) {
    return UsageBmcCheckInvar();
  }
  if (opt_handling_res != SUCCESS) return 1;
  /* ----------------------------------------------------------------------- */


  /* make sure the model hierarchy has been flattened */
  if (cmp_struct_get_flatten_hrc (cmps) == 0) {
    fprintf (nusmv_stderr, "The hierarchy must be flattened before. Use "\
	     "the \"flatten_hierarchy\" command.\n");
    return 1;
  }

  /* make sure bmc has been set up */
  if (cmp_struct_get_bmc_setup(cmps) == 0) {
    fprintf (nusmv_stderr, "Bmc must be setup before. Use "\
	     "the \"bmc_setup\" command.\n");
    return 1;
  }

  /* prepare the list of properties if no property was selected: */
  if (invarprop == NIL_PTR(Prop_Ptr)) {
    lsList props = Prop_Db_GetPropsOfType(Prop_Invar);
    lsGen  iterator; 
    Prop_Ptr prop;

    nusmv_assert(props != LS_NIL);

    lsForEachItem(props, iterator, prop) {
      Bmc_GenSolveInvar((Prop_Ptr)prop, 
			(bCreateDimacs) ? GEN_DIMACS_SOLVE : SOLVE,
			dimacsFilename);
    }

    lsDestroy(props, NULL); /* the list is no longer needed */
  }
  else {
    /* its time to generate dimacs: */
    Bmc_GenSolveInvar(invarprop, 
		      (bCreateDimacs) ? GEN_DIMACS_SOLVE : SOLVE, 
		      dimacsFilename);
  }

  return 0; 
}

/**Function********************************************************************

  Synopsis           [Usage string for command check_invar_bmc]

  Description        []

  SideEffects        [None]

  SeeAlso            [Bmc_CommandCheckInvarBmc]

******************************************************************************/
static int UsageBmcCheckInvar(void)
{
  fprintf(nusmv_stderr, "\nUsage: check_invar_bmc [-h | -n idx | -p \"formula\"] [-o filename]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage\n");
  fprintf(nusmv_stderr, "  -n idx\tChecks the INVAR property specified with <idx>\n");
  fprintf(nusmv_stderr, "  -p \"formula\"\tChecks the specified INVAR propositional property\n");
  fprintf(nusmv_stderr, "\t\tIf no property is specified, checks all INVAR properties.\n"); 
  fprintf(nusmv_stderr, "  -o filename\tGenerates dimacs output file too. <filename> may contain patterns.\n\n");

  return 1;
}



/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [Bmc commands options handling for commands with 
  -k and -l options]

  Description        [Handles options of bmc commands 
  {check,gen}_ltlspec_bmc\[_onepb\]. pProperty will contain the property must
  be checked out, or NULL if all properties must be checked. Initial value
  of ltlspec *must* be NULL<BR>
  This function will check the values specified for length and loopback are 
  compatible with their constrains. <BR>
  dimacsSpecified will be put to one (1) if the -o option is specified, zero 
  (0) otherwise. <BR>
  Returns GENERIC_ERROR if an error has occurred;
  Returns SUCCESS_REQUIRED_HELP if -h options had been specified; 
  Returns SUCCESS in all other cases. 
  ]

  SideEffects        [pProperty might change if one or more valid 
  properties are specified. dimacsSpecified will change if -o option 
  is specified]

  SeeAlso            []

******************************************************************************/
static outcome 
bmc_CommandsWith_k_l_OptionsHandling(const int argc, const char** argv, 
				     Prop_Ptr* pProperty,
				     Prop_Type type, 
				     int* pb_length, 
				     int* user_loop, 
				     boolean* dimacsSpecified, 
				     char* dimacsFilename, 
				     const size_t dimacsFilename_len) 
{
  int c;
  int prop_idx;
  char* strFormula = NULL;
  /* to keep requested loopback string: if were specified, option -l should 
   be evaluated after -k option is evaluated */
  char* szLoopback = NIL(char); 
  
  nusmv_assert(*pProperty == NIL_PTR(Prop_Ptr)); /* precondition contract */  

  util_getopt_reset();
  while((c = util_getopt((int)argc, (char**)argv, "hn:k:l:o:p:")) != EOF) {
    switch (c) {
    case 'h':
      return SUCCESS_REQUIRED_HELP;

    case 'n':
      {
	char* strPropIdx = NIL(char);
	
	/* check if a formula has been already specified: */
	if (*pProperty != NIL_PTR(Prop_Ptr) || (strFormula != NULL)) {
	  bmcErrorPropertyAlreadySpecified();
	  return GENERIC_ERROR;
	}
	
	strPropIdx = util_strsav(util_optarg);

	/* check if property idx is ok */
	prop_idx = Prop_GetPropIdxFromString(strPropIdx);
	if (prop_idx == -1) {
	  /* error messages have been already shown */
	  return GENERIC_ERROR;
	}
	  
	/* here property idx is ok */
	*pProperty = Prop_Db_GetNum(prop_idx);
	if ( Prop_CheckType(*pProperty, type) != 0 ) {
	  return GENERIC_ERROR;
	}
	
	break;
      } /* case 'n' */

    case 'p':
      {
	/* check if a formula has been already specified: */
	if (*pProperty != NIL_PTR(Prop_Ptr) || (strFormula != NULL)) {
	  bmcErrorPropertyAlreadySpecified();
	  return GENERIC_ERROR;	  
	}
	strFormula = util_strsav(util_optarg);
	break;
      }
	 
    case 'k':
      {
	char* strNumber = util_strsav(util_optarg);
	int k;
	if (util_str2int(strNumber, &k) != 0) {
	  bmcErrorInvalidNumber(strNumber);
	  return GENERIC_ERROR;
	}
	  
	/* k and l constrains are going to be check out of switch statement */ 
	*pb_length = k; break;  
      }
       
    case 'l':
      {
	szLoopback = util_strsav(util_optarg);
	/* checking of loopback value is delayed after command line processing
	   to allow any -k option evaluation before 
	   (see the cheking code below)
	*/
	break;
      }
      
    case 'o':
      {
	char* szFilename = util_strsav(util_optarg);
	strncpy(dimacsFilename, szFilename, dimacsFilename_len);
	*dimacsSpecified = true;
	break;
      }	    

    default:
      return GENERIC_ERROR;
    } /* switch case */
  } /* end of cmd line processing */
  
  /* checks if there are unexpected options: */
  if (argc != util_optind) {
    fprintf(nusmv_stderr, "You specified one or more invalid options.\n\n");
    return GENERIC_ERROR;
  }  

  
  /* Formula checking and commitment: */
  if (strFormula != NULL) {
    int idx = Prop_Db_PropParseAndAdd(strFormula, type);
    if (idx == -1) {
      bmcErrorInvalidProperty(strFormula);
      return GENERIC_ERROR;
    }
    else {
      /* index is ok */
      nusmv_assert(pProperty == NIL_PTR(Prop_Ptr));
      *pProperty = Prop_Db_GetNum(idx);
    }
  }

  /* ------------------------------------------------------------------------*/
  /* Checking of k,l constrains:                                             */
  if (szLoopback != NIL(char)) {
    outcome res; 
    *user_loop = Bmc_Utils_ConvertLoopFromString(szLoopback, &res);
    
    if (res != SUCCESS) {
      bmcErrorInvalidNumber(szLoopback);
      return GENERIC_ERROR;
    }
    
    if (Bmc_Utils_Check_k_l(*pb_length, 
			    Bmc_Utils_RelLoop2AbsLoop(*user_loop, *pb_length)) 
	!= SUCCESS) {
      bmcErrorInvalid_k_l(*pb_length, *user_loop);
      return GENERIC_ERROR;
    }
    
    *user_loop = Bmc_Utils_RelLoop2AbsLoop(*user_loop, *pb_length);
    if (Bmc_Utils_Check_k_l(*pb_length, *user_loop) != SUCCESS) {
      bmcErrorInvalid_k_l(*pb_length, *user_loop);
      return GENERIC_ERROR;
    }
  }
  /* ------------------------------------------------------------------------*/
  
  return SUCCESS;
}



/**Function********************************************************************

  Synopsis           [Bmc commands options handling for commands without 
  -k and -l options]

  Description        [Handles options of bmc commands 
  {check,gen}_invar_bmc\[_onepb\]. pProperty will contain the property must
  be checked out, or NULL if all properties must be checked. Initial value
  of ltlspec *must* be NULL<BR>
  dimacsSpecified will be put to one (1) if the -o option is specified, zero 
  (0) otherwise. <BR>
  Returns GENERIC_ERROR if an error has occurred;
  Returns SUCCESS_REQUIRED_HELP if -h options had been specified; 
  Returns SUCCESS in all other cases. 
  ]

  SideEffects        [pProperty might change if one or more 
  valid properties were specified. 
  dimacsSpecified will change if -o option were specified.]

  SeeAlso            []

******************************************************************************/
static outcome 
bmc_CommandsWithout_k_l_OptionsHandling(const int argc, const char** argv, 
					Prop_Ptr* pProperty, 
					Prop_Type type, 
					boolean* dimacsSpecified, 
					char* dimacsFilename, 
					const size_t dimacsFilename_len) 
{
  int c;
  int prop_idx;
  char* strFormula = NULL;

  nusmv_assert(*pProperty == NIL_PTR(Prop_Ptr)); /* precondition contract */  

  util_getopt_reset();
  while((c = util_getopt((int)argc, (char**)argv, "hn:p:o:")) != EOF) {
    switch (c) {
    case 'h':
      return SUCCESS_REQUIRED_HELP;

    case 'n':
      {
	char* strPropIdx = NIL(char);
	
	/* check if a formula has been already specified: */
	if (*pProperty != NIL_PTR(Prop_Ptr) || (strFormula != NULL)) {
	  bmcErrorPropertyAlreadySpecified();
	  return GENERIC_ERROR;
	}
	
	strPropIdx = util_strsav(util_optarg);

	/* check if property idx is ok */
	prop_idx = Prop_GetPropIdxFromString(strPropIdx);
	if (prop_idx == -1) {
	  /* error messages have been already shown */
	  return GENERIC_ERROR;
	}
	  
	/* here property idx is ok */
	*pProperty = Prop_Db_GetNum(prop_idx);
	if ( Prop_CheckType(*pProperty, type) != 0 ) {
	  return GENERIC_ERROR;
	}
	
	break;
      } /* case 'n' */

    case 'p':
      {
	/* check if a formula has been already specified: */
	if (*pProperty != NIL_PTR(Prop_Ptr) || (strFormula != NULL)) {
	  bmcErrorPropertyAlreadySpecified();
	  return GENERIC_ERROR;	  
	}
	strFormula = util_strsav(util_optarg);
	break;
      }
	 
    case 'o':
      {
	char* szFilename = util_strsav(util_optarg);
	strncpy(dimacsFilename, szFilename, dimacsFilename_len);
	*dimacsSpecified = true;
	break;
      }	    

    default:
      return GENERIC_ERROR;
    } /* switch case */
  } /* end of cmd line processing */

  /* checks if there are unexpected options: */
  if (argc != util_optind) {
    fprintf(nusmv_stderr, "You specified one or more invalid options.\n\n");
    return GENERIC_ERROR;
  }  
    
  /* Formula checking and commitment: */
  if (strFormula != NULL) {
    int idx = Prop_Db_PropParseAndAdd(strFormula, type);
    if (idx == -1) {
      bmcErrorInvalidProperty(strFormula);
      return GENERIC_ERROR;
    }
    else {
      /* index is ok */
      nusmv_assert(pProperty == NIL_PTR(Prop_Ptr));
      *pProperty = Prop_Db_GetNum(idx);
    }
  }

  return SUCCESS;
}



static void bmcErrorInvalidProperty(const char* szFormula)
{
  if (szFormula != NIL(char)) {
    fprintf(nusmv_stderr, "Error: property \"%s\" is not valid\n", szFormula);
  }
  else {
    fprintf(nusmv_stderr, "Error: you given an empty property\n");
  }
}


static void bmcErrorPropertyAlreadySpecified()
{
  fprintf(nusmv_stderr, "Error: It is not possible to specify multiple properties to be checked \n");
  fprintf(nusmv_stderr, "      using \"-p <formula>\" and/or \"-n <property_idx>\" options\n\n"); 
}

static void bmcErrorInvalidNumber(const char* szNumber)
{
  if (szNumber != NIL(char)) {
    fprintf(nusmv_stderr, "Error: string \"%s\" is not valid. An integer was expected.\n", szNumber);
  }
  else {
    fprintf(nusmv_stderr, "Error: you given an empty string when integer was expected\n");
  }
}

static void bmcErrorInvalid_k_l(const int k, const int l)
{
  char buf[16];
  
  Bmc_Utils_ConvertLoopFromInteger(l, buf, sizeof(buf));
  fprintf(nusmv_stderr, 
	  "Error: length=%d and loopback=%s are not compatible.\n",
	  k, buf);
}


