/**CFile***********************************************************************

  FileName [bmcBmc.c]

  PackageName [bmc]

  Synopsis [High level functionalities layer]

  Description [User-commands directly use function defined in this module. 
  This is the highest level in the BMC API architecture.]

  SeeAlso  []

  Author   [Roberto Cavada]

  Copyright [ This file is part of the ``bmc'' package of NuSMV
  version 2.  Copyright (C) 2000-2001 by ITC-irst and University of
  Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it> or
  email to <nusmv-users@irst.itc.it>.  Please report bugs to
  <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to
  <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "bmcBmc.h"

#include "node.h"
#include "be.h"
#include "sat.h" /* for solver and result */

#include "bmcInt.h"
#include "bmcGen.h"
#include "bmcSat.h"
#include "bmcDimacs.h"
#include "bmcModel.h"
#include "bmcVarsMgr.h"
#include "bmcWff.h"
#include "bmcUtils.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


static void 
bmc_expandFilename ARGS((const int k, const int l,
			 const int prop_idx,
			 const char* filename_to_be_expanded,
			 char* filename_expanded,
			 const size_t filename_expanded_maxlen));




/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Performs simulation]

  Description        [Generate a problem with no property, and search 
  for a solution, which represents a simulation trace.]

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
void Bmc_Simulate(const Bmc_Fsm_ptr be_fsm, const int k)
{
  be_ptr prob; /* The problem in BE format */
  Bmc_Problem_ptr bmcProb; /* The problem for BMC */
  Sat_SatResult_ptr satResult; /* The SIM outcome */
  Sat_SatSolver_ptr solver = Sat_GetSelectedSolver(get_sat_solver(options));
  VarsMgr_ptr vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);
  

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr,
      "\nGenerating simulation trace of length %d (no loopback)\n",
      k);
  }

  prob = Bmc_Model_GetPathWithInit(be_fsm, k);
  bmcProb = Bmc_Problem_Create(vars_mgr, prob);
  satResult = Bmc_Sat_SolveProblem(vars_mgr, solver, bmcProb, k);

  if (Sat_SatResult_IsError(satResult)) {
    /* internal error */
    Bmc_Problem_Delete(bmcProb);
    Sat_SatResult_Delete(&satResult);
    nusmv_assert(false);
  }
  else if (Sat_SatResult_IsUnsatisfiable(satResult)) {
    fprintf(nusmv_stdout,
      "The model deadlocks before requested length %d!\n", k);
    Bmc_Problem_Delete(bmcProb);
    Sat_SatResult_Delete(&satResult);
  }
  else {
    Bmc_SatTrace_ptr trace;

    /* no other choices are supplied: */
    nusmv_assert(Sat_SatResult_IsSatisfiable(satResult));
    
    trace = Bmc_SatTrace_Create(bmcProb, satResult);

    fprintf(nusmv_stderr, "\n\n");
    print_explanation(Bmc_SatTrace_GetSymbModel(trace, vars_mgr, k),
		      all_symbols);

    Bmc_SatTrace_Delete(&trace);
    Bmc_Problem_Delete(bmcProb);
    Sat_SatResult_Delete(&satResult);
  }
}


/**Function********************************************************************

  Synopsis           [Given a LTL property generates and solve the problems
  for all Ki (k_min<=i<=k_max). If bIncreaseK is 0 then k_min==k_max==k and
  only one problem is generated. If bIncreaseK is 1 then k_min == 0 and
  k_max == k.
  Each problem Ki takes into account of all possible loops from k_min to Ki
  if loopback is '*' (BMC_ALL_LOOPS). <BR>
  Also see the Bmc_GenSolve_Action possible values]

  Description        []

  SideEffects        []

  SeeAlso            [Bmc_GenSolve_Action]

******************************************************************************/
void Bmc_GenSolveLtl(Prop_Ptr ltlprop, 
		     const int k, const int relative_loop, 
		     const boolean bIncreaseK, 
		     const Bmc_GenSolve_Action action, 
		     const char* dimacsFilename) 
{
  node_ptr ltlspec;   /* The corresponding wff */
  node_ptr fltlspec;  /* wff after flattening */
  node_ptr bltlspec;  /* Its booleanization */
  Bmc_Fsm_ptr be_fsm; /* The corresponding be fsm  */
  VarsMgr_ptr vars_mgr;

  /* ----------------------------------------------------------------------*/
  /* Here a property was selected                                          */
  /* ----------------------------------------------------------------------*/
  int k_max = k;
  int k_min = 0;
  int increasingK;

  if (bIncreaseK == false) k_min = k_max;

  /* checks that a property was selected: */
  nusmv_assert(ltlprop != (Prop_Ptr)NULL);

  /* checks if it has been already checked: */
  if (Prop_GetStatus(ltlprop) != Prop_Unchecked) {
    /* aborts this check */
    return;
  }

  /* calculates the property be fsm: */
  ltlspec = Prop_GetProp(ltlprop);
  fltlspec = Compile_FlattenSexpExpandDefine(ltlspec, Nil);
  bltlspec = detexpr2bexpr(fltlspec);
  bltlspec = Bmc_Wff_MkNnf(Bmc_Wff_MkNot(bltlspec));
  be_fsm = Prop_GetBeFsmCoi(ltlprop);
  vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);

  /* Start problems generations: */
  for(increasingK = k_min; increasingK <= k_max; ++increasingK) {
    int l;
    char szLoop[16]; /* to keep loopback string */
    be_ptr prob; /* The problem in BE format */
    Bmc_Problem_ptr bmcProb; /* The problem for BMC */

    /* the loopback value could be depending on the length
       if it were relative: */
    l = Bmc_Utils_RelLoop2AbsLoop(relative_loop, increasingK);

    /* prints a verbose message: */
    Bmc_Utils_ConvertLoopFromInteger(relative_loop, szLoop, sizeof(szLoop)); 

    if (opt_verbose_level_gt(options, 0)) {
      if (Bmc_Utils_IsNoLoopback(l)) {
	fprintf(nusmv_stderr,
		"\nGenerating problem with bound %d, no loopback...\n",
		increasingK);
      }
      else if (Bmc_Utils_IsAllLoopbacks(l)) {
	fprintf(nusmv_stderr,
		"\nGenerating problem with bound %d, all possible loopbacks...\n",
		increasingK);
      }
      else {
	/* l can be negative iff loopback from the user pov is < -length */
	if ((l < increasingK) && (l>=0)) {
	  fprintf(nusmv_stderr,
		  "\nGenerating problem with bound %d, loopback %s...\n",
		  increasingK, szLoop);
	}
      }
    } /* verbose messages */

    /* checks for loopback vs k compatibility */
    if (Bmc_Utils_IsSingleLoopback(l) && ((l >= increasingK) || (l<0))) {
      fprintf(nusmv_stderr,
	      "\nWarning: problem with bound %d and loopback %s is not allowed: skipped\n",
	      increasingK, szLoop);
      continue;
    }

    /* generates the problem: */
    prob = Bmc_Gen_LtlProblem(be_fsm, bltlspec, increasingK, l);

    /* creates the CNF problem: */
    bmcProb = Bmc_Problem_Create(vars_mgr, prob);

    /* DIMACS Dumping */
    if ((action == GEN_DIMACS) || (action == GEN_DIMACS_SOLVE)) {
      char dimacsFilenameExpanded[DIMAX_FILENAME_MAXLEN];
      int loop = Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options),
						 NULL);
      bmc_expandFilename(increasingK, 
			 loop, 
			 Prop_GetIndex(ltlprop),
			 dimacsFilename,
			 dimacsFilenameExpanded,
			 sizeof(dimacsFilenameExpanded)-1);

      Bmc_Dimacs_DumpProblemFilename(vars_mgr, bmcProb,
				     dimacsFilenameExpanded,
					increasingK);
    }

    /* SAT problem solving */
    if ((action == SOLVE) || (action == GEN_DIMACS_SOLVE)) {
      Sat_SatResult_ptr satResult;
      Sat_SatSolver_ptr solver = 
	Sat_GetSelectedSolver(get_sat_solver(options));

      satResult = Bmc_Sat_SolveProblem(vars_mgr, solver, bmcProb, increasingK);

      /* SIM result processing: */
      if (Sat_SatResult_IsError(satResult)) {
	/* solver internal error */
	Bmc_Problem_Delete(bmcProb);
	Sat_SatResult_Delete(&satResult);
	internal_error("Sorry, solver answered with a fatal Internal Failure during problem solving.\n");
      }
      else if (Sat_SatResult_IsUnsatisfiable(satResult)) {
	char szLoopMsg[16]; /* for loopback part of message */
	memset(szLoopMsg, 0, sizeof(szLoopMsg));

	if (Bmc_Utils_IsAllLoopbacks(l)) {
	  strncpy(szLoopMsg, "", sizeof(szLoopMsg)-1);
	}
	else if (Bmc_Utils_IsNoLoopback(l)) {
	  strncpy(szLoopMsg, " and no loop", sizeof(szLoopMsg)-1);
	}
	else {
	  /* loop is Natural: */
	  strncpy(szLoopMsg, " and loop at ", sizeof(szLoopMsg)-1);
	  strncat(szLoopMsg, szLoop, sizeof(szLoopMsg)-1-strlen(szLoopMsg));
	}

	fprintf(nusmv_stdout,
		"-- no counterexample found with bound %d%s for ",
		increasingK, szLoopMsg);
	print_spec(nusmv_stdout, ltlspec);
	fprintf(nusmv_stdout, "\n");
  
	Bmc_Problem_Delete(bmcProb);
	Sat_SatResult_Delete(&satResult);
      }
      else {
	Bmc_SatTrace_ptr trace;

	/* no other choices are supplied: */
	nusmv_assert(Sat_SatResult_IsSatisfiable(satResult));
    
	trace = Bmc_SatTrace_Create(bmcProb, satResult);
	fprintf(nusmv_stdout, "-- ");
	print_spec(nusmv_stdout, ltlspec);
	fprintf(nusmv_stdout, "  is false\n");

	Prop_SetStatus(ltlprop, Prop_False);
	Prop_SetTrace(ltlprop, get_trace_number());

	print_explanation(Bmc_SatTrace_GetSymbModel(trace, vars_mgr, 
						    increasingK),
			  all_symbols);

	Bmc_SatTrace_Delete(&trace);
	Bmc_Problem_Delete(bmcProb);
	Sat_SatResult_Delete(&satResult);
	break; /* exits from the increasing k loop (found a solution!) */
      }
    } /* end of SAT invocation */

  } /* for all problems length */
}



/**Function********************************************************************

  Synopsis           [Generates DIMACS version and/or solve and INVARSPEC
  problems]

  Description        []

  SideEffects        []

  SeeAlso            [Bmc_GenSolvePbs]

******************************************************************************/
void Bmc_GenSolveInvar(Prop_Ptr invarprop,  
		       const Bmc_GenSolve_Action action,
		       const char* dimacsFilename)
{
  node_ptr invarspec;   /* The corresponding wff */
  node_ptr finvarspec;  /* wff after flattening*/
  node_ptr binvarspec;  /* Its booleanization */
  Bmc_Fsm_ptr be_fsm;   /* The corresponding be fsm  */
  be_ptr prob;             /* The problem in BE format */
  Bmc_Problem_ptr bmcProb; /* The problem for BMC */
  VarsMgr_ptr vars_mgr;

  /* checks that a property was selected: */
  nusmv_assert(invarprop != (Prop_Ptr)NULL);


  /* checks if it has been already checked: */
  if (Prop_GetStatus(invarprop) != Prop_Unchecked) {
    /* aborts this check */
    return;
  }

  /* gets the property's be fsm: */
  invarspec = Prop_GetProp(invarprop);
  finvarspec = Compile_FlattenSexpExpandDefine(invarspec, Nil);
  binvarspec = detexpr2bexpr(finvarspec);
  binvarspec = Bmc_Wff_MkNnf(binvarspec);
  be_fsm = Prop_GetBeFsmCoi(invarprop);
  vars_mgr = Bmc_Fsm_GetVarsManager(be_fsm);

  /* generates the problem: */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "\nGenerating invariant problem\n");
  }
  prob = Bmc_Gen_InvarProblem(be_fsm, binvarspec);

  /* creates the CNF problem: */
  bmcProb = Bmc_Problem_Create(vars_mgr, prob);

  /* DIMACS Dumping */
  if ((action == GEN_DIMACS) || (action == GEN_DIMACS_SOLVE)) {
    char dimacsFilenameExpanded[DIMAX_FILENAME_MAXLEN];
    bmc_expandFilename(1, Bmc_Utils_GetNoLoopback(),
		       Prop_GetIndex(invarprop),
		       dimacsFilename,
		       dimacsFilenameExpanded,
		       sizeof(dimacsFilenameExpanded)-1);

    Bmc_Dimacs_DumpInvarProblemFilename(vars_mgr, bmcProb,
					dimacsFilenameExpanded);
  }

  /* SIM invar problem solving: */
  if ((action == SOLVE) || (action == GEN_DIMACS_SOLVE)) {
    Sat_SatResult_ptr satResult;
    Sat_SatSolver_ptr solver = Sat_GetSelectedSolver(get_sat_solver(options));

    satResult = Bmc_Sat_SolveInvarProblem(vars_mgr, solver, bmcProb);

    /* SIM result processing: */
    if (Sat_SatResult_IsError(satResult)) {
      /* solver internal error */
      Bmc_Problem_Delete(bmcProb);
      Sat_SatResult_Delete(&satResult);
      internal_error("Sorry, solver answered with a fatal Internal Failure during problem solving.\n");
    }
    else if (Sat_SatResult_IsUnsatisfiable(satResult)) {
      fprintf(nusmv_stdout, "-- ");
      print_invar(nusmv_stdout, invarspec);
      fprintf(nusmv_stdout, "  is true\n");
      
      Prop_SetStatus(invarprop, Prop_True);
      Bmc_Problem_Delete(bmcProb);
      Sat_SatResult_Delete(&satResult);
    }
    else {
      Bmc_SatTrace_ptr trace;
      
      /* no other choices are supplied: */
      nusmv_assert(Sat_SatResult_IsSatisfiable(satResult));
      
      trace = Bmc_SatTrace_Create(bmcProb, satResult);
      fprintf(nusmv_stdout, "-- cannot prove the ");
      print_invar(nusmv_stdout, invarspec);
      fprintf(nusmv_stdout, ": the induction fails\n");
      print_explanation(Bmc_SatTrace_GetSymbModelInvar(trace, vars_mgr),
			all_symbols);
      
      Bmc_Problem_Delete(bmcProb);
      Sat_SatResult_Delete(&satResult);
    }

  } /* call solver */
}




/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis           [This is only a useful wrapper for easily call
  Bmc_Utils_ExpandMacrosInFilename]

  Description        []

  SideEffects        [None]

  SeeAlso            []

******************************************************************************/
static void
bmc_expandFilename(const int k, const int l,
		   const int prop_idx,
		   const char* filename_to_be_expanded,
		   char* filename_expanded,
		   const size_t filename_expanded_maxlen)
{
  char szBuffer[1024];
  char szLoopback[16];

  /* Prepares the structure for macro-expansion: */
  SubstString aSubstTable[] =  { SYMBOL_CREATE(),
         SYMBOL_CREATE(),
         SYMBOL_CREATE(),
         SYMBOL_CREATE(),
         SYMBOL_CREATE(),
         SYMBOL_CREATE()
  };

  /* customizes the table with runtime values: */
  Utils_StripPathNoExtension(get_input_file(options), szBuffer);
  Bmc_Utils_ConvertLoopFromInteger(l, szLoopback, sizeof(szLoopback));

  SYMBOL_ASSIGN(aSubstTable[0], "@F", string,  "%s", get_input_file(options));
  SYMBOL_ASSIGN(aSubstTable[1], "@f", string,  "%s", szBuffer);
  SYMBOL_ASSIGN(aSubstTable[2], "@k", integer, "%d", k);
  SYMBOL_ASSIGN(aSubstTable[3], "@l", string, "%s", szLoopback);
  if (prop_idx != BMC_NO_PROPERTY_INDEX) {
    SYMBOL_ASSIGN(aSubstTable[4], "@n", integer, "%d", prop_idx);
  }
  else {
    SYMBOL_ASSIGN(aSubstTable[4], "@n", string, "%s", "undef");
  }
  SYMBOL_ASSIGN(aSubstTable[5], "@@", string,  "%s", "@");

  Bmc_Utils_ExpandMacrosInFilename(filename_to_be_expanded,
           aSubstTable,
           sizeof(aSubstTable)/sizeof(aSubstTable[0]),
           filename_expanded,
           filename_expanded_maxlen);
}

/**AutomaticEnd***************************************************************/
