package edu.cmu.cs.radar.integ;

/**
 * Defines constants for MCA-related data stored by the Task Manager. Further details at: <a
 * href="http://radar-wm.isri.cmu.edu:8000/radar/wiki/MCA/PublicDataModel">MCA Public Data Model</a>
 * <p>
 * The keys use the following prefix namespace acronyms:
 * <ul>
 * <li>TPK: Task Property Key</li>
 * <li>TMPK: Task Meta-Property Key</li>
 * <li>TRK: Task Relationship Key</li>
 * <li>PPK: Phase Property Key</li>
 * <li>PRK: Phase Relationship Key</li>
 * </ul>
 * 
 * @author faulring
 */
public interface MCAConstants
{
	/**
	 * The prefix for MCA constants.
	 */
	public final String MCA_PREFIX = "urn:edu.cmu.cs.radar.mca.";

	/**
	 * The prefix for task property keys.
	 */
	public final String TPK_PREFIX = MCA_PREFIX + "task.property.key.";

	/**
	 * The amount of time (in seconds) that the user should expect to spend to complete this task.
	 * <p>
	 * Type: Integer
	 * <p>
	 * Required
	 */
	public final String TPK_PLANNED_DURATION = TPK_PREFIX + "planned-duration";

	/**
	 * A boolean flag indicating whether or not this task type is a critical task.
	 * <p>
	 * Type: Boolean
	 * <p>
	 * Default: false
	 */
	public final String TPK_CRITICAL = TPK_PREFIX + "critical";

	/**
	 * The number of tasks of this type that are expected during the two-hour test.
	 * <p>
	 * Type: Integer
	 * <p>
	 * Required
	 */
	public final String TPK_EXPECTED_COUNT = TPK_PREFIX + "expected-count";

	/**
	 * The number of tasks that will be spawned after completing a task instance of this type. The number can
	 * be an estimate.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_NUMBER_TASKS_WILL_SPAWN = TPK_PREFIX + "number-tasks-will-spawn";

	/**
	 * The learned partial dependency order of the task type.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_PARTIAL_ORDER = TPK_PREFIX + "partial-order";

	/**
	 * The learned sequential order for task types.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_SEQUENTIAL_ORDER = TPK_PREFIX + "sequential-order";

	/**
	 * The Advice Integrator combines task ordering information from MCA components to produce an list that
	 * specifies the order in which to perform the tasks. This property specifies the position of the task
	 * within that ordered list. The first task to perform has a value of 1 for this property.
	 * <p>
	 * Type: Integer
	 * <p>
	 * Required
	 */
	public final String TPK_SEQUENCE_POSITION = TPK_PREFIX + "sequence-position";

	/**
	 * The time-varying impact on the score of completing this task. Represented as an integer between 0 and
	 * 9, inclusive with 9 being the highest priority.
	 * <p>
	 * Type: Integer
	 * <p>
	 * Required
	 * <p>
	 * Meta Properties:
	 * <ul>
	 * <li>TMPK_MIN_VALUE: Minimum allowable priority value (inclusive)
	 * <li>TMPK_MAX_VALUE: Maximum allowable priority value (inclusive)
	 * </ul>
	 */
	public final String TPK_PRIORITY = TPK_PREFIX + "priority";

	/**
	 * The time at which the user should start working on the task.
	 * <p>
	 * Type: Date
	 * <p>
	 * Required
	 */
	public final String TPK_PLANNED_START_TIME = TPK_PREFIX + "planned-start-time";

	/**
	 * A boolean flag indicating whether or not the MCA recommends that the user not perform this task.
	 * <p>
	 * Type: Boolean
	 * <p>
	 * Default: false
	 */
	public final String TPK_SHED = TPK_PREFIX + "shed";

	/**
	 * A boolean flag indicating whether or not the user has chosen to defer handling this task until later.
	 * <p>
	 * Type: Boolean
	 */
	public final String TPK_DEFERRED = TPK_PREFIX + "deferred";

	/**
	 * The time that the user spent working on the task. This should only be set when the difference between
	 * the start and end times is greater than the actual time the user spent working on the task, which can
	 * happen when the user interleaves tasks. If not set for a completed task, the actual duration should be
	 * the difference between the start and end time. The duration is represented in milliseconds.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_ACTUAL_DURATION = TPK_PREFIX + "actual-duration";

	/**
	 * The number of key presses while this task instance was focused.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_KEY_PRESS_EVENTS = TPK_PREFIX + "key-press-events";

	/**
	 * The number of mouse button presses while this task instance was focused.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_MOUSE_BUTTON_PRESS_EVENTS = TPK_PREFIX + "mouse-button-press-events";

	/**
	 * The number of mouse movement events while this task instance was focused.
	 * <p>
	 * Type: Integer
	 */
	public final String TPK_MOUSE_MOVEMENT_EVENTS = TPK_PREFIX + "mouse-movement-events";

	/**
	 * A boolean flag indicating whether or not the user should work on this task at the present time.
	 * Generally used to recommend that the user <b>not</b> work on a task.
	 * <p>
	 * Possible values of this property:
	 * <ul>
	 * <li>A false value means that the user should not work on this task. For example, the OPTIMIZE-SCHEDULE
	 * might have a false value for this property during the last 20 minutes of the test. If the user chooses
	 * to do the task, a warning should be issued. </li>
	 * <li>A true value means that the user should do this task now. For example, the PERFORM-VOT task might
	 * have a null for this property during the first hour of the test, then a true value during the second
	 * hour.</li>
	 * <li>A null value (that is, the property is not set) indicates no recommendation either way.</li>
	 * </ul>
	 * <p>
	 * Type: Boolean
	 */
	public final String TPK_RECOMMENDED = TPK_PREFIX + "recommended";

	/**
	 * A boolean flag indicating that this task is in the "expected" state. Such tasks are ones that learned
	 * task models predict will appear, but have not yet done so because their prerequisites have not been
	 * achieved. For example, a PUBLISH-SCHEDULE task is expected after an OPTIMIZE-SCHEDULE, but will not
	 * actually appear until after the schedule has changed.
	 * <p>
	 * Type: Boolean
	 */
	public final String TPK_EXPECTED = TPK_PREFIX + "expected";

	/**
	 * The prefix for task meta-property keys.
	 */
	public final String TMPK_PREFIX = MCA_PREFIX + "task.meta-property.key.";

	/**
	 * The minimum allowable property value (inclusive).
	 */
	public final String TMPK_MIN_VALUE = TMPK_PREFIX + "min-value";

	/**
	 * The maximum allowable property value (inclusive).
	 */
	public final String TMPK_MAX_VALUE = TMPK_PREFIX + "max-value";

	/**
	 * Prefix for task relationship keys.
	 */
	public final String TRK_PREFIX = MCA_PREFIX + "task.relationship.key.";

	/**
	 * The phase containing this task.
	 * <p>
	 * Opposite: PRK_TASKS
	 */
	public final String TRK_PHASE = TRK_PREFIX + "phase";

	/**
	 * Phases will be represented by ITask objects with this type.
	 */
	public final String TYPE_PHASE = "PHASE";

	/**
	 * The prefix for phase property keys.
	 */
	public final String PPK_PREFIX = MCA_PREFIX + "phase.property.key.";

	/**
	 * The sequential position of the phase starting with 1.
	 * <p>
	 * Type: Integer
	 * <p>
	 * Required
	 */
	public final String PPK_SEQUENCE_POSITION = PPK_PREFIX + "sequence-position";

	/**
	 * A boolean flag indicating whether or not this is the active phase. There may only be one active phase
	 * at a time.
	 * <p>
	 * Type: Boolean
	 * <p>
	 * Required
	 */
	public final String PPK_ACTIVE = PPK_PREFIX + "active";

	/**
	 * The time at which the user should start working on the tasks in this phase.
	 * <p>
	 * Type: Date
	 * <p>
	 * Required
	 */
	public final String PPK_PLANNED_START_TIME = PPK_PREFIX + "planned-start-time";

	/**
	 * The amount of time (in seconds) that the user should expect to spend to complete all the tasks in this
	 * phase.
	 * <p>
	 * Type: Integer>
	 * <p>
	 * Required
	 */
	public final String PPK_PLANNED_DURATION = PPK_PREFIX + "planned-duration";

	/**
	 * Prefix for phase relationship keys.
	 */
	public final String PRK_PREFIX = MCA_PREFIX + "phase.relationship.key.";

	/**
	 * All task instances that belong to this phase.
	 * <p>
	 * Opposite: TRK_PHASE
	 */
	public final String PRK_TASKS = PRK_PREFIX + "tasks";
}