/**
 * txt2etree - Convert a list of octant addresses to an etree
 * 
 * Each octant address is a text line of the form:
 *
 * <x> <y> <z> <level> <type> <value>
 * 
 * where <x>, <y>, and <z> are unsigned integer coordinates, <level>
 * is the octant level [0,..,31], <type> is either 0 (interior) or 1
 * (leaf), and <value> is the payload (C structure) of each octant
 */


#include <stdio.h>
#include <stdlib.h>



/* $begin txt2etree */
#include "etree.h"

typedef struct payload_t {
    int32_t val;
    char tag;
} payload_t;
    
int main(int argc, char **argv)
{
    etree_t *ep;
    etree_addr_t addr;
    int type, count=0;
    payload_t payload;
    char buf[ETREE_MAXBUF];

    /* $end txt2etree */
    if (argc != 2) {
        fprintf(stderr, "usage: %s etree < input.txt\n", argv[0]);
        exit(0);
    }
    /* $begin txt2etree */
    /* Create an empty 3d etree */
    ep = etree_open(argv[1], O_RDWR|O_CREAT|O_TRUNC, 1, sizeof(payload_t), 3); 

    /* $end txt2etree */
    if (ep == NULL) {
        fprintf(stderr, "Could not open etree file %s\n", argv[1]); 
        exit(1);
    }

    /* $begin txt2etree */
    /* Register the schema for the etree to make it portable */
    if (etree_registerschema(ep, "int32_t val; char tag;") != 0) {
        fprintf(stderr, "%s\n", etree_strerror(etree_errno(ep)));
        exit(1);
    }

    /* Insert each octant into the tree */
    while (fgets(buf, ETREE_MAXBUF, stdin)) {
        /* read an octant from the input, code skipped */
        /* $end txt2etree */

        sscanf(buf, "%u %u %u %d %d %d %c", &addr.x, &addr.y, &addr.z, 
               &addr.level,  &type, &payload.val, &payload.tag);
        addr.type = (type == 0) ? ETREE_INTERIOR : ETREE_LEAF;

        /* Ignore comment or blank lines */
        if ((buf[0] == '#') || (buf[0] == '\n') || (buf[0] == '\t')) {
            continue;
        }
        
        /* $begin txt2etree */
        count++;
        if (etree_insert(ep, addr, &payload) != 0) {
            fprintf(stderr, "%s\n", etree_strerror(etree_errno(ep))); 
            exit(1);
        }
        /* echo the address and payload of the octant, code skipped */
        /* $end txt2etree */
        printf("Loaded %s = %d %c\n", etree_straddr(ep, buf, addr), 
               payload.val, payload.tag);
        /* $begin txt2etree */
    }
    printf("Loaded %d octants\n", count);

    /* Add a comment to the dataset */
    if (etree_setappmeta(ep, "Dataset Generated by txt2etree") != 0) {
        fprintf(stderr, "%s\n", etree_strerror(etree_errno(ep))); 
        exit(1);
    }
    etree_close(ep);
    return 0;
}
/* $end txt2etree */


