(load "eubutil")

;; HZOSC
;;
;; rate of hz must be less than or equal to sound-srate, so
;; force resampling and issue a warning if necessary. snd-fmosc can
;; handle upsampling cases internally.
;;
;; hz must be a sound.
(defun hzosc (hz &optional (sound *table*) (phase 0.0))
  (let ((hz-srate (snd-srate hz)))
    (cond ((< *SOUND-SRATE* hz-srate)
	   (format t "Warning: down-sampling hz in hzosc-v~%")
	   (setf hz (snd-down *SOUND-SRATE* hz))))
    (scale-db (get-loud)
      (snd-fmosc 
	(car sound)		; samples for table
	(cadr sound)		; step represented by table
	*SOUND-SRATE*		; output sample rate
	0.0			; dummy carrier
	(local-to-global 0)	; starting time
	hz			; frequency
	phase))))		; phase

; hz is a scalar
(defun hzosc-dur (hz dur &optional (sound *table*) (phase 0.0))
  (hzosc (const hz dur) sound phase)
)

;;; fixed-parameter filters based on snd-biquad

; remember that snd-biquad uses the opposite sign convention for a_i's 
; than Matlab does.

; convenient biquad: normalize a0, and use zero initial conditions.
(defun biquad (x b0 b1 b2 a0 a1 a2)
  (let ((a0r (/ 1.0 a0)))
    (snd-biquad x (* a0r b0) (* a0r b1) (* a0r b2) 
		             (* a0r a1) (* a0r a2) 0 0)))

; biquad with Matlab sign conventions for a_i's.
(defun biquad-m (x b0 b1 b2 a0 a1 a2)
  (biquad x b0 b1 b2 a0 (- a1) (- a2)))

; two-pole lowpass
(defun lowpass2 (x hz &optional (q 0.7071))
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (cw (cos w))
	 (sw (sin w))
	 (alpha (* sw (sinh (/ 0.5 q))))
	 (a0 (+ 1.0 alpha))
	 (a1 (* -2.0 cw))
	 (a2 (- 1.0 alpha))
	 (b1 (- 1.0 cw))
	 (b0 (* 0.5 b1))
	 (b2 b0))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; two-pole highpass
(defun highpass2 (x hz &optional (q 0.7071))
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (cw (cos w))
	 (sw (sin w))
	 (alpha (* sw (sinh (/ 0.5 q))))
	 (a0 (+ 1.0 alpha))
	 (a1 (* -2.0 cw))
	 (a2 (- 1.0 alpha))
	 (b1 (- -1.0 cw))
	 (b0 (* -0.5 b1))
	 (b2 b0))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; two-pole bandpass.  max gain is unity.
(defun bandpass2 (x hz q)
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (cw (cos w))
	 (sw (sin w))
	 (alpha (* sw (sinh (/ 0.5 q))))
	 (a0 (+ 1.0 alpha))
	 (a1 (* -2.0 cw))
	 (a2 (- 1.0 alpha))
	 (b0 alpha)
	 (b1 0.0)
	 (b2 (- alpha)))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; two-pole notch.
(defun notch2 (x hz q)
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (cw (cos w))
	 (sw (sin w))
	 (alpha (* sw (sinh (/ 0.5 q))))
	 (a0 (+ 1.0 alpha))
	 (a1 (* -2.0 cw))
	 (a2 (- 1.0 alpha))
	 (b0 1.0)
	 (b1 (* -2.0 cw))
	 (b2 1.0))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; two-pole allpass.
(defun allpass2 (x hz q)
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (cw (cos w))
	 (sw (sin w))
	 (k (exp (* -0.5 w (/ 1.0 q))))
	 (a0 1.0)
	 (a1 (* -2.0 cw k))
	 (a2 (* k k))
	 (b0 a2)
	 (b1 a1)
	 (b2 1.0))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; bass shelving EQ.  gain in dB; Fc is halfway point.
; response becomes peaky at slope > 1.
(defun eq-lowshelf (x hz gain &optional (slope 1.0))
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (sw (sin w))
	 (cw (cos w))
	 (A (expt 10.0 (/ gain (* 2.0 20.0))))
	 (b (sqrt (- (/ (+ 1.0 (square A)) slope) (square (- A 1.0)))))
	 (apc (* cw (+ A 1.0)))
	 (amc (* cw (- A 1.0)))
	 (bs (* b sw))

	 (b0 (*      A (+ A  1.0 (- amc)    bs  )))
	 (b1 (*  2.0 A (+ A -1.0 (- apc)        )))
	 (b2 (*      A (+ A  1.0 (- amc) (- bs) )))
	 (a0           (+ A  1.0    amc     bs  ))
	 (a1 (* -2.0   (+ A -1.0    apc         )))
	 (a2           (+ A  1.0    amc  (- bs) )))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; treble shelving EQ.  gain in dB; Fc is halfway point.
; response becomes peaky at slope > 1.
(defun eq-highshelf (x hz gain &optional (slope 1.0))
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (sw (sin w))
	 (cw (cos w))
	 (A (expt 10.0 (/ gain (* 2.0 20.0))))
	 (b (sqrt (- (/ (+ 1.0 (square A)) slope) (square (- A 1.0)))))
	 (apc (* cw (+ A 1.0)))
	 (amc (* cw (- A 1.0)))
	 (bs (* b sw))

	 (b0 (*      A (+ A  1.0    amc     bs  )))
	 (b1 (* -2.0 A (+ A -1.0    apc         )))
	 (b2 (*      A (+ A  1.0    amc  (- bs) )))
	 (a0           (+ A  1.0 (- amc)    bs  ))
	 (a1 (*  2.0   (+ A -1.0 (- apc)        )))
	 (a2           (+ A  1.0 (- amc) (- bs) )))
    (biquad-m x b0 b1 b2 a0 a1 a2)))

; midrange EQ.  gain in dB, width in octaves (half-gain width).
(defun eq-band (x hz gain width)
  (let* ((w (* 2.0 Pi (/ hz (snd-srate x))))
	 (sw (sin w))
	 (cw (cos w))
	 (J (sqrt (expt 10.0 (/ gain 20.0))))
	 (g (* sw (sinh (* 0.5 (log 2.0) width (/ w sw)))))
	 (b0 (+ 1.0 (* g J)))
	 (b1 (* -2.0 cw))
	 (b2 (- 1.0 (* g J)))
	 (a0 (+ 1.0 (/ g J)))
	 (a1 (- b1))
	 (a2 (- (/ g J) 1.0)))
    (biquad x b0 b1 b2 a0 a1 a2)))

; see failed attempt in eub-reject.lsp to do these with higher-order fns:

; four-pole Butterworth lowpass
(defun lowpass4 (x hz)
  (lowpass2 (lowpass2 x hz 0.60492333) hz 1.33722126))

; six-pole Butterworth lowpass
(defun lowpass6 (x hz)
  (lowpass2 (lowpass2 (lowpass2 x hz 0.58338080) 
		                  hz 0.75932572) 
	                          hz 1.95302407))

; eight-pole Butterworth lowpass
(defun lowpass8 (x hz)
  (lowpass2 (lowpass2 (lowpass2 (lowpass2 x hz 0.57622191)
				            hz 0.66045510) 
	                                    hz 0.94276399)
	                                    hz 2.57900101))

; four-pole Butterworth highpass
(defun highpass4 (x hz)
  (highpass2 (highpass2 x hz 0.60492333) hz 1.33722126))

; six-pole Butterworth highpass
(defun highpass6 (x hz)
  (highpass2 (highpass2 (highpass2 x hz 0.58338080) 
			             hz 0.75932572) 
	                             hz 1.95302407))

; eight-pole Butterworth highpass
(defun highpass8 (x hz)
  (highpass2 (highpass2 (highpass2 (highpass2 x hz 0.57622191)
				                hz 0.66045510) 
	                                        hz 0.94276399)
	                                        hz 2.57900101))
