import sys
import time


class Timer:

    def __init__(self):
        """
        Initialize varaibles.
        """
        self.startTime = None
        self.endTime = None

    def start(self):
        """
        Start the timer.
        """
        self.startTime = time.time()

    def stop(self):
        """
        Stop the timer.
        """
        self.endTime = time.time()

    @staticmethod
    def __form__(d, form):
        """
        Format the output of the timer.
        d: number of seconds (difference between start and stop)
        form: normal = HH:MM:SS, s = SS..S, m = MM..M, h = HH..H
        Returns a string.
        """
        if form == 'normal':
            secs = int(d % 60)
            mins = int((d/60) % 60)
            hrs = int(d/3600)
            return "{:02}:{:02}:{:02}".format(hrs, mins, secs)
        if form == 's':
            return str(round(d, 2))
        elif form == 'm':
            return str(round(d/60, 2))
        elif form == 'h':
            return str(round(d/3600, 2))

    def getTime(self, form='normal'):
        """
        Get the current time on the timer.
        If it hasn't been stopped, then result is instantaneous.
        If it has been stopped, the the result is from the last stop.
        Returns a formatted string.
        """
        if self.endTime is None:
            diff = time.time() - self.startTime
            return Timer.__form__(diff, form)
        else:
            diff = self.endTime - self.startTime
            return Timer.__form__(diff, form)


class Progress:

    def __init__(self, maximum, out='stderr'):
        """
        Inititalize the progress bar with the expected maximum number of items.
        maximum: int, max number of expected items

        Writing to stderr ensures that the progress bar always overwrites
        itself, which is useful when running the script locally from the command
        line, but will sometimes cause issues when running from a 'job' on a
        remote server.  Writing to stdout will alleviate these issues but end up
        printing a new progress statement every time update() is called.
        """
        self.max = int(maximum)
        if out == 'stderr':
            self.out = sys.stderr
            self.end = ''
        elif out == 'stdout':
            self.out = sys.stdout
            self.end = '\n'
        self.timer = Timer()

    def start(self):
        """
        Start the progress bar (and timer).
        """
        self.timer.start()
        self.update(0)
        return self

    def update(self, progress):
        """
        Update progress bar with number of items finished.
        progress: current number of items finished
        """
        self.current = progress
        percent = self.current/self.max
        timeElapsed = self.timer.getTime()
        secElapsed = float(self.timer.getTime('s'))
        if self.current != 0:
            eta = Timer.__form__(((secElapsed/self.current) * self.max) - secElapsed, 'normal')
        else:
            eta = ''
        outStr = "{:.2%} ( {:} / {:} ), {:} elapsed, ETA: {:}{:<}\r".format(percent, self.current, self.max, timeElapsed, eta, '')
        outStr += self.end
        self.out.write(outStr)

    def finish(self):
        """
        Stop the timer, update the bar to the expected maximum, and move output
        to a new line.
        """
        self.timer.stop()
        self.update(self.max)
        self.out.write("\n\r")
