class ResultsConsumer(object):
    """
    Collect, organize, and analyze the results of the domain architecture
    simulator.
    """

    def __init__(self):
        """
        Initialize the ResultsConsumer with the parameters from the engine being
        analyzed.
        """
        self.statCounts = 0
        self.stats = {}

    def update(self, info):
        """
        Update stats
        """
        self.statCounts += 1
        self.stats[self.statCounts-1] = info

    @staticmethod
    def consume(engine):
        """
        Gather individual information/stats about the results of the simulation.
        """
        # Get histories and store them locally
        domArchHist = engine.getDomainArchHistory()
        eventHist = engine.getEventHistory()

        # Gather information about the results
        info = {}
        # Length(s)
        ##lengthList=[]
        ##for i in range(len(domArchHist)):
            ##lengthList.append(str(len(domArchHist[i])-2)) # Ignore null domains
        ##info['length list'] = ' '.join(lengthList)
        lastArch = domArchHist[-1]
        info['last DA'] = lastArch[1:-1]
        finalDALength = len(lastArch[1:-1])
        info['final length'] = finalDALength  # Ignore null domains
        #info['final length NR'] = len(set(lastArch[1:-1]))
        info['avg length'] = sum( [len(da)-2 for da in domArchHist] ) / len(domArchHist)
        for i in range(2,len(lastArch)-1):
            if lastArch[i] == lastArch[i-1]:
                finalDALength = finalDALength - 1 # Merge the same adjacent domains
        info['final compact length'] = finalDALength

        # Rates
        inserts = [0, 0]  # [events, occurred]
        deletes = [0, 0]
        for i in range(len(eventHist)):
            if eventHist[i][1] == 'ins':
                inserts[0] += 1
                if len(domArchHist[i+1]) > len(domArchHist[i]):
                    inserts[1] += 1
            elif eventHist[i][1] in ['del', 'ext']:
                deletes[0] += 1
                if len(domArchHist[i+1]) < len(domArchHist[i]):
                    deletes[1] += 1
        if inserts[0] == 0:
            info['insert rate'] = 0
            info['success freq insert'] = 0
        else:
            info['insert rate'] = inserts[0]/len(eventHist)
            info['success freq insert'] = inserts[1]/inserts[0]
        if deletes[0] == 0:
            info['delete rate'] = 0
            info['success freq delete'] = 0
        else:
            info['delete rate'] = deletes[0]/len(eventHist)
            info['success freq delete'] = deletes[1]/deletes[0]
        if inserts[0] == 0 and deletes[0] == 0:
            info['success freq event'] = 0
        else:
            info['success freq event'] = (inserts[1]+deletes[1])/(len(eventHist))

        # Max copy number
        copies = 0
        for domain in lastArch[1:-1]:
            copies = max((lastArch[1:-1].count(domain), copies))
        info['max copy number'] = copies

        # Generations
        info['generations'] = len(domArchHist)-1

        # Starting domains retained
        info['starting domains retained'] = ResultsConsumer._wereDomainsRetained(domArchHist)

        # Extinction (and generations)
        info['extinct'] = False
        info['extinction times'] = 0
        for i in range(1, len(domArchHist)-1):
            if len(domArchHist[i]) == 3 and len(domArchHist[i+1]) == 2:
                info['extinct'] = True
                break
        for i in range(1, len(domArchHist)-1):
            if len(domArchHist[i]) == 3 and eventHist[i][1] == 'del' and eventHist[i][4] == 0:
                info['extinction times'] += 1
        if info['extinct']:
            info['generations to extinction'] = len(domArchHist[1:i])
        else:
            info['generations to extinction'] = 0

        # Return this information to facilitate on-the-fly results writing
        # to disk
        return domArchHist, eventHist, info

    def analyze(self):
        """
        Perform aggregate analysis of results.
        """
        aggInfo = {}

        # Get max, median, mean, min, std dev, std err for final length
        aggInfo['final length'] = {}
        finLens = [self.stats[i]['final length'] for i in range(len(self.stats)) if not self.stats[i]['extinct']]
        if len(finLens) == 0:  # Everything went extinct
            finLenM4 = (0, 0, 0, 0)
            finLenSDSE = (0, 0)
        else:
            finLenM4 = ResultsConsumer._getM4(finLens)
            finLenSDSE = ResultsConsumer._getStandardDevErr(finLens)
        aggInfo['final length']['max'] = finLenM4[0]
        aggInfo['final length']['median'] = finLenM4[1]
        aggInfo['final length']['mean'] = finLenM4[2]
        aggInfo['final length']['min'] = finLenM4[3]
        aggInfo['final length']['std dev'] = finLenSDSE[0]
        aggInfo['final length']['std err'] = finLenSDSE[1]

        # Get max, median, mean, min, std dev, std err for final compact length
        aggInfo['final compact length'] = {}
        finComLens = [self.stats[i]['final compact length'] for i in range(len(self.stats)) if not self.stats[i]['extinct']]
        if len(finComLens) == 0:  # Everything went extinct
            finComLenM4 = (0, 0, 0, 0)
            finComLenSDSE = (0, 0)
        else:
            finComLenM4 = ResultsConsumer._getM4(finComLens)
            finComLenSDSE = ResultsConsumer._getStandardDevErr(finComLens)
        aggInfo['final compact length']['max'] = finComLenM4[0]
        aggInfo['final compact length']['median'] = finComLenM4[1]
        aggInfo['final compact length']['mean'] = finComLenM4[2]
        aggInfo['final compact length']['min'] = finComLenM4[3]
        aggInfo['final compact length']['std dev'] = finComLenSDSE[0]
        aggInfo['final compact length']['std err'] = finComLenSDSE[1]

        # Get max, median, mean, min, std dev, std err for generations
        aggInfo['generations'] = {}
        gens = [self.stats[i]['generations'] for i in range(len(self.stats)) if not self.stats[i]['extinct']]
        if len(gens) == 0:  # Everything went extinct
            gensM4 = (0, 0, 0, 0)
            gensSDSE = (0, 0)
        else:
            gensM4 = ResultsConsumer._getM4(gens)
            gensSDSE = ResultsConsumer._getStandardDevErr(gens)
        aggInfo['generations']['max'] = gensM4[0]
        aggInfo['generations']['median'] = gensM4[1]
        aggInfo['generations']['mean'] = gensM4[2]
        aggInfo['generations']['min'] = gensM4[3]
        aggInfo['generations']['std dev'] = gensSDSE[0]
        aggInfo['generations']['std err'] = gensSDSE[1]

        # Get max, median, mean, min, std dev, std err for extinction times
        aggInfo['extinction times'] = {}
        extTime = [self.stats[i]['extinction times'] for i in range(len(self.stats))]
        if len(extTime) == 0: # Never tried to extinct
            extTimeM4 = (0, 0, 0, 0)
            extTimeSDSE = (0, 0)
        else:
            extTimeM4 = ResultsConsumer._getM4(extTime)
            extTimeSDSE = ResultsConsumer._getStandardDevErr(extTime)
        aggInfo['extinction times']['max'] = extTimeM4[0]
        aggInfo['extinction times']['median'] = extTimeM4[1]
        aggInfo['extinction times']['mean'] = extTimeM4[2]
        aggInfo['extinction times']['min'] = extTimeM4[3]
        aggInfo['extinction times']['std dev'] = extTimeSDSE[0]
        aggInfo['extinction times']['std err'] = extTimeSDSE[1]

        # Get max, median, mean, min, std dev, std err for generations to extinction
        aggInfo['generations to extinction'] = {}
        gensToExt = [self.stats[i]['generations to extinction'] for i in range(len(self.stats)) if self.stats[i]['extinct']]
        if len(gensToExt) == 0:  # Nothing went extinct
            gensToExtM4 = (0, 0, 0, 0)
            gensToExtSDSE = (0, 0)
        else:
            gensToExtM4 = ResultsConsumer._getM4(gensToExt)
            gensToExtSDSE = ResultsConsumer._getStandardDevErr(gensToExt)
        aggInfo['generations to extinction']['max'] = gensToExtM4[0]
        aggInfo['generations to extinction']['median'] = gensToExtM4[1]
        aggInfo['generations to extinction']['mean'] = gensToExtM4[2]
        aggInfo['generations to extinction']['min'] = gensToExtM4[3]
        aggInfo['generations to extinction']['std dev'] = gensToExtSDSE[0]
        aggInfo['generations to extinction']['std err'] = gensToExtSDSE[1]

        # Get max, median, mean, min, std dev, std err for rate of insertion
        aggInfo['insert rate'] = {}
        insRate = [self.stats[i]['insert rate'] for i in range(len(self.stats))]
        if len(insRate) == 0:  # Never try to insert
            insRateM4 = (0, 0, 0, 0)
            insRateSDSE = (0, 0)
        else:
            insRateM4 = ResultsConsumer._getM4(insRate)
            insRateSDSE = ResultsConsumer._getStandardDevErr(insRate)
        aggInfo['insert rate']['max'] = insRateM4[0]
        aggInfo['insert rate']['median'] = insRateM4[1]
        aggInfo['insert rate']['mean'] = insRateM4[2]
        aggInfo['insert rate']['min'] = insRateM4[3]
        aggInfo['insert rate']['std dev'] = insRateSDSE[0]
        aggInfo['insert rate']['std err'] = insRateSDSE[1]

        # Get max, median, mean, min, std dev, std err for rate of deletion
        aggInfo['delete rate'] = {}
        delRate = [self.stats[i]['delete rate'] for i in range(len(self.stats))]
        if len(delRate) == 0:  # Never try to insert
            delRateM4 = (0, 0, 0, 0)
            delRateSDSE = (0, 0)
        else:
            delRateM4 = ResultsConsumer._getM4(delRate)
            delRateSDSE = ResultsConsumer._getStandardDevErr(delRate)
        aggInfo['delete rate']['max'] = delRateM4[0]
        aggInfo['delete rate']['median'] = delRateM4[1]
        aggInfo['delete rate']['mean'] = delRateM4[2]
        aggInfo['delete rate']['min'] = delRateM4[3]
        aggInfo['delete rate']['std dev'] = delRateSDSE[0]
        aggInfo['delete rate']['std err'] = delRateSDSE[1]

        # Get max, median, mean, min, std dev, std err for frequency of successful events
        aggInfo['success freq event'] = {}
        evenFre = [self.stats[i]['success freq event'] for i in range(len(self.stats))]
        if len(evenFre) == 0:
            evenFreM4 = (0, 0, 0, 0)
            evenFreSDSE = (0, 0)
        else:
            evenFreM4 = ResultsConsumer._getM4(evenFre)
            evenFreSDSE = ResultsConsumer._getStandardDevErr(evenFre)
        aggInfo['success freq event']['max'] = evenFreM4[0]
        aggInfo['success freq event']['median'] = evenFreM4[1]
        aggInfo['success freq event']['mean'] = evenFreM4[2]
        aggInfo['success freq event']['min'] = evenFreM4[3]
        aggInfo['success freq event']['std dev'] = evenFreSDSE[0]
        aggInfo['success freq event']['std err'] = evenFreSDSE[1]

        # Get average success freq, actual insert and delete rates
        aggInfo['success freq insert'] = sum([self.stats[i]['success freq insert'] for i in range(len(self.stats))]) / len(self.stats)
        aggInfo['success freq delete'] = sum([self.stats[i]['success freq delete'] for i in range(len(self.stats))]) / len(self.stats)

        # Percent stats: extinction, retained domains
        aggInfo['% extinct'] = sum([1 for i in range(len(self.stats)) if self.stats[i]['extinct']]) / len(self.stats)
        aggInfo['% starting domains retained'] = sum([1 for i in range(len(self.stats)) if self.stats[i]['starting domains retained']]) / len(self.stats)

        # Get length list from each simulation and final length list
        ##aggInfo['length lists']=[]
        ##for i in range(len(self.stats)):
            #aggInfo['length lists'].append(self.stats[i]['length list'])
        aggInfo['final length list'] = []
        for i in range(len(self.stats)):
            aggInfo['final length list'].append(self.stats[i]['final length'])
        aggInfo['final compact length list'] = []
        for i in range(len(self.stats)):
            aggInfo['final compact length list'].append(self.stats[i]['final compact length'])
        aggInfo['final DA list'] = []
        for i in range(len(self.stats)):
            aggInfo['final DA list'].append(self.stats[i]['last DA'])

        self.stats['aggregate'] = aggInfo

    @staticmethod
    def _wereDomainsRetained(domArchHist):
        """
        Return whether or not any of the domains from the starting domain
        architecture were retained in the final domain architecture.
        """
        flag = False
        for domain in domArchHist[0][1:-1]:
            if domain in domArchHist[-1][1:-1]:
                flag = True
        return flag

    @staticmethod
    def _getM4(numList):
        """
        Find and return the maximum, median, mean, and minimum of the number
        list.

        Return (max, median, mean, min)
        """
        nums = numList[:]
        nums.sort()
        maximum = nums[-1]
        median = nums[int(len(nums)/2)]
        mean = sum(nums)/len(nums)
        minimum = nums[0]
        return maximum, median, mean, minimum

    @staticmethod
    def _getStandardDevErr(numList):
        """
        Calculate the standard error and standard deviation for a list of
        numbers.

        Return (standard deviation, standard error)
        """
        # Calculate standard deviation
        mean = sum(numList)/len(numList)
        diffs = []
        for num in numList:
            diffs.append((mean-num)**2)
        stddev = (sum(diffs)/len(diffs))**.5
        # Calculate standard error
        stderr = (stddev/len(diffs))**.5
        return stddev, stderr

    def getStats(self):
        return self.stats

    def getAggregate(self):
        return self.stats['aggregate']


class ResultsWriter(object):
    """
    Write the results of an experiment to file.
    """

    import pickle
    import json

    @staticmethod
    def writeSingleSimulation(path, domArch, events, stats, name, compact=False, supercompact=False):
        """
        Write a JSON text file representing all othe output from a single
        simulation.

        path: the path to the file to be written
        domArch: the domain architecture history from the simulation
        events: the event history from the simulation
        stats: the statistics about the simulation from the ResultsConsumer
        name: the name of the file to write (.txt is automatically appended)
        compact: boolean to enable/disable compact output (no whitespace)
        """
        # Special case of writing the aggregate stats
        if domArch is None and events is None:
            temp = stats
        elif supercompact:  # simulation that needs further compacted
            tempDomArch = []
            tempEvents = []
            tempDomArch.append(domArch[0])
            for i in range(1, len(domArch)):
                if len(domArch[i-1]) != len(domArch[i]):
                    tempDomArch.append(domArch[i])
                    tempEvents.append(events[i-1])
            temp = {}
            temp['domArchHistory'] = tempDomArch
            temp['eventHistory'] = tempEvents
            temp['stats'] = stats
        else:  # Normal simulation
            temp = {}
            temp['domArchHistory'] = domArch
            temp['eventHistory'] = events
            temp['stats'] = stats

        if 'length lists' in stats.keys():
            maxEverLength = []
            outFile = open(path+"/length_lists.txt", 'w')
            for lengthList in stats['length lists']:
                maxEverLength.append(max(lengthList))
                outFile.write('\t'.join(lengthList.split())+'\n')
            outFile.close()
            print('max ever length ' + str(max(maxEverLength)))
            del stats['length lists']
        if 'final length list' in stats.keys():
            outFile = open(path+"/final_length_list.txt", 'w')
            for finalLength in stats['final length list']:
                outFile.write(str(finalLength)+'\n')
            outFile.close()
            del stats['final length list']
        if 'final compact length list' in stats.keys():
            outFile = open(path+"/final_compact_length_list.txt", 'w')
            for finalComLength in stats['final compact length list']:
                outFile.write(str(finalComLength)+'\n')
            outFile.close()
            del stats['final compact length list']
        if 'final DA list' in stats.keys():
            with open(path+'/final_DA_list.pkl', 'w+b') as daOutFile:
                ResultsWriter.pickle.dump(stats['final DA list'], daOutFile)
            outFile = open(path+"/final_DA_list.txt", 'w')
            for finalDA in stats['final DA list']:
                outFile.write(str(finalDA)+'\n')
            outFile.close()
            del stats['final DA list']
        with open(path+"/{}.txt".format(name), 'w') as f:
            if compact:
                ResultsWriter.json.dump(temp, f, sort_keys=True, indent=None)
            else:
                ResultsWriter.json.dump(temp, f, sort_keys=True, indent=4)


class ResultsReader(object):
    """
    Read the results of an experiment from a file.
    """

    import json
    import os

    @staticmethod
    def readSimulation(path):
        """
        Read a simulation file (with JSON content).

        path: the path to the file to be written
        """
        with open(path, 'r') as f:
            return ResultsWriter.json.load(f)

    @staticmethod
    def loadDirectory(dirPath, loadAll=False, progress=False):
        """
        Load all of the simulation files from a directory.  By default, do
        not load anything with a non-numeric filename.  These files
        (aggregate.txt) will not be simulation results and thus we're not
        interested in loading them for graphical analysis.

        dirPath: directory path
        loadAll: flag to include loading of files that are NOT simulation
                 data
        """
        fileList = ResultsReader.os.listdir(dirPath)
        results = {}
        for filename in fileList:
            filepath = dirPath+"/"+filename
            key = filename[:filename.find('.')]
            if not loadAll:  # load only simulation data files
                if key.isnumeric():
                    results[key] = ResultsReader.readSimulation(filepath)
        return results

    @staticmethod
    def readAggregate(dirPath):
        """
        Read the aggregate file from the directory (at 'dirPath').
        """
        return ResultsReader.readSimulation(dirPath+'/aggregate.txt')


class CustomCSVWriter(object):
    """
    A dictionary CSV Writer.
    """

    def __init__(self, filePath, headerKeys):
        """
        Open the file and write the header.

        filePath: path to the file
        headerKeys: ordered list of dict keys
        """
        self.f = open(filePath, 'w')
        self.headerKeys = headerKeys
        self.writeHeader()

    def _write(self, string):
        """
        Write one line to the CSV file (with new line ending).

        string: the line to write
        """
        self.f.write(string + "\n")

    def __exit__(self, exception_type, exception_val, trace):
        """
        Use a Python magic method for context managers to close the
        file object.
        """
        self.f.close()

    def getKeys(self):
        """
        Get the keys for the CSV header
        """
        return self.headerKeys

    def writeHeader(self):
        """
        Write the header, the ordered keys, for the CSV file.
        """
        s = ""
        for k in self.headerKeys:
            s += str(k)+','
        s = s[:-1]
        self._write(s)

    def writeRow(self, dictRow):
        """
        Write the contents of 'dictRow' that correspond to the keys
        given in the constructor.

        dictRow: the dictionary to write in one row
        """
        s = ""
        for key in self.headerKeys:
            try:
                s += str(dictRow[key]) + ','
            except KeyError:
                s += ','
        self._write(s[:-1])
