from Simulation.Abstract import Model


class Immortality(Model):
    """
    Domain architectures are immortal and never go extinct
    (aka never lose their last domain).
    """

    def __init__(self):
        self.resurrection = False

    def getEventEffects(self, domArch):
        """
        Get the event parameters for extinction.
        Return (event type string, whether extinction occurs).
        """
        return 'ext', False

    def __str__(self):
        return "Immortality"


class ProbabilityExtinctionModel(Model):
    """
    Use a probability of survival (P = C(0,D1,0) / sum_x C(0,Dx,0) ) to
    decide whether a domain architecture goes extinct.
    """
    import pickle
    from Parsing import NULLDOM
    from random import random
    
    def __init__(self, pathToTriplePkl, multiplier):
        """
        Count the triples requried to calculate the probability of survival.

        pathToTriplePkl: path to the count of triples (pkl) generated in the
                         pre-processing step
        multiplier: survival probability multiplier
        """
        self.multiplier = multiplier
        with open(pathToTriplePkl, 'r+b') as trplPkl:
            trplCounts = self.pickle.load(trplPkl)
        self.counts = {}
        for domTup in trplCounts:
            if domTup[0] == self.NULLDOM and domTup[2] == self.NULLDOM:
                self.counts[domTup] = trplCounts[domTup]
        self.sumCounts = sum(self.counts.values())

    def getEventEffects(self, domArch):
        """
        Get the success (True) or failure (False) of the extinction event.
        Return ('ext', bool)
        """
        probExt = self.random()
        if probExt > self._getProbOfSurvival(domArch):
            return 'ext', True
        else:
            return 'ext', False

    def _getProbOfSurvival(self, domArch):
        """
        Calcualte probability of survival for a given domain architecture.
        """
        return self.multiplier * (self.counts[tuple(domArch)]/self.sumCounts)


class PermanentExtinction(ProbabilityExtinctionModel):
    """
    See ProbabilityExtinctionModel.
    Resurrection is not allowed.
    """

    def __init__(self, **kwargs):
        ProbabilityExtinctionModel.__init__(self, **kwargs)
        self.resurrection = False

    def __str__(self):
        return "Permanent Extinction"


class Resurrection(ProbabilityExtinctionModel):
    """
    See ProbabilityExtinctionModel.
    Resurrection is allowed.
    """

    def __init__(self, **kwargs):
        ProbabilityExtinctionModel.__init__(self, **kwargs)
        self.resurrection = True

    def __str__(self):
        return "Resurrection"
