class Model(object):
    """
    An abstract data model for domain architecture simulation.
    """

    def __init__(self):
        """
        Create a new data model using some data to influence events.
        Data is probably a list of domain architectures.
        Any pre-processing (counting, etc.) should be done here.
        """
        raise NotImplementedError("Inherited from abstract class.")

    def getEventEffects(self, event):
        """
        Get the effects of an event (insertion, deletion, ...) using this model.
        event: string
        Return a tuple with the event effects (location, action, ...).
        """
        raise NotImplementedError("Inherited from abstract class.")


class InsertionEvent(object):
    """
    An abstract insertion event.  Defines the way an insertion is performed, in
    general.
    """

    def getEventParameters(domArch):
        """
        Determine the location of the insertion and the domain to be inserted.
        Return this information as a tuple, prefaced with the insertion keyword.
        """
        raise NotImplementedError("Inherited from abstract class.")

    def _getLocation(domArch):
        """
        Determine and return the location of the proposed insertion.
        """
        raise NotImplementedError("Inherited from abstract class.")

    def _getDomain(domArch):
        """
        Determine and return the domain to be inserted for the proposed
        insertion.
        """
        raise NotImplementedError("Inherited from abstract class.")


class DeletionEvent(object):
    """
    An abstract deletion event.
    Defines the way a deletion is performed, in general.
    """

    def getEventParameters(domArch):
        """
        Determine the location of the deletion.
        Return this information as a tuple, prefaced with the deletion keyword.
        """
        raise NotImplementedError("Inherited from abstract class.")

    def _getLocation(domArch):
        """
        Determine and return the location of the proposed deletion
        """
        raise NotImplementedError("Inherited from abstract class.")


class InvalidModelError(Exception):
    """
    An exception to raise when the simulator cannot identify the model
    specified.
    """

    def __init__(self, val):
        self.val = val

    def __str__(self):
        return repr(self.val)
