class Doubles(object):

    def getDoubles(domArchs):
        """
        Get a list of doubles from a list of domain architectures.

        domArchs: list of list of domain ID's (strings)

        Return a list of tuples (of strings)
        """
        dbls = []
        for arch in domArchs:
            for i in range(len(arch)-1):
                dbls.append( tuple(arch[i:i+2]) )
        return dbls

class Triples(object):

    def getTriples(domArchs):
        """
        Get a list of triples from a list of domain architectures.

        domArchs: list of list of domain ID's (strings)

        Return a list of tuples (of strings)
        """
        trpls = []
        for arch in domArchs:
            for i in range(len(arch)-2):
                trpls.append( tuple(arch[i:i+3]) )
        return trpls

class Domains(object):
    import Parsing

    def getDomainAlphabet(container):
        """
        Get the unique set of domains (the domain alphabet, \sigma) from a list of domain
        architectures or a dictionary of sequences.

        container: list of domain architectures (list of list of string ID's), or
                   dictionary of sequences (key = sequence ID, value = list of domain
                   info tuples)

        Return a set of domain ID strings.
        """
        if isinstance(container, type([])): # is a list of domain architectures
            domains = {domain for domArch in container for domain in domArch}
        elif isinstance(container, type({})): # is a dictionary of sequences
            domains = {domain[0] for domArch in container.values() for domain in domArch}
        else:
            raise Exception("Domains.getDomainAlphabet: parameter was not a list or dict")
        if Domains.Parsing.NULLDOM in domains:
            domains.remove(Domains.Parsing.NULLDOM)
        return domains

    def getDomainFrequency(domArchs):
        """
        Get the domain frequency of all domains in the specified
        domain architectures.  Domain frequency is the number of times
        a specific domain occurs divided by occurrences of all
        domains.

        domArchs: list of domain architectures (as lists of domains)
        """
        counts = {}
        allDomains = 0
        for da in domArchs:
            allDomains += len(da)-2 # don't count NULL domains
            for d in da:
                temp = counts.setdefault(d, 0)
                counts[d] = temp + 1
        if Domains.Parsing.NULLDOM in counts:
            del counts[Domains.Parsing.NULLDOM]
        domFreq = {k: counts[k]/allDomains for k in counts}
        return domFreq

    def iterate(domArchs):
        """
        A simple generator function for iterating over every domain in a set of
        domain architectures.
        """
        for arch in domArchs:
            for domain in arch:
                yield domain

# Defined outside of any classes because Python can only pickle top-level
# classes and functions, the pickling is a requirement of the multiprocessing
# library for Windows
def countDomainOccurrencesThread(inputQ, outputQ, progressQ, domainTuples):
    """
    Collect the tuples that a given domain participates in.  Specifically, take
    one domain from the input queue (until 'None' is reached), and iterate over
    all domain tuples, keeping track of the ones that contain the specified
    domain.  Add these tuples to a dictionary keyed on the domain, and add the
    dictionary to the output queue once all domains have been consumed.

    inputQ: Queue, contains all domains (strings)
    outputQ: Queue, output dictionaries go here
    progressQ: once a domain is done being checked, a 0 is placed here to track
               progress
    domainTuples: a list of domain tuples (doubles or triples) that are checked
                  for domain participation
    """
    domainOccurrence = {}
    for domain in iter(inputQ.get, None):
        temp = []
        for tupl in domainTuples:
            if domain in tupl:
                temp.append(tupl)
        domainOccurrence[domain] = temp
        progressQ.put(0)
    outputQ.put(domainOccurrence)
