import json
import os

required = ['dataPath', 'model', 'branchLength',
            'lambda', 'replicates',  'expType',
            'startDomains', 'pseudoCount','insertionBias']

defaults = {'model': 'uniform',
            'branchLength': 100,
            'lambda': 1.0,
            'replicates': 1,
            'expType': 'normal',
            'startDomains': [],
            'pseudoCount': 0.05,
            'insertionBias': 0.5,
            'extra': ''}


def getExperimentDescriptor(filename, check=True):
    """
    Open the supplied text file and read the JSON content.

    filename: String of the path to the file
    check: boolean to check if all required arguments are present and
           insert defaults if they are not
    """
    with open(filename, 'r') as f:
        exp = json.load(f)
    # Check dataPath for trailing slash (and remove it)
    if exp['dataPath'].endswith('/') or exp['dataPath'].endswith('\\'):
        exp['dataPath'] = exp['dataPath'][:-1]
    # Check for presence of all required parameters
    if check:
        _check(exp)
    return exp


def writeExperimentDescriptor(filename, expDesc):
    """
    Write an experiment descriptor to file (as JSON).

    filename: String of the path to the file
    expDesc: the experiment descriptor (dict)
    """
    with open(filename, 'w') as f:
        json.dump(expDesc, f, indent=0)


def _check(expDict):
    """
    Check to see if the required arguments have been set.  If they
    haven't, set them to the defaults.

    expDict: the experiment argument dictionary
    """
    for item in required:
        if item not in expDict:
            try:
                _default(expDict, item)
            except KeyError:
                raise Exception("Attribute {} not in the experiment descriptor!".format(item))


def _default(expDict, param):
    """
    Set one entry of the experiment argument dictionary to the default
    value.

    expDict: the experiment argument dictionary
    param: String, the entry name
    """
    expDict[param] = defaults[param]


def convertExperimentToArgs(expDesc):
    """
    Convert the experiment descriptor dictionary into a Simulator Engine
    compatible dictionary of keyword arguments (kwargs).
    Return the keyword argument dictionary.
    """
    kwargs = {}
    kwargs['expType'] = expDesc['expType']
    # Set all of the variable parameters normally
    if expDesc['expType'] == 'normal':
        kwargs['eventRate'] = expDesc['lambda']
        kwargs['branchLength'] = expDesc['branchLength']
        kwargs['replicates'] = expDesc['replicates']
        kwargs['pseudoCount'] = expDesc['pseudoCount']
        kwargs['insertionBias'] = expDesc['insertionBias']
    elif expDesc['expType'] == 'batch':
        # Capture the argument range
        kwargs['start'] = expDesc['start']
        kwargs['stop'] = expDesc['stop']
        kwargs['step'] = expDesc['step']
        # For branchLength and eventRate, capture the complement of
        # parameters
        if expDesc['branchLength'] == 'batch':
            kwargs['batch'] = 'branchLength'
            kwargs['eventRate'] = expDesc['lambda']
            kwargs['replicates'] = expDesc['replicates']
            kwargs['pseudoCount'] = expDesc['pseudoCount']
            kwargs['insertionBias'] = expDesc['insertionBias']
        elif expDesc['lambda'] == 'batch':
            kwargs['batch'] = 'eventRate'
            kwargs['branchLength'] = expDesc['branchLength']
            kwargs['replicates'] = expDesc['replicates']
            kwargs['pseudoCount'] = expDesc['pseudoCount']
            kwargs['insertionBias'] = expDesc['insertionBias']
        elif expDesc['replicates'] == 'batch':
            kwargs['batch'] = 'replicates'
            kwargs['branchLength'] = expDesc['branchLength']
            kwargs['eventRate'] = expDesc['lambda']
            kwargs['pseudoCount'] = expDesc['pseudoCount']
            kwargs['insertionBias'] = expDesc['insertionBias']
        elif expDesc['pseudoCount'] == 'batch':
            kwargs['batch'] = 'pseudoCount'
            kwargs['branchLength'] = expDesc['branchLength']
            kwargs['eventRate'] = expDesc['lambda']
            kwargs['replicates'] = expDesc['replicates']
            kwargs['insertionBias'] = expDesc['insertionBias']
        elif expDesc['insertionBias'] == 'batch':
            kwargs['batch'] = 'insertionBias'
            kwargs['branchLength'] = expDesc['branchLength']
            kwargs['eventRate'] = expDesc['lambda']
            kwargs['replicates'] = expDesc['replicates']
            kwargs['pseudoCount'] = expDesc['pseudoCount']
        else:
            print("Error: misuse of batch mode. Please check your experiment discription.")
            print("Exiting...")
            os.exit(0)
    else:
        print("Error: expType can only be normal or batch. Please check your experiment discription.")
        print("Exiting...")
        os.exit(0)            

    # All other parameters that are not dependent on experiment type
    kwargs['dataset'] = expDesc['dataPath']
    kwargs['model'] = expDesc['model']
    kwargs['startDomArch'] = expDesc['startDomains']
    # Optional arguments
    if 'use_pickle' in expDesc:
        kwargs['use_pickle'] = expDesc['use_pickle']
    if 'extra' in expDesc:
        kwargs['extra'] = expDesc['extra']
    return kwargs


def convertArgsToExperiment(kwargs):
    exp = {}
    exp['expType'] = "normal"
    exp['lambda'] = kwargs['eventRate']
    exp['branchLength'] = kwargs['branchLength']
    exp['replicates'] = kwargs['replicates']
    exp['model'] = kwargs['model']
    exp['dataPath'] = kwargs['dataset']
    exp['startDomains'] = kwargs['startDomArch']
    exp['pseudoCount'] = kwargs['pseudoCount']
    exp['insertionBias'] = kwargs['insertionBias']
    if 'use_pickle' in kwargs:
        exp['use_pickle'] = kwargs['use_pickle']
    if 'extra' != '':
        exp['extra'] = kwargs['extra']

    return exp
