import argparse
from copy import deepcopy
from multiprocessing import Process, Pipe, Manager, current_process, Pool, cpu_count
import os
import shutil
import sys
import time
from queue import Empty

sys.path.append(os.path.dirname(os.path.realpath(__file__)) + "/lib")
from Parsing import ExperimentJSON
from Simulation import X_Simulator
#import X_Simulator
from Simulation.Results import ResultsConsumer, ResultsWriter
from Simulation.DomainCountContext import DomainCountContextModel
from Util import Progress


def parseArgs():
    """
    Parse command-line arguments.
    """
    parser = argparse.ArgumentParser(description="Generate domain architectures according to experimental parameters.")
    parser.add_argument('exp_descriptor', action='store', type=str, default=None,
                        help="Path to the file containing the experimental parameters.")
    parser.add_argument('-p', '--processes', action='store', type=int, default=1, dest='procs',
                        help="Number of processes to use for simulator engines.")
    parser.add_argument('-n', '--noprogress', action='store_true', default=False,
                        help="Disable progress tracking and printout.")
    parser.add_argument('-D', '--delete', action='store_true', default=False,
                        help="Delete any old results associated with the specified experiment")
    parser.add_argument('-c', '--compact', action='store_true', default=False, dest='compact',
                        help="Make output compact by removing whitespace.")
    parser.add_argument('-s', '--supercompact', action='store_true', default=False,
                        help="Make output compact by only recording successful events.")
    return parser.parse_args()


def processHandler(procs, kwargs, noprogress, compact, supercompact):
    """
    Handles the multi-processes for a single experiment

    procs: The number of worker processes
    kwargs: keyword argument dictionary
    noprogress: boolean for disable progress tracking
    compact: boolean for compact output
    supercompact: boolean for further compact output
    """
    # Create multiprocessing management, queues, pool and the result consumer
    procManager = Manager()
    statsQ = procManager.Queue() # Managed Queue for stats
    simulationPool=Pool(processes = procs + 1, maxtasksperchild = 50)

    print("Launching simulations...")
    # Initialize and start the ResultsConsumer process
    aggStats = simulationPool.apply_async(ResultsConsumerHandler,args=(statsQ,
                                                            kwargs['replicates'],
                                                            noprogress))
    # Initialize and start the simulation processes
    for i in range(kwargs['replicates']):
        simulationPool.apply_async(EngineHandler,args=(statsQ, kwargs, compact, supercompact, i))

    # Begins doing analysis and close the processes pool
    aggregateStats = aggStats.get() #Get the results consumer
    simulationPool.close()
    simulationPool.join()
    print("Writing results...")
    ResultsWriter.writeSingleSimulation(kwargs['output'], None, None,
                                        aggregateStats, "aggregate")
    ExperimentJSON.writeExperimentDescriptor(kwargs['output']+"/experiment.txt", ExperimentJSON.convertArgsToExperiment(kwargs))


def EngineHandler(statsQ, kwargs, compact, supercompact, counter):
    """
    (Target function for the Process that does simulations) Handles creating
    Simulator Engines and placing their output in the output queue.

    statsQ: Managed Queue for stats
    kwargs: keyword argument dictionary for Simulator Engine
    compact: boolean for compact output
    supercompact: boolean for further compact output
    counter: the number of simulations that have be launched
    """
    z = len(str(kwargs['replicates']))
    eng = X_Simulator.Engine(**kwargs)
    eng.run()
    da, events, stats = ResultsConsumer.consume(eng)
    ResultsWriter.writeSingleSimulation(kwargs['output'], da, events, stats, str(counter).zfill(z), compact, supercompact)
    statsQ.put(stats)


def ResultsConsumerHandler(statsQ, expected, noprogress):
    """
    (This is the target function for for the Process that reads the simulation
    results) Check the output queue and feed any new objects to the
    ResultsConsumer until all objects have been processed.  Once everything has
    been processed, send the ResultsConsumer down the pipe back to the wrapper
    (for analysis and writing results to disk).

    statsQ: Managed Queue for stats
    expected: Expected number of objects to process
    noprogress: boolean for disable progress tracking
    """
    resCon = ResultsConsumer()
    counter = resCon.statCounts
    while counter < expected:
        # if the output queue is not empty, process one available engine/result
        if not statsQ.empty():
            resCon.update(statsQ.get())
            counter = resCon.statCounts
            if not noprogress:
                print(str(counter) + '/' + str(expected) + '\r', end = '')
        # Otherwise, wait
        else:
            time.sleep(1)
    # Analyze the results in aggregate, and send the aggregate back
    print('analyzing...')
    resCon.analyze()
    aggStats = resCon.getAggregate()
    return aggStats


def checkOutputDirectory(expDesc, args, extra=""):
    """
    Use a command-line argument or user input to decide whether to delete
    existing results, if they exist.  If the user specifies deletion, delete the
    directory and its contents, and recreate it (empty).  If the user does not
    want the directory to be deleted, then print a message and exit.  Otherwise,
    just create it.

    Output folder -> "output" directory next to the experiment descriptor

    expDesc: the experiment descriptor (dict)
    args: the command line arguments
    extra: any extra directory characters
    """
    outputPath = os.path.dirname(args.exp_descriptor)+'/output'+extra
    if "decision" in dir(args):
        if args.decision == "yes":
            args.delete = True
        else:
            args.delete = False
    if os.path.exists(outputPath):
        if args.delete:
            shutil.rmtree(outputPath)
        else:
            args.decision = input("Delete the existing results? [yes|no] ")
            if args.decision == 'yes':
                shutil.rmtree(outputPath)
            else:
                print("Please move your old results to a new location and run the simulator again.")
                print("Exiting")
                sys.exit()
    os.mkdir(outputPath)
    return outputPath


def prepareModel(kwargs):
    """
    Create a pickled model for use during the experiment.  Instead of the model
    being recreated for every iteration, we load the pickled model instead.
    This is at least as fast as recreating the model (and often faster), while
    giving us peace of mind that the model used is identical in all experiments.

    Also adds 'model_pkl' key to kwargs and assigns it a value of the file path
    of the pickled model.

    kwargs: the experiment descriptor as a dictionary
    """
    # This is highly repetitive and non-DRY...but it works and it's easy
    if kwargs['model'] == 'domaincountcontext':
        print("Building the initial Domain Model...")
        model = DomainCountContextModel(kwargs['dataset'])
        model.pickle()
        kwargs['model_pkl'] = model.pickleName
    del model


if __name__ == "__main__":
    # Parse command-line arguments and start timing
    args = parseArgs()
    startTime = time.time()

    # Parse JSON experiment descriptor into a dictionary
    print("Reading experiment file and checking filesystem...")
    expDesc = ExperimentJSON.getExperimentDescriptor(args.exp_descriptor)
    kwargs = ExperimentJSON.convertExperimentToArgs(expDesc)
    # Create the keyword argument dictionaries for each experiment
    kwargList = []
    if kwargs['expType'] == 'normal':
        kwargs['output'] = os.path.abspath(checkOutputDirectory(expDesc, args,extra=kwargs['extra']))
        kwargList.append(kwargs)
    elif kwargs['expType'] == 'batch':
        it = X_Simulator.FloatIter(kwargs['start'], kwargs['stop'], kwargs['step'])
        for i in it:
            clone = deepcopy(kwargs)
            clone[kwargs['batch']] = i
            extra = "_{}-{}".format(kwargs['batch'][0:2], i)
            clone['output'] = os.path.abspath(checkOutputDirectory(expDesc, args, extra=extra))
            kwargList.append(clone)

    # Begin running experiments
    counter = 1
    print("Going to run {} simulations".format(len(kwargList)))
    for kwargs in kwargList:
        print("\nRunning Simulation {} of {}...".format(counter, len(kwargList)))
        counter += 1
        ########## Set up environment ##########
        print("Setting up simulation environment...")
        # Prepare the model pickle, if necessary
        prepareModel(kwargs)

        ########## Run ##########
        processHandler(args.procs, kwargs, args.noprogress, args.compact, args.supercompact)
    timeSpent = round(time.time() - startTime, 2)
    print('Done.\nTime spent: ' + str(timeSpent) + 's.')
