-- Companion code to
--    Focusing on Binding and Computation
--    Daniel R. Licata, Noam Zeilberger, Robert Harper

open import lib.Prelude
open List using (_++_)  
open List.InLists
open List.Subsets

module focbind.Language where

module Types where

  Atom : Set 
  Atom = String
  
  data Kind : Set where
    Type⁺ : Kind
    Type⁻ : Kind

  mutual
    Pos = Type Type⁺
    Neg = Type Type⁻
    
    data Rule : Set where
      _⇐_ : Atom -> List Pos -> Rule
  
    data Type : Kind -> Set where
      X⁺  : Atom -> Pos
      ↓   : Neg -> Pos
      1⁺  : Pos
      _*_ : Pos -> Pos -> Pos
      0⁺  : Pos
      _+_ : Pos -> Pos -> Pos
      D⁺  : Atom -> Pos
      _⇒_ : Rule -> Pos -> Pos
      □ : Pos -> Pos

      X⁻  : Atom -> Neg
      ↑   : Pos -> Neg
      _∧_ : Rule -> Neg -> Neg
      ⊤   : Neg
      _→_ : Pos -> Neg -> Neg
      _&_  : Neg -> Neg -> Neg
      ◇    : Neg -> Neg
--    _ν_ : Atom -> Neg -> Neg
--    ∀W [R1 ... Rn] A- = ν X. A- & (R1 ∧ X) ... (Rn ∧ X)
--    but I don't feel like defining atom substitution.
      ∀W  : List Rule -> Neg -> Neg

  head : Rule -> Atom
  head (R ⇐ _ ) = R

  infixr 15 _→_
  infixr 15 _⇒_
  infixl 15 _⇐_
  infixr 15 _&_
  infixr 15 _∧_
  infixr 15 _+_
  infixr 15 _*_

module Contexts where    
  open Types

  RCtx : Set
  RCtx = List Rule

  ax : Atom -> Rule
  ax a = a ⇐ []
  
  C : Set -> Set
  C A = RCtx × A

  <_>_ : {A : Set} -> RCtx -> A -> C A
  < Ψ > x = Ψ , x
  infixr 13 <_>_
  
  data Hyp : Set where
    _atom⁺ : Atom -> Hyp
    _true⁻ : C Neg -> Hyp
  
  data Conc : Set where
    _true⁺ : C Pos -> Conc
    _atom⁻ : Atom -> Conc
  
  LCtx : Set
  LCtx = List Hyp
  
  UCtx : Set
  UCtx = List (List Hyp)

module Focus (InΣ : Types.Rule -> Set) where
  open Types public
  open Contexts public

  module Pats where  
    mutual   
      data CPats : LCtx -> C (List Pos) -> Set where
        C[]   : { Ψ : RCtx} -> CPats [] (< Ψ > [])
        _C:_  : {Δ1 : LCtx} { Δ2 : LCtx } {Ψ : RCtx} {A : Pos} {As : List Pos}
                -> CPat Δ1 (< Ψ > A) -> CPats Δ2 (< Ψ > As)
                -> CPats (Δ2 ++ Δ1) (< Ψ > A :: As)
      infixr 10 C:

    
      data CPat : LCtx -> C Pos -> Set where
        Cx⁺ : {Ψ : RCtx} {X : Atom} -> CPat [ X atom⁺ ] (< Ψ > X⁺ X)
        Cx⁻ :  {Ψ : RCtx} {A- : Neg} -> CPat [ (< Ψ > A-) true⁻ ] (< Ψ > ↓ A-)
        C<> :  {Ψ : RCtx} -> CPat [] (< Ψ > 1⁺)
        Cpair :  {Δ1 : LCtx} {Δ2 : LCtx} {Ψ : RCtx} {A : Pos} {B : Pos}
                -> CPat Δ1 (< Ψ > A) -> CPat Δ2 (< Ψ > B)
                -> CPat (Δ2 ++ Δ1) (< Ψ > A * B)
        Cinl : {Ψ : RCtx} {Δ : LCtx} {A B : Pos} -> CPat Δ (< Ψ > A) -> CPat Δ (< Ψ > A + B)
        Cinr : {Ψ : RCtx} {Δ : LCtx} {A B : Pos} -> CPat Δ (< Ψ > B) -> CPat Δ (< Ψ > A + B)
        Cc   : {Ψ : RCtx} {Δ : LCtx} {P : Atom} {As : List Pos} ->
               InΣ (P ⇐ As) -> CPats Δ (< Ψ > As) -> CPat Δ (< Ψ > D⁺ P)
        Cv   :  {Ψ : RCtx} {Δ : LCtx} {P : Atom} {As : List Pos} ->
               (P ⇐ As) ∈ Ψ -> CPats Δ (< Ψ > As) -> CPat Δ (< Ψ > D⁺ P)
        Cλ   :  {R : Rule} {Ψ : RCtx} {Δ : LCtx} {A : Pos} 
             -> CPat Δ (< R :: Ψ > A) -> CPat Δ (< Ψ > R ⇒ A)
        Cbox : {Ψ : RCtx} {Δ : LCtx} {A+ : Pos} -> CPat Δ (< [] > A+) -> CPat Δ (< Ψ > □ A+)
        
    data DPat : LCtx -> C Neg -> Conc -> Set where
      De⁻  : forall {Ψ X} -> DPat [] (< Ψ > X⁻ X) (X atom⁻)
      De⁺  : forall {Ψ A+} -> DPat  [] (< Ψ > ↑ A+) ( (< Ψ > A+) true⁺)
      Dapp : {Ψ : RCtx} {Δ1 : LCtx} {Δ2 : LCtx} {A+ : Pos} {B- : Neg} {γ : Conc}
               -> CPat Δ1 (< Ψ > A+) -> DPat Δ2 (< Ψ > B-) γ -> DPat (Δ2 ++ Δ1) (< Ψ > A+ → B-) γ
      Dfst   : {Ψ : RCtx} {Δ : LCtx} {A- : Neg} {B- : Neg} {γ : Conc}
                -> DPat Δ (< Ψ > A-) γ -> DPat  Δ (< Ψ > A- & B-) γ
      Dsnd   : {Ψ : RCtx} {Δ : LCtx} {A- : Neg} {B- : Neg} {γ : Conc}
                -> DPat Δ (< Ψ > B-) γ -> DPat  Δ (< Ψ > A- & B-) γ
      Dunpack :  {R : Rule} {Ψ : RCtx} {Δ : LCtx} {A- : Neg} {γ : Conc}
               -> DPat Δ (< R :: Ψ > A-) γ -> DPat Δ (< Ψ > R ∧ A-) γ
      Dundia :  {Ψ : RCtx} {Δ : LCtx} {A- : Neg} {γ : Conc}
               -> DPat Δ (< [] > A-) γ -> DPat Δ (< Ψ > ◇ A-) γ
      Dcur     : {Ψ : RCtx} {Rs : List Rule} {Δ : LCtx} {A- : Neg} {γ : Conc}
               -> DPat Δ (< Ψ > A-) γ -> DPat Δ (< Ψ > ∀W Rs A-) γ
      Dextend  : {Ψ : RCtx} {R : Rule} {Rs : List Rule} {Δ : LCtx} {A- : Neg} {γ : Conc}
               -> (R ∈ Rs) -> DPat Δ (< Ψ > R ∧ (∀W Rs A-)) γ -> DPat Δ (< Ψ > ∀W Rs A-) γ

  open Pats public

  data Judge : Set where
    RFoc : C Pos -> Judge           -- right-focus
    LInv : Conc -> Conc -> Judge    -- left-inversion
    RInv : Hyp -> Judge             -- right-inversion
    LFoc : C Neg -> Conc -> Judge   -- left-focus
    Neu  : Conc -> Judge            -- neutral sequent
    Asms : LCtx -> Judge            -- multi-inversion (substitution)

  data _⊢_ : UCtx -> Judge -> Set where
    Val⁺ : forall {Γ C+ Δ} -> CPat Δ C+ -> Γ ⊢ Asms Δ -> Γ ⊢ RFoc C+
  
    Ke⁻ : forall {Γ X- } -> Γ ⊢ LInv (X- atom⁻) (X- atom⁻)
    Cont⁺ : forall {Γ C+ γ} -> ({Δ : LCtx} -> CPat Δ C+ -> (Δ :: Γ) ⊢ (Neu γ)) -> Γ ⊢ LInv (C+ true⁺) γ
    -- non-canonical:
    Ke⁺ : forall {C+ Γ} -> Γ ⊢ LInv (C+ true⁺) (C+ true⁺)
    K⁺K⁺   : forall {γ0 γ1 γ Γ} -> Γ ⊢ LInv γ0 γ1  ->  Γ ⊢ LInv γ1 γ  -> Γ ⊢ LInv γ0 γ
  
    Cont⁻ : forall {Δ C- γ0 γ Γ} -> DPat Δ C- γ0  ->  Γ ⊢ Asms Δ  ->  Γ ⊢ LInv γ0 γ  -> Γ ⊢ LFoc C- γ
    -- non-canonical:
    K⁻K⁺ : forall {C- γ0 γ Γ} -> Γ ⊢ LFoc C- γ0  ->  Γ ⊢ LInv γ0 γ  -> Γ ⊢ LFoc C- γ
  
    Vx⁺ : forall {Γ X+ } -> (X+ atom⁺) ∈∈ Γ -> Γ ⊢ RInv (X+ atom⁺)
    Val⁻ : forall {Γ C- } -> ({Δ : LCtx} {γ : Conc} -> DPat Δ C- γ -> Δ :: Γ ⊢ Neu  γ) -> Γ ⊢ RInv (C- true⁻)
    -- noncanonical:
    Vx⁻ : forall {Γ C- } -> ((C- true⁻) ∈∈ Γ) -> Γ ⊢ RInv (C- true⁻)
    Vfix : forall {Γ C- } -> ([ C- true⁻ ] :: Γ) ⊢ RInv (C- true⁻) -> Γ ⊢ RInv (C- true⁻)
  
    Rf : forall {Γ C+} -> Γ ⊢ RFoc C+ -> Γ ⊢ Neu (C+ true⁺)
    Lf : forall {Γ C- γ} -> ((C- true⁻) ∈∈ Γ) -> Γ ⊢ LFoc C- γ -> Γ ⊢ Neu γ
    -- noncanonical:
    Cut⁻  : forall {Γ C- γ} -> Γ ⊢ RInv (C- true⁻) -> Γ ⊢ LFoc C- γ -> Γ ⊢ Neu γ
    Cut⁺  : forall {Γ C+ γ} -> Γ ⊢ RFoc C+ -> Γ ⊢ LInv (C+ true⁺) γ -> Γ ⊢ Neu γ
    EK⁺   : forall {Γ γ0 γ} -> Γ ⊢ Neu γ0 -> Γ ⊢ LInv γ0 γ -> Γ ⊢ Neu γ
  
    Sub  : {Γ : UCtx} -> {Δ : LCtx} -> Everywhere (\ α -> Γ ⊢ RInv α) Δ -> Γ ⊢ Asms Δ
    -- non-canonical:
    Ids  : forall {Γ Δ} -> (Δ ⊆LS Γ) -> Γ ⊢ Asms Δ
    _+s+_ : forall {Γ Δ1 Δ2} -> Γ ⊢ Asms Δ1 -> Γ ⊢ Asms Δ2 -> Γ ⊢ Asms (Δ2 ++ Δ1)

