\ignore{
\begin{code}

{-# OPTIONS --no-termination-check #-}

open import lib.Prelude
open List.In
open List.InLists
open List.SW
open List using (_++_)  
open import foc.posdep.Intuit

module foc.posdep.IntuitProperties where

module Properties where
 open Focus

 module SimpleIdent where
\end{code}}

\subsection{Identity}

There are two main identity principles, which are defined mutually.  The
negative identity says that an assumption of \ttt{C⁻} can be expanded
into a negative value (right inversion).  The positive identity says
that there is an identity continuation (left inversion) from \ttt{C⁺} to
\ttt{C⁺}.  There are also three auxiliary principles: The first defines
identity for assumptions α, by delegating either to the negative
identity theorem, or to the primitive rule \ttt{Vx⁺} for atoms.  The
second maps this across all assumptions in a context, producing a
substitution.  The third expands conclusions γ, by delegating either to
positive identity or to the rule \ttt{Ke⁻} for atoms.

The positive identity is defined to be a continuation that, when given a
constructor pattern, forms the value composed of that pattern under the
identity substitution (a recursive call).  The negative identity is
defined to be a value that, when given a destructor pattern, observes
the designated variable with that destructor, under the identity
substitution, and followed by the identity left-inversion (both
recursive calls).

Here we write \ttt{s0} and \ttt{sS} as constructors for de Bruijn indices into Γ:
\begin{verbatim}
s0 : α ∈ Δ -> α ∈∈ (Δ :: Γ)
sS : α ∈ Γ -> α ∈  (Δ :: Γ)
\end{verbatim}

The identity principles are defined as follows:

\begin{code}
  mutual
   Ke⁺ : forall {K Γ} {C⁺ : Type⁺ K} 
       -> Γ ⊢ LInv (C⁺ true⁺) (C⁺ true⁺)
   Ke⁺ ~ Cont⁺ (\c -> R (Val⁺ c (Ids s0)))  
 
   Vx⁻ : forall {Γ C⁻} 
       -> ((C⁻ true⁻) ∈∈ Γ) -> Γ ⊢ RInv (C⁻ true⁻)
   Vx⁻ x ~ Val⁻ (\d -> L (sS x) (Cont⁻ d (Ids s0) Ke)) 

   Vx : forall {Γ α} -> (α ∈∈ Γ) -> Γ ⊢ RInv α
   Vx {α = x+ atom⁺} x ~ Vx⁺ x
   Vx {α = C- true⁻} x ~ Vx⁻ x
 
   Ids  : forall {Δ Γ} 
        -> ({α : Hyp} -> (α ∈ Δ) -> (α ∈∈ Γ)) 
        -> Γ ⊢ Asms Δ
   Ids subset ~ Sub (\{α} i -> Vx (subset i))

   Ke : forall {Γ γ} -> Γ ⊢ LInv γ γ
   Ke {γ = x- atom⁻} ~ Ke⁻
   Ke {γ = C+ true⁺} ~ Ke⁺
\end{code}

Do these functions terminate?  There are several circumstances in which
we can answer this question:

First, if the pattern judgements have the property that they decompose a
type into syntactically smaller types (i.e., whenever Δ ⊩ A⁺, every
negative type in Δ is a subexpression of A⁺, and similarly for
destructor patterns), then identity is total.  This is because the
proofs of identity make recursive calls only on the assumptions and
conclusions coming from patterns.  We proved this theorem in previous
work \citep{lzh08focbind}.

However, not all types have this property. For example, \ttt{dom}
violates it, because \ttt{dom} is decomposed into \ttt{dom → ↑ dom}.
This property is also violated by more innocuous types, such as streams
specified as an inductive type with a suspended tail---i.e. the solution to
\ttt{str ≅ elt * ↓↑(str)}.  The patterns for \ttt{str} produce
assumptions of \ttt{↑str}.  And indeed, these types pose a problem for
the identity theorem as described above: for example, the negative
identity at \ttt{(dom → ↑ dom)} makes a recursive call to the negative
identity at
\ttt{(dom → ↑ dom)}!  

One solution is to disallow these types.  A better solution is to treat
the focusing judgement \ttt{Γ ⊢ J} \emph{coinductively}, following
\citet{girard01locus}, in which case identity is \emph{productive}.  The
η-expansions for types like \ttt{dom} and streams are infinitely deep,
but they always respond to a single observation in a finite amount of
time.  We have taken this solution in the Agda code: In Figure
\ref{fig:focus-agda-dep}, we used an Agda \ttt{codata} declaration for
the focusing judgement.  Above, we used \verb|~| instead of \ttt{=} for
the equations defining identity, which is Agda syntax for a function
whose termination should be checked by coinduction towards the result,
rather than induction over the argument.  Agda successfully checks these
definitions, because all of the recursive calls occur under
constructors.

\subsection{Cut}

We present the code for cut admissibility in Figure \ref{fig:cut}.  We
first require a weakening lemma, whose code we elide; the type \ttt{Γ
⊆SS Γ'} classifies proofs that Γ is a subset of Γ'.

The most fundamental cuts, \ttt{Cut⁺} and \ttt{Cut⁻}, put a value up
against a continuation.  A positive cut is reduced by applying the
meta-function given in the continuation to the constructor pattern given
in the value, and then applying the value's substitution to the result.
A negative cut is reduced by applying the meta-function given in the
value to the destructor pattern given in the continuation, and then (1)
substituting into the result and (2) composing the resulting expression
substitution with the positive continuation.  The next three cut
principles, \ttt{EK⁺} and \ttt{K⁻K⁺} and \ttt{K⁺K⁺}, compose expressions
and continuations.  Finally, we have a substitution lemma; note that
\ttt{Γ - i} removes the element of Γ given by the index, and that
\ttt{List.SW.here} tests whether the index into Γ is in the Δ being
substituted for, and in the first case gives the index into Δ, and in
the second case gives the index into \ttt{Γ - i}.  We elide the
straightforward cases of substitution.  Both composition and
substitution refer back to the principal cuts: composing a value with a
positive continuation causes a positive cut, whereas substituting a
negative value in for a variable causes a negative cut.

This cut admissibility procedure does not always terminate,
corresponding to the fact that our language admits non-terminating
run-time programs.  We show an example of a looping program using
\ttt{dom} below.  

Because cut is defined on well-typed syntax, it intrinsically ensures
subject reduction, and the fact that cut can only fail by
non-termination (i.e., that it coverage checks) is tantamount to a
progress lemma.  It is simple to adapt the cut admissibility procedure
to an operational semantics on closed terms \citep{l08thesprop}.

\begin{figure}
\ignore{
\begin{code}
 module WeakenΓ where
  open List.Subsets
\end{code}
}

\begin{code}
  weaken : { Γ Γ' : CtxCtx } { J : FocJudg }
         -> Γ ⊆SS Γ' -> Γ ⊢ J -> Γ' ⊢ J
\end{code}

\ignore{
\begin{code}
  weaken wv (Val⁺ p σ) ~ Val⁺ p (weaken wv σ)

  weaken wv (Val⁻ φ) ~ Val⁻ (\k -> weaken (List.Subsets.shiftwv wv _) (φ k))
  weaken wv (Vx⁺ x) ~ (Vx⁺ (wv x))

  weaken wv (R v) ~ R (weaken wv v)
  weaken wv (L x s) ~ L (wv x) (weaken wv s)

  weaken wv (Cont⁻ d σ c) ~ Cont⁻ d (weaken wv σ) (weaken wv c)

  weaken wv Ke⁻ ~ Ke⁻
  weaken wv (Cont⁺ φ) ~ Cont⁺ (\p -> weaken (List.Subsets.shiftwv wv _ ) (φ p))

  weaken {Γ} {Γ'} wv (Sub vars) ~ Sub (\ {α} i -> weaken wv (vars i)) 

 module Cut where
  open WeakenΓ
\end{code}}

\begin{code}
  mutual 
   Cut⁺  : forall {Γ K γ} {C⁺ : Type⁺ K}
         -> Γ ⊢ RFoc C⁺ 
         -> Γ ⊢ LInv (C⁺ true⁺) γ 
         -> Γ ⊢ Neu γ
   Cut⁺ (Val⁺ c σ) (Cont⁺ φ⁺) ~ [ i0 ← σ ] (φ⁺ c)

   Cut⁻  : forall {Γ C⁻ γ} 
        -> Γ ⊢ RInv (C⁻ true⁻) -> Γ ⊢ LFoc C⁻ γ 
        -> Γ ⊢ Neu γ
   Cut⁻ (Val⁻ φ⁻) (Cont⁻ d σ k⁺) ~
     EK⁺ ([ i0 ← σ ] (φ⁻ d)) k⁺

   EK⁺   : forall {Γ γ₀ γ} 
         -> Γ ⊢ Neu γ₀ -> Γ ⊢ LInv γ₀ γ 
         -> Γ ⊢ Neu γ
   EK⁺ (R v) k⁺ ~ Cut⁺ v k⁺
   EK⁺ (L x k⁻) k⁺ ~ L x (K⁻K⁺ k⁻ k⁺)

   K⁺K⁺ : forall {γ₀ γ₁ γ Γ} 
        -> Γ ⊢ LInv γ₀ γ₁ -> Γ ⊢ LInv γ₁ γ 
        -> Γ ⊢ LInv γ₀ γ
   K⁺K⁺ (Cont⁺ φ⁺) k₂⁺ ~ 
     Cont⁺ (\c -> EK⁺ (φ⁺ c) (weaken sS k₂⁺))
   K⁺K⁺ Ke⁻ k₂⁺ ~ k₂⁺

   K⁻K⁺ : forall {C⁻ γ₀ γ Γ} 
        -> Γ ⊢ LFoc C⁻ γ₀  ->  Γ ⊢ LInv γ₀ γ  
        -> Γ ⊢ LFoc C⁻ γ
   K⁻K⁺ (Cont⁻ d σ k⁺) k₂⁺ ~ (Cont⁻ d σ (K⁺K⁺ k⁺ k₂⁺))

   [_←_]_ : forall {Δ Γ J} -> (i : Δ ∈ Γ) 
           -> (Γ - i) ⊢ Asms Δ -> Γ ⊢ J 
           -> (Γ - i) ⊢ J
   [ x₀ ← σ₀ ] L y k⁻ with List.SW.here? x₀ y | σ₀
   ... | Inl newy | Sub f ~ Cut⁻ (f newy) ([ x₀ ← σ₀ ] k⁻) 
   ... | Inr newy | _     ~ L newy ([ x₀ ← σ₀ ] k⁻)
\end{code}

\ignore{
\begin{code}
   [ x₀ ← σ₀ ] Val⁺ c σ ~ Val⁺ c ([ x₀ ← σ₀ ] σ) 
   [ x₀ ← σ₀ ] Ke⁻ ~ Ke⁻
   [ x₀ ← σ₀ ] Cont⁺ φ ~ 
     Cont⁺ (\ c -> [ iS x₀ ← weaken sS σ₀ ] (φ c))
   [ x₀ ← σ₀ ] Cont⁻ d σ k⁺ ~ 
     Cont⁻ d ([ x₀ ← σ₀ ] σ) ([ x₀ ← σ₀ ] k⁺)
   [ x₀ ← Sub f ] Vx⁺ y with List.SW.here? x₀ y
   ... | Inl newy ~ f newy
   ... | Inr newy ~ Vx⁺ newy
   [ x₀ ← σ₀ ] Val⁻ φ ~ 
     Val⁻ (\ c -> [ iS x₀ ← weaken sS σ₀ ] (φ c))
   [ x₀ ← σ₀ ] R v⁺ ~ R ([ x₀ ← σ₀ ] v⁺)
   [ x₀ ← σ₀ ] Sub f ~ Sub (\ c -> [ x₀ ← σ₀ ] (f c)) 
\end{code}}

\caption{Cut admissibility procedure}
\label{fig:cut}
\end{figure}

\subsection{Type Equality}

\ignore{
\begin{code}
 module Closed where

   closed : {A+ : Type⁺ PPos} {Δ : Ctx} -> Δ ⊩ A+ -> Id Δ []
   closed {1⁺} C<> = Refl
   closed {TypesPats._*_ A B} (Cpair{Δ1} {Δ2} c1 c2) with closed {A} {Δ1} c1 | closed {B} {Δ2} c2
   closed {TypesPats._*_ A B} (Cpair{.[]} {.[]} c1 c2)  | Refl      | Refl = Refl
   closed {TypesPats.Σ⁺ A τ} { Δ } (Cdpair p1 p2) = closed {τ p1} { Δ } p2
   closed {TypesPats.0⁺} () 
   closed {TypesPats._+_ A B} { Δ } (Cinl p) = closed {A} { Δ } p
   closed {TypesPats._+_ A B} { Δ } (Cinr p) = closed {B} { Δ } p
   closed {TypesPats.nat} Czero = Refl
   closed {TypesPats.nat} { Δ } (Csucc p) = closed {nat} { Δ } p
   closed {TypesPats.vec A+ Czero} Cnil = Refl
   closed {TypesPats.vec A (Csucc n)} (Ccons{Δ1} {Δ2} c1 c2) with closed {A} {Δ1} c1 | closed {vec A n} {Δ2} c2
   closed {TypesPats.vec A (Csucc n)} (Ccons{.[]} {.[]} c1 c2)  | Refl      | Refl = Refl

   close : {A+ : Type⁺ PPos} {Δ : Ctx} -> Δ ⊩ A+ -> [] ⊩ A+
   close {A} { Δ } p with closed {A} { Δ } p
   close {_} {.[]} p | Refl = p

 module Force where
   force : forall { Γ J } -> (Γ ⊢ J) -> (Γ ⊢ J)
   force (Val⁺ x y) = Val⁺ x y
   force Ke⁻ = Ke⁻
   force (Cont⁺ y) = Cont⁺ y
   force (Cont⁻ y y' y0) = (Cont⁻ y y' y0)
   force (Vx⁺ y) = (Vx⁺ y) 
   force (Val⁻ y) = (Val⁻ y)
   force (R y) = (R y)
   force (L y y') = (L y y')
   force (Sub y) = (Sub y)

   forceid : forall { Γ J } (t : Γ ⊢ J) -> Id t (force t)
   forceid (Val⁺ _ _) = Refl
   forceid Ke⁻ = Refl
   forceid (Cont⁺ y) = Refl
   forceid (Cont⁻ y y' y0) = Refl
   forceid (Vx⁺ y) = Refl
   forceid (Val⁻ y) = Refl
   forceid (R y) = Refl
   forceid (L y y') = Refl
   forceid (Sub y) = Refl

   --     uncong : forall {Γ1 J1 Γ2 J2} (t1 : Γ1 ⊢ J1) (t2 : Γ2 ⊢ J2) -> HId (force t1) (force t2) -> HId t1 t2
   --     uncong t1 t2 eq = Id.Het.trans (forceid t1) (Id.Het.trans eq (Id.Het.sym (forceid t2)))

 module Invert where

   unCpair-nil : forall {K Δ} {A B : Type⁺ K} -> Id Δ [] -> (c : Δ ⊩ (A * B)) 
         -> Σ \(c1 : [] ⊩ A) -> Σ \(c2 : [] ⊩ B) -> HId{Δ ⊩ (A * B)} c {[] ⊩ (A * B)} (Cpair{_}{_}{[]}{[]} c1 c2)
   unCpair-nil p (Cpair{Δ1}{Δ2} c1 c2) with Δ1  | Δ2  | List.invert-append-nil{_}{Δ2}{Δ1} p
   ...                              | .[] | .[] | Refl , Refl = c1 , c2 , HRefl

   unCcons-nil : forall {K Δ} {A : Type⁺ K} {n : [] ⊩ nat{PPos}} -> Id Δ [] -> (c : Δ ⊩ vec A (Csucc n)) 
         -> Σ \(c1 : [] ⊩ A) -> Σ \(c2 : [] ⊩ vec A n) -> HId{Δ ⊩ vec A (Csucc n)} c {[] ⊩ vec A (Csucc n)} (Ccons c1 c2)
   unCcons-nil p (Ccons{Δ1}{Δ2} c1 c2) with Δ1  | Δ2  | List.invert-append-nil{_}{Δ2}{Δ1} p
   ...                              | .[] | .[] | Refl , Refl = c1 , c2 , HRefl



 module SyntacticEquality where

   -- syntactic equality of proofs, treating the metafns extensionally and ignoring the types

   data EqCPat : forall {Δ1 Δ2 K1 K2} {A1 : Type⁺ K1} {A2 : Type⁺ K2} -> Δ1 ⊩ A1 -> Δ2 ⊩ A2 -> Set where 
     ≡Cx⁺   : forall {X Y} -> EqCPat{[ X atom⁺ ]}{[ Y atom⁺ ]}{Pos}{Pos}{X⁺ X}{X⁺ Y} Cx⁺ Cx⁺
     ≡Cx⁻   : forall {A1- A2- } -> EqCPat{[ A1- true⁻ ]}{[ A2- true⁻ ]}{Pos}{Pos}{↓ A1- }{↓ A2- } Cx⁻ Cx⁻
     ≡C<>   : forall {K1 K2} -> EqCPat{[]}{[]}{K1}{K2}{1⁺{K1}}{1⁺{K2}} C<> C<>
     ≡Cpair : forall {Δ1 Δ1' Δ2 Δ2' K1 K2} {A1+ : Type⁺ K1} {A2+ : Type⁺ K2} {B1+ : Type⁺ K1} {B2+ : Type⁺ K2}  
            {c1 : Δ1 ⊩ A1+} {c1' : Δ1' ⊩ B1+} {c2 : Δ2 ⊩ A2+} {c2' : Δ2' ⊩ B2+}
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1+}{A2+} c1 c2
            -> EqCPat{Δ1'}{Δ2'}{K1}{K2}{B1+}{B2+} c1' c2'
            -> EqCPat{Δ1' ++ Δ1}{Δ2' ++ Δ2}{K1}{K2}{A1+ * B1+}{A2+ * B2+} (Cpair c1 c1') (Cpair c2 c2') 
     ≡Cdpair : forall {Δ1 Δ2 K1 K2} {A1+ : Type⁺ PPos} {A2+ : Type⁺ PPos} 
                      {τ1+ : [] ⊩ A1+ -> Type⁺ K1} {τ2+ : [] ⊩ A2+ -> Type⁺ K2}  
            {c1 : [] ⊩ A1+} {c2 : [] ⊩ A2+} {c1' : Δ1 ⊩ (τ1+ c1)} {c2' : Δ2 ⊩ (τ2+ c2)}
            -> EqCPat{[]}{[]}{PPos}{PPos}{A1+}{A2+} c1 c2
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{τ1+ c1}{(τ2+ c2)} c1' c2'
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{Σ⁺ A1+ τ1+}{Σ⁺ A2+ τ2+} (Cdpair c1 c1') (Cdpair c2 c2') 
     ≡Cinl : forall {Δ1 Δ2 K1 K2} {A1+ : Type⁺ K1} {A2+ : Type⁺ K2} {B1+ : Type⁺ K1} {B2+ : Type⁺ K2}  
            {c1 : Δ1 ⊩ A1+} {c2 : Δ2 ⊩ A2+} 
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1+}{A2+} c1 c2
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1+ + B1+}{A2+ + B2+} (Cinl c1) (Cinl c2) 
     ≡Cinr : forall {Δ1 Δ2 K1 K2} {A1+ : Type⁺ K1} {A2+ : Type⁺ K2} {B1+ : Type⁺ K1} {B2+ : Type⁺ K2}  
            {c1 : Δ1 ⊩ B1+} {c2 : Δ2 ⊩ B2+} 
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{B1+}{B2+} c1 c2
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1+ + B1+}{A2+ + B2+} (Cinr c1) (Cinr c2) 
     ≡Czero : forall {K1 K2} -> EqCPat{[]}{[]}{K1}{K2}{nat{K1}}{nat{K2}} Czero Czero
     ≡Csucc : forall {Δ1 Δ2 K1 K2}   
            {c1 : Δ1 ⊩ nat{K1}} {c2 : Δ2 ⊩ nat{K2}} 
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{nat}{nat} c1 c2
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{nat}{nat} (Csucc c1) (Csucc c2)
     ≡Cnil  : forall {K1 K2} {A1+ : Type⁺ K1} {A2+ : Type⁺ K2} 
            -> EqCPat{[]}{[]}{K1}{K2}{vec A1+ Czero}{vec A2+ Czero} Cnil Cnil
     ≡Ccons : forall {Δ1 Δ1' Δ2 Δ2' K1 K2} {A1+ : Type⁺ K1} {A2+ : Type⁺ K2} 
            {n1 : [] ⊩ nat{PPos}} {c1 : Δ1 ⊩ A1+} {c1' : Δ1' ⊩ vec A1+ n1} 
            {n2 : [] ⊩ nat{PPos}} {c2 : Δ2 ⊩ A2+} {c2' : Δ2' ⊩ vec A2+ n2} 
            -> EqCPat{[]}{[]}{K1}{K2}{nat}{nat} n1 n2
            -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1+}{A2+} c1 c2
            -> EqCPat{Δ1'}{Δ2'}{K1}{K2}{vec A1+ n1}{vec A2+ n2} c1' c2'
            -> EqCPat{Δ1' ++ Δ1}{Δ2' ++ Δ2}{K1}{K2}{vec A1+ (Csucc n1)}{vec A2+ (Csucc n2)} 
                     (Ccons c1 c1') (Ccons c2 c2') 
     ≡Cdom  : EqCPat{[ (dom → ↑ dom) true⁻ ]}{[ (dom → ↑ dom) true⁻ ]}{Pos}{Pos}{dom}{dom} Cdom Cdom

   eqcpat-resp : forall {S K} (A : Type⁺ K) (c1 : [] ⊩ A) (c2 : [] ⊩ A)
               -> EqCPat{[]}{[]}{K}{K}{A}{A} c1 c2 
               -> (f : [] ⊩ A -> S)
               -> Id (f c1) (f c2)
   eqcpat-resp (A * B) p1 p2 pateq f with Invert.unCpair-nil{_}{[]}{A}{B} Refl p1 | Invert.unCpair-nil{_}{[]}{A}{B} Refl p2  
   eqcpat-resp (A * B) .(Cpair c1 c1') .(Cpair c2 c2') (≡Cpair e1 e2) f | (c1 , c1' , HRefl) | (c2 , c2' , HRefl) = 
         Id.trans (eqcpat-resp A c1 c2 e1 (\h -> f (Cpair h c1'))) (eqcpat-resp B c1' c2' e2 (\h -> f (Cpair c2 h)))
   eqcpat-resp (Σ⁺ A τ) (Cdpair c1 c1') (Cdpair c2 c2') (≡Cdpair e e') f with c2 | eqcpat-resp A c1 c2 e τ 
   ... | .c1 | Refl = eqcpat-resp (τ c1) c1' c2' e' (\h -> f (Cdpair c1 h)) 
   eqcpat-resp (A + B) (Cinl c1) (Cinl c2) (≡Cinl e) f = eqcpat-resp A c1 c2 e (\c -> f (Cinl c)) 
   eqcpat-resp (A + B) (Cinr c1) (Cinr c2) (≡Cinr e) f = eqcpat-resp B c1 c2 e (\c -> f (Cinr c)) 
   eqcpat-resp (A + B) (Cinr c1) (Cinl c2) () _
   eqcpat-resp (A + B) (Cinl c1) (Cinr c2) () _
   eqcpat-resp 1⁺ C<> C<> _ f = Refl
   eqcpat-resp (X⁺ x) () () _ _
   eqcpat-resp (↓ A-) () () _ _
   eqcpat-resp dom    () () _ _
   eqcpat-resp 0⁺     () () _ _
   eqcpat-resp nat Czero Czero _ f = Refl
   eqcpat-resp nat (Csucc c1) (Csucc c2) (≡Csucc e) f = eqcpat-resp nat c1 c2 e (\c -> f (Csucc c)) 
   eqcpat-resp nat Czero (Csucc _) () _
   eqcpat-resp nat (Csucc _) Czero () _
   eqcpat-resp (vec A+ Czero) Cnil Cnil _ f = Refl
   eqcpat-resp (vec A+ (Csucc n)) p1 p2 pateq f with Invert.unCcons-nil{_}{[]}{A+} Refl p1 | Invert.unCcons-nil {_}{[]}{A+} Refl p2  
   eqcpat-resp (vec A+ (Csucc n)) .(Ccons c1 c1') .(Ccons c2 c2') (≡Ccons _ e1 e2) f | (c1 , c1' , HRefl) | (c2 , c2' , HRefl) = 
         Id.trans (eqcpat-resp A+ c1 c2 e1 (\h -> f (Ccons h c1'))) (eqcpat-resp (vec A+ n) c1' c2' e2 (\h -> f (Ccons c2 h)))

   eqcpat-idΔ : forall {Δ1 Δ2 K} {A : Type⁺ K} {c1 : Δ1 ⊩ A} {c2 : Δ2 ⊩ A} -> EqCPat{Δ1}{Δ2}{K}{K}{A}{A} c1 c2 -> Id Δ1 Δ2
   eqcpat-idΔ ≡Cx⁺ = Refl
   eqcpat-idΔ ≡Cx⁻ = Refl
   eqcpat-idΔ (≡Cpair e1 e2) with eqcpat-idΔ e1 | eqcpat-idΔ e2
   ...                          | Refl          | Refl = Refl
   eqcpat-idΔ {A = Σ⁺ A+ τ+} {c1 = (Cdpair c1 c1')} {c2 = (Cdpair c2 c2')} (≡Cdpair e1 e2) with c2 | eqcpat-resp A+ c1 c2 e1 τ+ 
   ... | .c1 | Refl = eqcpat-idΔ e2 
   eqcpat-idΔ (≡Cinl e) = eqcpat-idΔ e
   eqcpat-idΔ (≡Cinr e) = eqcpat-idΔ e
   eqcpat-idΔ (≡C<>) = Refl
   eqcpat-idΔ (≡Czero) = Refl
   eqcpat-idΔ (≡Cnil) = Refl
   eqcpat-idΔ (≡Cdom) = Refl
   eqcpat-idΔ (≡Csucc e) = eqcpat-idΔ e
   eqcpat-idΔ (≡Ccons _ e1 e2) with eqcpat-idΔ e1 | eqcpat-idΔ e2
   ...                            | Refl          | Refl = Refl

   eqcpat-refl : forall { Δ K } (A : Type⁺ K) (c : Δ ⊩ A) -> EqCPat{Δ}{Δ}{K}{K}{A}{A} c c
   eqcpat-refl 1⁺ C<> = ≡C<>
   eqcpat-refl (TypesPats.X⁺ y) TypesPats.CPats.Cx⁺ = ≡Cx⁺
   eqcpat-refl (TypesPats.↓ y) TypesPats.CPats.Cx⁻ = ≡Cx⁻
   eqcpat-refl TypesPats.1⁺ TypesPats.CPats.C<> = ≡C<>
   eqcpat-refl (TypesPats._*_ y y') (TypesPats.CPats.Cpair y0 y1) = ≡Cpair (eqcpat-refl y y0) (eqcpat-refl y' y1)
   eqcpat-refl (TypesPats.Σ⁺ A⁺ y) (TypesPats.CPats.Cdpair c y') = ≡Cdpair (eqcpat-refl A⁺ c) (eqcpat-refl (y c) y')
   eqcpat-refl TypesPats.0⁺ ()
   eqcpat-refl (TypesPats._+_ y y') (TypesPats.CPats.Cinl y0) = ≡Cinl (eqcpat-refl y y0)
   eqcpat-refl (TypesPats._+_ y y') (TypesPats.CPats.Cinr y0) = ≡Cinr (eqcpat-refl y' y0)
   eqcpat-refl TypesPats.nat TypesPats.CPats.Czero = ≡Czero
   eqcpat-refl TypesPats.nat (TypesPats.CPats.Csucc y) = ≡Csucc (eqcpat-refl nat y)
   eqcpat-refl (TypesPats.vec A+ Czero) Cnil = ≡Cnil
   eqcpat-refl (TypesPats.vec A+ (Csucc n)) (Ccons y y') = ≡Ccons (eqcpat-refl nat n) (eqcpat-refl A+ y)
                                                            (eqcpat-refl (vec A+ n) y')
   eqcpat-refl TypesPats.dom TypesPats.CPats.Cdom = ≡Cdom
   
   eqcpat-sym : forall {Δ1 Δ2 K1 K2} {A1 : Type⁺ K1} {A2 : Type⁺ K2} {c1 : Δ1 ⊩ A1} {c2 : Δ2 ⊩ A2}
              -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1}{A2} c1 c2 -> EqCPat{Δ2}{Δ1}{K2}{K1}{A2}{A1} c2 c1
   eqcpat-sym ≡Cx⁺ = ≡Cx⁺
   eqcpat-sym ≡Cx⁻ = ≡Cx⁻
   eqcpat-sym ≡C<> = ≡C<>
   eqcpat-sym (≡Cpair e1 e2) = ≡Cpair (eqcpat-sym e1) (eqcpat-sym e2) 
   eqcpat-sym (≡Cdpair e1 e2) = ≡Cdpair (eqcpat-sym e1) (eqcpat-sym e2)
   eqcpat-sym (≡Cinl e1) = ≡Cinl (eqcpat-sym e1) 
   eqcpat-sym (≡Cinr e2) = ≡Cinr (eqcpat-sym e2) 
   eqcpat-sym ≡Czero = ≡Czero
   eqcpat-sym (≡Csucc e2) = ≡Csucc (eqcpat-sym e2) 
   eqcpat-sym ≡Cnil = ≡Cnil
   eqcpat-sym (≡Ccons e1 e2 e3) = ≡Ccons (eqcpat-sym e1)  (eqcpat-sym e2) (eqcpat-sym e3)
   eqcpat-sym ≡Cdom = ≡Cdom

   eqcpat-trans : forall {Δ1 Δ2 Δ3 K1 K2 K3} {A1 : Type⁺ K1} {A2 : Type⁺ K2} {A3 : Type⁺ K3} 
                 {c1 : Δ1 ⊩ A1} {c2 : Δ2 ⊩ A2} {c3 : Δ3 ⊩ A3}
              -> EqCPat{Δ1}{Δ2}{K1}{K2}{A1}{A2} c1 c2 -> EqCPat{Δ2}{Δ3}{K2}{K3}{A2}{A3} c2 c3  
              -> EqCPat{Δ1}{Δ3}{K1}{K3}{A1}{A3} c1 c3  
   eqcpat-trans ≡Cx⁺ ≡Cx⁺ = ≡Cx⁺
   eqcpat-trans ≡Cx⁻ ≡Cx⁻ = ≡Cx⁻
   eqcpat-trans ≡C<> ≡C<> = ≡C<>
   eqcpat-trans (≡Cpair e1 e2) (≡Cpair e1' e2') = ≡Cpair (eqcpat-trans e1 e1') (eqcpat-trans e2 e2') 
   eqcpat-trans (≡Cdpair e1 e2) (≡Cdpair e1' e2') = ≡Cdpair (eqcpat-trans e1 e1') (eqcpat-trans e2 e2')
   eqcpat-trans (≡Cinl e1) (≡Cinl e1') = ≡Cinl (eqcpat-trans e1 e1') 
   eqcpat-trans (≡Cinr e2) (≡Cinr e2') = ≡Cinr (eqcpat-trans e2 e2') 
   eqcpat-trans ≡Czero ≡Czero = ≡Czero
   eqcpat-trans (≡Csucc e2) (≡Csucc e2') = ≡Csucc (eqcpat-trans e2 e2') 
   eqcpat-trans ≡Cnil ≡Cnil = ≡Cnil
   eqcpat-trans (≡Ccons e1 e2 e3) (≡Ccons e1' e2' e3') = ≡Ccons (eqcpat-trans e1 e1')  (eqcpat-trans e2 e2') (eqcpat-trans e3 e3')
   eqcpat-trans ≡Cdom ≡Cdom = ≡Cdom

   data EqDPat : forall {Δ1 Δ2 A1 A2 γ1 γ2} -> Δ1 ⊩ A1 > γ1 -> Δ2 ⊩ A2 > γ2 -> Set where
     ≡De⁻ : forall {X Y } -> EqDPat{[]}{[]}{X⁻ X}{X⁻ Y} De⁻ De⁻
     ≡De⁺ : forall {A1+ A2+} -> EqDPat{[]}{[]}{↑ A1+}{↑ A2+} De⁺ De⁺
     ≡Dapp : forall {Δ1 Δ1' Δ2 Δ2' γ1 γ2 A1- A2- } {A1+ : Type⁺ Pos} {A2+ : Type⁺ Pos} 
            {c1 : Δ1 ⊩ A1+} {d1 : Δ1' ⊩ A1- > γ1} {c2 : Δ2 ⊩ A2+} {d2 : Δ2' ⊩ A2- > γ2}
            -> EqCPat{Δ1}{Δ2}{Pos}{Pos}{A1+}{A2+} c1 c2
            -> EqDPat d1 d2
            -> EqDPat{Δ1' ++ Δ1}{Δ2' ++ Δ2}{A1+ → A1- }{A2+ → A2- } (Dapp c1 d1) (Dapp c2 d2) 
     ≡Ddapp : forall {Δ1 Δ2 γ1 γ2} {A1+ : Type⁺ PPos} {A2+ : Type⁺ PPos} {τ1- : [] ⊩ A1+ -> Type⁻} {τ2- : [] ⊩ A2+ -> Type⁻}
            {c1 : [] ⊩ A1+} {d1 : Δ1 ⊩ (τ1- c1) > γ1} {c2 : [] ⊩ A2+} {d2 : Δ2 ⊩ (τ2- c2) > γ2}
            -> EqCPat{[]}{[]}{PPos}{PPos}{A1+}{A2+} c1 c2
            -> EqDPat d1 d2
            -> EqDPat{Δ1}{Δ2}{ Π⁻ A1+ τ1- }{ Π⁻ A2+ τ2- } (Ddapp c1 d1) (Ddapp c2 d2) 
     ≡Dfst : forall {Δ1 Δ2 γ1 γ2 A1- A2- B1- B2- } 
            {d1 : Δ1 ⊩ A1- > γ1} {d2 : Δ2 ⊩ A2- > γ2}
            -> EqDPat d1 d2
            -> EqDPat{Δ1}{Δ2}{A1- & B1- }{A2- & B2- } (Dfst d1) (Dfst d2) 
     ≡Dsnd : forall {Δ1 Δ2 γ1 γ2 A1- A2- B1- B2- } 
            {d1 : Δ1 ⊩ B1- > γ1} {d2 : Δ2 ⊩ B2- > γ2}
            -> EqDPat d1 d2
            -> EqDPat{Δ1}{Δ2}{A1- & B1- }{A2- & B2- } (Dsnd d1) (Dsnd d2) 

   eqdpat-idΔ : forall {Δ1 Δ2 γ1 γ2} {A : Type⁻} {d1 : Δ1 ⊩ A > γ1} {d2 : Δ2 ⊩ A > γ2} 
              -> EqDPat d1 d2 -> Id Δ1 Δ2
   eqdpat-idΔ ≡De⁺ = Refl
   eqdpat-idΔ ≡De⁻ = Refl
   eqdpat-idΔ (≡Dfst e) = eqdpat-idΔ e
   eqdpat-idΔ (≡Dsnd e) = eqdpat-idΔ e
   eqdpat-idΔ (≡Dapp e1 e2) with eqcpat-idΔ e1 | eqdpat-idΔ e2
   ...                         | Refl          | Refl = Refl
   eqdpat-idΔ {A = Π⁻ A+ τ- } {d1 = (Ddapp c1 d1)} {d2 = (Ddapp c2 d2')} (≡Ddapp e1 e2) with c2 | eqcpat-resp A+ c1 c2 e1 τ-
   ... | .c1 | Refl = eqdpat-idΔ e2 

   eqdpat-idγ : forall {Δ1 Δ2 γ1 γ2} {A : Type⁻} {d1 : Δ1 ⊩ A > γ1} {d2 : Δ2 ⊩ A > γ2} 
              -> EqDPat d1 d2 -> Id γ1 γ2
   eqdpat-idγ ≡De⁺ = Refl
   eqdpat-idγ ≡De⁻ = Refl
   eqdpat-idγ (≡Dfst e) = eqdpat-idγ e
   eqdpat-idγ (≡Dsnd e) = eqdpat-idγ e
   eqdpat-idγ (≡Dapp e1 e2) = eqdpat-idγ e2
   eqdpat-idγ {A = Π⁻ A+ τ- } {d1 = (Ddapp c1 d1)} {d2 = (Ddapp c2 d2')} (≡Ddapp e1 e2) with c2 | eqcpat-resp A+ c1 c2 e1 τ-
   ... | .c1 | Refl = eqdpat-idγ e2 

   eqdpat-refl : forall {Δ A γ} (d : Δ ⊩ A > γ) -> EqDPat d d
   eqdpat-refl De⁺ = ≡De⁺
   eqdpat-refl De⁻ = ≡De⁻
   eqdpat-refl (Dfst d) = ≡Dfst (eqdpat-refl d)
   eqdpat-refl (Dsnd d) = ≡Dsnd (eqdpat-refl d)
   eqdpat-refl (Dapp c d) = ≡Dapp (eqcpat-refl _ c) (eqdpat-refl d)
   eqdpat-refl (Ddapp c d) = ≡Ddapp (eqcpat-refl _ c) (eqdpat-refl d)

   eqdpat-sym : forall {Δ1 Δ2 A1 A2 γ1 γ2} {d1 : Δ1 ⊩ A1 > γ1} {d2 : Δ2 ⊩ A2 > γ2}
              -> EqDPat d1 d2 -> EqDPat d2 d1
   eqdpat-sym ≡De⁺ = ≡De⁺
   eqdpat-sym ≡De⁻ = ≡De⁻
   eqdpat-sym (≡Dfst e) = ≡Dfst (eqdpat-sym e)
   eqdpat-sym (≡Dsnd e) = ≡Dsnd (eqdpat-sym e)
   eqdpat-sym (≡Dapp e1 e2) = ≡Dapp (eqcpat-sym e1) (eqdpat-sym e2)
   eqdpat-sym (≡Ddapp e1 e2) = ≡Ddapp (eqcpat-sym e1) (eqdpat-sym e2)

   eqdpat-trans : forall {Δ1 Δ2 Δ3 A1 A2 A3 γ1 γ2 γ3} {d1 : Δ1 ⊩ A1 > γ1} {d2 : Δ2 ⊩ A2 > γ2} {d3 : Δ3 ⊩ A3 > γ3}
              -> EqDPat d1 d2 -> EqDPat d2 d3 -> EqDPat d1 d3
   eqdpat-trans ≡De⁺ ≡De⁺ = ≡De⁺
   eqdpat-trans ≡De⁻ ≡De⁻ = ≡De⁻
   eqdpat-trans (≡Dfst e) (≡Dfst e') = ≡Dfst (eqdpat-trans e e')
   eqdpat-trans (≡Dsnd e) (≡Dsnd e') = ≡Dsnd (eqdpat-trans e e')
   eqdpat-trans (≡Dapp e1 e2) (≡Dapp e1' e2') = ≡Dapp (eqcpat-trans e1 e1') (eqdpat-trans e2 e2') 
   eqdpat-trans (≡Ddapp e1 e2) (≡Ddapp e1' e2') = ≡Ddapp (eqcpat-trans e1 e1') (eqdpat-trans e2 e2')

   data EqIn : {α1 : Hyp} {Δ1 : Ctx} {α2 : Hyp} {Δ2 : Ctx} -> α1 ∈ Δ1 -> α2 ∈ Δ2 -> Set where
     ≡i0 : forall {α1 Δ1 α2 Δ2} -> EqIn{α1}{(α1 :: Δ1)}{α2}{(α2 :: Δ2)} i0 i0
     ≡iS : forall {α1 Δ1 α2 Δ2 α1' α2'} 
         {i1 : α1 ∈ Δ1}{i2 : α2 ∈ Δ2} -> EqIn i1 i2
         -> EqIn{α1}{(α1' :: Δ1)}{α2}{(α2' :: Δ2)} (iS i1) (iS i2)

   eqin-refl : {α1 : Hyp} {Δ1 : Ctx} -> (x : α1 ∈ Δ1) -> EqIn x x 
   eqin-refl i0 = ≡i0
   eqin-refl (iS x) = ≡iS (eqin-refl x)

   eqin-sym : {α1 : Hyp} {Δ1 : Ctx} {α2 : Hyp} {Δ2 : Ctx} {x1 : α1 ∈ Δ1} {x2 : α2 ∈ Δ2} -> EqIn x1 x2 -> EqIn x2 x1
   eqin-sym ≡i0 = ≡i0
   eqin-sym (≡iS e) = ≡iS (eqin-sym e)

   eqin-trans : {α1 : Hyp} {Δ1 : Ctx} {α2 : Hyp} {Δ2 : Ctx} {α3 : Hyp} {Δ3 : Ctx}  
                 {x1 : α1 ∈ Δ1} {x2 : α2 ∈ Δ2} {x3 : α3 ∈ Δ3} -> EqIn x1 x2 -> EqIn x2 x3 -> EqIn x1 x3
   eqin-trans ≡i0 ≡i0 = ≡i0
   eqin-trans (≡iS e1) (≡iS e2) = ≡iS (eqin-trans e1 e2)

   eqin-idα : {α1 α2 : Hyp} {Δ : Ctx} {x1 : α1 ∈ Δ} {x2 : α2 ∈ Δ} -> EqIn x1 x2 -> Id α1 α2
   eqin-idα ≡i0 = Refl
   eqin-idα (≡iS e) = eqin-idα e

   data EqInIn : {α1 : Hyp} {Γ1 : CtxCtx} {α2 : Hyp} {Γ2 : CtxCtx} -> α1 ∈∈ Γ1 -> α2 ∈∈ Γ2 -> Set where
     ≡s0 : forall {α1 Γ1 α2 Γ2 Δ1 Δ2} -> 
         {i1 : α1 ∈ Δ1}{i2 : α2 ∈ Δ2} -> EqIn i1 i2
         -> EqInIn{α1}{(Δ1 :: Γ1)}{α2}{(Δ2 :: Γ2)} (s0 i1) (s0 i2)
     ≡sS : forall {α1 Γ1 α2 Γ2 Δ1 Δ2} 
         {s1 : α1 ∈∈ Γ1}{s2 : α2 ∈∈ Γ2} -> EqInIn s1 s2
         -> EqInIn{α1}{(Δ1 :: Γ1)}{α2}{(Δ2 :: Γ2)} (sS s1) (sS s2)

   eqinin-refl : {α1 : Hyp} {Γ1 : CtxCtx} -> (x : α1 ∈∈ Γ1) -> EqInIn x x 
   eqinin-refl (s0 x) = ≡s0 (eqin-refl x)
   eqinin-refl (sS x) = ≡sS (eqinin-refl x)

   eqinin-sym : {α1 : Hyp} {Γ1 : CtxCtx} {α2 : Hyp} {Γ2 : CtxCtx} {x1 : α1 ∈∈ Γ1} {x2 : α2 ∈∈ Γ2} -> EqInIn x1 x2 -> EqInIn x2 x1
   eqinin-sym (≡s0 e) = ≡s0 (eqin-sym e)
   eqinin-sym (≡sS e) = ≡sS (eqinin-sym e)

   eqinin-trans : {α1 : Hyp} {Γ1 : CtxCtx} {α2 : Hyp} {Γ2 : CtxCtx} {α3 : Hyp} {Γ3 : CtxCtx}  
                 {x1 : α1 ∈∈ Γ1} {x2 : α2 ∈∈ Γ2} {x3 : α3 ∈∈ Γ3} -> EqInIn x1 x2 -> EqInIn x2 x3 -> EqInIn x1 x3
   eqinin-trans (≡s0 e) (≡s0 e') = ≡s0 (eqin-trans e e')
   eqinin-trans (≡sS e) (≡sS e') = ≡sS (eqinin-trans e e')

   codata Eq : forall { J1 J2 Γ1 Γ2 } -> Γ1 ⊢ J1 -> Γ2 ⊢ J2 -> Set where
     ≡Val⁺ : forall {Γ1 K1 Δ1 Γ2 K2 Δ2} {C1+ : Type⁺ K1} {C2+ : Type⁺ K2} 
             {c1 : Δ1 ⊩ C1+} {c2 : Δ2 ⊩ C2+} {σ1 : Γ1 ⊢ Asms Δ1} {σ2 : Γ2 ⊢ Asms Δ2}
        -> EqCPat{Δ1}{Δ2}{K1}{K2}{C1+}{C2+} c1 c2 -> Eq σ1 σ2 -> Eq{RFoc C1+}{RFoc C2+} (Val⁺ c1 σ1) (Val⁺ c2 σ2)
     ≡Ke⁻ : forall {Γ1 x1 Γ2 x2}
          -> Eq (Ke⁻{Γ1}{x1}) (Ke⁻{Γ2}{x2})
     ≡Cont⁺ : forall {Γ1 K1 Γ2 K2 γ1 γ2} {C1+ : Type⁺ K1} {C2+ : Type⁺ K2} 
            {φ1 : {Δ1 : Ctx} -> Δ1 ⊩ C1+ -> (Δ1 :: Γ1) ⊢ Neu γ1} {φ2 : {Δ2 : Ctx} -> Δ2 ⊩ C2+ -> (Δ2 :: Γ2) ⊢ Neu γ2} -> 
             ({Δ1 : Ctx} (c1 : Δ1 ⊩ C1+) {Δ2 : Ctx} (c2 : Δ2 ⊩ C2+) -> EqCPat{Δ1}{Δ2}{K1}{K2}{C1+}{C2+} c1 c2 -> Eq (φ1 c1) (φ2 c2))
            -> Eq {LInv (C1+ true⁺) γ1}{LInv (C2+ true⁺) γ2} (Cont⁺ φ1) (Cont⁺ φ2)
     ≡Cont⁻ : forall {Γ1 Γ2 Δ1 Δ2 γ01 γ02 γ1 γ2} {C1- : Type⁻} {C2- : Type⁻} 
            {d1 : Δ1 ⊩ C1- > γ01} {d2 : Δ2 ⊩ C2- > γ02} 
            {σ1 : Γ1 ⊢ Asms Δ1} {σ2 : Γ2 ⊢ Asms Δ2}
            {k1 : Γ1 ⊢ LInv γ01 γ1} {k2 : Γ2 ⊢ LInv γ02 γ2} 
            -> EqDPat d1 d2 -> Eq σ1 σ2 -> Eq k1 k2 -> Eq{LFoc C1- γ1}{LFoc C2- γ2} (Cont⁻ d1 σ1 k1) (Cont⁻ d2 σ2 k2)
     ≡Vx⁺ : forall {Γ1 x1 Γ2 x2}
          {i1 : (x1 atom⁺) ∈∈ Γ1} {i2 : (x2 atom⁺) ∈∈ Γ2} 
          -> EqInIn i1 i2
          -> Eq (Vx⁺{Γ1}{x1} i1) (Vx⁺{Γ2}{x2} i2)
     ≡Val⁻ : forall {Γ1 Γ2 C1- C2- }
            {φ1 : {Δ1 : Ctx} {γ1 : Conc} -> Δ1 ⊩ C1- > γ1 -> (Δ1 :: Γ1) ⊢ Neu γ1} 
            {φ2 : {Δ2 : Ctx} {γ2 : Conc} -> Δ2 ⊩ C2- > γ2 -> (Δ2 :: Γ2) ⊢ Neu γ2} -> 
             ({Δ1 : Ctx} {γ1 : Conc} (d1 : Δ1 ⊩ C1- > γ1) {Δ2 : Ctx} {γ2 : Conc} (d2 : Δ2 ⊩ C2- > γ2) -> EqDPat d1 d2 -> Eq (φ1 d1) (φ2 d2))
            -> Eq {RInv (C1- true⁻)}{RInv (C2- true⁻)} (Val⁻ φ1) (Val⁻ φ2)
     ≡R : forall {Γ1 K1 Γ2 K2} {C1+ : Type⁺ K1} {C2+ : Type⁺ K2} 
        {v1 : Γ1 ⊢ RFoc C1+} {v2 : Γ2 ⊢ RFoc C2+}
        -> Eq v1 v2 -> Eq{Neu (C1+ true⁺)}{Neu (C2+ true⁺)} (R v1) (R v2)
     ≡L : forall {Γ1 Γ2 γ1 γ2} {C1- : Type⁻} {C2- : Type⁻} 
        {x1 : ((C1- true⁻) ∈∈ Γ1)} {x2 : ((C2- true⁻) ∈∈ Γ2)} {k1 : Γ1 ⊢ LFoc C1- γ1} {k2 : Γ2 ⊢ LFoc C2- γ2}
        -> EqInIn x1 x2 -> Eq k1 k2 -> Eq (L x1 k1) (L x2 k2)
     ≡Sub : {Γ1 : CtxCtx} {Δ1 : Ctx} {s1 : {α : Hyp} -> α ∈ Δ1 -> Γ1 ⊢ RInv α}
            {Γ2 : CtxCtx} {Δ2 : Ctx} {s2 : {α : Hyp} -> α ∈ Δ2 -> Γ2 ⊢ RInv α}
        -> ({α1 : Hyp} (x1 : α1 ∈ Δ1) {α2 : Hyp} (x2 : α2 ∈ Δ2) -> EqIn x1 x2 -> Eq (s1 x1) (s2 x2))
        -> Eq{Asms Δ1}{Asms Δ2} (Sub s1) (Sub s2)

   equnforce : forall {Γ1 Γ2 J1 J2} (t1 : Γ1 ⊢ J1) (t2 : Γ2 ⊢ J2) -> Eq (Force.force t1) (Force.force t2) -> Eq t1 t2
   equnforce t1 t2 eq with (Force.force t1) | Force.forceid t1 | (Force.force t2) | Force.forceid t2
   ...  | .t1 | Refl | .t2 | Refl = eq

 module Subtyping where
  open SyntacticEquality public
  mutual 
\end{code}}

The focusing rules have the property that two flows of type information
never meet, except at atomic propositions \ttt{X⁺} and \ttt{X⁻}: in
fully η-expanded form, all of the type equality tests are pushed down to
base type.  (For related phenomena, see
LFR~\citep{lovaspfenning08refinement}, where subtyping at higher types
is characterized by an identity coercion, and
OTT~\citep{altenkirch+07ott}, where an $\eta$-expanded identity coercion
is induced by proofs of type equality).  The cut principles described
above have too little type information: the principal type of the cut
must be annotated or guessed.  On the other hand, the identity
principles may be used in a situation where two different flows of type
information meet.  In such a circumstance, for example, we may desire a
more general positive identity that \ttt{LInv γ γ}, if the surrounding
context requires \ttt{LInv γ γ'}, for two different types γ and γ'.  But
when does the identity coercion defined above suffice to map one type to
a different type?

For our language's types, this happens when there is an intensional mismatch in the Agda functions used to
specify \ttt{Σ⁺} and \ttt{Π⁻}.  For example, given an addition function
\ttt{plus*} on two patterns of type \ttt{nat}, the types 
\verb|(Σ⁺ \n -> Σ⁺ \m -> vec nat (plus* m n))| and 
\verb|(Σ⁺ \n -> Σ⁺ \m -> vec nat (plus* n m))| will be intensionally different, but extensionally
equal, in the sense that they have the same patterns.  In this case, 
the same implementation of \ttt{Ke⁺} that we gave above will work as a coercion from one of these types to the other.  

We codify this with a subtyping relation that is sufficient to define
the identity coercion.  Informally, the subtyping relation for positive
types is defined as follows: \ttt{A1⁺ <: A2⁺} iff for every pattern
\ttt{Δ1 ⊩ A1⁺}, the same pattern has type \ttt{Δ2 ⊩ A2⁺} where \ttt{Δ1
<: Δ2}.  The definition is then extended to contexts, assumptions,
conclusions, and negative types (which have a contravariant flip).  What
does it mean for the same pattern to have two different types?  If we
annotated the pattern judgement with raw proof terms (e.g., Δ ⊩ c : A⁺),
then we would require the same proof term \ttt{c} in both cases.  In our
intrinsic encoding, we can define an auxiliary judgement \ttt{EqCPat c1
c2} which relates two patterns, with two different types, as long as they
have the same structure.  We also require a similar judgement for
destructor patterns.  The two important cases of subtyping are the
following:

\begin{code}
   codata _<:⁺_ : {K : PKind} -> Type⁺ K -> Type⁺ K -> Set where
     Sub⁺ : forall {K} {A1+ A2+ : Type⁺ K} -> 
          ({ Δ1 : Ctx} (c1 : Δ1 ⊩ A1+) -> 
              Σ \ Δ2 -> Σ \(c2 : Δ2 ⊩ A2+) 
                -> EqCPat{Δ1}{Δ2}{K}{K}{A1+}{A2+} c1 c2 
                   × Δ1 <:Δ Δ2)
          -> A1+ <:⁺ A2+
   codata _<:⁻_ : Type⁻ -> Type⁻ -> Set where
     Sub⁻ : forall {A1- A2- } ->
            ({Δ2 : Ctx} {γ2 : Conc} (d2 : Δ2 ⊩ A2- > γ2)
              -> Σ \Δ1 -> Σ \γ1 -> Σ \(d1 : Δ1 ⊩ A1- > γ1) -> 
                EqDPat d1 d2 
                × ((Δ1 , γ1) <:Δγ (Δ2 , γ2))) 
          -> A1- <:⁻ A2-
\end{code}

\ignore{
\begin{code}
   codata _<:Δ_ : Ctx -> Ctx -> Set where
     SubΔ : forall {Δ1 Δ2} -> ({α2 : Hyp} (x1 : α2 ∈ Δ2) -> Σ \ α1 -> Σ \ (x2 : α1 ∈ Δ1) -> EqIn x1 x2 × (α1 <:α α2)) -> Δ1 <:Δ Δ2
   codata _<:α_ : Hyp -> Hyp -> Set where
     Subα⁺ : {x+ : Atom} -> (x+ atom⁺) <:α (x+ atom⁺)
     Subα⁻ : {A- B- : Type⁻} -> A- <:⁻ B- -> (A- true⁻) <:α (B- true⁻)
   codata _<:γ_ : Conc -> Conc -> Set where
     Subγ⁻ : {x- : Atom} -> (x- atom⁻) <:γ (x- atom⁻)
     Subγ⁺ : forall {K} {A+ B+ : Type⁺ K} -> A+ <:⁺ B+ -> (A+ true⁺) <:γ (B+ true⁺)
   codata _<:Δγ_ : (Ctx × Conc) -> (Ctx × Conc) -> Set where
     SubΔγ : forall { Δ1 Δ2 γ1 γ2} -> Δ2 <:Δ Δ1 -> γ1 <:γ γ2 -> (Δ1 , γ1) <:Δγ (Δ2 , γ2)
\end{code}
}

\ignore{
\begin{code}
  unsubΔγ : forall {p1 p2} -> p1 <:Δγ p2 -> (fst p2) <:Δ (fst p1) × (snd p1) <:γ (snd p2)
  unsubΔγ (SubΔγ p1 p2) = p1 , p2
  
  unsub⁺ : forall {K} {A1+ A2+ : Type⁺ K} -> A1+ <:⁺ A2+ ->
             ({ Δ1 : Ctx} (c1 : Δ1 ⊩ A1+) -> Σ \ Δ2 -> Σ \(c2 : Δ2 ⊩ A2+) -> EqCPat c1 c2 × Δ1 <:Δ Δ2)
  unsub⁺ (Sub⁺ f) = f

  <:-refl : {A : Set} -> (A -> A -> Set) -> Set
  <:-refl lt = (x : _) -> lt x x

  <:-trans : {A : Set} -> (A -> A -> Set) -> Set
  <:-trans lt = {x : _} {y : _} {z : _} -> lt x y -> lt y z -> lt x z

  mutual
   <:Δ-refl : <:-refl _<:Δ_
   <:Δ-refl Δ ~ SubΔ (\ {α} x -> (_ , x , eqin-refl x , <:α-refl α))

   <:α-refl : <:-refl _<:α_
   <:α-refl (X atom⁺) ~ Subα⁺
   <:α-refl (A- true⁻) ~ Subα⁻ (<:⁻-refl A-)

   <:⁻-refl : <:-refl _<:⁻_
   <:⁻-refl A- ~ Sub⁻ (\{Δ} {γ} d -> Δ , γ , d , eqdpat-refl d , SubΔγ (<:Δ-refl Δ) (<:γ-refl γ)) 

   <:γ-refl : <:-refl _<:γ_
   <:γ-refl (X atom⁻) ~ Subγ⁻  
   <:γ-refl (A true⁺) ~ Subγ⁺ (<:⁺-refl A)

   <:⁺-refl : forall {K} (A : Type⁺ K) -> A <:⁺ A
   <:⁺-refl A ~ Sub⁺ (\ {Δ} c -> Δ , c , eqcpat-refl _ c , <:Δ-refl Δ)

 module Ident where
  open Subtyping

  mutual
\end{code}
}

The subtyping relation suffices to define the identity coercion; the
code is essentially the same as above, aside from the need to push the
subtyping proofs through.  

\begin{code}
    Ke⁺ : forall {K Γ} {C1+ C2+ : Type⁺ K} 
        -> C1+ <:⁺ C2+ 
        -> Γ ⊢ LInv (C1+ true⁺) (C2+ true⁺)
\end{code}

\ignore{
\begin{code}
    Ke⁺ {_} {Γ} {C1+} {C2+} (Sub⁺ f) ~ Cont⁺ (\ c -> R (Val⁺ (fst (snd (f c))) (Ids (snd (snd (snd (f c)))) s0)))
\end{code}
}

\begin{code}
    Vx⁻  : forall {Γ C1- C2- } 
         -> C1- <:⁻ C2- 
         -> ((C1- true⁻) ∈∈ Γ) -> Γ ⊢ RInv (C2- true⁻)
\end{code}

In summary, our focused formalism has squeezed the balloon so that
proofs of equality are only needed in one spot: to show that an identity
coercion exists between apparently different types.  So far, we have
manually constructed the subtyping proofs when necessary.  However, we
have also proved that the subtyping proofs are unique, in the sense that
any two proofs of \ttt{A <: B} determine the same identity coercion
(where ``same'' means structural equality of proof terms---a judgement
similar to \ttt{EqCPat} but for the focusing judgement).  This licenses
the use of theorem proving to find proofs of subtyping, as the code
determined by the proofs will be the same.  We intend to explore ways of
exploiting this in future work.

\ignore{
\begin{code}
    Vx⁻ (Sub⁻ f) x ~ Val⁻ (\d ->
                              L (sS x)
                              (Cont⁻ (fst (snd (snd (f d)))) 
                              (Ids (fst (unsubΔγ (snd (snd (snd (snd (f d))))))) s0)
                              (Ke (snd (unsubΔγ (snd (snd (snd (snd (f d))))))))))

    Vx : forall {Γ α1 α2} -> α1 <:α α2 -> (α1 ∈∈ Γ) -> Γ ⊢ RInv α2
    Vx Subα⁺ x ~ Vx⁺ x
    Vx (Subα⁻ s) x ~ Vx⁻ s x
    
    Ke : forall {Γ γ1 γ2} -> γ1 <:γ γ2 -> Γ ⊢ LInv γ1 γ2
    Ke Subγ⁻ ~ Ke⁻
    Ke (Subγ⁺ s) ~ Ke⁺ s
           
    Ids  : forall {Γ Δ1 Δ2} -> Δ1 <:Δ Δ2 
         -> ({h : Hyp} -> (h ∈ Δ1) -> (h ∈∈ Γ)) 
         -> Γ ⊢ Asms Δ2
    Ids (SubΔ f) subset ~ Sub (\ {_} h -> Vx (snd (snd (snd (f h)))) (subset (fst (snd (f h)))))

 open Ident

 module Unique where
   open Subtyping
   open Ident

   -- productivity checker doesn't pass this because the equnforce retyping is in the way
   mutual
       unique⁻ : forall {Γ1 Γ2 A1- A2- B1- B2- } (sub1 : A1- <:⁻ B1-) (sub2 : A2- <:⁻ B2-) 
               -> Id A1- A2- -> Id B1- B2-
               -> {i1 : (A1- true⁻) ∈∈ Γ1} {i2 : (A2- true⁻) ∈∈ Γ2} -> EqInIn i1 i2 
               -> Eq (Force.force (Vx⁻ sub1 i1)) (Force.force (Vx⁻ sub2 i2))
       unique⁻ (Sub⁻ f1) (Sub⁻ f2) Refl Refl r ~ 
         ≡Val⁻ (\ {Δ1} {γ1} d1 {Δ2} {γ2} d2 eqd -> 
           ≡L (≡sS r) (≡Cont⁻
                       (eqdpat-trans (fst (snd (snd (snd (f1 d1)))))
                         (eqdpat-trans eqd (eqdpat-sym (fst (snd (snd (snd (f2 d2))))))))
                       (equnforce _ _
                         (uniqueΔ (fst (unsubΔγ (snd (snd (snd (snd (f1 d1)))))))
                                  (fst (unsubΔγ (snd (snd (snd (snd (f2 d2)))))))
                                  (eqdpat-idΔ eqd) 
                                  (eqdpat-idΔ (eqdpat-trans (fst (snd (snd (snd (f1 d1)))))
                                                (eqdpat-trans eqd (eqdpat-sym (fst (snd (snd (snd (f2 d2))))))))) 
                                  (\eq -> ≡s0 eq)))
                       (equnforce _ _
                         (uniqueγ (snd (unsubΔγ (snd (snd (snd (snd (f1 d1)))))))
                          (snd (unsubΔγ (snd (snd (snd (snd (f2 d2))))))) 
                          (eqdpat-idγ
                            (eqdpat-trans (fst (snd (snd (snd (f1 d1)))))
                             (eqdpat-trans eqd (eqdpat-sym (fst (snd (snd (snd (f2 d2))))))))) 
                          (eqdpat-idγ eqd)))))

       uniqueΔ : forall {Γ1 Γ2 Δ1 Δ2 Δ1' Δ2'} (sub1 : Δ1 <:Δ Δ1') (sub2 : Δ2 <:Δ Δ2')
               -> Id Δ1 Δ2 -> Id Δ1' Δ2' -> 
               {subset1 : ({h : Hyp} -> (h ∈ Δ1) -> (h ∈∈ Γ1))} 
               {subset2 : ({h : Hyp} -> (h ∈ Δ2) -> (h ∈∈ Γ2))}
               -> ({h1 : Hyp} {x1 : h1 ∈ Δ1} {h2 : Hyp} {x2 : h2 ∈ Δ2} -> EqIn x1 x2 -> EqInIn (subset1 x1) (subset2 x2)) 
               -> Eq (Force.force (Ids sub1 subset1)) (Force.force (Ids sub2 subset2))
       uniqueΔ (SubΔ s1) (SubΔ s2) Refl Refl id ~ 
         ≡Sub (\ {α1} x1 {α2} x2 idx -> equnforce _ _ 
           (uniqueα (snd (snd (snd (s1 x1)))) (snd (snd (snd (s2 x2))))
                    (eqin-idα (eqin-trans (eqin-sym (fst (snd (snd (s1 x1)))))
                                (eqin-trans idx (fst (snd (snd (s2 x2)))))))
                    (eqin-idα idx) 
                    (id
                      (eqin-trans (eqin-sym (fst (snd (snd (s1 x1)))))
                       (eqin-trans idx (fst (snd (snd (s2 x2)))))))
           ))

       uniqueα : forall {Γ1 Γ2 α1 α1' α2 α2' } (sub1 : α1 <:α α1') (sub2 : α2 <:α α2') 
               -> Id α1 α2 -> Id α1' α2'
               -> {i1 : α1 ∈∈ Γ1} {i2 : α2 ∈∈ Γ2} -> EqInIn i1 i2 
               -> Eq (Force.force (Vx sub1 i1)) (Force.force (Vx sub2 i2))
       uniqueα Subα⁺ Subα⁺ Refl Refl id ~ ≡Vx⁺ id
       uniqueα (Subα⁻ s1) (Subα⁻ s2) Refl Refl id ~ unique⁻ s1 s2 Refl Refl id
       uniqueα (Subα⁻ _ ) Subα⁺ () p2 _ 
       uniqueα Subα⁺ (Subα⁻ _ ) () p2 _ 

       uniqueγ : forall {Γ1 Γ2 γ1 γ1' γ2 γ2' } (sub1 : γ1 <:γ γ1') (sub2 : γ2 <:γ γ2') 
               -> Id γ1 γ2 -> Id γ1' γ2'
               -> Eq{_}{_}{Γ1}{Γ2} (Force.force (Ke sub1)) (Force.force (Ke sub2))
       uniqueγ Subγ⁻ Subγ⁻ _ _ ~ ≡Ke⁻
       uniqueγ (Subγ⁺ s1) (Subγ⁺ s2) Refl Refl ~ unique⁺ s1 s2 Refl Refl
       uniqueγ Subγ⁻ (Subγ⁺ _) () _ 
       uniqueγ (Subγ⁺ _) Subγ⁻ () _ 

       unique⁺ : forall {Γ1 Γ2 K} {A1 A1' A2 A2' : Type⁺ K} (sub1 : A1 <:⁺ A1') (sub2 : A2 <:⁺ A2') 
               -> Id A1 A2 -> Id A1' A2'
               -> Eq{_}{_}{Γ1}{Γ2} (Force.force (Ke⁺ sub1)) (Force.force (Ke⁺ sub2))
       unique⁺ (Sub⁺ sub1) (Sub⁺ sub2) Refl Refl ~ 
         ≡Cont⁺ (\ {Δ1} c1 {Δ2} c2 eqpat -> 
          ≡R (≡Val⁺ (eqcpat-trans (eqcpat-sym (fst (snd (snd (sub1 c1))))) (eqcpat-trans eqpat (fst (snd (snd (sub2 c2))))))
                    (equnforce _ _ (uniqueΔ (snd (snd (snd (sub1 c1)))) (snd (snd (snd (sub2 c2)))) 
                                            (eqcpat-idΔ eqpat) 
                                            (eqcpat-idΔ (eqcpat-trans (eqcpat-sym (fst (snd (snd (sub1 c1)))))
                                                          (eqcpat-trans eqpat (fst (snd (snd (sub2 c2))))))) 
                                            ≡s0))))

   {-
   lemma : {K : PKind} {A1 A2 : Type⁺ PPos} {τ1 : [] ⊩ A1 -> Type⁺ K} {τ2 : [] ⊩ A2 -> Type⁺ K}
         (s : A1 <:⁺ A2)
         -> ((c : [] ⊩ A1) -> τ1 c <:⁺ τ2 (Closed.close {A2} (fst (snd (unsub⁺ s c)))))
         -> ({ Δ : Ctx} (c :  Δ ⊩ (Σ⁺ A1 τ1))  -> 
               Σ \ Δ' -> Σ \(c2 : Δ' ⊩ (Σ⁺ A2 τ2)) -> HId c c2 × Δ <:Δ Δ')
   --          -> Σ⁺ A1 τ1 <:⁺ Σ⁺ A2 τ2
   lemma {_} {A1} {A2} s s' (Cdpair p1 p2) = 
     _ , 
     Cdpair (Closed.close {A2} (fst (snd (unsub⁺ s p1))))
            (fst (snd (unsub⁺ (s' p1) p2))) , 
     {! -- fst (snd (snd (unsub⁺ s p1))) 
       fst (snd (snd (unsub⁺ (s' p1) p2))) !} , 
     snd (snd (snd (unsub⁺ (s' p1) p2)))
   -}

\end{code}
}

