#!/usr/bin/env ruby

##
# To run:  ruby dot-test.rb -v
# (You want the verbose flag;  it makes the testing more clear)
##

$LOAD_PATH << (File.dirname(__FILE__))

require 'test/unit'
require 'thread'
require 'tempfile'

require 'unit_test_hack'
require 'spawn_buffered'

class GTCD < SpawnBufferedReader
    attr_reader :cachedir, :sock, :port_base, :logpath
    def initialize(extra_args = nil, port_base = 15000, 
		   sockname_override = nil)
	@port_base = port_base
	@cachedir_base = "tmp" + @port_base.to_s
	@cachedir = "/tmp/" + @cachedir_base
	@sock = "/tmp/gtcd_" + @port_base.to_s
	if (sockname_override)
	    @sock = sockname_override
	end

	ENV["DOT_CACHE_HOME"] = @cachedir_base
	ea = extra_args || ""

	@logpath = "/tmp/gtcd-#{@port_base}"
	super("gtcd/gtcd -p #{@sock} -t #{@port_base} #{ea}", @logpath)
	@keep_log = true
    end

    def keep_log
	@keep_log = true
    end

    def gcp_args
	"-p #{sock}"
    end

    def kill
	super
	system("rm -rf #{@cachedir}")
	File.unlink(@sock)
	if (!@keep_log)
	    File.unlink(@logpath)
	end
    end
end

module DOTTestUtil
    def create_random_file(size)
	t = Tempfile.new("dottest")
	dr = File.open("/dev/urandom", "r")
	while (size > 0)
	    toread = [size, 4096].min
	    t.syswrite(dr.sysread(toread))
	    size -= toread
	end
	return t
    end

    def do_gcp_put(gtcd, path)
	run_gcp = "gcp/gcp " + gtcd.gcp_args + " -P #{path} 2>/dev/null"
	put_res = `#{run_gcp}`
	assert_match(/^PUT_OID:([^:]+):/, put_res, "output of gcp put incorrect")
	oid = ""
	if (put_res =~ /^PUT_OID:([^:]+):/)
	    oid = $1
	end
	return oid
    end

    def do_gcp_get(gtcd, oid, outpath)
	run_gcp = "gcp/gcp " + gtcd.gcp_args + " dot://#{oid} #{outpath} 2>/dev/null"
	get_res = `#{run_gcp}`
	assert_equal($?, 0, "Exit code of gcp was non-zero")
    end

    def do_cross_gtc_put_get(p1, p2)
	oid = do_gcp_put(@g[0], p1)
	oid += ":127.0.0.1:#{@g[0].port_base.to_s}"
	do_gcp_get(@g[1], oid, p2)
	assert_files_equal(p1, p2)
    end
end

class TestDOT < CleanupTestCase
    include DOTTestUtil

    def setup_once
	@g = Array.new
	@g[0] = GTCD.new(nil, 12000, "/tmp/gtcd.sock") # a "normal" gtcd
	@g[1] = GTCD.new(nil, 15000)
	sleep(1)
	return @g # passed to setup_real each time
    end

    def setup_real(params)
	@g = params
	@t1 = Tempfile.new("dottest")
	@t2 = Tempfile.new("dottest")
    end

    def cleanup
	@g.each { |g| g.kill if g }
	super
    end

    def teardown
	@t1.close if @t1
	@t1.unlink if @t1
	@t2.close if @t2
	@t2.unlink if @t2
	@t1 = nil
	@t2 = nil
    end

    def test_gtcds_run
	@g.each { |g| assert(g.is_alive?, "gtcd was dead") }
    end

    def test_gcp_put
	assert(@g[0].is_alive?)
	@t1.write('a' * 500)
	@t1.close
	oid = do_gcp_put(@g[0], @t1.path)
	assert_equal($?, 0, "gcp put exit code was non-zero")
	assert(@g[0].is_alive?)
	assert_equal(oid, "e62ca5609e96073ffdc80ad480510d6de0a13f3e")
    end

    def test_gcp_put_get
	@t1.write('a' * 500)
	@t1.close
	@t2.close
	oid = do_gcp_put(@g[0], @t1.path)
	do_gcp_get(@g[0], oid, @t2.path)
	assert_files_equal(@t1.path, @t2.path)
    end

    def test_cross_gtc_put_get
	@t1.write('a' * 500)
	@t1.close
	@t2.close
	do_cross_gtc_put_get(@t1.path, @t2.path)
    end

    def test_random_put_gets
	return
	10.times do
	    begin
		t1 = create_random_file(rand(1000000))
		t1.close
		t2 = Tempfile.new("dottest")
		t2.close
		do_cross_gtc_put_get(t1.path, t2.path)
	    ensure
		t1.unlink if t1
		t2.unlink if t2
	    end
	end
    end

    def test_zero_byte_put_get
	@t1.close
	@t2.close
	do_cross_gtc_put_get(@t1.path, @t2.path)
	assert(@g[0].is_alive?, "gtcd1 dead after zero byte put/get")
	assert(@g[1].is_alive?, "gtcd2 dead after zero byte put/get")
    end

end

module ManySetup
    def setup_once
	@cdht = SpawnBufferedReader.new("cdht/cdht_server", "/tmp/cdht.log")
	gtcds = Array.new
	@g = Array.new
	@g[0] = GTCD.new("-S", 12000, "/tmp/gtcd.sock")
	(0..4).each do |x|
	    @g << GTCD.new("-S", 15000 + (x*1000))
	end
	sleep 2
	return [@cdht, @g].flatten
    end

    def setup_real(params)
	@cdht, *@g = params
	@t1 = Tempfile.new("dottest")
	@t2 = Tempfile.new("dottest")
    end


end

class TestSET < TestDOT
    include ManySetup
    # Inherits all DOT basic tests

    def cleanup
	@cdht.kill if @cdht
	super
    end

    def test_many_sources
	t1 = create_random_file(rand(1000000))
	t1.close
	oid = nil
	oldoid = nil
	@g[1..-1].each { |g|
	    oid = do_gcp_put(g, t1.path)
	    assert_equal(oid, oldoid, "OIDs from puts do not match") if oldoid
	    oldoid = oid
	}
	oidhints = "#{oid}:127.0.0.1:17000"
	do_gcp_get(@g[0], oidhints, @t2.path)
	assert_files_equal(t1.path, @t2.path)
    ensure
	t1.unlink if t1
    end

end

# This test doesn't actually work because the localhost source
# is too fast.  But it's a start.

class TestSET_Swarming < CleanupTestCase
    include ManySetup
    include DOTTestUtil

    def cleanup
	@g.each { |g| g.kill if g }
	@cdht.kill if @cdht
	super
    end
    
    def test_swarmers
	t1 = create_random_file(rand(1000000))
	t1.close
	oid = do_gcp_put(@g[0], t1.path)
	oidhints = "#{oid}:127.0.0.1:12000"
	threads = Array.new
	@g[1..-1].each { |g|
	    threads << Thread.new(g) { |target|
		t = Tempfile.new("dottest")
		t.close
		do_gcp_get(target, oidhints, t.path)
		# not clear that this assertion failure willl work!!
		assert_files_equal(t1.path, t.path);
	    }
	}
	threads.each { |t| t.join }
    end
end
