/*
 * Copyright (c) 2005-2006 Carnegie Mellon University and Intel Corporation.
 * All rights reserved.
 * See the file "LICENSE" for licensing terms.
 */

#ifndef _XFER_OPT_H_
#define _XFER_OPT_H_

#include "xferPlugin.h"
#include "gtcd.h"
#include "xferPlugin_gtc_prot.h"
#include "se_transfer.h"
#include "params.h"
#include "sha1.h"
#include "xferPlugin_gtc.h"

struct opt_result {
    ptr< vec<dot_descriptor> > missing_xdisk;
    ptr< vec<dot_descriptor> > missing_net;
    ptr<hv_vec > missing_xdisk_hints;
    ptr<hv_vec > missing_net_hints;
    str opid;
    ptr< vec<dot_descriptor> > prev_miss_net;
    bool cancel;

    opt_result() {
	missing_xdisk = NULL;
	missing_net = NULL;
	missing_xdisk_hints = NULL;
	missing_net_hints = NULL;
	prev_miss_net = NULL;
	cancel = false;
	opid = NULL;
    }
    ~opt_result() { }
};

class xferPlugin_opt;
class ds_entry;

class ht_entry {
public:
    str key;
    ihash_entry<ht_entry> link;
    
    ref<vec<oid_hint> > oidhint;
    ptr<vec<xdisk_hint> > hints;
    unsigned int counter;

    ptr<vec<ds_entry *> > desc_vec;

    double avail;

    //debug
    int num;

    ht_entry(str k, ref<vec<oid_hint> > oh, ds_entry *a);
    ~ht_entry();
    void add_desc(ds_entry *a);
    void remove_desc(ds_entry *a);
};

class Compare_ht_entry {
public:
    int operator()(const ht_entry *x, const ht_entry *y) {
	    return x->avail > y->avail;
    }
};

class ds_entry {
public:
    dot_desc desc;
    ihash_entry<ds_entry> link;

    vec<chunk_cb > ccb;
    dot_descriptor dd;

    bool ask;
    ht_entry *hint_ptr;

    int ordernum;
    //index in the hint ptr's vec
    int hint_index;

    ds_entry(chunk_cb cb, dot_descriptor dot_des, 
	     ref<vec<oid_hint> > oh, xferPlugin_opt *xp);
    ~ds_entry();
    void update_info(ref<vec<oid_hint> > oh, chunk_cb cb, xferPlugin_opt *xp);
    void print_hints();
    void delete_from_hint();
};

// class Compare_ds_entry {
// public:
//     int operator()(const ds_entry *x, const ds_entry *y) {
// 	    return x->adj > y->adj;
//     }
// };

struct item_info {
    str name;
    struct stat s;
};

class os_entry {
public:
    dot_desc desc;
    str id;
    ihash_entry<os_entry> link;
    tailq_entry<os_entry> tlink;
    
    double op_cost;
    double sum_p;
    double sum_cp;
    double cpb; //cost per block
    int it_num;
    //number of blocks
    //obtained by this operation
    unsigned int success;
    double time;

    xfer_op type;
    str path;
    
    virtual void get_cost() = 0;
    virtual void perform_op(CLOSURE) = 0;
    virtual float get_benefit(ht_entry *) = 0;
    virtual double get_xfer_cost(ht_entry *) = 0;
    virtual void dump_info() = 0;
    virtual bool is_chit_op() = 0;
    virtual ~os_entry() { };
};

class Compare_os_entry {
public:
    int operator()(const os_entry *x, const os_entry *y) {
	    return x->cpb > y->cpb;
    }
};

#include <vector>
#include <algorithm>
class heap_impl {
private:  
    std::vector<os_entry *> v;
public:
    void heap_insert(os_entry *ose);
    void heap_pop(os_entry *ose);
    os_entry *heap_top();
    void heap_make();
    void heap_sort(ptr<vec<str > > op);
    void print_vec(std::vector<os_entry *> &);
};

class opt_matrix {

public:
    
    ihash<const dot_desc, ds_entry, &ds_entry::desc, &ds_entry::link, dd_hash> desc_store;
    ihash<const str, os_entry, &os_entry::id, &os_entry::link> op_store;
    ihash<const str, ht_entry, &ht_entry::key, &ht_entry::link> hint_store;

    //queues
    //for bfs
    tailq<os_entry, &os_entry::tlink> hash_q;
    tailq<os_entry, &os_entry::tlink> stat_q;

    //pending ops list
    tailq<os_entry, &os_entry::tlink> pending_q;
    heap_impl *heap;
    
    //to keep sorted descs
    std::vector<ht_entry *> s;
    struct last {
	int hint;
	int desc;
    } index_set;

    opt_matrix() { heap = New heap_impl(); index_set.hint = 0; index_set.desc = -1; }
    ~opt_matrix() { delete heap; }
        
    void dump_table();
   
    void add_col(ds_entry *);
    void delete_col(dot_desc);
    void add_row(os_entry *);
    void delete_row_start(str);
    void delete_row_complete(str);

    void refresh_rows_complete();
    void refresh_rows_partial();
    os_entry *extract_min_row();
    void sort_rows(ptr<vec<str > > op);
    void sort_cols();
};

class xferPlugin_opt : public xferPlugin {
  
private:
    gtcd *m;
    //for optimization
    int cur_iteration;
    double cur_change_chunks;
    unsigned int prev_pending_chunks;
    ptr< vec<dot_descriptor> > prev_missing_net;
    bool disk_busy;
    int net_scheduled;
    //for debugging
    str path; unsigned int op_mode;
    
public:
    vec<xferPlugin*> xfplugins;
    
    bool configure(str s);
    /* This should only be called after initialization */
    void get_default_hint(oid_hint *hint);

    /* Calls from the GTC */
    void get_descriptors(ref<dot_oid_md> oid, ref<vec<oid_hint> > hints,
			 descriptors_cb cb, CLOSURE);
    void get_bitmap(ref<dot_oid_md> oid, ref<vec<oid_hint> > hints,
		    bitmap_cb cb, CLOSURE);
    void notify_descriptors(ref<dot_oid_md> oid, ptr<vec<dot_descriptor> > descs);
    void get_chunk(ref<dot_descriptor> d, ref<vec<oid_hint> > hints,
                   chunk_cb cb, CLOSURE);
    void get_chunks(ref< vec<dot_descriptor> > dv, ref<hv_vec > hints,
		    chunk_cb cb, CLOSURE);

    void cancel_chunk(ref<dot_descriptor> d);
    void cancel_chunks(ref< vec<dot_descriptor> > dv);

    void update_hints(ref< vec<dot_descriptor> > dv, ref<hv_vec > hints);

    void set_more_plugins(vec<xferPlugin*> xplist);
	
    xferPlugin_opt(gtcd *m, xferPlugin *next_xp);
    ~xferPlugin_opt() { }

    void update_rows(ht_entry *hte);
    
private:
    void schedule_optimization();
    void perform_optimization(CLOSURE);
    void desc_avail(ht_entry * dse, ptr<vec<str > > r);
    void cost_per_block(os_entry * ose);
    bool filter_computation();
    void pick_single_block(ptr<opt_result> res);
    void pick_block_alloc(ptr<opt_result> res, bool compute);
    void pick_xdisk_op(ptr<opt_result> res, bool compute);
    void pick_xdisk_bfs_op(ptr<opt_result> res);
    void pick_op_descs(callback<void, str, ptr<opt_result> >::ref cb, CLOSURE);
    void get_descriptors_cb(descriptors_cb cb1, str s, ptr<vec<dot_descriptor> > descs, bool end);
    void dump_rusage(struct rusage *rs1, struct rusage *rs2);
    void get_chunks_cb(str opid, unsigned int plugin, str s, ptr<desc_result> res);
    double get_change_chunks_new_ops();
    double get_change_chunks_del_op(str opid);
};


#endif /* _XFER_NET_H_ */
