/*
 * Copyright (c) 2005-2006 Carnegie Mellon University and Intel Corporation.
 * All rights reserved.
 * See the file "LICENSE" for licensing terms.
 */

#include "xferPlugin_gtc.h"
#include "parseopt.h"

#include <openssl/evp.h> /* Shouldn't need!  Abstract the OID contents */

/* Mac OS X ugly hack - adapt to missing header definition.  Poo. */
#ifndef HOST_NAME_MAX
#define HOST_NAME_MAX MAXHOSTNAMELEN
#endif

#define CONN_ENTRY_IDLE_SEC    600 
#define CONN_ENTRY_IDLE_NSEC   000

#define MAX_RPCS_IN_FLIGHT     50

#define DEFAULT_PORT           12000

static ihash<const dot_desc, stat_entry, &stat_entry::cid, &stat_entry::hlink, dd_hash> statCache;

ihash<const str, conn_entry, &conn_entry::key, &conn_entry::hlink> connCache;

stat_entry::stat_entry(const dot_desc o)
    : cid(o)
{
    count = 0;
    //warn << "stat_entry::stat_entry: Creating stat_entry for " << cid << "\n";
    statCache.insert(this);
}

stat_entry::~stat_entry()
{
    statCache.remove(this);
}

pending_conn_entry::pending_conn_entry(const str key)
    : key(key)
{
}

conn_entry::conn_entry(const str &k, ref<aclnt> c, int f)
    : key(k), c(c), refcount(1), fd(f)
{
    dwarn(DEBUG_XFER_GTC) << "conn_entry for " << k << "\n";
}

conn_entry::~conn_entry()
{
    dwarn(DEBUG_XFER_GTC) << "Destroying conn entry for " << key << "\n";
    for (wait_conn_ent *i = wait_cb.first; i; i = wait_cb.next(i)) {
	(*(i->cb))(NULL);
    }
}

xferPlugin_gtc::xferPlugin_gtc(gtcd *_m, xferPlugin *next_xp)
    : m(_m)
{
    assert(m);
    if (next_xp)
        fatal << __PRETTY_FUNCTION__ << " next_xp is not NULL\n"
              << "Make sure that this storage plugin comes last\n";

    sp = m->sp;
}

bool
xferPlugin_gtc::configure(str s)
{
    int port_num;
    if (!s || s == "")
        port_num = DEFAULT_PORT;
    else if (!convertint(s, &port_num)) {
        warn << "Cannot parse port number: " << s << "\n";
        return false;
    }

    dwarn(DEBUG_INIT|DEBUG_XFER_GTC) << "xferPlugin_gtc starting on port "
				     << port_num << "\n";
    xfer_gtc_listen_port = port_num;
   
    sock = inetsocket(SOCK_STREAM, xfer_gtc_listen_port);
    if (sock < 0)
        fatal("xfer_gtc inetsocket: %m\n");

    close_on_exec(sock);
    make_async(sock);

    if (!tcb) {
        tcb = delaycb(CONN_ENTRY_IDLE_SEC/10, CONN_ENTRY_IDLE_NSEC, 
                      wrap(this, &xferPlugin_gtc::reap_conn_entries));
    }

    listen(sock, 150);
    fdcb(sock, selread, wrap(this, &xferPlugin_gtc::accept_connection, sock));

    //delaycb(5, 0, wrap(this, &xferPlugin_gtc::dump_statcache));

    return true;
}

void 
xferPlugin_gtc::int_reap_conn_entries(conn_entry *conn)
{
    struct timeval tv;
    gettimeofday(&tv, NULL);

    if (conn->refcount == 0 &&
        tv.tv_sec - conn->tv.tv_sec > CONN_ENTRY_IDLE_SEC) {
        dwarn(DEBUG_XFER_GTC) << "deleting stale connection to "
			      << conn->key << "\n" ;
	connCache.remove(conn);
        delete conn;
    }
}

void 
xferPlugin_gtc::reap_conn_entries()
{
    // warn << "about to reap connection entries \n";
    connCache.traverse(wrap(this, &xferPlugin_gtc::int_reap_conn_entries));
    tcb = delaycb(CONN_ENTRY_IDLE_SEC/10, CONN_ENTRY_IDLE_NSEC, 
                  wrap(this, &xferPlugin_gtc::reap_conn_entries));
}

void 
xferPlugin_gtc::get_default_hint(oid_hint *hint)
{
    str hostname;
    str comp = "10"; //used in parsing default_hint in gtc plugin for emulab
    unsigned int port;
    strbuf buf;
    
    port = xfer_gtc_listen_port;

#if 1
    // XXX: Make more robust by fixing addr. selection heuristic
    // or by sending multiple addresses as multiple hints
    vec<in_addr> av;
    if (myipaddrs (&av)) {
        for (in_addr *ap = av.base (); ap < av.lim (); ap++) {
            if (ap->s_addr != htonl (INADDR_LOOPBACK)
		&& ap->s_addr != htonl (0)) {
                char s[64];
                if (inet_ntop(AF_INET, ap, s, sizeof(s))) {
#ifdef EMULAB
		  //warn << "xferPlugin_gtc::get_default_hint:: current is "
		  // << s <<"\n";
		    char *d, *hn = NULL;
		    
		    d = strdup(s); 
		    assert(d != NULL);
		    if ((hn = strchr(d, '.'))) {
			*hn++ = '\0';
			str temp = d;
			
			//warn << "xferPlugin_gtc::get_default_hint:: temp is "
			//   << temp <<"\n";
			
			if (temp == comp) {
			    hostname = str(s);
			    //  warn << "xferPlugin_gtc::get_default_hint:: hint "
			    // << hostname <<"\n";
			    break;
			}
		    }
#else
		    hostname = str(s);
		    break;
#endif
                }
            }
        }
    }
#else
    if ((hostname = myname())) {
	if (hostname == "localhost") {
	    hostname = "127.0.0.1";
	}
    }
    else {
	hostname = "";
    }
#endif

    dwarn(DEBUG_XFER_GTC) << "Hints: hostname:port is " << hostname << ":"
			  << port << "\n";

    buf << "gtc://" << hostname << ":" << port;
    hint->name = buf;
}

void
xferPlugin_gtc::dump_statcache()
{
    warn << "Dumping statCache\n" << debug_sep;
    stat_entry *se = statCache.first();
    while (se != NULL) {

	warn << se->cid << " " << se->count << "\n";
	se = statCache.next(se);
    }
    warn << debug_sep;
    
    delaycb(5, 0, wrap(this, &xferPlugin_gtc::dump_statcache));
}

void
xferPlugin_gtc::accept_connection(int s)
{
    struct sockaddr_in sin;
    socklen_t sinlen = sizeof(sin);
    bzero(&sin, sizeof(sin));
    
    int cs = accept(sock, (struct sockaddr *) &sin, &sinlen);
    if (cs < 0) {
        if (errno != EAGAIN)
            warn << "gtc accept failed; errno = " << errno << "\n";
        return;
    }
    tcp_nodelay(cs);
    make_async(cs);
    close_on_exec(cs);

    xferGtcConn *c = New xferGtcConn(cs, sin, this);
    subconnlist.insert_head(c);
}

void
xferPlugin_gtc::remote_get_descriptors_cb(svccb *sbp, unsigned int offset, str s, 
				          ptr< vec<dot_descriptor> > descs, bool end)
{
    xfergtc_get_descriptors_res res(false);
    /* XXX - we're ignoring end right now b/c storage always gives us
       everything */
    if (s) {
        *res.errmsg = s;
        sbp->replyref(res);
        return;
    }
    
    if (offset > 0 && offset >= descs->size()) {
        *res.errmsg = "Too large offset for GET_DESCRIPTORS";
        warn << *res.errmsg << "\n";
        sbp->replyref(res);
        return;
    }

    res.set_ok(true);
    // Determine how many descriptors to send,
    // assuming that all descriptors are the same size
    unsigned int maxsize = SEND_SIZE;
    // XXX - Figure out a way to get the size of a descriptor. The
    // number below is a kludge!
    unsigned int maxd = maxsize*2 / (EVP_MAX_MD_SIZE*5); // outstanding descs.
    unsigned int numd = min((unsigned int)(descs->size() - offset), maxd);
    
    /* warn("With a max size of %d, the max descs sent as %d, "
       "the number sent == %d, and size of desc == %d\n",
       maxsize, maxd, numd, descs[0].size()); */
    if (descs->size() - offset <= maxd) {
	res.resok->end = true;
    }
    else {
	res.resok->end = false;
    }
    res.resok->offset = offset;
    res.resok->descriptors.setsize(numd);
    
    for (unsigned int i = offset; i < offset + numd ; i++) {
	res.resok->descriptors[i-offset] = (*descs)[i];
	//warn("Set descriptor at position %d, offset %d to %s\n", i, 
	//     offset, 
	//     res.resok->descriptors[i-offset].desc.cstr());
    }
    res.resok->count = numd;
    sbp->replyref(res);
}

void
xferPlugin_gtc::remote_get_descriptors(svccb *sbp)
{
    xfergtc_get_descriptors_arg *arg = sbp->Xtmpl getarg<xfergtc_get_descriptors_arg>();
    ref<dot_oid_md> oid = New refcounted<dot_oid_md> (arg->oid);
    unsigned int offset = arg->offset;

    if (oid->id.size() < 1) {
	xfergtc_get_descriptors_res res(false);
        *res.errmsg = "Received invalid OID for GET_DESCRIPTORS";
        warn << *res.errmsg << "\n";
        sbp->replyref(res);
        return;
    }

    dwarn(DEBUG_XFER_GTC) << "GET_DESCRIPTORS w/ OID " << oid->id << " offset "
			  << offset << "\n";
    
    sp->get_descriptors(oid, wrap(this, &xferPlugin_gtc::remote_get_descriptors_cb,
                                  sbp, offset));
}

void
xferPlugin_gtc::remote_get_chunk(svccb *sbp)
{
    xfergtc_get_chunk_arg *arg = sbp->Xtmpl getarg<xfergtc_get_chunk_arg>();
    ref<dot_descriptor> d = New refcounted<dot_descriptor> (arg->desc);

    stat_entry *s = statCache[d->id];

    if (!s) {
	s = New stat_entry(d->id);
    }
    s->count++;

    // warn << "Remote get chunk " << *arg <<"\n";
    sp->get_chunk(d, wrap(this, &xferPlugin_gtc::remote_get_chunk_cb, sbp));
}

void
xferPlugin_gtc::remote_get_chunk_cb(svccb *sbp, str errmsg, ptr<desc_result> dres)
{
    xfergtc_get_chunk_arg *arg = sbp->Xtmpl getarg<xfergtc_get_chunk_arg>();
    xfergtc_get_chunk_res res(false);

    if (errmsg) {
        warn << "get_chunk from sp failed " << inet_ntoa(((sockaddr_in *)sbp->getsa())->sin_addr) << "  " << errmsg << "\n";
	*res.errmsg = errmsg;
	sbp->replyref(res);
	return;
    }
    ptr<suio> data = dres->data;

    if (arg->offset >= data->resid()) {
        warn << "Invalid offset\n";
        *res.errmsg = "Invalid offset for get_chunk";
        sbp->replyref(res);
        return;
    }

    res.set_ok(true);

    size_t num_bytes = min((dres->data->resid() - (size_t) arg->offset), 
                           SEND_SIZE);
    if (num_bytes < dres->data->resid() - arg->offset) {
        res.resok->end = false;
    }
    else {
        res.resok->end = true;
    }

    if (arg->offset > 0) {
        data->rembytes(arg->offset);
    }
    res.resok->offset = arg->offset;
    res.resok->data.setsize(num_bytes);
    // warn << "Sending back " << dres->data->resid() << " bytes\n";
    data->copyout(res.resok->data.base(), num_bytes);
    sbp->replyref(res);
}

/* Internal stuff */

void
xferPlugin_gtc::get_descriptors(ref<dot_oid_md> oid, ref<vec<oid_hint> > hints,
			        descriptors_cb cb, ptr<closure_t>)
{
    strbuf key;
    hint_res result;
    
    if (parse_hint((*hints)[0].name, "gtc", &result) < 0)
	fatal << "No hints to get descriptors from\n";
    
    key << result.hint.hostname << ":" << result.hint.port;

    conn_entry *conn = connCache[key];
    if (!connCache[key]) {
	tcpconnect(result.hint.hostname, result.hint.port, 
		   wrap(this, &xferPlugin_gtc::get_descriptors_clnt, oid, cb, key));
    }
    else {
        conn->refcount++;
        gettimeofday(&conn->tv, NULL);
	get_descriptors_int(oid, 0, cb, conn);
    }
}

void
xferPlugin_gtc::get_descriptors_clnt(ref<dot_oid_md> oid, descriptors_cb cb, str key,
			             int fd)
{
    if (fd == -1) {
	(*cb)("could not connect to remote host", NULL, true);
	return;
    }
    tcp_nodelay(fd);
    ref<axprt> x(axprt_stream::alloc(fd, MAX_PKTSIZE));
    ref<aclnt> c(aclnt::alloc(x, xfergtc_program_1));
    conn_entry *conn = New conn_entry(key, c, fd);
    connCache.insert(conn);
    gettimeofday(&conn->tv, NULL);
    get_descriptors_int(oid, 0, cb, conn);
}

void xferPlugin_gtc::get_descriptors_int(ref<dot_oid_md> oid, int offset, 
                                         descriptors_cb cb, conn_entry *conn)
{
    xfergtc_get_descriptors_arg darg;
    ref<xfergtc_get_descriptors_res> dres = 
        New refcounted<xfergtc_get_descriptors_res>;
    darg.oid = *oid;
    darg.offset = offset;
    conn->c->call(XFERGTC_PROC_GET_DESCRIPTORS, &darg, dres, 
                  wrap(this, &xferPlugin_gtc::get_desc_internal_cb, cb, oid, conn, 
                       dres));
    
}

void
xferPlugin_gtc::get_desc_internal_cb(descriptors_cb cb, ref<dot_oid_md> oid,
                                     conn_entry *conn,
                                     ref<xfergtc_get_descriptors_res> res, 
                                     clnt_stat err)
{
    if (err) {
        strbuf sb;
        sb << __PRETTY_FUNCTION__ << " XFERGTC_PROC_GET_DESCRIPTORS RPC failure: " 
           << err << "\n";
	connCache.remove(conn);
        delete conn;
        (*cb)(sb, NULL, true);
        return;
    }
    if (!res->ok) {
        strbuf sb;
        sb << __PRETTY_FUNCTION__ <<" XFERGTC_PROC_GET_DESCRIPTORS returned:\n`" 
           << *res->errmsg << "'\n";
        (*cb)(sb, NULL, true);
        return;
    }
    

    ptr< vec<dot_descriptor> > descptr = New refcounted<vec<dot_descriptor> >;
    descptr->setsize(res->resok->count);

    for (unsigned int i = 0; i < res->resok->count ;i++) {
        (*descptr)[i] = res->resok->descriptors[i];
	// warn << res->resok->descriptors[i].desc << " " << i << "\n";
    }

    if (!res->resok->end) {
	get_descriptors_int(oid, res->resok->count + res->resok->offset, cb,
                            conn);
    }
    else {
	struct timeval tv;
	gettimeofday(&tv, NULL);
	printf("Time to get descs - %.2f\n", timeval_diff(&conn->tv, &tv));

        conn->refcount--;
    }
    (*cb)(NULL, descptr, res->resok->end);
}

void
xferPlugin_gtc::get_chunk_int(ref<dot_descriptor> d, chunk_cb cb, size_t offset,
                       ref<suio> data, str s, conn_entry *conn)
{
    if (s) {
	(*cb) (s, NULL);
    }

    xfergtc_get_chunk_arg arg;
    ref<xfergtc_get_chunk_res> res = New refcounted<xfergtc_get_chunk_res>;
    arg.desc = *d;
    arg.offset = offset;
    //    warn << "calling transfer for " << d->desc << "\n";
    conn->c->call(XFERGTC_PROC_GET_CHUNK, &arg, res, 
                  wrap(this, &xferPlugin_gtc::get_chunk_internal_cb, cb, d, data, res,
                       conn));
}

void
xferPlugin_gtc::establish_connection(str key, int fd)
{
    pending_conn_entry *pce = pendingConnCache[key];
    if (!pce) {
	fatal << "Unable to find entry in pending Conn Cache";
    }

    if (fd == -1) {
	for (unsigned int i = 0; i < pce->pending_cbs.size(); i++) {
	    (*pce->pending_cbs[i])("could not connect to remote host", NULL);
	}
	pendingConnCache.remove(pce);
	delete pce;
	return;
    }
    tcp_nodelay(fd);
    ref<axprt> x(axprt_stream::alloc(fd, MAX_PKTSIZE));
    ref<aclnt> c(aclnt::alloc(x, xfergtc_program_1));
    conn_entry *conn = New conn_entry(key, c, fd);
    connCache.insert(conn);
    gettimeofday(&conn->tv, NULL);

    for (unsigned int i = 0; i < pce->pending_cbs.size(); i++) {
	(*pce->pending_cbs[i])(NULL, conn);
    }
    pendingConnCache.remove(pce);
    delete pce;
}

void
xferPlugin_gtc::get_connection(str host, int port, conn_cb cb)
{
    strbuf key;

    key << host << ":" << port;
    conn_entry *conn = connCache[key];
    if (conn) {
	(*cb)(NULL, conn);
	return;
    }
    pending_conn_entry *pce = pendingConnCache[key];
    if (pce) {
	pce->pending_cbs.push_back(cb);
	return;
    }
    pce = New pending_conn_entry(key);
    pendingConnCache.insert(pce);
    pendingConnCache[key]->pending_cbs.push_back(cb);

    tcpconnect(host, port, wrap(this, &xferPlugin_gtc::establish_connection, key));
}

void
xferPlugin_gtc::get_chunk(ref<dot_descriptor> d, ref<vec<oid_hint> > hints,
			  chunk_cb cb, ptr<closure_t>)
{
    strbuf key;
    hint_res result;
    
    if (parse_hint((*hints)[0].name, "gtc", &result) < 0)
	fatal << "No hints to get chunk from\n";
    
    key << result.hint.hostname << ":" << result.hint.port;
    
    // warn << "xferPlugin_gtc::get_chunk for " << d->desc << "\n" ;
    conn_entry *conn = connCache[key];
    if (!conn) {
	ref<suio> data = New refcounted<suio>;
	get_connection(result.hint.hostname, result.hint.port, wrap(this, &xferPlugin_gtc::get_chunk_int, d, cb,
					0, data));
    }
    else {
        if (conn->refcount >= MAX_RPCS_IN_FLIGHT) {
	    conn->wait_cb.insert_tail(New wait_conn_ent(wrap(this, &xferPlugin_gtc::get_chunk, d, hints, cb)));
        }
        else {
            conn->refcount++;
            gettimeofday(&conn->tv, NULL);
            ref<suio> data = New refcounted<suio>;
            get_chunk_int(d, cb, 0, data, NULL, conn);
        }
    }
}


void
xferPlugin_gtc::get_chunks(ref< vec<dot_descriptor> > dv, ref<hv_vec > hints,
			   chunk_cb cb, ptr<closure_t>)
{
  
  //warn << "get_chunks received\n";

    // When we want to send chunks in the reverse order from the
    // request, we actually need 'int i' and NOT 'unsigned int i'
  //for (int i = dv->size()-1; i >= 0; i--) {
  for (unsigned int i = 0; i < dv->size(); i++) {
    ref<dot_descriptor> d = New refcounted<dot_descriptor> ;
    d->id = ((*dv)[i]).id;
    d->length = ((*dv)[i]).length;
    get_chunk(d, (*hints)[i], cb);
  }
}

void
xferPlugin_gtc::get_chunk_clnt(ref<dot_descriptor> d, chunk_cb cb, str key, 
			int fd)
{
    if (fd == -1) {
	(*cb)("could not connect to remote host", NULL);
	return;
    }
    tcp_nodelay(fd);
    ref<axprt> x(axprt_stream::alloc(fd, MAX_PKTSIZE));
    ref<aclnt> c(aclnt::alloc(x, xfergtc_program_1));
    conn_entry *conn = New conn_entry(key, c, fd);
    connCache.insert(conn);
    gettimeofday(&conn->tv, NULL);

    ref<suio> data = New refcounted<suio>;
    get_chunk_int(d, cb, 0, data, NULL, conn);
}

void
xferPlugin_gtc::check_outstanding(conn_entry *conn)
{
    wait_conn_ent *n = conn->wait_cb.first;
    if (n) {
	wait_conn_cb w = n->cb;
	conn->wait_cb.remove(n);
	(*w)(NULL);
    }
}

void
xferPlugin_gtc::get_chunk_internal_cb(chunk_cb cb, ref<dot_descriptor> d, 
                               ref<suio> data, ref<xfergtc_get_chunk_res> res,
			       conn_entry *conn, clnt_stat err)
{
    if (err) {
        strbuf sb;
        sb << __PRETTY_FUNCTION__ <<" XFERGTC_PROC_GET_CHUNK RPC failure: " 
           << err << "\n";
	connCache.remove(conn);
        delete conn;
        (*cb)(sb, NULL);
        return;
    }
    if (!res->ok) {
        strbuf sb;
        sb << __PRETTY_FUNCTION__ << " XFERGTC_PROC_GET_CHUNK returned:\n`" 
           << *res->errmsg << "'\n";
        (*cb)(sb, NULL);
        return;
    }
    // warn << "Got reply for " << d->id << "\n";

    data->copy(res->resok->data.base(), res->resok->data.size());
    if (res->resok->end) {
#if 1
	unsigned char digest[EVP_MAX_MD_SIZE];
	EVP_MD_CTX desc_hash;
	unsigned int diglen;

	EVP_MD_CTX_init(&desc_hash);
	EVP_DigestInit(&desc_hash, EVP_sha1());

	EVP_DigestUpdate(&desc_hash, res->resok->data.base(), data->resid());
	EVP_DigestFinal(&desc_hash, digest, &diglen);

	if (memcmp((char *)digest, d->id.base(), diglen)) {
	    strbuf sb;
	    sb << __PRETTY_FUNCTION__ << " XFERGTC_PROC_GET_CHUNK returned invalid data\n";
	    (*cb)(sb, NULL);
	    return;
	}
#endif
        ref<desc_result> dres = New refcounted<desc_result> (d, data, false);
        conn->refcount--;
        (*cb)(NULL, dres);
        check_outstanding(conn);
    }
    else {
        get_chunk_int(d, cb, data->resid(), data, NULL, conn);
    }
}

void
xferPlugin_gtc::cancel_chunk(ref<dot_descriptor> d)
{

}
void
xferPlugin_gtc::cancel_chunks(ref< vec<dot_descriptor> > dv )
{

}

void 
xferPlugin_gtc::notify_descriptors(ref<dot_oid_md> oid, ptr<vec<dot_descriptor> > descs)
{
  
}

void 
xferPlugin_gtc::update_hints(ref< vec<dot_descriptor> > dv, ref<hv_vec > hints)
{
}

xferPlugin_gtc::~xferPlugin_gtc()
{
    warn << "xferPlugin_gtc destructor\n";
}

void
xferPlugin_gtc::dispatch(xferGtcConn *helper, svccb *sbp)
{
    if (!sbp) {
        warnx("xferPlugin_gtc: dispatch(): client closed connection\n");
        subconnlist.remove(helper);
        return;
    }

    switch(sbp->proc()) {
    case XFERGTC_PROC_GET_CHUNK:
        remote_get_chunk(sbp);
        break;
    case XFERGTC_PROC_GET_DESCRIPTORS:
	remote_get_descriptors(sbp);
	break;
    case XFERGTC_PROC_GET_BITMAP:
	remote_get_bitmap(sbp);
	break;
    default:
        sbp->reject(PROC_UNAVAIL);
        break;
    }
}

xferGtcConn::xferGtcConn(int fd, const sockaddr_in &sin, xferPlugin_gtc *parent)
    : x(axprt_stream::alloc(fd, MAX_PKTSIZE)),
      c(asrv::alloc(x, xfergtc_program_1, wrap(parent, &xferPlugin_gtc::dispatch, 
                                               this)))
{
    ipaddr = sin.sin_addr;
    tcpport = ntohs (sin.sin_port);
    
    warnx("xferGtc: Accepted connection from %s:%d\n", inet_ntoa(ipaddr), tcpport);
}

void
xferPlugin_gtc::get_bitmap(ref<dot_oid_md> oid, ref<vec<oid_hint> > hints,
			   bitmap_cb cb, ptr<closure_t>)
{
    strbuf key;
    hint_res result;
    
    if (parse_hint((*hints)[0].name, "gtc", &result) < 0)
	fatal << "No hints to get bitmap from\n";
    
    key << result.hint.hostname << ":" << result.hint.port;

    conn_entry *conn = connCache[key];
    if (!connCache[key]) {
	ref<bitvec> bmp = New refcounted<bitvec>;
	tcpconnect(result.hint.hostname, result.hint.port, 
		   wrap(this, &xferPlugin_gtc::get_bitmap_clnt, oid, cb, key, bmp));
    }
    else {
	if (conn->refcount >= MAX_RPCS_IN_FLIGHT) {
	    conn->wait_cb.insert_tail(New wait_conn_ent(wrap(this, &xferPlugin_gtc::get_bitmap, oid, hints, cb)));
	    //fatal << "xferPlugin_gtc::get_bitmap: Why so many RPCs?\n";
        }
	else {
	    conn->refcount++;
	    gettimeofday(&conn->tv, NULL);
	    ref<bitvec> bmp = New refcounted<bitvec>;
	    get_bitmap_int(oid, 0, cb, conn, bmp);
	}
    }
}

void
xferPlugin_gtc::get_bitmap_clnt(ref<dot_oid_md> oid, bitmap_cb cb, str key,
				ref<bitvec> bmp, int fd)
{
    if (fd == -1) {
	(*cb)("could not connect to remote host", NULL);
	return;
    }
    tcp_nodelay(fd);
    ref<axprt> x(axprt_stream::alloc(fd, MAX_PKTSIZE));
    ref<aclnt> c(aclnt::alloc(x, xfergtc_program_1));
    conn_entry *conn = New conn_entry(key, c, fd);
    connCache.insert(conn);
    gettimeofday(&conn->tv, NULL);
    get_bitmap_int(oid, 0, cb, conn, bmp);
}

void xferPlugin_gtc::get_bitmap_int(ref<dot_oid_md> oid, int offset, 
				    bitmap_cb cb, conn_entry *conn,
				    ref<bitvec> bmp)
{
    xfergtc_get_bitmap_arg darg;
    ref<xfergtc_get_bitmap_res> dres = 
        New refcounted<xfergtc_get_bitmap_res>;
    darg.oid = *oid;
    darg.offset = offset;

    conn->c->call(XFERGTC_PROC_GET_BITMAP, &darg, dres, 
                  wrap(this, &xferPlugin_gtc::get_bitmap_internal_cb, cb, oid, conn, 
                       dres, bmp));
    
}

void
xferPlugin_gtc::get_bitmap_internal_cb(bitmap_cb cb, ref<dot_oid_md> oid,
				       conn_entry *conn,
				       ref<xfergtc_get_bitmap_res> res, 
				       ref<bitvec> bmp, clnt_stat err)
{
    if (err) {
        strbuf sb;
        sb << __PRETTY_FUNCTION__ << " XFERGTC_PROC_GET_BITMAP RPC failure: " 
           << err << "\n";
	connCache.remove(conn);
        delete conn;
        (*cb)(sb, NULL);
        return;
    }

    if (!res->ok) {
        strbuf sb;
        sb << __PRETTY_FUNCTION__ <<" XFERGTC_PROC_GET_BITMAP returned:\n`" 
           << *res->errmsg << "'\n";
        (*cb)(sb, NULL);
        return;
    }

    ref<bitvec> bmp_tmp = New refcounted<bitvec>(res->resok->count);

    // warn << "---------------BEFORE\n";
    
    //     for (unsigned int i = 0; i < res->resok->bmp.size(); i++) {
    // 	warn << hexdump(&(((res->resok->bmp))[i]), 1) << " " ;
    //     }
    
    //     warn << "\n";
    //     warn << "---------------BEFORE\n";
    
    ref<bmp_data> bmpref = New refcounted<bmp_data>(res->resok->bmp);
    convert_to_bitvec(bmpref, res->resok->count, bmp_tmp);

    /* XXX - should validate returned offset. */
    bmp->zsetsize(res->resok->count + res->resok->offset);
    for (unsigned int i = 0; i < res->resok->count; i++) {
	(*bmp)[(i + res->resok->offset)] = (int)((*bmp_tmp)[i]);
    }

    if (!res->resok->end) {
	get_bitmap_int(oid, res->resok->count + res->resok->offset, cb,
		       conn, bmp);
    }
    else {
	struct timeval tv;
	gettimeofday(&tv, NULL);
	printf("Time to get bitmap - %.2f\n", timeval_diff(&conn->tv, &tv));

        conn->refcount--;

	(*cb)(NULL, bmp);
	check_outstanding(conn);
    }
}

void
xferPlugin_gtc::remote_get_bitmap(svccb *sbp)
{
    xfergtc_get_bitmap_arg *arg = sbp->Xtmpl getarg<xfergtc_get_bitmap_arg>();
    ref<dot_oid_md> oid = New refcounted<dot_oid_md> (arg->oid);
    unsigned int offset = arg->offset;

    if (oid->id.size() < 1) {
	xfergtc_get_bitmap_res res(false);
        *res.errmsg = "Received invalid OID for GET_BITMAP";
        warn << *res.errmsg << "\n";
        sbp->replyref(res);
        return;
    }

    dwarn(DEBUG_XFER_GTC|DEBUG_SET) << "GET_BITMAP w/ OID " << oid->id
				    << " offset " << offset << "\n";

    sp->get_bitmap(oid, wrap(this, &xferPlugin_gtc::remote_get_bitmap_cb,
			     sbp, offset));
}

void
xferPlugin_gtc::remote_get_bitmap_cb(svccb *sbp, unsigned int offset, str s, 
				     ptr<bitvec > bmp)
{
    //warn << "xferPlugin_gtc::remote_get_bitmap_cb: Got back bitmap " << bmp->size() << "\n";
    //print_bitvec(bmp);
    
    xfergtc_get_bitmap_res res(false);
    /* XXX - we're ignoring end right now b/c storage always gives us
       everything */
    if (s) {
        res.set_ok(false);
        *res.errmsg = s;
        sbp->replyref(res);
        return;
    }

    ref< bmp_data > bmp_ret = New refcounted< bmp_data >;
    
    convert_from_bitvec(bmp_ret, (unsigned int) bmp->size(), bmp);

    if (offset > 0 && offset >= bmp_ret->size()) {
        *res.errmsg = "Too large offset for GET_BITMAP";
        warn << *res.errmsg << "\n";
        sbp->replyref(res);
        return;
    }

    res.set_ok(true);

    /*
     * We're limited in the number of bits we can send back by
     * the RPC max size.  That's a lot, but perhaps not enough
     * for a huge file.  Support an offset that MUST BE A MULTIPLE
     * OF 8.
     */
    unsigned int maxbits = (SEND_SIZE / 2) * 8; /* Conservative */

    //warn << "Maxbits " << maxbits <<"\n";

    unsigned int totalbits = bmp->size() - offset;
    unsigned int sendbits = totalbits;

    if (totalbits > maxbits) {
	sendbits = maxbits;
	res.resok->end = false;
    }
    else {
	res.resok->end = true;
    }
    unsigned int byte_offset = offset / 8;
    unsigned int sendbytes = sendbits / 8;
    if (sendbits % 8)
	sendbytes++;

    res.resok->offset = offset;
    res.resok->bmp.set(bmp_ret->base() + byte_offset, sendbytes);

    // warn << "---------------AFTER\n";
    
    //     for (unsigned int i = 0; i < res.resok->bmp.size(); i++) {
    // 	warn << hexdump(&(((res.resok->bmp))[i]), 1) << " " ;
    //     }
    
    //     warn << "\n";
    
    res.resok->count = sendbits;
    res.resok->num_descs = bmp->size();
    
    sbp->replyref(res);
}

bool
convert_to_bitvec(ref<bmp_data> bmp, int desc_count, ptr<bitvec> bmp_ret)
{
    
    unsigned int dc = static_cast<unsigned int>(desc_count);

    //clear the bitvector
    bmp_ret->zsetsize(desc_count);
    bmp_ret->setrange(0, bmp_ret->size(), 0);

    for (unsigned int i = 0; i < dc; i++) {
	/* unpack and repack the bit string.  This is a little silly,
	 * but it's safe. */
	(*bmp_ret)[i] = (bmp->base()[i / 8] & (1 << ((i%8) & 0x07)));
    }

    return true;
}

bool
convert_from_bitvec(ref<bmp_data> bmp_ret, unsigned int desc_count, ptr<bitvec> bmp)
{
    char *bytes;
    int nbytes = desc_count / 8;
    if (desc_count % 8) nbytes++;

    bmp_ret->setsize(nbytes);
    bytes = bmp_ret->base();
    bzero(bytes, nbytes);
    
    for (unsigned int i = 0; i < desc_count; i++) {
	if ((*bmp)[i])
	    bytes[i / 8] |= (1 << ((i % 8) & 0x07));
    }
    
    return true;
}
